<?php
/****************************************************************
 * IBM Confidential
 *
 * Collaboration Source Materials
 *
 * (C) Copyright IBM Corp. 2015
 *
 * The source code for this program is not published or otherwise
 * divested of its trade secrets, irrespective of what has been
 * deposited with the U.S. Copyright Office
 *
 ***************************************************************/

require_once('include/EntryPoint.php');

/**
 * Performs cleanup actions against a UCD server
 * 
 * @class   CleanUCD
 * @author  Marco Bassi         marcobas@ie.ibm.com
 */
class CleanUCD extends EntryPoint {
    protected $type;
    protected $pos;
    protected $clean;
    
    public function __construct(){
        parent::__construct();
        $this->setReturn('json');
        $this->type = array(
                        'default',
                        'application',
                        'component',
                        'applicationProcess',
                        'componentProcess',
                        'genericProcess',
                        'postProcessScript'
        );
        
        $this->pos = array('start','end');
        $this->loadConfig();
    }
    
    /**
     * 
     * Load clean.config.php configuration
     *
     */
    protected function loadConfig(){
        include 'config/clean.config.php';
        $this->clean = $config['clean'];
    }
    
    /**
     * 
     * Check if a name matches the rules defined in clean.config.php
     *
     * @param string $name
     * @param string $type
     * @return boolean
     * 
     */
    public function matchRule($name, $type) {
        // Type not supported or empty name
        $trimmedName = str_replace(' ','',$name);
        if (!in_array($type, $this->type) || 
            empty($trimmedName)
        ) {
            return false;
        }
        $exclude['rule'] = '';
        $exclude['pos'] = $this->clean['exclude']['position'];
        $include['rule'] = '';
        $include['pos'] = $this->clean['include']['position'];
        
        // Check exclude - precedence over include
        $found = false;
        if ( !empty($this->clean['exclude']['rule']['default']) ) {
            $exclude['rule'] = $this->clean['exclude']['rule']['default'];
        }
        if ( !empty($this->clean['exclude']['rule'][$type]) ) {
            $exclude['rule'] = $this->clean['exclude']['rule'][$type];
        }
        
        if (!empty($exclude['rule'])) {
            $found = $this->matchPosition($name, $exclude['rule'], $exclude['pos']);
            if ($found) {
                // Exclude rule is matched: DO NOT CLEAN
                return false;
            }
        }
        
        // Check include
        $found = false;
        if ( !empty($this->clean['include']['rule']['default']) ) {
            $include['rule'] = $this->clean['include']['rule']['default'];
        }
        if ( !empty($this->clean['include']['rule'][$type]) ) {
            $include['rule'] = $this->clean['include']['rule'][$type];
        }
        
        if (!empty($include['rule'])) {
            $found = $this->matchPosition($name, $include['rule'], $include['pos']);
            if ($found) {
                // Include rule is matched: CLEAN
                return true;
            }
        }
        
        // Shouldn't go here, but in case, DO NOT CLEAN
        return false;
    }
    
    /**
     * 
     * Match position for rule
     *
     * @param string $name
     * @param string $rule
     * @param string $position
     * @return boolean
     */
    private function matchPosition($name, $rule, $position) {
        if ( !empty($position) && !in_array($position, $this->pos) ) {
            return false;
        }
        
        switch ($rule) {
            case 'start' :
                if (strpos($name,$rule) === 0 )
                    return true;
                break;
            case 'end' :
                $nl = strlen($name);
                $rl = strlen($rule);
                if (strpos($name,$rule) === ($nl-$rl) )
                    return true;
                break;
            default :
                if ( strpos($name,$rule) !== false )
                    return true;
                break;
        }
        
        return false;
    }
    
    /**
     * 
     * Checki clean rules and return if a particular cleanup is needed, based on empty or not rules
     *
     * @param string $type
     * @return boolean
     */
    private function doClean( $type ) {
        if ( ! empty( $this->clean['include']['rule']['default'] ) ) {
            return true;
        } else {
            if ( ! empty( $this->clean['include']['rule'][$type] ) ) {
                return true;
            } else {
                return false;
            }
        }
        
        return false;
    }

/*******************
 * BATCH FUNCTIONS
*******************/
    
    /**
     * Clean ALL
     */
    public function cleanAll() {
        $errors = false;
        
        // APPLICATIONS
        if ( $this->doClean( 'application' ) ) {
            Utils::CLIinfo( 'Cleaning applications...' );
            $result = $this->cleanApplications();
            if ($result === false) {
                Utils::CLIerror( 
                    "Failed to clean applications." );
                $errors = true;
            }
        } else {
            Utils::CLIout( "[SKIP] No rule defined for 'application'." );
        }
        
        // APPLICATION PROCESSES
        if ( $this->doClean( 'applicationProcess' ) ) {
            Utils::CLIinfo( 'Cleaning applications processes...' );
            $responseAPI = $this->udclient()->application()->getApplications();
            if ($responseAPI === false) {
                Utils::CLIerror( 
                    "Failed to retrieve the list of applications" );
                $errors = true;
            } else {
                $applications = Utils::outputToArray( $responseAPI );
                foreach ( $applications as $application ) {
                    $result = $this->cleanApplicationProcesses( $application ['name'] );
                    if ($result === false) {
                        Utils::CLIerror( 
                            "Failed to clean processes for application '{$application ['name']}'." );
                        $errors = true;
                    }
                }
            }
        } else {
            Utils::CLIout( "[SKIP] No rule defined for 'applicationProcess'." );
        }
        
        // COMPONENTS
        if ( $this->doClean( 'component' ) ) {
            Utils::CLIinfo( 'Cleaning components...' );
            $result = $this->cleanComponents();
            if ($result === false) {
                Utils::CLIerror( 
                    "Failed to clean components." );
                $errors = true;
            }
            
            // COMPONENT PROCESSES
            Utils::CLIinfo( 'Cleaning components processes...' );
            $responseAPI = $this->udclient()->component()->getComponents();
            if ($responseAPI === false) {
                Utils::CLIerror( 
                    "Failed to retrieve the list of getComponents" );
                $errors = true;
            } else {
                $components = Utils::outputToArray( $responseAPI );
                foreach ( $components as $component ) {
                    $result = $this->cleanComponentProcesses( $component ['name'] );
                    if ($result === false) {
                        Utils::CLIerror(
                        "Failed to clean processes for component '{$component ['name']}'." );
                        $errors = true;
                    }
                }
            }
        } else {
            Utils::CLIout( "[SKIP] No rule defined for 'component'." );
        }
        
        // GENERIC PROCESSES
        if ( $this->doClean( 'genericProcess' ) ) {
            Utils::CLIinfo( 'Cleaning generic processes...' );
            $result = $this->cleanGenericProcesses();
            if ($result === false) {
                Utils::CLIerror( 
                    "Failed to clean generic processes." );
                $errors = true;
            }
        } else {
            Utils::CLIout( "[SKIP] No rule defined for 'genericProcess'." );
        }
        
        // POST-PROCESSING SCRIPTS
        if ( $this->doClean( 'postProcessScript' ) ) {
            Utils::CLIinfo( 'Cleaning post-processing scripts...' );
            $result = $this->cleanPostProcessingScripts();
            if ($result === false) {
                Utils::CLIerror( 
                    "Failed to clean post-processing scripts." );
                $errors = true;
            }
        } else {
            Utils::CLIout( "[SKIP] No rule defined for 'postProcessScript'." );
        }
        
        // Result
        if ($errors) {
            Utils::CLIerror( 
                "Encountered some errors during cleaning process, please check messages and log files." );
            return false;
        } else {
            Utils::CLIout( 
                "[SUCCESS] Applications, components, processes and post-processing scripts successfully cleaned." );
            return true;
        }
    }
    
    /**
     * Clean all Applications
     * @return boolean
     */
    public function cleanApplications() {
        $responseAPI = $this->udclient()->application()->getApplications();
        if ($responseAPI === false) {
            Utils::CLIerror( 
                "Failed to retrieve the list of applications" );
            Utils::CLIdebug( __CLASS__ . '::' . __FUNCTION__ );
            return false;
        }
        $applications = Utils::outputToArray( $responseAPI );
        foreach ( $applications as $application) {
            $this->cleanApplication( $application );
        }
        return true;
    }
    
    /**
     * Clean all Components
     * @return boolean
     */
    public function cleanComponents() {
        $responseAPI = $this->udclient()->component()->getComponents();
        if ($responseAPI === false) {
            Utils::CLIerror( 
                "Failed to retrieve the list of components" );
            Utils::CLIdebug( __CLASS__ . '::' . __FUNCTION__ );
            return false;
        }
        $components = Utils::outputToArray( $responseAPI );
        foreach ( $components as $component) {
            $this->cleanComponent( $component );
        }
        return true;
    }
    
    /**
     * Clean all Application Processes
     * 
     * @param string $application
     * @return boolean
     * 
     */
    public function cleanApplicationProcesses($applicationName) {
        if (empty($applicationName)){
            Utils::CLIerror( "Application name is missing or empty" );
            Utils::CLIdebug( __CLASS__ . '::' . __FUNCTION__ );
            return false;
        }
        
        if($this->udclient()->application()->getApplication($applicationName) === false) {
            Utils::CLIerror( "Application '{$applicationName}' doens't exist" );
            Utils::CLIdebug( __CLASS__ . '::' . __FUNCTION__ );
            return false;
        }
        
        Utils::CLIout("  Cleaning processes for application {$applicationName}");
        $output = $this->rest()->application()->getApplicationProcesses($applicationName);
        $appProcesses = Utils::outputToArray($output);
        
        foreach($appProcesses as $process) {
            $this->cleanApplicationProcess($process);
        }
        return true;
    }
    
    /**
     * Clean all Component Processes
     *
     * @param string $component
     *
     */
    public function cleanComponentProcesses($componentName) {
        if (empty( $componentName )) {
            Utils::CLIerror( "Component name is missing or empty" );
            Utils::CLIdebug(__CLASS__ . '::' . __FUNCTION__);
            return false;
        }
        $responseAPI = $this->rest()->component()->getComponentRest( $componentName );
        if ($responseAPI === false) {
            Utils::CLIerror( 
                "Component '{$componentName}' doens't exist" );
            Utils::CLIdebug( __CLASS__ . '::' . __FUNCTION__ );
            return false;
        }
        Utils::CLIout("  Cleaning processes for component {$componentName}");
        $responseAPI = Utils::outputToArray( $responseAPI );
        $compProcesses = $responseAPI ['processes'];
        foreach ( $compProcesses as $process ) {
            $path = $process ['path'];
            // $path: "components/ID-component/processes/ID-process"
            $path = explode( "/", $path );
            $processID = $path [3];
            $process ['id'] = $processID;
            $this->cleanComponentProcess( $process );
        }
        return true;
    }
    /**
     * Clean all Generic Processes
     */
    public function cleanGenericProcesses() {
        $responseAPI = $this->rest()->process()->getProcesses();
        if ($responseAPI === false) {
            Utils::CLIerror( 
                "Failed to retrieve the list of processes" );
            Utils::CLIdebug( __CLASS__ . '::' . __FUNCTION__ );
            return false;
        }
        $processes = Utils::outputToArray( $responseAPI );
        foreach ( $processes as $process ) {
            
            $this->cleanGenericProcess( $process );
        }
        return true;
    }
    /**
     * Clean all Post Processing Scripts
     */
    public function cleanPostProcessingScripts() {
        $this->setReturn( 'json' );
        $responseAPI = $this->rest()->process()->getPostProcessingScripts();
        if ($responseAPI === false) {
            Utils::CLIerror(
            "Failed to retrieve the list of post-processing scripts" );
            Utils::CLIdebug( __CLASS__ . '::' . __FUNCTION__ );
            return false;
        }
        
        $scripts = Utils::outputToArray($responseAPI);
        foreach ($scripts as $script) {
            // By default, $script includes 'name' and 'id', no need to create a new variable
            $this->cleanPostProcessingScript($script);
        }
        
        return true;
    }
    
/*******************
 * SINGLE FUNCTIONS
 *******************/
    
    /** 
     * Clean one single Application
     * 
     * @param object $application
     * @return boolean
     */
    private function cleanApplication($application) {
        if ($this->matchRule( $application ['name'], 'application' )) {
            $this->setReturn( 'json' );
            $result = $this->rest()->application()->deleteApplication( $application ['id'] );
            if (is_array( $result )) {
                Utils::CLIout( 
                    "  [SUCCESS] Successfully cleaned application '{$application['name']}'" );
                return true;
            } else {
                Utils::CLIerror(
                    "Problem cleaning application '{$application['name']}'" );
                Utils::CLIdebug( 
                    __CLASS__ . '::' . __FUNCTION__ .
                    ":: Application '{$application['name']}' ({$application ['id']})" );
            return false;
        }
        } else {
            Utils::CLIdebug( 
                __CLASS__ . '::' . __FUNCTION__ .
                ":: Application '{$application['name']}' did not match the rules. No cleaning needed" );
            return null;
        }
    }
    
    /**
     * Clean one single Component
     * 
     * @param object $component
     * @return boolean
     */
    private function cleanComponent($component) {
    if ($this->matchRule( $component ['name'], 'component' )) {
            $this->setReturn( 'json' );
            $result = $this->rest()->component()->deleteComponent( $component ['id'] );
            if (is_array( $result )) {
                Utils::CLIout( 
                    "  [SUCCESS] Successfully cleaned component '{$component['name']}'" );
                return true;
            } else {
                Utils::CLIerror( 
                    "Problem cleaning component '{$component['name']}'" );
                Utils::CLIdebug(
                    __CLASS__ . '::' . __FUNCTION__ .
                    ":: Component '{$component['name']}' ({$component['id']})" );
            return false;
        }
        } else {
            Utils::CLIdebug( 
                __CLASS__ . '::' . __FUNCTION__ .
                ":: Component '{$component['name']}' did not match the rules. No cleaning needed" );
            return null;
        }
    }
    /**
     * Clean one single Application Process
     * 
     * @param object $process
     * @return boolean - true|false|null (there is nothing to do)
     * 
     */
    private function cleanApplicationProcess($process) {
        if($this->matchRule($process['name'], 'applicationProcess')) {
            $this->setReturn('json');
            $result = $this->rest()->application()->deleteApplicationProcess($process['id']);
            if(is_array($result)) {
                Utils::CLIout( 
                    "  [SUCCESS] Successfully cleaned application process '{$process['name']}'" );
                return true;
            } else {
                Utils::CLIerror( 
                    "Problem cleaning application process '{$process['name']}'" );
                Utils::CLIdebug(
                    __CLASS__ . '::' . __FUNCTION__ .
                    ":: Application process '{$process['name']}' ({$process['id']})" );
                return false;
            }
        } else {
            Utils::CLIdebug( 
                __CLASS__ . '::' . __FUNCTION__ .
                ":: Application process '{$process['name']}' did not match the rules. No cleaning needed" );
            return null;
        }
    }

    /**
     * Clean one single Component Process
     * 
     * @param object $process
     * @return boolean - true|false|null (there is nothing to do)
     */
    private function cleanComponentProcess($process) {
        if($this->matchRule($process['name'], 'componentProcess')) {
            $this->setReturn('json');
            $result = $this->rest()->component()->deleteComponentProcess($process['id']);
            if(is_array($result)) {
                Utils::CLIout( 
                    "  [SUCCESS] Successfully cleaned component process '{$process['name']}'" );
                return true;
            } else {
                Utils::CLIerror(
                    "Problem cleaning component process '{$process['name']}'" );
                Utils::CLIdebug( 
                    _CLASS__ . '::' . __FUNCTION__ .
                    ":: Component process '{$process['name']}' ({$process['id']})" );
                return false;
            }
        } else {
            Utils::CLIdebug( 
                __CLASS__ . '::' . __FUNCTION__ .
                ":: Component process '{$process['name']}' did not match the rules. No cleaning needed." );
            return null;
        }
    }

    /**
     * Clean one single Generic Process
     * 
     * @param object $process
     * @return boolean - true|false|null (there is nothing to do)
     */
    private function cleanGenericProcess($process) {
    if($this->matchRule($process['name'], 'genericProcess')) {
            $this->setReturn('json');
            $result = $this->rest()->process()->deleteGenericProcess($process['id']);
            if(is_array($result)) {
                Utils::CLIout(
                    "  [SUCCESS] Successfully cleaned processs '{$process['name']}'" );
                return true;
            } else {
                Utils::CLIerror( 
                    "Problem cleaning process '{$process['name']}'" );
                Utils::CLIdebug(
                    __CLASS__ . '::' . __FUNCTION__ .
                    ":: Process: '{$process['name']}' ({$process['id']})" );
                return false;
            }
        } else {
            Utils::CLIdebug( 
                __CLASS__ . '::' . __FUNCTION__ .
                ":: Process '{$process['name']}' did not match the rules. No cleaning needed." );
            return null;
        }
    }
    /**
     * Clean one single Post Processing Script
     * 
     * @param object $script
     * @return boolean
     * 
     */
    private function cleanPostProcessingScript($script) {
        if($this->matchRule($script['name'], 'postProcessScript')) {
            $this->setReturn('json');
            $result = $this->rest()->process()->deletePostProcessingScript($script['id']);
            if(is_array($result)) {
                Utils::CLIout(
                    "  [SUCCESS] Successfully cleaned post-processing script '{$script['name']}'" );
                return true;
            } else {
                Utils::CLIerror(
                    "Problem cleaning post-processing script '{$script['name']}'" );
                Utils::CLIdebug(
                    __CLASS__ . '::' . __FUNCTION__ .
                    ":: Script: '{$script['name']}' ({$script['id']})" );
                return false;
            }
        } else {
            Utils::CLIdebug(
                __CLASS__ . '::' . __FUNCTION__ .
                ":: Post-processing script '{$script['name']}' did not match the rules. No cleaning needed." );
            return null;
        }
    }
}