<?php
/****************************************************************
 * IBM Confidential
*
* SFA100-Collaboration Source Materials
*
* (C) Copyright IBM Corp. 2014
*
* The source code for this program is not published or otherwise
* divested of its trade secrets, irrespective of what has been
* deposited with the U.S. Copyright Office
*
***************************************************************/

require_once('include/Extensions/RestoreUCD.php');
require_once('include/Extensions/BackupUCD.php');
require_once('Component.php');


/**
 * Manages snapshot promotion between uDeploy servers
 *
 * @class   Snapshot
 * @author  Marco Bassi         marcobas@ie.ibm.com
*/
class Snapshot extends RestoreUCD {
    protected  $component; 
    public function __construct(){
        parent::__construct();
        $this->setReturn('json');
        $this->component = new Component();
    }
    
    /**
     *
     * Promotes a snapshot from a uDeploy server (origin) to another uDeploy server (destination).
     * Compares component versions in the snapshot between the two uDeploy servers and updates the destination server with the missing version.
     * Creates a copy of the snapshot into the destination server.
     *
     * @param string $snapshot      Snapshot name
     * @param array $config         array(
     *                                  'origin_alias'         => 'alias name for origin server',
     *                                  'destination_alias'       => 'alias name for destination server'
     *                              )
     * @param array $artifacts      $artifacts['component_name']['base'] = 'path of directory/file to upload to component version'
     *                              $artifacts['component_name']['include'] => array (
     *                                                                              'file to include',
     *                                                                              ...
     *                                                                         )
     *                               $artifacts['component_name']['exclude'] => array (
     *                                                                              'file to exclude',
     *                                                                              ...
     *                                                                              )
     * @return array|boolean
     */
    public function promoteSnapshot($snapshot, $config = null, $artifacts = null, $force = null) {
        if (empty($config)) {
            $config = 'config/promote.config.php';
            if (file_exists($config)) {
                include $config;
            } else {
                Utils::CLIerror("No server configuration set.");
            }
        }
        if (empty($artifacts)) {
            $artifacts = 'config/artifacts.config.php';
            if (file_exists($artifacts)) {
                include $artifacts;
            }
        }
        
        if (!self::validateServerConfig($config)) {
            Utils::CLIerror("Configuration error. Please check the messages above and provide a valid configuration.");
            return false;
        }
        
        $result = array();
        $snapshotFailures = 0;
        
        // Setup connection to first server (origin)
        if (!$this->setupServer('origin', $config, true)) {
            Utils::CLIerror("Cannot connect to origin server.");
            return false;
        }
        
        // Retrieve snapshot from origin uDeploy server
        Utils::CLIinfo("Retrieving requested snapshot '{$snapshot}' from '{$this->application}' application on origin server");
        $output = $this->rest()->snapshot()->getSnapshotsInApplication($this->application);
        $appSnapshots = Utils::outputToArray($output);
        $snapshotId = null;
        foreach ($appSnapshots as $appSnapshot) {
            if ($appSnapshot['name'] == $snapshot) {
                $snapshotId = $appSnapshot['id'];
                break;
            }
        }
        if (empty($snapshotId)) {
            Utils::CLIerror("Requested snapshot '{$snapshot}' in '{$this->application}' application NOT FOUND on origin server.");
            return false;
        }
        
        // Get component and component version list, as defined in the snapshot
        $output = $this->rest()->component()->getComponentVersionsInSnapshot($snapshotId);
        $originComponents = Utils::outputToArray($output);
        
        
        // Verify that snapshot doesn't already exist on destination server
        Utils::CLIinfo("Verifying that snapshot '{$snapshot}' doesn't exists on destination server");
        if (!$this->setupServer('destination', $config, true)) {
            Utils::CLIerror("Cannot connect to destination server.");
            return false;
        }
        
        $output = $this->rest()->snapshot()->getSnapshotsInApplication($this->application);
        $checkSnapshots = Utils::outputToArray($output);
        $checkSnapshotId = null;
        foreach ($checkSnapshots as $checkSnapshot) {
            if ($checkSnapshot['name'] == $snapshot) {
                $checkSnapshotId = $checkSnapshot['id'];
                break;
            }
        }
        if (!empty($checkSnapshotId)) {
            Utils::CLIerror("Requested snapshot '{$snapshot}' in '{$this->application}' application ALREADY EXISTS on destination server.");
            return false;
        }
        
        // Compare components between origin and destination, verify if every component exists in destination. Create if missing
        Utils::CLIinfo("Verifying existing components in destination server");
        $appDestComponents = $this->udclient()->component()->getComponentsInApplication($this->application);
        if ($appDestComponents) {
            $appDestComponents = Utils::outputToArray($appDestComponents);
        } else {
            Utils::CLIerror("Error retrieving components for application '{$this->application}' on destination server.");
            return false;
        }
        
        // All components in destination server
        $allDestComponents = $this->udclient()->component()->getComponents();
        if ($allDestComponents) {
            $allDestComponents = Utils::outputToArray($allDestComponents);
        } else {
            Utils::CLIerror("Error retrieving all components from destination server.");
            return false;
        }
        
        $toCreateComponents = array();
        $toAddComponents = array();
        // Loop components required for snapshot
        foreach ($originComponents as $component) {
            $componentExists = false;
            $componentIsAdded = false;
            // Check if current component is available in destination server
            foreach ($appDestComponents as $appDestComponent) {
                if ($component['name'] == $appDestComponent['name']) {
                    $componentExists = true;
                    // Check if current component is added to the target application
                    foreach($allDestComponents as $destComponent) {
                        if ($component['name'] == $destComponent['name']) {
                            $componentIsAdded = true;
                            break;
                        }
                    }
                    break;
                }
            }
            
            if (!$componentExists) {
                // Component doesn't exist: need to create it
                $toCreateComponents[$component['id']] = $component['name'];
            }
            if (!$componentIsAdded) {
                // Component is not added to target applicaiton: need to add it
                $toAddComponents[$component['id']] = $component['name'];
            } 
        } // END Loop components
        
        
        if (!empty($toCreateComponents)) {
            Utils::CLIwarning("The following components are missing and need to be created:");
            foreach ($toCreateComponents as $id => $name){
                Utils::CLIout("  $name");
            }
            
            return false;
        } else {
            Utils::CLIout("  All components are available on destination server");
        }
        
        if (!empty($toAddComponents)) {
            Utils::CLIwarning("The following components are missing from '{$this->application}' application:");
            foreach ($toAddComponents as $id => $name){
                Utils::CLIout("  $name");
            }
            return false;
        } else {
            Utils::CLIout("  All components are associated to '{$this->application}' application");
        }
        Utils::CLIout("");
        
        // Export artifacts 
        Utils::CLIinfo("Exporting snapshot artifacts");
        if (!$this->setupServer('origin', $config, true)) {
            Utils::CLIerror("Cannot connect to origin server.");
            return false;
        }
        Utils::CLIout("  (This may take a while)", true);
        $this->setReturn('file');
        
        $backupOutput = $this->output;
        $cwd = getcwd();
        $this->setOutput("{$cwd}/tmp", true);
        $result['snapshotExport'] = $this->rest()->snapshot()->exportSnapshot($snapshotId, $this->application, $snapshot);
        if ($result['snapshotExport'] === false) {
            Utils::CLIerror("Cannot download snapshot zip file from origin server.");
            return false;
        }
        $snapshotZip = "{$this->output}/{$this->application}.{$snapshot}.zip";
        $this->log->info("Snapshot zip: {$snapshotZip}");
        
        $snapshotDir = "{$this->output}/" . time();
        Utils::CLIinfo("Extracting snapshot artifacts to {$snapshotDir}");
        $unzip = "unzip -q -o {$snapshotZip} -d {$snapshotDir}";
        Utils::CLIinfo("unzip command: ".$unzip);
        Utils::CLIinfo("");
        exec($unzip, $output, $return);
        
        $this->log->info(print_r($output,true));
        
        if ($return != 0) {
            Utils::CLIerror("Cannot unzip snapshot artifacts.");
            return false;
        }
        
        $snapshotSubDirs = scandir($snapshotDir);
        
        $this->setOutput($backupOutput, true);
        
        $this->setReturn('json');
        
        // Preparing new Snapshot data
        Utils::CLIinfo("Preparing snapshot for destination server");
        if (!$this->setupServer('destination',$config, true)) {
            Utils::CLIerror("Cannot connect to destination server.");
            return false;
        }
        
        $newSnapshot['snapshot']['name'] = $snapshot;
        $newSnapshot['application'] = $this->application;
        $newSnapshot['snapshot']['description'] = "{$this->application} snapshot for build '{$snapshot}'";
        $newSnapshot['components'] = array();
        
        Utils::CLIout("  Name: {$newSnapshot['snapshot']['name']}");
        Utils::CLIout("  Application: {$newSnapshot['application']}");
        Utils::CLIout("  Description: {$newSnapshot['snapshot']['description']}");
        $this->log->info("Components: ");
        
        // Compare component versions set in snapshot with destination's component versions. Create if missing
        foreach ($originComponents as $component) {
            $name = $component['name'];
            foreach ($component['desiredVersions'] as $ver) {
                $version = $ver['name'];
                $versions = $version;
                if (array_key_exists($name, $newSnapshot['components'])) {
                    if (is_array($newSnapshot['components'][$name])) {
                        $versions = $newSnapshot['components'][$name];
                        array_push($versions, $version);
                    } else {
                        $versions = array($newSnapshot['components'][$name], $version);
                    }
                }
                $newSnapshot['components'][$name] = $versions;
                $this->log->log("  {$name} ({$version})");
                
                // Verify if any artifact and add it
                foreach ($snapshotSubDirs as $dir) {
                    if (strpos($dir,"{$version}-{$ver['id']}") !== false) {
                        $base =  "{$snapshotDir}/{$dir}";
                        $newSnapshot['artifacts'][$name][$version] = array('base' => $base);
                        $this->log->log("    Artifacts: {$base}");
                        break;
                    }
                }
            }
        }
        Utils::CLIout("\n");
        
        if (!$this->setupServer('destination', $config, true)) {
            Utils::CLIerror("Cannot connect to destination server.");
            return false;
        }
        $result['versionsAndSnapshot'] = self::createVersionsAndSnapshot($newSnapshot, $force);
        return $result;
    }
    
    /**
     * 
     * Validate server configuration used by promoteSnapshot() 
     *
     * @param array $config
     * @return boolean
     */
    public function validateServerConfig($config) {
        if (empty($config)) {
            Utils::CLIerror("Configuration is empty. Please provide a valid configuration");
            return false;
        }
        $return = true;
        if (empty($config['origin_alias'])) {
            Utils::CLIerror("No origin_alias set for origin server");
            $return = false;
        }
        if (empty($config['destination_alias'])) {
            Utils::CLIwarning("No destination_alias for destination server");
            $return = false;
        }
        return $return;
    }
    
    /**
     *
     * Creates component versions and add them into snapshot. 
     * Accepts an array or file for configuration 
     * 
     * Config (php or json version) should look like
     * 
     * $config['application'] = "Application Name"
     * $config['snapshot'] = array(
     *                          "name" => "Snapshot name",
     *                          "description" => "Snapshot description",
     *                      )
     * $config['components'] = array(
     *                          "Component Name" => "Component Version",
     *                          ...
     *                      )
     *
     * @param array|string $config
     */
    public function createVersionsAndSnapshot($config, $force = true) {
        if (!is_array($config) && file_exists($config)) {
            // Argument is a file
            $ext = pathinfo($config, PATHINFO_EXTENSION);
            if ($ext == 'php') {
                include ($config);
            } else if ($ext == 'json') {
                $config = json_decode(file_get_contents($config),true);
            } else {
                Utils::CLIerror("Invalid file. Please pass a php or json file.");
                return false;
            }
        } else if (!is_array($config)) {
            Utils::CLIerror("Invalid argument. Please use a php or json file or a php array.");
            return false;
        }
        
        // Retrieve component list for application
        $appDestComponents = $this->udclient()->component()->getComponentsInApplication($config['application']);
        $appDestComponentList = array();
        if ($appDestComponents) {
            $appDestComponents = Utils::outputToArray($appDestComponents);
            foreach ($appDestComponents as $comp) {
                array_push($appDestComponentList, $comp['name']);
            }
        } else {
            Utils::CLIerror("Error retrieving components for application '{$config['application']}' on destination server.");
            return false;
        }
        
        // Check if all components in snapshot are available in application
        $missingComp = false;
        foreach ($config['components'] as $name => $versions) {
            if (! in_array($name, $appDestComponentList)) {
                Utils::CLIerror("Component '{$name}' is not available in application '{$config['application']}'");
                $missingComp = true;
            }
        }
        if ($missingComp) {
            Utils::CLIout("\n");
            Utils::CLIerror("Cannot create Snapshot due to missing components.");
            return false;
        }
        
        // Snapshot creation
        $createSnapshot = false;
        
        $snapshot['name'] = $config['snapshot']['name'];
        $snapshot['application'] = $config['application'];
        $snapshot['description'] = $config['snapshot']['description'];
        $snapshot['versions'] = array();
        
        // Check if snapshot already exits
        $snapshotExists = false;
        $output = $this->rest()->snapshot()->getSnapshotsInApplication($snapshot['application']);
        $checkSnapshots = Utils::outputToArray($output);
        
        foreach ($checkSnapshots as $checkSnapshot) {
            if ($checkSnapshot['name'] == $snapshot['name']) {
                $snapshotExists = true;
                break;
            }
        }
        if ($snapshotExists) {
            Utils::CLIerror("Requested snapshot '{$snapshot['name']}' is already available in '{$config['application']}' application.");
            return true;
        }
        
        // Create component version for every component in config
        Utils::CLIinfo("Creating component versions \n");
        // Login component to the current server
        $promote_config = $this->getPromoteConfig();
        $this->component->setupServer('destination', $promote_config);
        foreach ($config['components'] as $name => $versions) {
            if (!is_array($versions)) {
                $versions = array($versions);
            }
            foreach ($versions as $version) {
                // Check if the component version already exists
                if ($this->component->versionExists($name,$version)) {
                    Utils::CLIout("  [SKIPPING] {$name} ({$version})");
                    array_push($snapshot['versions'], array( $name => $version ));
                    continue;
                }
                // Version is missing: create it
                $result['components'][$name] = $this->udclient()->component()->createVersion($name,$version);
                if ($result['components'][$name] !== false) {
                    Utils::CLIout("  [SUCCESS] {$name} ({$version})");
                    array_push($snapshot['versions'], array( $name => $version ));
                    $createSnapshot = true;
                    
                    // Upload artifacts, if any
                    if (array_key_exists('artifacts', $config)) {
                        $artifacts = $config['artifacts'];
                        if (array_key_exists($name, $artifacts)) {
                            // Check if more than one version
                            if (array_key_exists($version, $artifacts[$name])) {
                                $artifact = $artifacts[$name][$version];
                            } else {
                                $artifact = $artifacts[$name];
                            }
                            
                            if (array_key_exists('base', $artifact)) {
                                // Check artifacts base directory
                                $base = $artifact['base'];
                                if (!is_dir($base)) {
                                    Utils::CLIerror("Artifacts base directory '{$base}' is missing or not accessible");
                                    return false;
                                }
                                
                                // Check artifacts to include
                                $include = null;
                                $includeArray = array();
                                if (array_key_exists('include', $artifact)) {
                                    $include = $artifact['include'];
                                    $includeError = false;
                                    if (! is_array($include)) {
                                        array_push($includeArray, $include);
                                    } else {
                                        $includeArray = $include;
                                    }
                                    foreach ($includeArray as $includeFile) {
                                        $includeFile = str_replace('*','',$includeFile);
                                        $includeFile = str_replace('\/','/',$includeFile);
                                        $path = "{$base}/{$includeFile}";
                                        if (! file_exists($path)) {
                                            Utils::CLIerror("Artifacts file/dir '{$includeFile}' is missing or not accessible in base directory '{$base}'");
                                            $includeError = true;
                                        }
                                    }
                                    if ($includeError) {
                                        return false;
                                    }
                                }
                                
                                // Artifacts to exclude
                                $exclude = null;
                                if (array_key_exists('exclude', $artifact)) {
                                    $exclude = $artifact['exclude'];
                                }
                                
                                $result['artifacts'][$name][$version] = $this->rest()->component()->addVersionFiles($name, $version, $base, $include, $exclude);
                            } else {
                                $this->log->debug("Trying to upload artifacts for component '{$name}', version '{$version}', but no base path has been set. Skipping.");
                            }
                        }
                    } // End of artifacts upload
                } else {
                    // Error during component version creation
                    Utils::CLIerror("  [FAILURE] {$name} ({$version})");
                }
            }
        }
        
        $result['snapshot'] = true;
        // At least one new component version has been created. Create a new snapshot
        if ($createSnapshot || $force) {
            $file = Utils::createJsonFile($snapshot);
            $result['snapshot'] = Utils::outputToArray($this->udclient()->snapshot()->createSnapshot($file));
            $this->log->log(print_r($result['snapshot'],true));
            if ($result['snapshot']) {
                Utils::CLIout("\n\n  [SUCCESS] Snapshot created ({$snapshot['name']})");
                Utils::CLIout("\n\n  [LINK] {$this->weburl}/#snapshot/{$result['snapshot']['id']}");
            } else {
                Utils::CLIout("\n\n  [FAILURE] Snapshot was not created ({$snapshot['name']})");
                $result['snapshot'] = false;
            }
            unlink($file);
        } else {
            Utils::CLIout("\n\n  [SKIPPED] Snapshot not created. All component versions are included in a previous snapshot.");
        }
        
        return $result;
    }
    
    /**
     *
     * Download configuration from one origin server and promote it (upgrade) up to the destination server
     * @param boolean $cleanEnvironments        If true, will remove the environments 
     *                                          imported with the application upgrade, 
     *                                          keeping the existing ones available before upgrade
     *
     */
    public function promoteConfig( $sourceDir = null, $cleanEnvironments = false ){
        if (empty($sourceDir)) {
            $config = 'config/promote.config.php';
            if (file_exists($config)) {
                include $config;
            } else {
                Utils::CLIerror("No server configuration set.");
                return false;
            }
            if (!self::validateServerConfig($config)) {
                Utils::CLIerror("Configuration error. Please check the messages above and provide a valid configuration.");
                return false;
            }
            Utils::CLIinfo("Promoting from server {$config['origin_alias']}");
        }
        else {
            Utils::CLIinfo("Promoting from directory {$sourceDir}");
        }
        
        $backupsRootDir = $this->output;
        $result = array();
        
        // Set backup/restore objects
        $backup = new BackupUCD();
        $restore = new RestoreUCD();
        if (empty($sourceDir)) {
            // Setup connection to first server (origin)
            if (!$backup->setupServer('origin', $config, true)) {
                Utils::CLIerror("Cannot connect to origin server.");
                return false;
            }
            
            // Get data from origin
            Utils::CLIinfo("Retrieving data from origin server");
            // Setup connection to first server (origin)
            if (!$this->setupServer('origin', $config, true)) {
                Utils::CLIerror("Cannot connect to origin server.");
                return false;
            }
            $sourceDir = getcwd() . '/tmp/' . date('Ymd-His');
            $backup->setOutput($sourceDir, true);
            $result['backup'] = $backup->backup();
            $resourceTemplates = $this->rest()->resource()->getResourceTemplates();
            $resourceTemplates = Utils::outputToArray( $resourceTemplates );
            $origResources = array();
            foreach ( $resourceTemplates as $res ) {
                array_push( $origResources, $res['name'] );
            }
            
            // Setup connection to second server (destination)
            if (!$restore->setupServer('destination', $config, true)) {
                Utils::CLIerror("Cannot connect to destination server.");
                return false;
            }
        }
        
        $restore->setOutput($sourceDir,true);
        $this->setOutput($sourceDir, true);
        
        // Get existing environments
        if ($cleanEnvironments) {
            Utils::CLIinfo("Retrieve existing environments");
            $this->setReturn('json');
            $existingEnvs = array();
            $apps = Utils::outputToArray($this->udclient()->application()->getApplications());
            foreach ($apps as $app) {
                $envs = Utils::outputToArray($this->udclient()->environment()->getEnvironmentsInApplication($app['id']));
                foreach ($envs as $env) {
                    if (!in_array($env['id'], $existingEnvs)) {
                        array_push($existingEnvs, $env['id']);
                        Utils::CLIout("{$env['name']} ({$env['id']})");
                    }
                }
            }
        }
        
        // Upgrade generic processes
        Utils::CLIinfo("Upgrading processes");
        $processes = "{$this->output}/processes";
        $result['upgrade']['processes'] = $restore->restoreAllProcesses($processes, true);
        
        // Upgrade components
        Utils::CLIinfo("Upgrading components");
        $components = "{$this->output}/components";
        $result['upgrade']['components'] = $restore->importAllComponents($components, true);
        
        // Upgrade application
        Utils::CLIinfo("Upgrading applications");
        $applications = "{$this->output}/applications";
        $result['upgrade']['applications'] = $restore->upgradeAllApplications($applications);
        
        // Import missing resource templates
        Utils::CLIinfo("Importing missing resource templates");
        $resources = "{$this->output}/resourceTemplates";
        $resourceTemplates = $this->rest()->resource()->getResourceTemplates();
        $resourceTemplates = Utils::outputToArray( $resourceTemplates );
        $toCreateResources = $origResources;
        
        foreach ( $resourceTemplates as $res ) {
            // Remove existing tempate files
            $filePath = "{$resources}/{$res['name']}.json";
            if ( file_exists ( $filePath ) ) {
                Utils::CLIdebug("Deleting not needed resource template file '{$filePath}'");
                unlink ( $filePath );
            }
            // Get remaining template files
            if ( in_array ( $res['name'], $toCreateResources ) ) {
                $key = array_search( $res['name'], $toCreateResources );
                if ( $key !== false ) {
                    unset ( $toCreateResources[$key] );
                }
            }
        }
        
        if ( !empty ($toCreateResources) ) {
            $result['upgrade']['resources'] = $restore->restoreAllResourceTemplates( $resources );
        } else {
            Utils::CLIinfo("Resource templates up to date");
        }
        
        // Cleanup unwanted new environments
        if ($cleanEnvironments) {
            Utils::CLIinfo("Cleaning environments");
            $this->setReturn('json');
            $apps = Utils::outputToArray($this->udclient()->application()->getApplications());
            foreach ($apps as $app) {
                Utils::CLIout("  {$app['name']}");
                $envs = Utils::outputToArray($this->udclient()->environment()->getEnvironmentsInApplication($app['id']));
                foreach ($envs as $env) {
                    if (!in_array($env['id'], $existingEnvs)) {
                        $result['remove_envs'][$env['id']] = $this->udclient()->environment()->deleteEnvironment($env['id']);
                        Utils::CLIout("    [REMOVE] {$env['name']} ({$env['id']})");
                    }
                }
            }
        }
        if ( !empty ($toCreateResources) ) {
            Utils::CLIinfo("One or more new resource template has been promoted.");
            Utils::CLIout("[ACTION REQUIRED] Please check in UrbanCode web UI if any blueprint creation is required.");
            Utils::CLIout("\n  Blueprint creation is available from the tab 'Blueprints' in the application view.");
            Utils::CLIout("    - On the left, click 'Create new blueprint'");
            Utils::CLIout("    - Set a unique name and a description for your blueprint");
            Utils::CLIout("    - Choose the resource template you wan to use");
            Utils::CLIout("    - Click on 'save'. Blueprint will then be available to be used with environments. ");
        }
        
        Utils::CLIout("\n");
        
        // Restore backup dir
        $this->setOutput($backupsRootDir, true);
        
        return $result;
    }
    
    /**
     * Set a snapshot PASSED or FAILED
     *
     * @param snapshot name
     * @param snapshot status [PASSED | FAILED] case insensitive
     * 
     * @return boolean
     */
    public function setSnapshotStatus($snapshotName, $status) {
        $error = false;
        // Check status argument
        if (empty($status)) {
            Utils::CLIerror( "Missing snapshot status." );
            $errors = true;
        }
        
        $status = strtoupper( $status );
        if ($status != "PASSED" && $status != "FAILED") {
            Utils::CLIerror( "Passed Status: '$status' is not supported" );
            $errors = true;
        }
        
        if ($errors === true){
            Utils::CLIinfo( "Possible options:" );
            Utils::CLIout( "  PASSED" );
            Utils::CLIout( "  FAILED".PHP_EOL );
            return false;
        }
        
        Utils::CLIinfo( "Getting snapshot info.." );
        $ucc = new EntryPoint();
        $this->setReturn( 'json' );
        $snapJson = $ucc->rest()->snapshot()->getSnapshotsInApplication( $this->application );
        if ($snapJson == false) {
            Utils::CLIerror( 
                    __CLASS__ . '::' . __FUNCTION__ .
                             ":: Failed to get info about snapshots from application: {$this->application}" );
            return false;
        }
        // Output to Array for easy editing
        $snapsArray = Utils::outputToArray( $snapJson );
        // Looking for conflicts snapshot
        $snapsArray = $this->matchSnapshot( $snapsArray, $snapshotName );
        if (empty($snapsArray)) {
            Utils::CLIinfo( 
                    __CLASS__ . '::' . __FUNCTION__ .
                             ":: No snapshot named: $snapshotName from application: {$this->application}" );
            return false;
        }
        if (count( $snapsArray ) > 1) {
            Utils::CLIinfo( 
                    __CLASS__ . '::' . __FUNCTION__ .
                             ":: Conflict detected for '{$snapshotName}', please use a more specific name. Snapshot collected: " );
            $this->printSnapshots($snapsArray);
            return false;
        }
        $snapArray = $snapsArray[0];
        // Try to override the old status
        $statusOld = substr( $snapArray ['name'], - 6 );
        if ($statusOld === "PASSED" || $statusOld === "FAILED") {
            $snapArray ['name'] = str_replace( $statusOld, $status, 
                    $snapArray ['name'] );
        } else {
            $snapArray ['name'] = "{$snapArray ['name']}_{$status}";
        }
        $snapArray ['existingId'] = $snapArray ['id'];
        $nameResult = $snapArray ['name'];
        $snapJson = json_encode( $snapArray );
        $response = $ucc->setSnapshotName( $snapJson );
        if ($response == false) {
            Utils::CLIerror( 
                    __CLASS__ . '::' . __FUNCTION__ .
                             ":: Failed to edit the snapshot: $snapshotName, in : {$snapshotName}_{$status}" );
            return false;
        }
        $response = $ucc->addSnapshotStatus( $snapArray ['id'], $status );
        if (! is_array( $response )) {
            Utils::CLIerror( 
                    __CLASS__ . '::' . __FUNCTION__ .
                             ":: Failed to add the snapshot status: $status to $nameResult" );
            return false;
        }
        
        Utils::CLIresult( 
                "Success. Snapshot: '$snapshotName' now marked : '$nameResult' and in status: '$status'" );
        return true;
    }
    
    /**
     * Return all the snapshots with matchs with name in param
     *
     * @param snapshots array
     * @param snapshot name
     * 
     * @return array
     */
    private function matchSnapshot($snapsArray, $snapshotName) {
        $snapshotReturn = array ();
        foreach ( $snapsArray as $entry ) {
            
            if ($entry ['name'] === $snapshotName) {
                $snapshotReturn [] = $entry;
                continue;
            }
            $status = substr( $entry ['name'], - 6 );
            $nameNoStatus = substr( $entry ['name'], 0, strlen($entry ['name']) - 7 );
            if (($status == "FAILED" || $status == "PASSED") &&
                     ($nameNoStatus == $snapshotName)) {
                $snapshotReturn [] = $entry;
            }
        }
        return $snapshotReturn;
    }
    
    /**
     * Print out snapshots available
     */
    private function printSnapshots($snapsArray) {
        foreach ( $snapsArray as $entry ) {
            Utils::CLIout($entry['name']);
        }
        Utils::CLIout('');
    }

    /**
     * Return a list of snapshot based on application id or name
     * 
     * @param string $json_file_path
     * @return boolean | array
     */
    public function getList($sapplication) {
        $this->setReturn('array');
        $return = $this->rest()->snapshot()->getSnapshotsInApplication($sapplication);
        if ($return === false){
            Utils::CLIerror( "Failed to retrieve snapshots list" );
            Utils::CLIdebug( __CLASS__ . '::' . __FUNCTION__ .": Failed to retrieve snapshots list for application '{$$sapplication}'" );
            return false;
        }
        if (empty($return)){
            return null;
        }
        return Utils::outputToArray($return);
    }

    /**
     * Create a Snapshot based on an array object or json file path
     */
    public function create($snapshot) {
        $file_path = null;
        // Check if $snapshot arg is file path or array obj
        if (is_array( $snapshot ) || is_object( $snapshot )) {
            $file_path = Utils::createJsonFile( $snapshot );
        } else {
            if (file_exists($snapshot) === false) {
                Utils::CLIerror( "Json file does not exist" );
                Utils::CLIdebug( __CLASS__ . '::' . __FUNCTION__.": Json file '{$snapshot}' does not exist" );
                return false;
            }
            if (Utils::validateJSON( file_get_contents( $snapshot ) ) === false) {
                Utils::CLIerror( "Json file has wrong format" );
                Utils::CLIdebug( __CLASS__ . '::' . __FUNCTION__.": Json file '{$snapshot}' has wrong format" );
                return false;
            }
            $file_path = $snapshot;
        }
        $result = Utils::outputToArray( $this->udclient()->snapshot()->createSnapshot( $file_path ) );
        if ($result) {
            is_string( $snapshot ) === false ? unlink( $file_path ) : null;
            return true;
        } else {
            Utils::CLIerror("Failed to create snapshot " );
            return false;
        }
        return false;
    }
}