<?php
/****************************************************************
 * IBM Confidential
 *
 * SFA100-Collaboration Source Materials
 *
 * (C) Copyright IBM Corp. 2014
 *
 * The source code for this program is not published or otherwise
 * divested of its trade secrets, irrespective of what has been
 * deposited with the U.S. Copyright Office
 *
 ***************************************************************/

require_once('include/EntryPoint.php');

/**
 * Restore uDeploy from files created by BackupUCD class
 * 
 * @class   RestoreUCD
 * @author  Marco Bassi         marcobas@ie.ibm.com
 * @author  Kostadin Ivanov     kostadii@ie.ibm.com
 */
class RestoreUCD extends EntryPoint {
    
    protected $source = '';
    
    public function __construct(){
        parent::__construct();
        $this->setReturn('json');
    }
    
    /**
     * Run a FULL UCD RESTORE process from a complete UCD backup.
     * This includes:
     *      - restoring of UCD Applications and their corresponding Components,
     *      - restoring of UCD component Templates.
     *
     * @param string  $backup_dir   String, holding the FULL PATH to a UCD backup directory
     * @param boolean $upgrade      Boolean FLAG, used to define, whether the specified ComponentTemplate
     *                              will be CREATED or UPGRADED.
     *                              DEFAULT value: FALSE (RESTORE (create))
     *
     * @return array    Array, holding the results from 'restore' processes of UCD Applications and
     *                  Component Templates
     */
    public function restoreAll ( $backup_dir, $upgrade = false, $cleanEnvironments = false ) {
        $result = array();
        $this->source = $backup_dir;
        
        // Set Action: RESTORE (create) or UPGRADE
        $process = $upgrade ? 'UPGRADE' : 'RESTORE (create)';
        
        $restore_watch =  time();
        StopWatch::getInstance()->start($restore_watch);
        // Prompt message for Backup Process
        Utils::CLIinfo( '### START of UrbanCode ' . $process . ' process ###' );
        Utils::CLIstart( $process.' process: '. date("d-m-Y h:i:s"));
        Utils::CLIinfo( 'Server to ' . $process . ':', false, 4 );
        Utils::CLIdest( $this->weburl, false, 6 );
        Utils::CLIinfo( $process . ' from backup folder:', false, 4 );
        Utils::CLIdest( $backup_dir, false, 6 );
        
        Utils::CLIout("  Checking source directory");
        $validJson = Utils::validateDirectoryJSON( $backup_dir );
        
        if (!$validJson) {
            Utils::CLIerror("Source directory didn't pass json validation check");
            return false;
        }
        
        // Get existing environments
        if ($cleanEnvironments) {
            $getEnv_watch = time();
            StopWatch::getInstance()->start($getEnv_watch);
            Utils::CLIinfo("Retrieve existing environments");
            $this->setReturn('json');
            $existingEnvs = array();
            $apps = Utils::outputToArray($this->udclient()->application()->getApplications());
            foreach ($apps as $app) {
                $envs = Utils::outputToArray($this->udclient()->environment()->getEnvironmentsInApplication($app['id']));
                foreach ($envs as $env) {
                    if (!in_array($env['id'], $existingEnvs)) {
                        array_push($existingEnvs, $env['id']);
                        Utils::CLIout("{$env['name']} ({$env['id']})");
                    }
                }
            }
            StopWatch::getInstance()->stop($getEnv_watch);
            Utils::CLIcompleted('Retrieve existing environments: '. StopWatch::getInstance()->get($getEnv_watch));
        }
        
        // Restore tags
        $restoreTags_watch =  time();
        StopWatch::getInstance()->start($restoreTags_watch);
        $result['restore_tags'] = $this->restoreAllTags( $backup_dir . '/tags' );
        StopWatch::getInstance()->stop($restoreTags_watch);
        Utils::CLIcompleted( 'Restore tags: '. StopWatch::getInstance()->get($restoreTags_watch));
        
        // Restore all post processing scripts
        $postProcessingScripts_watch =  time();
        StopWatch::getInstance()->start($postProcessingScripts_watch);
        $result['post_processing_scripts'] = $this->restorePostProcessingScripts( $backup_dir . '/settings');
        StopWatch::getInstance()->stop($postProcessingScripts_watch);
        Utils::CLIcompleted( 'Post Processing Scripts: '. StopWatch::getInstance()->get($postProcessingScripts_watch));
        
        // Restore all general processes
        $generalProcesses_watch =  time();
        StopWatch::getInstance()->start($generalProcesses_watch);
        $result['processes'] = $this->restoreAllProcesses( $backup_dir . '/processes' );
        StopWatch::getInstance()->stop($generalProcesses_watch);
        Utils::CLIcompleted( 'General Processes: '. StopWatch::getInstance()->get($generalProcesses_watch));
        
        // Restore ALL ComponentTemplates
        $restoreComponentTemplates_watch =  time();
        StopWatch::getInstance()->start($restoreComponentTemplates_watch);
        $result['restore_component_templates'] = $this->importAllComponentTemplates( $backup_dir . '/componentTemplates', $upgrade );
        StopWatch::getInstance()->stop($restoreComponentTemplates_watch);
        Utils::CLIcompleted( 'Restore Component Templates: '. StopWatch::getInstance()->get($restoreComponentTemplates_watch));
        
        // Restore ALL Components
        $restoreComponents_watch =  time();
        StopWatch::getInstance()->start($restoreComponents_watch);
        $result['restore_components'] = $this->importAllComponents( $backup_dir . '/components', $upgrade );
        StopWatch::getInstance()->stop($restoreComponents_watch);
        Utils::CLIcompleted( 'Restore Components: '. StopWatch::getInstance()->get($restoreComponents_watch));
        
        // Restore ALL Applications (includes All APPS + underlying App components, blueprints)
        $restoreAllApplications_watch =  time();
        StopWatch::getInstance()->start($restoreAllApplications_watch);
        $result['restore_applications'] = $this->restoreAllApplications( $backup_dir . '/applications', true, false );
        StopWatch::getInstance()->stop($restoreAllApplications_watch);
        Utils::CLIcompleted( 'Restore Applications: '. StopWatch::getInstance()->get($restoreAllApplications_watch));
        
        // Restore resource templates
        $restoreResourceTemplates_watch =  time();
        StopWatch::getInstance()->start($restoreResourceTemplates_watch);
        $result['restore_resource_templates'] = $this->restoreAllResourceTemplates( $backup_dir . '/resourceTemplates' );
        StopWatch::getInstance()->stop($restoreResourceTemplates_watch);
        Utils::CLIcompleted( 'Restore Resource Templates: '. StopWatch::getInstance()->get($restoreResourceTemplates_watch));
        
        // Restore all resources
        $restoreResources_watch =  time();
        StopWatch::getInstance()->start($restoreResources_watch);
        $result['restore_resources'] = $this->restoreAllResources( $backup_dir . '/resources', $upgrade );
        StopWatch::getInstance()->stop($restoreResources_watch);
        Utils::CLIcompleted( 'Restore Resources: '. StopWatch::getInstance()->get($restoreResources_watch));

        // Restore ALL Blueprints
        $restoreBlueprints =  time();
        StopWatch::getInstance()->start($restoreBlueprints);
        $result['restore_blueprints'] = $this->restoreBlueprints( $backup_dir . '/blueprints' );
        StopWatch::getInstance()->stop($restoreBlueprints);
        Utils::CLIcompleted( 'Restore Blueprints: '. StopWatch::getInstance()->get($restoreBlueprints));
        
        // Cleanup unwanted new environments
        if ($cleanEnvironments) {
            $cleanEnv_watch =  time();
            StopWatch::getInstance()->start($cleanEnv_watch);
            Utils::CLIinfo("Cleaning environments");
            $this->setReturn('json');
            $apps = Utils::outputToArray($this->udclient()->application()->getApplications());
            foreach ($apps as $app) {
                Utils::CLIout("  {$app['name']}");
                $envs = Utils::outputToArray($this->udclient()->environment()->getEnvironmentsInApplication($app['id']));
                foreach ($envs as $env) {
                    if (!in_array($env['id'], $existingEnvs)) {
                        $result['remove_envs'][$env['id']] = $this->udclient()->environment()->deleteEnvironment($env['id']);
                        Utils::CLIout("    [REMOVE] {$env['name']} ({$env['id']})");
                    }
                }
            }
            StopWatch::getInstance()->stop($cleanEnv_watch);
            Utils::CLIcompleted('Cleanup unwanted new environments: '. StopWatch::getInstance()->get($cleanEnv_watch));
        }
        
        StopWatch::getInstance()->stop($restore_watch);
        Utils::CLIcompleted( $process .' process: '. StopWatch::getInstance()->get($restore_watch));
        // Prompt message for end of process and return
        Utils::CLIinfo( '### END of UrbanCode ' . $process . ' process ###' );
        return $result;
    }
    
    /**
     * Giving a folder path, will look for all available applications and restore them all, components included.
     * Folder should be in the format BackupUCD is creating it
     * OUTPUT_DIRECTORY
     * - APPLICATION_NAME
     *   - APPLICATION_NAME.json
     *   - comonents
     *     - COMPONENT_1.json
     *     - COMPONENT_2.json
     *
     * @param string  $applicationsBackupsDir    FULL PATH to a directory with UrbanCode Applications backups
     * @param boolean $upgrade                   Boolean FLAG, used to define, whether the specified ComponentTemplate
     *                                           will be CREATED or UPGRADED.
     *                                           DEFAULT value: FALSE (RESTORE (create))
     * @param boolean $full                      Full restore with components (true) or simple creation (false)
     *
     * @return array | boolean   Array with the result from Restore/Upgrade result
     *                           for each Application OR
     *                           FALSE in case of a problem with specified $applicationsBackupsDir
     */
    public function restoreAllApplications ($applicationsBackupsDir, $upgrade = false, $full = true) {
        
        // Set Action: RESTORE (create) or UPGRADE
        $process = $upgrade ? 'UPGRADE' : 'RESTORE (create)';
        Utils::CLIinfo( $process . ' UrbanCode Applications from:', true );
        Utils::CLIdest( $applicationsBackupsDir, false );
        
        Utils::CLIout("  Checking source directory");
        $validJson = Utils::validateDirectoryJSON( $applicationsBackupsDir );
        
        if (!$validJson) {
            Utils::CLIerror("Source directory didn't pass json validation check");
            return false;
        }
        
        if ( is_dir( $applicationsBackupsDir ) ) {
            $result = array();
            
            $existingApplictions = array();
            $applications = Utils::outputToArray($this->udclient()->application()->getApplications());
            
            foreach ($applications as $application) {
                array_push($existingApplictions, $application['name']);
            }
            
            $appsDirs = scandir($applicationsBackupsDir);
            foreach ($appsDirs as $subdir) {
                // Skip: '.' and '..' pointers
                if ( !in_array($subdir, array('.', '..')) ) {
                    
                    // Check if application exists
                    if (in_array($subdir, $existingApplictions)) {
                        Utils::Cliout(" [SKIPPING] Application {$subdir} already exists");
                        continue;
                    }
                    
                    $subdir = "{$applicationsBackupsDir}/{$subdir}";
                    
                    if ( is_dir($subdir) ) {
                        if ($full) {
                            $result[$subdir] = self::restoreApplication($subdir, $upgrade);
                        } else {
                            $result[$subdir] = self::simpleApplicationRestore($subdir,$upgrade);
                        }
                    }
                }
            }
        } else {
            Utils::CLIerror( 'Problem with Applications backup folder: ', true );
            Utils::CLIdest( $applicationsBackupsDir, false );
            Utils::CLIerror( 'Backup Folder DOES NOT exist or specified PATH is wrong.', false );
            $result =  false;
        }
        
        return $result;
    }
    
    /**
     * 
     * Create/Upgrade application and try to add components (already created) to that application based on {$application}.components.json file
     *
     * @param string $application       path to application backup folder
     * @param boolean $upgrade          If true, perform an upgrade
     */
    public function simpleApplicationRestore($application, $upgrade = false) {
        
        $result = array();
        if (is_dir($application)) {
            $files = scandir($application);
            
            foreach ($files as $file) {
                $applicationFilepath = "{$application}/{$file}";
                /*
                 * Skip: '.', '..' files and dirs
                * Process ONLY Application JSON file, components will be handled later
                */
                if ( !is_dir($applicationFilepath) && !in_array($file, array('.', '..')) ) {
                    
                    // Validate application JSON file
                    if ( Utils::validateJSON(file_get_contents($applicationFilepath)) ) {
                        if ($upgrade) {
                            Utils::CLIinfo( 'Process Application UPGRADE file:', true );
                            Utils::CLIdest($applicationFilepath, false );
                            /*
                             * @NOTE: upgradeApplication - upgrade App->Process(es) 
                             *        and related ComponentTemplate(es) with paramter "UPGRADE_IF_EXISTS"
                             */
                            $result['application'] = $this->rest()->application()->upgradeApplication($applicationFilepath, 'UPGRADE_IF_EXISTS', 'UPGRADE_IF_EXISTS');
                            
                            }
                        // Create application
                        else {
                            Utils::CLIinfo( 'Process Application RESTORE file:', true);
                            Utils::CLIdest($applicationFilepath, false );
                            $output = $this->udclient()->application()->createApplication($applicationFilepath);
                            
                            if ($output !== false) {
                                $result['application'] = Utils::outputToArray($output);
                                
                                // Add components to application
                                $result['components'] = self::simpleAddComponentToApplication($application);
                                
                                $result['processes'] = self::restoreAllApplicationProcesses(
                                        $result['application']['id'], 
                                        $applicationFilepath,
                                        $upgrade);
                            } else {
                                $result['application'] = false;
                                $result['blueprints']  = false;
                                $result['processes']   = false;
                                Utils::CLIerror('Application RESTORE process -FAILED');
                                Utils::CLIinfo('For more details, please check: "' . $this->log->getLogFile() .'"', false);
                            }
                        }
                    }
                }
            }
        }
        
        return $result;
    }
    
    /**
     * 
     * Add components to application from application backup folder
     *
     * @param string $application       Application folder
     */
    public function simpleAddComponentToApplication($application) {
        $dir = "{$application}/components";
        $files = scandir($dir);
        
        $applicationName = explode("/", $application);
        $applicationName = end($applicationName);
        
        foreach ($files as $file) {
            $filepath = "{$application}/components/{$file}";
            if ( !is_dir($filepath) && !in_array($file, array('.', '..')) ) {
                if ( Utils::validateJSON(file_get_contents($filepath)) ) {
                    $components = json_decode(file_get_contents($filepath),true);
                    foreach ($components as $component) {
                        Utils::CLIinfo("Add component '{$component['name']}' to application '{$applicationName}'");
                        $result[$component['name']] = $this->udclient()->component()->addComponentToApplication($component['name'],$applicationName);
                    }
                }
            }
        }
        
        return $result;
    }
    
    /**
     * Will restore an application using the json file which path is passed as argument.
     * If $application is a path to a directory, restore the Application file plus all the related components
     *
     * @param string $application
     * @return boolean
     */
    public function restoreApplication ($application, $upgrade = false) {
        $result = array();
        $applicationFilepath = '';
        // Prompt start of action
        Utils::CLIinfo( ($upgrade ? 'Upgrade' : 'Create') . ' Application:', true );
        Utils::CLIdest($application, false );
        
        // Restore applcation and components
        if (is_dir($application)) {
            // Check application file
            $files = scandir($application);
            
            foreach ($files as $file) {
                $applicationFilepath = "{$application}/{$file}";
                
                /*
                 * Skip: '.', '..' files and dirs
                 * Process ONLY Application JSON file, components will be handled later
                 */
                if ( !is_dir($applicationFilepath) && !in_array($file, array('.', '..')) ) {
                    Utils::CLIinfo( 'Process Application BACKUP file:', false );
                    Utils::CLIdest($applicationFilepath, false );
                    
                    // Validate application JSON file
                    if ( Utils::validateJSON(file_get_contents($applicationFilepath)) ) {
                        // Upgrade application
                        if ($upgrade) {
                            /*
                             * @NOTE: upgradeApplication - will also upgrade App->Process(es)
                             *        and related ComponentTemplate(es) if there are such.
                             */
                            $result['application'] = $this->rest()->application()->upgradeApplication($applicationFilepath,'UPGRADE_IF_EXISTS', 'UPGRADE_IF_EXISTS');
                            return $result;
                        }
                        // Create application
                        else {
                            $result['application'] = $this->udclient()->application()->createApplication($applicationFilepath);
                        }
                        
                    } else {
                        Utils::CLIerror('Specified Application JSON File is not a valid json file.', true );
                        $result['application'] = false;
                    }
                    
                    // Exit the loop as we processed the application JSON file
                    break;
                }
            }
            
            // Restore application components
            if ( is_dir("{$application}/components") && $result['application'] !== false ) {
                // clean up the Application (result from exec() or execRest()
                $result['application'] = Utils::outputToArray($result['application']);
                
                // Prompt start of action
                Utils::CLIinfo( ($upgrade ? 'Upgrade' : 'Create') . ' Application components from:', true );
                Utils::CLIdest( "{$application}/components", false );
                
                $components = scandir("{$application}/components");
                foreach ($components as $component) {
                    $componentpath = "{$application}/components/{$component}";
                    if ( !is_dir($componentpath) && Utils::validateJSON(file_get_contents($componentpath)) ) {
                        $componentResult = self::restoreComponent($componentpath, $upgrade);
                        $result['components'][$component]['result'] = $componentResult;
                        // If it's not an upgrade, assign component to application
                        if ($componentResult !== false && !$upgrade) {
                            $applicationID = $result['application']['id'];
                            $componentID = $componentResult['component']['id'];
                            $result['components'][$component]['action'] = $this->udclient()->component()->addComponentToApplication($componentID, $applicationID);
                        }
                    } else {
                        if ( $component != "." && $component != "..") {
                            Utils::CLIinfo("Skipping file '{$componentpath}': not a valid JSON file.");
                            $result['components'][$component]['action'] = false;
                        }
                    }
                }
            }
            
            // Restore blueprints
            if ( is_dir("{$application}/blueprints") && $result['application'] !== false ) {
                $applicationID = $result['application']['id'];
                $blueprintResult = self::restoreAllBlueprints("{$application}/blueprints", $applicationID);
                $result['blueprints'] = $blueprintResult;
            }
            
            // Restore Application Process(es)
            if ( $result['application'] !== false) {
                // Application process(es) have been upgrade in the upgradeApplication request, by default
                if ( $upgrade === true ) {
                    $result['processes'] = true;
                // Create Application process(es)
                } else {
                    $result['processes'] = self::restoreAllApplicationProcesses(
                            $result['application']['id'],
                            $applicationFilepath,$upgrade);
                }
                
            }
        }
        
        // Restore application only
        else if (is_file($application)) {
            if ( !is_dir($application) && Utils::validateJSON(file_get_contents($application)) ) {
                // Upgrade application
                if ($upgrade) {
                    $result['application'] = $this->rest()->application()->upgradeApplication($application, 'UPGRADE_IF_EXISTS', 'UPGRADE_IF_EXISTS');
                }
                // Create application
                else {
                    $result['application'] = $this->udclient()->application()->createApplication($application);
                }
            }
        }
        
        // Verify result and prompt success/fail message
        if ( $result['application'] === false ) {
            Utils::CLIerror( 'Application ' . ($upgrade ? 'UPGRADE' : 'CREATE')
                    . ' FAILED to complete.', true );
        } else {
            Utils::CLIinfo( 'Application ' . ($upgrade ? 'UPGRADE' : 'CREATE') 
                    . ' completed - SUCCESS.', true );
        }
        
        return $result;
    }
    
    public function restoreAllComponents ($componentsBackupsDir, $upgrade = false) {
        // Set Action: RESTORE (create) or UPGRADE
        $process = $upgrade ? 'UPGRADE' : 'RESTORE (create)';
        Utils::CLIinfo( $process . ' UrbanCode Components from:', true );
        Utils::CLIdest( $componentsBackupsDir, false );
        
        Utils::CLIout("  Checking source directory");
        $validJson = Utils::validateDirectoryJSON( $componentsBackupsDir );
        
        if (!$validJson) {
            Utils::CLIerror("Source directory didn't pass json validation check");
            return false;
        }
        
        if ( is_dir( $componentsBackupsDir ) ) {
            $result = array();
            
            $componentsBackups = scandir($componentsBackupsDir);
            foreach ($componentsBackups as $componentFile) {
                // Process ONLY 'Component.json' backup files
                if ( strpos($componentFile, '.json') !== false && $componentFile != '.json' ) {
                    // Set component filepath
                    $componentFilePath = $componentsBackupsDir . '/' . $componentFile;
                    // Get component Name
                    $componentName = end(explode('/', $componentFile));
                    // Remove the .json part from the component name
                    $componentName = str_ireplace('.json', '', $componentName);
                    
                    $result[$componentName] = self::restoreComponent($componentFilePath, $upgrade);
                }
            }
        } else {
            Utils::CLIerror( 'Problem with Components backup folder: ', true );
            Utils::CLIdest( $componentsBackupsDir, false );
            Utils::CLIerror( 'Backup Folder DOES NOT exist or specified PATH is wrong.', false );
            $result =  false;
        }
        return $result;
    }
    
    /**
     * Will restore a component using the json file which path is passed as argument
     * If upgrade argument is set to true, it will try to upgrade the existing component instead of creating a new one.
     * 
     * @param string    $component  Path to a Component JSON formatted file
     * @param bool      $upgrade    Flag to set whether to UPGRADE or RESTORE (Create) a Component
     * 
     * @return boolean|array Boolean FALSE in case of a problem with Component UPGRADE/RESTORE process. 
     *                       Array holding the details of the Component UPGRADE/RESTORE process steps,
     *                       on SUCCESS.
     */
    public function restoreComponent ($component, $upgrade = false){
        // Set $result
        $result = array();
        
        // Set Action: RESTORE (create) or UPGRADE
        $restoreType = $upgrade ? 'UPGRADE' : 'RESTORE (create)';
        
        Utils::CLIinfo( $restoreType . ' UrbanCode Component from:', true );
        Utils::CLIdest( $component, false );
        
        // Validate file
        if ( is_file($component) && Utils::validateJSON(file_get_contents($component)) ) {
            // Upgrade component
            if ($upgrade) {
                $result['component'] = $this->rest()->component()->upgradeComponent($component);
                if ($result['component'] === false){
                    $result = false;
                }
            }
            // Restore component. NOTE: Processes need to be restored in case of restore
            else {
                $exec = $this->udclient()->component()->createComponent($component);
                if ($exec !== false) {
                    $result['component'] = Utils::outputToArray($exec);
                    
                    // Restore component processes
                    $content = json_decode(file_get_contents($component),true);
                    $componentID = $result['component']['id'];
                    $processes = $content['processes'];
                    foreach ($processes as $process){
                        // Temp file name is process name with md5 encoded time appended, to make filename unique
                        $tempFile = $process['name'] . "-" . md5(time());
                        $tmpProcessFile = "{$this->udcpath}/{$tempFile}.json";
                        $fileCreated = Utils::writeToFile($tmpProcessFile, json_encode($process));
                        
                        // Create Component Process
                        if ( $fileCreated !== false ) {
                            $exec = $this->udclient()->component()->createComponentProcess($componentID,$tmpProcessFile);
                            // delete temporary file from system
                            unlink($tmpProcessFile);
                            // ComponentProcess creation - failed
                            if ($exec === false) {
                                Utils::CLIwarning('There was a problem with Component-Process creation.', true);
                                Utils::CLIwarning('Component Restore NOT Fully completed.', false);
                                // set result and exit for loop as Component was NOT fully restored
                                $result = false;
                                break;
                            }
                            // else: ComponentProcess created - OK. Continue as usual
                            $result['process'][$process['name']] = Utils::outputToArray($exec);
                        // There was a problem with creation of temporary ComponentProcess file
                        } else {
                            Utils::CLIwarning('There was a problem with creation of Component-Process temp file: ', true);
                            Utils::CLIdest($tmpProcessFile, false );
                            Utils::CLIwarning('Component Restore NOT Fully completed.', false);
                            $this->log('FAILED to create temporary file: ' . $tmpProcessFile);
                            $result = false;
                            break;
                        }
                    } // end of creating Component-process(es)
                // There was a problem with Component creation
                } else {
                    Utils::CLIwarning('Component Create - FAILED to complete.', true);
                    $result = false;
                }
            } // end of Component: upgrade / restore process
            
        } else {
            Utils::CLIerror("File '{$component}' is not a valid json file.\n");
            $this->log->error("File '{$component}' is not a valid json file.\n");
            $result = false;
        }
        
        // Check $result and prompt correspondingly
        if ( $result !== false ) {
            Utils::CLIinfo('Component '
                    . ($upgrade ? 'UPGRADED' : 'RESTORED') . '- SUCCESS', true );
        } else {
            Utils::CLIwarning('Component '
                    . $restoreType . ' process - FAILED to complete' , true);
            Utils::CLIinfo('For more details, please check: "'
                    . $this->log->getLogFile() . '"', true );
        }
        
        // Return
        return $result;
    }
    
    /**
     * Import / Upgrade multiple Components.
     * Will scan the provided components directory and will attempt to
     * Import / Upgrade each available Component JSON file by calling
     * THIS Class' function: importAComponent()
     *
     * @param string  $componentsBackupsDir Full Path to a Directory, which holds one or more
     *                                      Component JSON formatted files.
     * @param boolean $upgrade              Boolean TRUE / FALSE - flag to set wether to run: UPGRADE or IMPORT process
     *
     * @return boolean|array    Boolean FALSE in case of a problem with the specified $componentsBackupsDir or
     *                          Array holding the results from importAComponent() calls for each
     *                          available in the specified $componentsBackupsDir Componente JSON file
     */
    public function importAllComponents ($componentsBackupsDir, $upgrade = false) {
        $result = array();
        // Set Action: IMPORT or UPGRADE
        $process = $upgrade ? 'UPGRADE' : 'IMPORT';
        
        Utils::CLIinfo( $process . ' UrbanCode Components from:', true );
        Utils::CLIdest( $componentsBackupsDir, false );
        
        Utils::CLIout("  Checking source directory");
        $validJson = Utils::validateDirectoryJSON( $componentsBackupsDir );
        
        if (!$validJson) {
            Utils::CLIerror("Source directory didn't pass json validation check");
            return false;
        }
        
        if ( is_dir( $componentsBackupsDir ) ) {
            $componentsBackups = scandir($componentsBackupsDir);
            
            $existingComponents = array();
            $components = Utils::outputToArray($this->udclient()->component()->getComponents());
            
            foreach ($components as $component) {
                array_push($existingComponents, $component['name']);
            }
            
            foreach ($componentsBackups as $componentFile) {
                // Process ONLY 'ComponentTemplate.json' files
                if ( strpos($componentFile, '.json') !== false && $componentFile != '.json' ) {
                    // Set component filepath
                    $componentFilePath = $componentsBackupsDir . '/' . $componentFile;
                    // Get component Name
                    $componentName = end(explode('/', $componentFile));
                    // Remove the .json part from the component name
                    $componentName = str_ireplace('.json', '', $componentName);
                    
                    if (in_array($componentName, $existingComponents) && !$upgrade) {
                        Utils::CLIout("  [SKIPPING] {$componentName}");
                        continue;
                    }
                    
                    $result[$componentName] = self::importAComponent($componentFilePath, $upgrade);
                    
                    if ($result[$componentName] === false) {
                        Utils::CLIout("  [FAILURE] {$componentName}");
                    } else {
                        Utils::CLIout("  [SUCCESS] {$componentName}");
                    }
                } // else skip this file, as it is not a proper json file
            }
        } else {
            Utils::CLIerror( 'Problem with Components backup folder: ', true );
            Utils::CLIdest( $componentsBackupsDir, false );
            Utils::CLIerror( 'Backup Folder DOES NOT exist or specified PATH is wrong.', false );
            
            $result = false;
        }
        
        return $result;
    }
    
    /**
     * Import / Upgrade a Component to UrbanCode Server.
     * Will validate the provided $componentFile file
     * and will delegate its IMPORT / UPGRADE processing to
     * the EntryPoint: importComponent request.
     *
     * @param string  $componentFile    Full Path to a Component JSON file
     * @param boolean $upgrade          Boolean TRUE / FALSE flag, used to set whether to
     *                                  UPGRADE or IMPORT the Component
     *
     * @return boolean|array    Array holding the result from Component IMPORT / UPGRADE process or
     *                          Boolean FALSE in case of an error.
     */
    public function importAComponent ($componentFile, $upgrade = false){
        $result     = array();
        $success    = true;
        $resultType = $upgrade ? 'upgrade_result' : 'import_result';
        
        // Prompt start of action
        Utils::CLIinfo( ($upgrade ? 'Upgrade' : 'Import') . ' Component from:', true );
        Utils::CLIdest($componentFile, false );
        
        // Validate file
        if ( is_file($componentFile) ) {
            // Get file contents
            $fileContents = file_get_contents($componentFile);
            
            if ( Utils::validateJSON($fileContents) ) {
                $result[$resultType] = $this->rest()->component()->importComponent($componentFile, $upgrade);
                
                if ( $result[$resultType] === false ) {
                    // There was some problem - error logs are handled in the execRest() call
                    $success = false;
                } // else request was process OK -=> just continue as usual
            } else {
                Utils::CLIerror('Specified Component JSON File is not a valid json file.', true );
                $this->log->error('Specified Component JSON file: "'
                        . $componentFile . '" is NOT a valid JSON file');
                $result = false;
            }
        } else {
            Utils::CLIerror('Specified Component File is not a valid file.', true );
            $this->log->error('Specified Component file: "'
                    . $componentFile . '" is NOT a valid file');
            $result = false;
        }
        
        // Prompt END of action with SUCCESS / FAIL result
        if ($result !== false && $success !== false) {
            Utils::CLIinfo('Component ' . ($upgrade ? 'UPRGADED' : 'IMPORTED')
                    . ' - SUCCESS', false );
        } else {
            Utils::CLIwarning('Component '
                    . ($upgrade ? 'UPGRADE' : 'IMPORT')
                    . ' - FAILED to complete', true );
            Utils::CLIinfo('For more details, please check: "' . $this->log->getLogFile() .'"', false );
        }
        
        // Return
        return $result;
    }
    
    /**
     * Restore (CREATE/UPGRADE) uDeploy ComponentTemplates, based on JSON formatted backup files
     * of UCD Component Templates.
     *
     * @param string $templatesDir FULL PATH to a directory with Component Templates JSON formatted backup files
     * @param boolean $upgrade     Boolean FLAG, used to define, whether the specified ComponentTemplate
     *                             will be CREATED or UPGRADED.
     *                             DEFAULT value: FALSE (RESTORE (create))
     *
     * @return array | boolean    Array holding the result(s) from the ComponentTemplate(s) create/upgrade
     *                            uDeploy API request(s) OR
     *                            FALSE when the specified $templatesDir file is NOT a valid directory
     */
    public function restoreAllComponentTemplates ($templatesDir, $upgrade = false) {
        
        // Set Action: RESTORE (create) or UPGRADE
        $process = $upgrade ? 'UPGRADE' : 'RESTORE (create)';
        Utils::CLIinfo( $process . ' UrbanCode Component Templates in:', true );
        Utils::CLIdest( $templatesDir, false );
        
        Utils::CLIout("  Checking source directory");
        $validJson = Utils::validateDirectoryJSON( $templatesDir );
        
        if (!$validJson) {
            Utils::CLIerror("Source directory didn't pass json validation check");
            return false;
        }
        
        if ( is_dir( $templatesDir ) ) {
            $result = array();
            
            $templatesFiles = scandir($templatesDir);
            
            foreach ($templatesFiles as $templateFile) {
                // Process ONLY 'ComponentTemplate.json' files
                if ( strpos($templateFile, '.json') !== false && $templateFile != '.json' ) {
                    $templateRestoreFile   = $templatesDir . '/' . $templateFile;
                    $result[$templateFile] = self::restoreComponentTemplate($templateRestoreFile, $upgrade);
                } // else skip this file, as it is not a proper json file
            }
        } else {
            Utils::CLIerror( 'Problem with Component Templates backup folder: ', true );
            Utils::CLIdest( $templatesDir, false );
            Utils::CLIerror( 'Component Templates Backup Folder DOES NOT exist or specified PATH is wrong.', false );
            
            $result = false;
        }
        
        return $result;
    }
    
    /**
     * Restore (CREAT/UPGRADE) a uDeploy ComponentTemplate.
     *
     * @param string $componentTemplate String, holding the FULL PATH to a JSON file, 
     *                                  which holds the ComponentTemplate to be CREATED/UPDATED
     * @param string $upgrade           Boolean FLAG, used to define, whether the specified ComponentTemplate
     *                                  will be CREATED or UPGRADED.
     *                                  DEFAULT value: FALSE (CREATE)
     *                                  
     * @return array | boolean   Array holding the result from the ComponentTemplate create/upgrade 
     *                           uDeploy API request OR 
     *                           FALSE when the specified componentTemplate file is NOT a valid JSON file
     */
    public function restoreComponentTemplate ($componentTemplate, $upgrade = false){
        
        $result  = array();
        $success = true;
        
        // Prompt start of action
        Utils::CLIinfo( ($upgrade ? 'Upgrade' : 'Create') . ' Component Template from:', true );
        Utils::CLIdest($componentTemplate, false );
        
        // Validate file
        if ( is_file($componentTemplate) ) {
            // Get file contents
            $fileContents = file_get_contents($componentTemplate);
            
            if ( Utils::validateJSON($fileContents) ) {
                // Upgrade $componentTemplate
                if ($upgrade) {
                    $result['upgradeResult'] = $this->rest()->component()->upgradeComponentTemplate($componentTemplate);
                    
                    /*
                     * In case of error, result will hold just the error message
                     * When result was Ok it will hold response as: 
                     * '<textarea>SOME JSON FORMATTED INFORMATION</textarea>'
                     * and also should have the following substring: '"status":"ok"'
                     */
                    if ( $result['upgradeResult'] === false ) {
                        // There was some problem
                        $success = false;
                    } // else request was process OK -=> just continue as usual
                // Restore $componentTemplate
                } else {
                    $result['createResult'] = $this->rest()->component()->createComponentTemplate($componentTemplate);
                    
                    if ( $result['createResult'] === false ) {
                        // There was some problem
                        $success = false;
                    } // else request was process OK -=> just continue as usual
                }
                
                // Prompt END of action with SUCCESS / FAIL result
                if ($success) {
                    Utils::CLIinfo('Component Template '
                            . ($upgrade ? 'UPRGADED' : 'CREATED')
                            . ' - SUCCESS', false );
                } else {
                    Utils::CLIinfo('Component Template '
                            . ($upgrade ? 'UPGRADE' : 'CREATE')
                            . ' - FAILED to complete', true );
                }
                return $result;
            } else {
                Utils::CLIerror("Specified Template JSON File is not a valid json file.", true );
                return false;
            }
        } else {
            Utils::CLIerror("Specified Template File is not a valid file.", true );
            return false;
        }
    }
    
    /**
     * Import / Upgrade multiple ComponentTemplates.
     * Will scan the provided templates directory and will attempt to
     * Import / Upgrade each available Template JSON file by calling
     * THIS Class' function: importAComponentTemplate()
     * 
     * @param string  $templatesDir Full Path to a Directory, which holds one or more
     *                              ComponentTemplates JSON formatted files.
     * @param boolean $upgrade      Boolean TRUE / FALSE - flag to set wether to run: UPGRADE or IMPORT process
     * 
     * @return boolean|array    Boolean FALSE in case of a problem with the specified $templatesDir or
     *                          Array holding the results from importAComponentTemplate() calls for each
     *                          available in the specified $templatesDir ComponentTemplate JSON file
     */
    public function importAllComponentTemplates ($templatesDir, $upgrade = false) {
        
        $result = array();
        // Set Action: IMPORT or UPGRADE
        $process = $upgrade ? 'UPGRADE' : 'IMPORT';
        
        Utils::CLIinfo( $process . ' UrbanCode Component Templates from:', true);
        Utils::CLIdest( $templatesDir, false);
        
        Utils::CLIout("  Checking source directory");
        $validJson = Utils::validateDirectoryJSON( $templatesDir );
        
        if (!$validJson) {
            Utils::CLIerror("Source directory didn't pass json validation check");
            return false;
        }
        
        if ( is_dir( $templatesDir ) ) {
            // Get existing component templates
            $templates = $this->rest()->component()->getComponentTemplates();
            $templates = Utils::outputToArray($templates);
            $existingTemplates = array();
            
            foreach ($templates as $template) {
                array_push($existingTemplates, $template['name']);
            }
            
            $templatesFiles = scandir($templatesDir);
            
            foreach ($templatesFiles as $templateFile) {
                // Process ONLY 'ComponentTemplate.json' files
                if ( strpos($templateFile, '.json') !== false && $templateFile != '.json' ) {
                    $templateBackupFile   = $templatesDir . '/' . $templateFile;
                    
                    // Check if component template exists
                    $currentTemplate = json_decode(file_get_contents($templateBackupFile), true);
                    if (in_array($currentTemplate['name'], $existingTemplates)) {
                        Utils::CLIout("  [SKIPPING] {$currentTemplate['name']}");
                        continue;
                    }
                    
                    $result[$templateFile] = self::importAComponentTemplate($templateBackupFile, $upgrade);
                } // else skip this file, as it is not a proper json file
            }
        } else {
            Utils::CLIerror( 'Problem with Component Templates backup folder: ', true );
            Utils::CLIdest( $templatesDir, false );
            Utils::CLIerror( 'Component Templates Backup Folder DOES NOT exist or specified PATH is wrong.', false );
            
            $result = false;
        }
        
        return $result;
    }
    
    /**
     * Import / Upgrade a ComponentTemplate to UrbanCode Server.
     * Will validate the provided $componentTemplate file
     * and will delegate its IMPORT / UPGRADE processing to
     * the EntryPoint: importComponentTemplate request.
     * 
     * @param string  $componentTemplate Full Path to a ComponentTemplate JSON file
     * @param boolean $upgrade          Boolean TRUE / FALSE flag, used to set whether to
     *                                  UPGRADE or IMPORT the Template
     * 
     * @return boolean|array    Array holding the result from ComponentTemplate IMPORT / UPGRADE process or
     *                          Boolean FALSE in case of an error.
     */
    public function importAComponentTemplate ($componentTemplate, $upgrade = false){
        $result     = array();
        $success    = true;
        $resultType = $upgrade ? 'upgrade_result' : 'import_result';
        
        // Prompt start of action
        Utils::CLIinfo( ($upgrade ? 'Upgrade' : 'Import') . ' Component Template from:', true );
        Utils::CLIdest($componentTemplate, false );
        
        // Validate file
        if ( is_file($componentTemplate) ) {
            // Get file contents
            $fileContents = file_get_contents($componentTemplate);
            
            if ( Utils::validateJSON($fileContents) ) {
                
                $result[$resultType] = $this->rest()->component()->importComponentTemplate($componentTemplate, $upgrade);
                
                if ( $result[$resultType] === false ) {
                    // There was some problem - error logs are handled in the execRest() call
                    $success = false;
                } // else request was process OK -=> just continue as usual
            } else {
                Utils::CLIerror('Specified Template JSON File is not a valid json file.', true );
                $this->log->error('Specified ComponentTemplate JSON file: "'
                        . $componentTemplate . '" is NOT a valid JSON file');
                $result = false;
            }
        } else {
            Utils::CLIerror('Specified Template File is not a valid file.', true );
            $this->log->error('Specified ComponentTemplate file: "'
                    . $componentTemplate . '" is NOT a valid file');
            $result = false;
        }
        
        // Prompt END of action with SUCCESS / FAIL result
        if ($result !== false && $success !== false) {
            Utils::CLIinfo('Component Template '
                    . ($upgrade ? 'UPRGADED' : 'IMPORTED')
                    . ' - SUCCESS', false );
        } else {
            Utils::CLIwarning('Component Template '
                    . ($upgrade ? 'UPGRADE' : 'IMPORT')
                    . ' - FAILED to complete', true );
            Utils::CLIinfo('For more details, please check: "' . $this->log->getLogFile() .'"', false );
        }
        
        // Return
        return $result;
    }
    
    public function restoreResource($resource, $upgrade = false) {
        // Prompt start of action
        Utils::CLIinfo( ($upgrade ? 'Upgrade' : 'Create') . ' Resource from:', true );
        Utils::CLIdest( $resource, false );
        
        if (!file_exists($resource) || is_dir($resource)) {
            Utils::CLIerror("File set as resource doesn't exist or is not accessible.");
            return false;
        }
        
        if ( $upgrade ) {
            $result = $this->rest()->resource()->upgradeResource($resource);
        } else {
            $result = $this->udclient()->resource()->createResource($resource);
        }
        
        if ( $result === false ) {
            Utils::CLIwarning('Resource ' . ($upgrade ? 'UPGRADE' : 'CREATE')
                    . ' process - FAILED to COMPLETE', true );
            Utils::CLIinfo('For more details, please check: ' . $this->log->getLogFile(), false );
        } else {
            Utils::CLIinfo( 'Resource ' . ($upgrade ? 'UPGRADE' : 'CREATE') . ' process completed - SUCCESS', true );
        }
        
        return $result;
    }
    
    public function restoreAllResources($resourcesDir, $upgrade = false) {
        
        // Prompt start of action
        Utils::CLIinfo( ' Restore All Resources from:', true);
        Utils::CLIdest( $resourcesDir, false );
        Utils::CLIout("  Checking source directory");
        $validJson = Utils::validateDirectoryJSON( $resourcesDir );
        
        if (!$validJson) {
            Utils::CLIerror("Source directory didn't pass json validation check");
            return false;
        }
        
        if ( is_dir( $resourcesDir ) ) {
            $result = array();
            $resources = scandir($resourcesDir);
            
            // Get existing resources
            $existingResources = array();
            $existingRes = Utils::outputToArray($this->udclient()->resource()->getResources());
            foreach ($existingRes as $res) {
                array_push($existingResources, $res['name']);
            }
            
            
            foreach ($resources as $resource) {
                // Process ONLY 'ComponentTemplate.json' files
                if ( strpos($resource, '.json') !== false && $resource != '.json' ) {
                    $resourceFile = $resourcesDir . '/' . $resource;
                    $resourceContent = json_decode(file_get_contents($resourceFile), true);
                    
                    // Check if resource exists
                    if (in_array($resourceContent['name'], $existingResources) && !$upgrade) {
                        Utils::CLIout("  [SKIPPING] {$resourceContent['name']}");
                        continue;
                    }
                    
                    $result[$resourceContent['name']] = self::restoreResource($resourceFile, $upgrade);
                    
                    if ($result[$resourceContent['name']] === false) {
                        Utils::CLIout("  [FAILURE] {$resourceContent['name']}");
                    } else {
                        Utils::CLIout("  [SUCCESS] {$resourceContent['name']}");
                    }
                    
                } // else skip this file, as it is not a proper json file
            }
        } else {
            Utils::CLIerror( 'Problem with Resources backup folder: ', true );
            Utils::CLIdest( $resourcesDir, false );
            Utils::CLIerror( 'Resources Backup Folder DOES NOT exist or specified PATH is wrong.', false );
            
            $result = false;
        }
        return $result;
    }
    
    public function restoreBlueprints($blueprintsDir) {
        if ( is_dir( $blueprintsDir ) ) {
            $result = array();
            $apps = scandir($blueprintsDir);
            foreach ($apps as $app) {
                if (!is_dir("{$blueprintsDir}/{$app}") || $app == "." || $app == "..") {
                    continue;
                }
                $output = $this->udclient()->application()->getApplication($app);
                if ($output !== false) {
                    $application = Utils::outputToArray($output);
                    $appId =  $application['id'];
                    
                    $result[$app] = $this->restoreAllBlueprints("{$blueprintsDir}/{$app}",$appId);
                } else {
                    Utils::CLIwarning("Failed to restore blueprints from {$blueprintsDir}/{$app}");
                }
            }
        }
        return $result;
    }
    
    /**
     * 
     * Creates a new blueprint associated to the application set as argument
     *
     * @param string $applicationId
     * @param string $blueprintName
     * @param string $templateId
     * @param string $description
     * @return boolean
     */
    public function restoreBlueprint($applicationId, $blueprintName, $templateId, $description = '') {
        $error = false;
        if (empty($applicationId)) {
            Utils::CLIerror("Application ID required");
            $error = true;
        }
        if (empty($blueprintName)) {
            Utils::CLIerror("Blueprint name required");
            $error = true;
        }
        if (empty($templateId)) {
            Utils::CLIerror("Template ID required");
            $error = true;
        }
        if ($error) {
            return false;
        }
        $json = array(
            'applicationId' => $applicationId,
            'name' => $blueprintName,
            'resourceTemplateId' => $templateId,
            'description' => $description,
        );
        $json = Utils::createJsonFile($json);
        $output = $this->rest()->resource()->createBlueprint($json);
        $result = Utils::outputToArray($output);
        if ($result !== false) {
            Utils::CLIinfo("Creation of blueprint '{$blueprintName}' - SUCCESS");
        } else {
            Utils::CLIinfo("Creation of blueprint '{$blueprintName}' faliled");
        }
        unlink($json);
        return $result;
    }
    
    public function restoreAllBlueprints ($blueprintsDir, $applicationId) {
        Utils::CLIinfo( 'RESTORE UrbanCode blueprints in:', true );
        Utils::CLIdest( $blueprintsDir, false );
        
        Utils::CLIout("  Checking source directory");
        $validJson = Utils::validateDirectoryJSON( $blueprintsDir );
        
        if (!$validJson) {
            Utils::CLIerror("Source directory didn't pass json validation check");
            return false;
        }
        
        $files = scandir($blueprintsDir);
        $result = array();
        
        // Get existing blueprints
        $blueprints = Utils::outputToArray($this->rest()->resource()->getBlueprintsInApplication($applicationId));
        $existingBlueprints = array();
        foreach ($blueprints as $blueprint) {
            array_push($existingBlueprints, $blueprint['name']);
        }
        
        foreach ($files as $file) {
            if ( $file == "." || $file == ".." || is_dir("{$blueprintsDir}/{$file}")) {
                continue;
            }
            $filePath = $blueprintsDir . "/" . $file;
            if (Utils::validateJSON( file_get_contents($filePath) ) ) {
                $blueprint = json_decode( file_get_contents($filePath), true );
                
                // Chek if blueprint already exists
                if (in_array($blueprint['name'], $existingBlueprints)) {
                    Utils::CLIout("  [SKIPPING] {$blueprint['name']}");
                    continue;
                }
                
                $blueprintName = $blueprint['name'];
                $description = $blueprint['description'];
                
                // Get template ID from web server
                $resourceTemplates = $this->rest()->resource()->getResourceTemplates();
                $resourceTemplates = Utils::outputToArray($resourceTemplates);
                foreach ($resourceTemplates as $resource) {
                    if ($resource['name'] == $blueprint['template']['name']) {
                        $templateId = $resource['id'];
                        break;
                    }
                }
                if (!empty($templateId)) {
                    $result[$blueprintName] = self::restoreBlueprint($applicationId, $blueprintName, $templateId, $description);
                    if ($result[$blueprintName] === false) {
                        Utils::CLIout("  [FAILURE] {$blueprintName}");
                        continue;
                    } else {
                        Utils::CLIout("  [SUCCESS] {$blueprintName}");
                    }
                    
                    if (array_key_exists('tags',$blueprint)) {
                        foreach ($blueprint['tags'] as $tag) {
                            $tagFileContent['ids'] = array($result[$blueprintName]['id']);
                            $tagFileContent['name'] = $tag['name'];
                            $tagFileContent['objectType'] = 'Resource';
                            $tagFile = Utils::createJsonFile($tagFileContent);
                            
                            $result["{$blueprintName}_tags"] = $this->rest()->system()->createTag($tagFile);
                            
                            unlink($tagFile);
                        }
                    }
                } else {
                    Utils::CLIwarning("Cannot find resource template {$blueprint['template']['name']} in destinations server when restoring blueprint '{$blueprintName}'");
                    $result[$blueprintName] = false;
                }
            } else {
                Utils::CLIinfo("Skipping file {$file}. Not a valid JSON file.");
            }
        }
        
        // NOT IMPLEMENTED ON PURPOSE - Task 53661
        /*
        foreach ($result as $resultBlueprintName => $resultBlueprint )  {
            $resourceFile = "{$blueprintsDir}/resources/{$resultBlueprintName}.resources.json";
            if (file_exists($resourceFile)) {
                Utils::CLIinfo("Restoring file {$resourceFile}");
                // Get complete blueprint resources structure
                $blueprintParent = $this->getBlueprintResources($resultBlueprint['id']);
                $blueprintParent = Utils::outputToArray($blueprintParent);
                // Restore blueprint resources
                $resourceFile = json_decode(file_get_contents($resourceFile),true);
                $output = self::restoreAllBlueprintResources($resourceFile,$resultBlueprint['id'],$blueprintParent);
                $result[$resultBlueprintName]['resources'] = $output;
            }
        }
        */
        return $result;
    }
    
    /**
     * 
     * Create resource template structure from backup file
     *
     * @param string $resourceTemplateFile
     * @return array|boolean
     */
    public function restoreResourceTemplate($resourceTemplateFile, $resourceDir = '') {
        if (!file_exists($resourceTemplateFile) || is_dir($resourceTemplateFile)) {
            Utils::CLIerror("File {$resourceTemplateFile} doesn't exist or cannot be read.");
            return false;
        }
        Utils::CLIinfo( 'RESTORE UrbanCode resource templates in:', true );
        Utils::CLIdest( $resourceTemplateFile, false );
        $result = array();
        
        // Get existing resource templates
        $templates = Utils::outputToArray($this->rest()->resource()->getResourceTemplates());
        $existingTemplates = array();
        foreach ($templates as $template){
            array_push($existingTemplates, $template['name']);
        }
        
        $jsonTemplateFile = file_get_contents($resourceTemplateFile);
        if ( Utils::validateJSON($jsonTemplateFile) ) {
            $resourceTemplates = json_decode($jsonTemplateFile, true );
            foreach ($resourceTemplates as $template) {
                // Check if the resource template already exists
                if (in_array($template['name'], $existingTemplates)){
                    Utils::CLIout("  [SKIPPING] {$template['name']}");
                    continue;
                }
                
                // Create main resource template
                Utils::CLIout("  [RESOURCE] {$template['name']}");
                
                $fileContent['name'] = $template['name'];
                $fileContent['description'] = $template['resourceTemplate']['description'];
                $file = Utils::createJsonFile($fileContent);
                $output = $this->rest()->resource()->createResourceTemplate($file);
                // Remove temporary file
                unlink($file);
                if ( $output !== false ) {
                    $result[$template['name']] = Utils::outputToArray($output);
                    
                    // Get resource template so to get the correct resource ID
                    $output = $this->rest()->resource()->getResourceTemplate($result[$template['name']]['id']);
                    $output = Utils::outputToArray($output);
                    
                    // Scan and create children
                    if (array_key_exists('children',$template)) {
                        $children = $template['children'];
                        $parentId = $output[0]['id'];
                        $result[$template['name']]['children'] = self::restoreResourceTemplateChild($children, $parentId, $resourceDir);
                    }
                } else {
                    $result[$template['name']] = false;
                    Utils::CLIwarning("Failed to create resource template {$template['name']}. Skipping all related children (if any).");
                }
            }
        }
        
        Utils::CLIout("");
        return $result;
    }
    
    /**
     * 
     * Recursively scan the resource template array so to create all resource children 
     *
     * @param array $resourceTemplateArray
     * 
     * @return array|boolean
     */
    public function restoreResourceTemplateChild($resourceTemplateArray, $parentId, $resourceDir = '') {
        $result = array();
        foreach ($resourceTemplateArray as $child) {
            // If component, skip: it will be created afterwards
            if (array_key_exists('role',$child)
                && array_key_exists('specialType',$child['role'])
                && $child['role']['specialType'] == 'COMPONENT')
            {
                continue;
            }
            // Create temporary template child file
            $fileContent['name'] = $child['name'];
            $fileContent['description'] = $child['description'];
            $fileContent['parentId'] = $parentId;
            
            // Is child an agent placeholder?
            if (array_key_exists('role',$child) 
                && array_key_exists('specialType', $child['role']) 
                && $child['role']['specialType'] == "AGENT_PLACEHOLDER") 
            {
                // Agent Placeholder
                $fileContent['agentPlaceholder'] = true;
                $outputType = '    [AGENT]';
                
            } else {
                // Resource group
                $fileContent['inheritTeam'] = $child['inheritTeam'];
                $outputType = '   [GROUP]';
            }
            
            $file = Utils::createJsonFile($fileContent);
            $output = $this->rest()->resource()->createResourceTemplateChild($file);
            
            if (is_file($file)) {
                // Remove temporary file
                unlink($file);
            }
            if ( $output !== false ) {
                
                Utils::CLIout("{$outputType} {$child['name']}");
                $result[$child['name']] = Utils::outputToArray($output);
                
                // Assign tags
                foreach ($child['tags'] as $tag) {
                    $tagFileContent['ids'] = array($result[$child['name']]['id']);
                    $tagFileContent['name'] = $tag['name'];
                    $tagFileContent['objectType'] = 'Resource';
                    $tagFile = Utils::createJsonFile($tagFileContent);
                    
                    $result["{$child['name']}_tags"] = $this->rest()->system()->createTag($tagFile);
                    
                    unlink($tagFile);
                }
                
                if (array_key_exists('agentPlaceholder',$fileContent) && 
                    $fileContent['agentPlaceholder']) {
                    
                    if (empty($resourceDir)) {
                        $resourceDir = "{$this->source}/resourceTemplates/resources";
                    }
                    $file = "{$resourceDir}/{$child['id']}.json";
                    $resources = json_decode(file_get_contents($file),true);
                    self::restoreResourceTemplateResources($resources, $result[$child['name']]['id']);
                }
                // Check if there are children
                if (array_key_exists('children',$child)) {
                    $children = $child['children'];
                    $childParentId = $result[$child['name']]['id'];
                    $result[$child['name']]['children'] = self::restoreResourceTemplateChild($children, $childParentId, $resourceDir);
                }
            } else {
                $result[$child['name']] = false;
                Utils::CLIwarning("Failed to create resource template {$child['name']}. Skipping all related children (if any).");
            }
        }
        
        return $result;
    }
    
    public function restoreResourceTemplateResources($resources, $parentId) {
        foreach ($resources as $resource) {
            if (array_key_exists('role',$resource)
                && array_key_exists('specialType',$resource['role'])
                && $resource['role']['specialType'] == 'COMPONENT') 
            {
                $output = $this->rest()->resource()->getResourceRole($resource['name']);
                $output = Utils::outputToArray($output);
                if (!empty($output) && $output !== false) {
                    $tempFileContent['roleId'] = $output[0]['id'];
                    $tempFileContent['description'] = $resource['description'];
                    $tempFileContent['name'] = $resource['name'];
                    $tempFileContent['parentId'] = $parentId;
                    Utils::CLIout("     [COMPONENT] {$resource['name']}");
                    $tempFile = Utils::createJsonFile($tempFileContent);
                    $result[$resource['name']] = $this->rest()->resource()->createResourceTemplateChild($tempFile);
                    unlink($tempFile);
                    
                    // Assign tags
                    foreach ($resource['tags'] as $tag) {
                        $tagFileContent['ids'] = array($result[$resource['name']]['id']);
                        $tagFileContent['name'] = $tag['name'];
                        $tagFileContent['objectType'] = 'Resource';
                        $tagFile = Utils::createJsonFile($tagFileContent);
                    
                        $result["{$resource['name']}_tags"] = $this->rest()->system()->createTag($tagFile);
                    
                        unlink($tagFile);
                    }
                }
            }
        }
    }
    
    /**
     * 
     * restore all resource templates from a directory
     *
     * @param string $templatesDir
     * 
     * @return boolean|array
     */
    public function restoreAllResourceTemplates($templatesDir) {
        Utils::CLIinfo("Restoring Resource Templates from {$templatesDir}");
        
        if ( is_dir( $templatesDir ) ) {
            Utils::CLIout("  Checking source directory");
            $validJson = Utils::validateDirectoryJSON( $templatesDir );
            
            if (!$validJson) {
                Utils::CLIerror("Source directory didn't pass json validation check");
                return false;
            }
            
            $result = array();
            $templatesFiles = scandir($templatesDir);
            
            foreach ($templatesFiles as $file) {
                if ( $file == '.' || $file ==  '..' || is_dir("{$templatesDir}/{$file}")) {
                    continue;
                }
                $filePath = "{$templatesDir}/{$file}";
                $templateContent = json_decode(file_get_contents($filePath), true);
                
                $result[$file] = self::restoreResourceTemplate($filePath);
            }
            
        } else {
            Utils::CLIerror( 'Problem with Resource Templates backup folder: ', true );
            Utils::CLIdest( $templatesDir, false );
            Utils::CLIerror( 'Resource Templates Backup Folder DOES NOT exist or specified PATH is wrong.', false );
            
            $result = false;
        }
        
        return $result;
    }
    
    /**
     * Create / Upgrade Application Process(es), based on Application backup File.
     * This function will read the Application backup file and use it as an Object,
     * in order to retrieve the Application Process(es) to be restored.
     * 
     * NOTE: $applicationID should refer to the Application on the DESTINATION Server,
     *       and it might differ from the one in the specified $applicationBackupFile.
     * 
     * @param string  $applicationID         String holding the Application Name or ID, or the Application,
     *                                       which Processes are goingt to be CREATED / UPGRADED
     * @param string  $applicationBackupFile String holding the PATH to an Application backup file,
     *                                       which will be used to obtain the Application-Process(es)
     *                                       to be CREATED/UPGRADED
     * @param boolean $upgrade               Boolean Flag, used to define, wether to UPGRADE or CREATE
     *                                       Application Process(es).
     *                                       DEFAULT value: FALSE (CREATE)
     * 
     * @return mixed Array holding the result from each Application Process CREEATE/UPGRADE action.
     *               Boolean FALSE in case of a Problem with reading the specified $applicationBackupFile,
     *               or any other error.
     *               NULL if there was NO Processes to be CREATED/UPGRADED for the specified Application.
     *               
     */
    public function restoreAllApplicationProcesses($applicationID, $applicationBackupFile, $upgrade = false) {
        
        $result = array();
        
        // Set Action: UPGRADE or CREATE
        $actionType = $upgrade ? 'UPGRADE' : 'CREATE';
        
        Utils::CLIinfo( $actionType . ' UrbanCode Application Processes from Application backup file:', true );
        Utils::CLIdest( $applicationBackupFile, false );
        
        /*
         * Read Application backup File. restoreAllApplicationProcesses
         * NOTE: this MIGHT NOT hold the correct Application ID
         */
        $appData = file_get_contents($applicationBackupFile);
        // There was problem reading the Application backup file
        if ($appData === false) {
            Utils::CLIerror('There was a problem with reading of Application backup file');
            Utils::CLIerror('Application processes CANNOT be RESTORED', false);
            $result = false;
        // OK to continue with App-Process(es) restore
        } else {
            $appObject = Utils::getAsObject($appData);
            // There was a problem
            if (is_null($appObject)) {
                Utils::CLIerror('Application data COULD NOT be transfered to an Object');
                Utils::CLIerror('File might be corrupted or NOT in CORRECT JSON format', false);
                $result = false;
            // OK to continue with App-Process(es) restore
            } else {
                if (isset($appObject->processes) && count($appObject->processes) > 0) {
                    foreach($appObject->processes as $appProcess) {
                        /* 
                         * Add Application ID to the $appProcess data.
                         * This is required for the createApplicationProcess request.
                         * USE $applicationID, as the $appObject->id as read from the
                         * $applicationBackupFile might differ on the DEST Server
                         */
                        if (!isset($appProcess->application)) {
                            $appProcess->application = $applicationID;
                        } else {
                            $appProcess->application = $applicationID;
                        }
                        // Save temporary AppProcess file
                        $processTmpFile = Utils::createJsonFile($appProcess);
                        // There was problem with saving of file
                        if ( $processTmpFile === false) {
                            Utils::CLIwarning('FAILED to store Application Process: "'
                                    . $appProcess->name . '" temporary file');
                            $result[$appProcess->name] = false;
                        // Create / upgrade Application Process
                        } else {
                            Utils::CLIinfo('Application Process: "' . $appProcess->name . '" temporary file CREATED');
                            Utils::CLIinfo('Proceed with Application Process ' . ($upgrade? 'UPGRADE' : 'CREATE')
                                    . ' operation', false);
                            $result[$appProcess->name] = self::restoreApplicationProcess($processTmpFile, $upgrade);
                            // Delete $processTmpFile only in case of success
                            if ($result[$appProcess->name] !== false) {
                                unlink($processTmpFile);
                            } else {
                                // Prompt location of $processTmpFile for further debugging
                                Utils::CLIinfo('Application Process: "' . $appProcess->name . '" temporary file located in:');
                                Utils::CLIdest($processTmpFile, false);
                            }
                        }
                    }
                } else {
                    Utils::CLIinfo('There is NO Application Process(es) to be ' . $actionType . 'D');
                    $result = null;
                }
            }
        }
        
        return $result;
    }
    
    public function restoreApplicationProcess($processBackupFile, $upgrade = false) {
        $result     = array();
        $success    = true;
        $resultType = $upgrade ? 'upgrade_result' : 'create_result';
        
        // Prompt start of action
        Utils::CLIinfo( ($upgrade ? 'Upgrade' : 'Create') . ' Application Process from:', true );
        Utils::CLIdest($processBackupFile, false );
        
        // Validate file
        if ( is_file($processBackupFile) ) {
            // Get file contents
            $fileContents = file_get_contents($processBackupFile);
        
            if ( Utils::validateJSON($fileContents) ) {
                // Upgrade App-Process
                if (!$upgrade) {
                    $result[$resultType] = $this->udclient()->application()->createApplicationProcess($processBackupFile);
                }
                
                // Validate Result
                if ( $result[$resultType] === false ) {
                    // There was some problem - error logs are handled in the exec() call
                    $success = false;
                } // else CREATE / UPGRADE went OK - messages will be handled later
            } else {
                Utils::CLIerror('Specified Application Process JSON File is not a valid json file.', true );
                $this->log->error('Specified Application Process JSON file: "'
                        . $processBackupFile . '" is NOT a valid JSON file');
                $result[$resultType] = false;
            }
        } else {
            Utils::CLIerror('Specified Application Process File is not a valid file.', true );
            $this->log->error('Specified Application Process file: "'
                    . $processBackupFile . '" is NOT a valid file');
            $result[$resultType] = false;
        }
        
        // Prompt END of action with SUCCESS / FAIL result
        if ($result[$resultType] !== false && $success !== false) {
            Utils::CLIinfo('Application Process ' . ($upgrade ? 'UPRGADED' : 'CREATED')
                    . ' - SUCCESS', false );
        } else {
            Utils::CLIwarning('Application Process ' . ($upgrade ? 'UPGRADE' : 'CREATE')
                    . ' - FAILED to complete', true);
            Utils::CLIinfo('For more details, please check: "'
                    . $this->log->getLogFile() .'"', false );
        }
        
        // Return
        return $result;
    }
    
    /**
     * 
     * restores all post processing scripts from exported file
     *
     * @param unknown $file
     * @return boolean|array
     */
    public function restorePostProcessingScripts($settingsDir) {
        Utils::CLIinfo("Restoring Post Processing Scritps from {$settingsDir}");
        $result = array();
        
        // Check directory
        if (is_dir($settingsDir)) {
            Utils::CLIout("  Checking source directory");
            $validJson = Utils::validateDirectoryJSON( $settingsDir );
            
            if (!$validJson) {
                Utils::CLIerror("Source directory didn't pass json validation check");
                return false;
            }
            
            // Get existing post-processing scripts
            $pps = Utils::outputToArray($this->rest()->process()->getPostProcessingScripts());
            $existingScripts = array();
            foreach ($pps as $ppscript) {
                array_push($existingScripts, $ppscript['name']);
            }
            
            $files = scandir($settingsDir);
            //Scan directory for files
            foreach ($files as $file) {
                if ($file == '.' || $file == '..') {
                    continue;
                }
                $file = "{$settingsDir}/{$file}";
                if (is_file($file)) {
                    // Get file contents
                    $fileContents = file_get_contents($file);
                    
                    if ( Utils::validateJSON($fileContents) ) {
                        $scripts = json_decode($fileContents,true);
                        foreach ($scripts as $script) {
                            // Check if script already exists
                            if (in_array($script['name'], $existingScripts)) {
                                Utils::CLIout("  [SKIPPING] {$script['name']}");
                                continue;
                            }
                            
                            // Restore scripts
                            $tempfile = Utils::createJsonFile($script);
                            $output = $this->rest()->process()->createPostProcessingScript($tempfile);
                            unlink($tempfile);
                            if ($output === false) {
                                Utils::CLIout("  [FAILURE] {$script['name']}");
                                $result[$file][$script['name']] = false;
                            } else {
                                Utils::CLIout("  [SUCCESS] {$script['name']}");
                                $result[$file][$script['name']] = Utils::outputToArray($output);
                            }
                        }
                    } else {
                        Utils::CLIwarning("File {$file} is not a valid JSON file. Skipping.", true );
                    }
                } else {
                    Utils::CLIerror('Specified Post Processing Scripts File is not a valid file.', true );
                    $this->log->error('Specified Post Processing Scripts file: "'
                            . $file . '" is NOT a valid file');
                    $result[$file] = false;
                }
            }
        } else {
            Utils::CLIerror('Specified settings directory is not valid.', true );
            $this->log->error('Specified settings directory is not valid: "'
                    . $$settingsDir . '" is NOT a valid file');
            $result = false;
        }
        
        return $result;
    }
    
    /**
     * 
     * Add resources to blueprint, getting info from exported blueprint resource file
     *
     * @param string $blueprintId
     * @param string $name
     * @param string $parentId
     * @param string $roleId
     * @param string $description
     * @param array $roleProperties
     * @param array $teamMappings
     * @param boolean $useImpersonation
     */
    public function restoreBlueprintResource ($blueprintId,$name,$parentId,$roleId,$description='',$roleProperties=null,$teamMappings=null,$useImpersonation=false) {
        // Setup content for json file
        $content['blueprintId'] = $blueprintId;
        $content['name'] = $name;
        $content['parentId'] = $parentId;
        $content['roleId'] = $roleId;
        $content['description'] = $description;
        if (!empty($roleProperties)) {
            $content['roleProperties'] = json_decode($roleProperties,true);
        } else {
            $content['roleProperties'] = '';
        }
        if (!empty($teamMappings)) {
            $content['teamMappings'] = json_decode($teamMappings,true);
        } else {
            $content['teamMappings'] = '';
        }
        $content['useImpersonation'] = $useImpersonation;
        
        // Create json file for blueprint resource
        $file = Utils::createJsonFile($content);
        $result = $this->rest()->resource()->createBlueprintResource($file);
        unlink($file);
        
        return $result;
    }
    
    public function restoreAllBlueprintResources ($blueprintResources, $blueprintId, $parentBlueprint) {
        $result = array();
        $count = 0;
        foreach ($blueprintResources as $resource) {
            // Check if any child
            if (array_key_exists('children', $resource)) {
                // Loop children, looking for resource to be created
                foreach ($resource['children'] as $children) {
                    if (array_key_exists('role',$children)
                        && array_key_exists('specialType',$children['role'])
                        && $children['role']['specialType'] == 'COMPONENT') 
                    {
                        $output = $this->rest()->resource()->getResourceRole($children['name']);
                        $output = Utils::outputToArray($output);
                        if (!empty($output) && $output !== false) {
                            $roleId = $output[0]['id'];
                        } else {
                            $roleId = '';
                        }
                        $result[$resource['name']]['children'][$children['name']] = 
                            self::restoreBlueprintResource(
                                    $blueprintId,
                                    $children['name'],
                                    $parentBlueprint[$count]['id'],
                                    $roleId,
                                    $resource['description']
                            );
                    }
                }
                // If parent doesn't have children, then continue with resource one 
                if (!array_key_exists('children', $parentBlueprint[$count])) {
                    $parent = $resource['children'];
                } else {
                    $parent = $parentBlueprint[$count]['children'];
                }
                
                $result[$resource['name']]['children'] = self::restoreAllBlueprintResources($resource['children'], $blueprintId, $parent);
                
                $count++;
            }
        }
        
        return $result;
    }
    
    public function restoreAllTags( $dir ) {
        Utils::CLIinfo( "Restoring tags from {$dir}" );
        
        $result = array();
        $files = scandir($dir);
        
        // Retrieve existing tags
        $tagsComponent = Utils::outputToArray($this->rest()->system()->getTag('component'));
        $tagsResource = Utils::outputToArray($this->rest()->system()->getTag('resource'));
        
        $existingTags['component'] = array();
        $existingTags['resource'] = array();
        
        foreach ($tagsComponent as $tag) {
            array_push($existingTags['component'], $tag['name']);
        }
        
        foreach ($tagsResource as $tag) {
            array_push($existingTags['resource'], $tag['name']);
        }
        
        
        foreach ($files as $file) {
            if ($file != '.' && $file != '..') {
                $filePath = "{$dir}/{$file}";
                $json = file_get_contents($filePath);
                if (! Utils::validateJSON($json)) {
                    Utils::CLIwarning("File {$filePath} is not a valid JSON file");
                    continue;
                }
                $tags = json_decode($json, true);
                foreach ($tags as $tag) {
                    // Check if tag already exist
                    $type = strtolower($tag['objectType']);
                    if (in_array ($tag['name'], $existingTags[$type])) {
                        Utils::CLIout("  [SKIPPING] {$tag['name']}");
                        continue;
                    }
                    
                    $tag['url'] = "/rest/tag/{$tag['objectType']}";
                    $tag['ids'] = array();
                    $tagFile = Utils::createJsonFile($tag);
                    $result[$tag['name']] = $this->rest()->system()->createTag($tagFile);
                    
                    if ($result[$tag['name']] !== false) {
                        Utils::CLIout("  [SUCCESS] {$tag['name']}");
                    } else {
                        Utils::CLIout("  [FAILURE] {$tag['name']}");
                    }
                    
                    unlink($tagFile);
                }
                
                Utils::CLIout("");
            }
        }
        return $result;
    }
    
    public function restoreAllProcesses( $dir, $upgrade = false ) {
        Utils::CLIinfo( "Restoring processes from {$dir}" );
        
        $result = array();
        $files = scandir($dir);
        
        // Retrieve existing processes
        $processes = Utils::outputToArray($this->rest()->process()->getProcesses());
        $existingProcesses = array();
        
        foreach ($processes as $process) {
            array_push($existingProcesses, $process['name']);
        }
        
        foreach ($files as $file) {
            if ($file != '.' && $file != '..' && !is_dir("{$dir}/{$file}")) {
                $filePath = "{$dir}/{$file}";
                $json = file_get_contents($filePath);
                if (! Utils::validateJSON($json)) {
                    Utils::CLIwarning("File {$filePath} is not a valid JSON file");
                    continue;
                }
                $process = json_decode($json, true);
                $doUpgrade = false;
                
                if (in_array($process['name'], $existingProcesses)) {
                    if ( !$upgrade ) {
                        Utils::CLIout("  [SKIPPING] {$process['name']}");
                        continue;
                    } else {
                        $doUpgrade = true;
                    }
                }
                
                $result[$process['name']] = $this->importProcess($filePath, $doUpgrade);
                
                if ($result[$process['name']] !== false) {
                    Utils::CLIout("  [SUCCESS] {$process['name']}");
                } else {
                    Utils::CLIout("  [FAILURE] {$process['name']}");
                }
            }
        }
        
        return $result;
    }
    
    /**
     * 
     * Assing a team to all components in uDeploy
     *
     * @param string $team
     * @return boolean
     */
    public function addTeamToAllComponents($team) {
        $teams = Utils::outputToArray($this->rest()->system()->getTeams());
        
        $teamExists = false;
        foreach ($teams as $singleTeam) {
            if ( $singleTeam['name'] == $team || $singleTeam['id'] == $team ) {
                $teamExists = true;
            }
        }
        
        if (!$teamExists) {
            Utils::CLIerror("Team {$team} doesn't exist");
            return false;
        }
        
        $components = $this->udclient()->component()->getComponents();
        $components = Utils::outputToArray($components);
        
        $result = true;
        foreach ($components as $component) {
            $success = $this->udclient()->component()->addComponentToTeam($component['id'], $team);
            if ($success === false) {
                Utils::CLIout("  [FAILURE] {$component['name']}");
                $result = false;
            } else {
                Utils::CLIout("  [SUCCESS] {$component['name']}");
            }
        }
        
        return $result;
    }
    
    /**
     * 
     * Assign a team to all resources in uDeploy
     *
     * @param string $team
     * @return boolean
     */
    public function addTeamToAllResources($team) {
        $teams = Utils::outputToArray($this->rest()->system()->getTeams());
        
        $teamExists = false;
        foreach ($teams as $singleTeam) {
            if ( $singleTeam['name'] == $team || $singleTeam['id'] == $team ) {
                $teamExists = true;
            }
        }
        
        if (!$teamExists) {
            Utils::CLIerror("Team {$team} doesn't exist");
            return false;
        }
        
        // Apply to all resources
        $resources = $this->udclient()->resource()->getResources();
        $resources = Utils::outputToArray($resources);
        
        $result = true;
        foreach ($resources as $resource) {
            $success = $this->udclient()->resource()->addResourceToTeam($resource['id'], $team);
            if ($success === false) {
                Utils::CLIout("  [FAILURE] {$resource['name']}");
                $result = false;
            } else {
                Utils::CLIout("  [SUCCESS] {$resource['name']}");
            }
        }
        
        return $result;
    }
    
    public function upgradeAllApplications( $applicationsBackupsDir ) {
        Utils::CLIinfo( 'Upgrade UrbanCode Applications from:', true );
        Utils::CLIdest( $applicationsBackupsDir, false );
        
        Utils::CLIout("  Checking source directory");
        $validJson = Utils::validateDirectoryJSON( $applicationsBackupsDir );
        
        if (!$validJson) {
            Utils::CLIerror("Source directory didn't pass json validation check");
            return false;
        }
        
        if ( is_dir( $applicationsBackupsDir ) ) {
            $result = array();
            
            $appsDirs = scandir($applicationsBackupsDir);
            foreach ($appsDirs as $subdir) {
                // Skip: '.' and '..' pointers
                if ( !in_array($subdir, array('.', '..')) ) {
                    $subdir = "{$applicationsBackupsDir}/{$subdir}";
                    
                    if ( is_dir($subdir) ) {
                        $result[$subdir] = self::restoreApplication($subdir, true);
                    }
                }
            }
        } else {
            Utils::CLIerror( 'Problem with Applications backup folder: ', true );
            Utils::CLIdest( $applicationsBackupsDir, false );
            Utils::CLIerror( 'Backup Folder DOES NOT exist or specified PATH is wrong.', false );
            $result =  false;
        }
        
        return $result;
    }
}