<?php
/**
 * **************************************************************
 * IBM Confidential
 *
 * Collaboration Source Materials
 *
 * (C) Copyright IBM Corp. 2014-2015
 *
 * The source code for this program is not published or otherwise
 * divested of its trade secrets, irrespective of what has been
 * deposited with the U.S. Copyright Office
 *
 * *************************************************************
 */
chdir(__DIR__);

require_once 'include/EntryPoint.php';

require_once 'include/Extensions/BackupUCD.php';
require_once 'include/Extensions/CleanUCD.php';
require_once 'include/Extensions/Fork.php';
require_once 'include/Extensions/Promote.php';
require_once 'include/Extensions/RestoreUCD.php';

require_once 'include/Layers/ApplicationProcess.php';
require_once 'include/Layers/Environment.php';
require_once 'include/Layers/Snapshot.php';

require_once 'include/Utils/Logger.php';
require_once 'include/Utils/Utils.php';

include 'config/ucd.config.php';

$return = array ();
$return_msg = '';

// Setup logger
$log = new Logger();
if (file_exists( 'config/logger.config.php' )) {
    include ('config/logger.config.php');
    $log->setLogFile( $config ['log'] );
    $log->setDebug( $config ['debug'] );
}

$valid_actions = array (
    'addTeamToComponent' => 'Add a team to one or more component',
    'addTeamToResource' => 'Add a team to one or more resource',
    'backup' => 'Run a complete back up of uDeploy.',
    'cleanAll' => 'Clean applications, components, processes and post-processing scripts.',
    'createEnvironment' => 'Create a new environment from a JSON file',
    'createSnapshot' => 'Create an application snapshot based on json file',
    'createVersion' => 'Create a new version for the component',
    'createVersionsAndSnapshot' => 'First create component versions, then add them into a snapshot',
    'exportEnvironmentProperties' => 'Export the environment variables of an environment',
    'forkApplication' => 'Creates a new copy of the application and it\'s contents replacing the suffix in the copy',
    'importEnvironmentProperties' => 'Import the environment variables of an application',
    'promoteApplication' => 'Promote a single application from a uDeploy origin server to a destination uDeploy server',
    'promoteConfig' => 'Promote (upgrade) configuration from a uDeploy origin server to a destination uDeploy server',
    'promoteSnapshot' => 'Promote a snapshot from a uDeploy origin server to a destination uDeploy server',
    'requestStatus' => 'Get status for a specified request ID',
    'restore' => 'Run a complete restore of uDeploy. Requires a backup source path as argument.',
    'restoreApplication' => 'Restore an application. Requires a path to a json file as argument.',
    'restoreComponent' => 'Restore a component. Requires a path to a json file as argument.',
    'restoreComponentTemplate' => 'Restore a ComponentTemplate. Requires a path to a json file as argument.',
    'restoreResourceTemplate' => 'Restore a resource template from a JSON file',
    'RTCbackup' => 'Cleanup RTC directory and backup files',
    'run' => 'Run an application process. Requires a path to a json file as argument.',
    'runAndWait' => 'Run a process and wait for completion',
    'setSnapshotStatus' => 'Set Snapshot status: SnapshotName Passed|Failed',
    'upgrade' => 'Run a complete upgrade of uDeploy. Requires a backup source path as argument',
    'upgradeApplication' => 'Upgrade an existing application. Requires a path to a json file or to a directory as argument.',
    'upgradeComponent' => 'Upgrade an existing component. Requires a path to a json file as argument.',
    'upgradeComponentTemplate' => 'Upgrade an existing ComponentTemplate. Requires a path to a json file as argument.',
    'ver' => 'Show current uCommand version and list of uDeploy versions compatibility',
    'waitRequestCompletion' => 'Wait for a specific application process request for completion',
);

// Prompt a start-up message to user
Utils::CLIout( "##################################" );
Utils::CLIout( "#            uCommand            #" );
Utils::CLIout( "# Urban Code Deploy PHP CLI Tool #" );
Utils::CLIout( "#    by SalesConnect - Dublin    #" );
Utils::CLIout( "##################################\n" );

// Check if arguments are passed
if (count( $argv ) == 1) {
    Utils::CLIerror( "Please provide a valid Action as Argument." );
    Utils::CLIout( "Usage: ", true );
    foreach ( $valid_actions as $action => $description ) {
        if ($action == "API") {
            continue;
        }
        Utils::CLIout( "\t{$action} => {$description}", true );
    }
    Utils::CLIout( "\n", true );
    exit( 1 );
}

// Valid arguments
switch ($argv [1]) {
    case 'addTeamToComponent' :
        if (empty( $argv [2] )) {
            Utils::CLIerror( "Missing team name/ID" );
            exit( 1 );
        }
        $team = $argv [2];
        
        if (empty( $argv [3] )) {
            $all = true;
        } else {
            $component = $argv [3];
        }
        
        $restore = new RestoreUCD();
        if ($all) {
            // Apply team to all components
            $result = $restore->addTeamToAllComponents( $team );
        } else {
            // Apply team to a single component
            $result = $restore->addComponentToTeam( $component, $team );
        }

        if ($result === false) {
            exit( 1 );
        }
        break;
    case 'addTeamToResource' :
        if (empty( $argv [2] )) {
            Utils::CLIerror( "Missing team name/ID" );
            exit( 1 );
        }
        $team = $argv [2];
    
        if (empty( $argv [3] )) {
            $all = true;
        } else {
            $resource = $argv [3];
        }
    
        $restore = new RestoreUCD();
        if ($all) {
            // Apply team to all resources
            $result = $restore->addTeamToAllResources( $team );
        } else {
            // Apply team to a single resource
            $result = $restore->addResourceToTeam( $resource, $team );
        }

        if ($result === false) {
            exit( 1 );
        }
        break;
    case "backup" :
        $backup = new BackupUCD();
        $return = $backup->backup();
        $config ['output'] = $backup->getOutput();
        $config ['json_check'] = true;
        break;
    case 'cleanAll' :
        $cleanUCD = new CleanUCD();
        $result = $cleanUCD->cleanAll();
        if ($result === false) {
            exit( 1 );
        }
        break;
    case 'createEnvironment' :
        if (empty( $argv [2] )) {
            Utils::CLIerror( "Missing environment JSON file" );
            exit( 1 );
        }
        
        $environment = new Environment();
        $result = $environment->create( $argv [2] );
        echo "";

        if ($result === false) {
            exit( 1 );
        }
        break;
    case 'createSnapshot' :
        $ucc = new EntryPoint();
        if (empty( $argv [2] )) {
            Utils::CLIerror( "Missing snapshot file." );
            exit( 1 );
        }
        
        $result = $ucc->udclient()->snapshot()->createSnapshot( $argv [2] );

        if ($result === false) {
            exit( 1 );
        }
        break;
    case 'createVersion' :
        $ucc = new EntryPoint();
        if (empty( $argv [2] )) {
            Utils::CLIerror( "Missing component name or ID." );
            exit( 1 );
        }
        if (empty( $argv [3] )) {
            Utils::CLIerror( "Missing version name." );
            exit( 1 );
        }
        
        if (isset( $argv [4] )) {
            $description = $argv [4];
        } else {
            $description = '';
        }
        
        $result = $ucc->udclient()->component()->createVersion( $argv [2], $argv [3], $description );

        if ($result === false) {
            exit( 1 );
        }
        break;
    case 'createVersionsAndSnapshot' :
        $snapshot = new Snapshot();
        if (empty( $argv [2] )) {
            Utils::CLIerror( "Missing configuration file." );
            exit( 1 );
        }
        
        if (! empty( $argv [3] ) && $argv [3] == 'force') {
            $force = true;
        } else {
            $force = false;
        }
        
        $result = $snapshot->createVersionsAndSnapshot( $argv [2], $force );
        Utils::CLIout( "\n" );
        if ($result === false) {
            exit( 1 );
        }
        break;
    case 'exportEnvironmentProperties' :
        
        $environment = new Environment();
        if (empty( $argv [2] )) {
            Utils::CLIerror( "Missing Environment id/name " );
            exit( 1 );
        }
        if (empty( $argv [3] )) {
            Utils::CLIwarning( "Missing the application name. " );
            Utils::CLIwarning( "The export method will retrieve the application name from the configuration " );
        }
        
        $result = $environment->exportProperties( $argv [2], $argv [3] );
        if ($result === false) {
            exit( 1 );
        }
        break;
    case 'forkApplication' :
        $isError = false;
        if (empty( $argv [2] )) {
            Utils::CLIerror( "Missing the application name" );
            $isError = true;
        }
        if (is_null( $argv [3] )) {
            Utils::CLIerror( "Missing the old suffix string" );
            $isError = true;
        }
        if (empty( $argv [4] )) {
            Utils::CLIerror( "Missing the new suffix string" );
            $isError = true;
        }
        
        if($isError) {
            echo PHP_EOL;
            echo 'Help:' . PHP_EOL;
            echo '    The forkApplication entry point is used by developers to create a new copy of an application including it\'s components and processes' . PHP_EOL;
            echo 'Usage:' . PHP_EOL;
            echo "    php ${argv[0]} forkApplication [Application Name] [Old Suffix] [New Suffix]". PHP_EOL;
            echo 'Required:' .PHP_EOL;
            echo '    Application Name - the name of the application to fork' . PHP_EOL;
            echo '    Old Suffix - The string at the end of the name to remove e.g. _22' . PHP_EOL;
            echo '    New Suffix - The string to add to the end of the name e.g. _33' . PHP_EOL;
            exit(1);
        }
        $fork = new Fork();
        $fork->setOldSuffix($argv[3]);
        $fork->setNewSuffix($argv[4]);
        $result = $fork->forkApplication($argv[2]);
        if ($result === false) {
            exit( 1 );
        }
        break;
    case 'importEnvironmentProperties' :
        if (empty( $argv [2] )) {
            Utils::CLIerror( "Missing the file name." );
            exit( 1 );
        }
        if (empty( $argv [3] )) {
            Utils::CLIerror( "Missing the environment id." );
            exit( 1 );
        }
        if (empty( $argv [4] )) {
            Utils::CLIerror( "Missing the Application name." );
        }
        
        $environment = new Environment();
        // Success | Error handling will be managed at the end of this script
        $return = $environment->importProperties( $argv [2], $argv [3], $argv[4] );
        break;
    case 'promoteApplication' :
        if (empty( $argv [2] )) {
            Utils::CLIerror( "Missing the application name." );
            exit( 1 );
        }
        
        $promote = new Promote();
        $return = $promote->application( $argv [2] );
        break;
    case 'promoteConfig' :
        $source = null;
        $clean = false;
        if (! empty( $argv [2] )) {
            if ($argv [2] == 'clean') {
                $clean = true;
            } else {
                if (is_dir( $argv [2] )) {
                    $source = $argv [2];
                } else {
                    Utils::CLIout( "Argument is not valid." );
                    exit( 1 );
                }
                
                if (! empty( $argv [3] ) && $argv [3] == 'clean') {
                    $clean = true;
                } else {
                    $clean = false;
                }
            }
        }
        
        $promote = new Snapshot();
        
        $return = $promote->promoteConfig( $source, $clean );

        if ($return === false) {
            exit( 1 );
        }
        break;
    case 'promoteSnapshot' :
        if (empty( $argv [2] )) {
            Utils::CLIerror( "Missing snapshot name." );
            exit( 1 );
        }
        $promote = new Promote();
        $snapshot = $argv [2];
        
        if (! empty( $argv [3] )) {
            $force = $argv [3];
        } else {
            $force = null;
        }
        $return = $promote->snapshot( $snapshot, $force );
        Utils::CLIout( "\n" );

        if ($return === false) {
            exit( 1 );
        }
        break;
    case 'requestStatus' :
        if (empty( $argv [2] )) {
            Utils::CLIerror( "Missing process request ID." );
            exit( 1 );
        }
        $appProcess = new ApplicationProcess();
        $appProcess->setRequestId( $argv [2] );
        
        $exit = $appProcess->status( true );

        if ($exit === false) {
            exit( 1 );
        }
        break;
    case 'restore' :
        if (empty( $argv [2] )) {
            Utils::CLIerror( "No source path for restore defined", true );
            exit( 1 );
        }
        if (! empty( $argv [3] ) && $argv [3] == 'clean') {
            $clean = true;
        } else {
            $clean = false;
        }
        $path = $argv [2];
        $restore = new RestoreUCD();
        $return = $restore->restoreAll( $path, false, $clean );
        
        if ($return === false) {
            exit( 1 );
        }
        
        $success = true;
        foreach ( $return as $val ) {
            if ($val === false) {
                $success = false;
            }
        }
        
        if ($success === false) {
            exit( 1 );
        }
        break;
    case 'restoreApplication' :
        if (empty( $argv [2] )) {
            Utils::CLIerror( "No source path for restore defined", true );
            exit( 1 );
        }
        $path = $argv [2];
        $restore = new RestoreUCD();
        $return = $restore->restoreApplication( $path );
        break;
    case 'restoreComponent' :
        if (empty( $argv [2] )) {
            Utils::CLIerror( "No source path for restore defined", true );
            exit( 1 );
        }
        $path = $argv [2];
        $restore = new RestoreUCD();
        $return = $restore->restoreComponent( $path );
        break;
    case 'restoreComponentTemplate' :
        if (empty( $argv [2] )) {
            Utils::CLIerror( "No source path for restore defined", true );
            exit( 1 );
        }
        $path = $argv [2];
        $restore = new RestoreUCD();
        $return = $restore->restoreComponentTemplate( $path );
        break;
    case 'restoreResourceTemplate' :
        if (empty( $argv [2] )) {
            Utils::CLIerror( "Missing template resource JSON file" );
            exit( 1 );
        }
        $baseDir = dirname( $argv [2] );
        $resourceDir = '';
        if (is_dir( $baseDir . "/resources" )) {
            $resourceDir = $baseDir . "/resources";
        }
        $restore = new RestoreUCD();
        $result = $restore->restoreResourceTemplate( $argv [2], $resourceDir );

        if ($result === false) {
            exit( 1 );
        }
        break;
    case 'RTCbackup' :
        $backup = new BackupUCD();
        Utils::CLIinfo( "Removing old backup data from {$config['output']}" );
        $cwd = getcwd();
        $cleanCommand = "cd {$config['output']}; rm -R `ls | grep -v 'build.xml' | grep -v 'component.xml'`";
        exec( $cleanCommand, $output, $result );
        chdir( $cwd );
        if ($result == 0) {
            $return = $backup->backup();
        } else {
            Utils::CLIerror( 
                    "Error while removing old backup data from RTC directory." );
            exit(1);
        }
        break;
    case 'run' :
        if (empty( $argv [2] ) || ! file_exists( $argv [2] )) {
            Utils::CLIerror( 
                    "Invalid or missing template for application process." );
            exit( 1 );
        }
        $appProcess = new ApplicationProcess();
        $process = $argv [2];
        $result = $appProcess->request( $process );
        if ($result === false) {
            exit( 1 );
        } else {
            Utils::CLIout( "{$result}", true, "id" );
            Utils::CLIout( 
                    "{$appProcess->getWeburl()}/#applicationProcessRequest/{$result}", 
                    true, "link" );
        }
        break;
    case 'runAndWait' :
        if (empty( $argv [2] ) || ! file_exists( $argv [2] )) {
            Utils::CLIerror( 
                    "Invalid or missing template for application process." );
            exit( 1 );
        }
        
        $process = $argv [2];
        $appProcess = new ApplicationProcess();
        $result = $appProcess->request( $process );
        if ($result === false) {
            exit( 1 );
        }
        Utils::CLIout( "{$result}", true, "id" );
        Utils::CLIout( 
                "{$appProcess->getWeburl()}/#applicationProcessRequest/{$result}", 
                true, "link" );
        if (! empty( $argv [3] )) {
            $timeout = $argv [3];
        } else {
            $timeout = true;
        }
        
        $exit = $appProcess->waitCompletion( $timeout );
        
        if ($exit === false) {
            exit( 1 );
        } else {
            exit( $exit );
        }
        
        break;
    case 'setSnapshotStatus' :
        $snapshot = new Snapshot();
        $result = $snapshot->setSnapshotStatus( $argv [2], $argv [3] );

        if ($result === false) {
            exit( 1 );
        }
        break;
    case 'upgrade' :
        if (empty( $argv [2] )) {
            Utils::CLIerror( "No source path for upgrade defined", true );
            exit( 1 );
        }
        $path = $argv [2];
        $restore = new RestoreUCD();
        $return = $restore->restoreAll( $path, true );
        break;
    case 'upgradeApplication' :
        if (empty( $argv [2] )) {
            Utils::CLIerror( "No source path for restore defined", true );
            exit( 1 );
        }
        $path = $argv [2];
        
        $upgradeTemplate = false;
        $upgradeProcess = false;
        if (! empty( $argv [3] )) {
            if ($argv [3] == "upgradeTemplate") {
                $upgradeTemplate = true;
            } else if ($argv [3] == "upgradeProcess") {
                $upgradeProcess = true;
            }
        }
        if (! empty( $argv [4] )) {
            if ($argv [4] == "upgradeTemplate") {
                $upgradeTemplate = true;
            } else if ($argv [4] == "upgradeProcess") {
                $upgradeProcess = true;
            }
        }
        
        $ucc = new EntryPoint();
        $return = $ucc->rest()->application()->upgradeApplication( $path, $upgradeTemplate, 
                $upgradeProcess );
        break;
    case 'upgradeComponent' :
        if (empty( $argv [2] )) {
            Utils::CLIerror( "No source path for restore defined", true );
            exit( 1 );
        }
        $path = $argv [2];
        
        $upgradeTemplate = false;
        $upgradeProcess = false;
        if (! empty( $argv [3] )) {
            if ($argv [3] == "upgradeTemplate") {
                $upgradeTemplate = true;
            } else if ($argv [3] == "upgradeProcess") {
                $upgradeProcess = true;
            }
        }
        if (! empty( $argv [4] )) {
            if ($argv [4] == "upgradeTemplate") {
                $upgradeTemplate = true;
            } else if ($argv [4] == "upgradeProcess") {
                $upgradeProcess = true;
            }
        }
        
        $ucc = new EntryPoint();
        $return = $ucc->rest()->component()->upgradeComponent( $path, $upgradeTemplate, 
                $upgradeProcess );
        break;
    case 'upgradeComponentTemplate' :
        if (empty( $argv [2] )) {
            Utils::CLIerror( "No source path for upgrade defined", true );
            exit( 1 );
        }
        $path = $argv [2];
        $restore = new RestoreUCD();
        $return = $restore->restoreComponentTemplate( $path, true );
        break;
    case 'vers' :
    case 'version' :
    case 'versions' :
    case 'ver' :
        include ('include/version.php');
        Utils::CLIout( "Version: {$ver['ucommand']}" );
        Utils::CLIout( "Supported uDeploy versions:" );
        foreach ( $ver ['ucd'] as $ucdver ) {
            Utils::CLIout( "  {$ucdver}" );
        }
        Utils::CLIout( "" );
        break;
    case 'waitRequestCompletion' :
        if (empty( $argv [2] )) {
            Utils::CLIerror( "Missing process request ID." );
            exit( 1 );
        }
        $appProcess = new ApplicationProcess();
        $appProcess->setRequestId( $argv [2] );
        
        if (! empty( $argv [3] )) {
            $timeout = $argv [3];
        } else {
            $timeout = true;
        }
        
        $exit = $appProcess->waitCompletion( $timeout );

        if ($exit === false) {
            exit( 1 );
        }
        break;
    default :
        Utils::CLIerror( '"' . $argv [1] . '" is Not a valid action', true );
        Utils::CLIinfo( "Please provide a valid Action as Argument." );
        Utils::CLIout( "Usage: ", true );
        foreach ( $valid_actions as $action => $description ) {
            if ($action == "API") {
                continue;
            }
            Utils::CLIout( "\t{$action} => {$description}", true );
        }
        Utils::CLIout( "\n", true );
        exit( 1 );
}

// Print final mesasge
echo PHP_EOL;
Utils::CLIresult("Action '{$argv[1]}' completed: ", false);

// JSON validation
if ($config ['json_check'] === true) {
    Utils::validateDirectoryJSON( $config ['output'] );
}

// Quick spot errors
if (Utils::quickErrorCheck($return)) {
    Utils::CLIout('         Success' . PHP_EOL, false);
    exit (0);
} else {
    Utils::CLIout('         with ERRORS' . PHP_EOL, false);
    exit (1);
}

Utils::CLIdebug( var_export( $return, true ) );
$log->log( var_export( $return, true ) );