<?php
/****************************************************************
 * IBM Confidential
*
* SFA100-Collaboration Source Materials
*
* (C) Copyright IBM Corp. 2014
*
* The source code for this program is not published or otherwise
* divested of its trade secrets, irrespective of what has been
* deposited with the U.S. Copyright Office
*
***************************************************************/

require_once('include/Layers/Application.php');
require_once('include/Layers/Environment.php');
require_once('include/Layers/Snapshot.php');
require_once('include/EntryPoint.php');

/**
 * Manages snapshot promotion between uDeploy servers
 *
 * @class   Promote
 * @author  Marco Bassi         marcobas@ie.ibm.com
*/
class Promote extends EntryPoint {
    
    protected $server;
    protected $promote;
    
    public function __construct(){
        parent::__construct();
        $load = $this->loadConfig();
        if ( $load === false ) {
            throw new Exception("Invalid promote configuration");
        }
    }
    
    /**
     * 
     * Retrieve information from promote.config.php and setup the servers
     *
     */
    protected function loadConfig () {
        include ("config/promote.config.php");
        $this->promote = $config;
        
        // Handle missing alias
        $aliasFailure = false;
        if ( ! isset ( $config['origin_alias'] ) 
            || empty ( $config['origin_alias'] ) ) {
            Utils::CLIerror("No origin alias set in config/promote.config.php");
            $aliasFailure = true;
        }
        if ( ! isset ( $config['destination_alias'] )
            || empty ( $config['destination_alias'] ) ) {
            Utils::CLIerror("No destination alias set in config/promote.config.php");
            $aliasFailure = true;
        }
        
        if ( $aliasFailure ) {
            return false;
        }
        
        // Load configs
        $configFailure = false;
        $origin = $config['origin_alias'];
        $destination = $config['destination_alias'];
        
        unset ( $config );
        
        // Load origin
        $serverFile = "config/servers/{$origin}.server.php";
        if ( is_file( $serverFile ) ) {
            include ( $serverFile );
            $this->server['origin'] = $config;
        } else {
            Utils::CLIerror( "No server file available for alias '{$origin}' ");
            $configFailure = true;
        }
        
        unset( $config );
        
        // Load destination
        $serverFile = "config/servers/{$destination}.server.php";
        if ( is_file( $serverFile ) ) {
            include ( $serverFile );
            $this->server['destination'] = $config;
        } else {
            Utils::CLIerror( "No server file available for alias '{$destination}' ");
            $configFailure = true;
        }
        
        if ( $configFailure ) {
            return false;
        }
        
        return true;
    }
    
    /**
     * 
     * Promote a single application
     *
     * @param string $application       application Name|ID
     * @param boolean $clean            If true, do not import new environments in destination
     * @return boolean
     */
    public function application ( $application, $clean = true ) {
        $runtimeWatch =  time();
        StopWatch::getInstance()->start( $runtimeWatch );
        
        Utils::CLIinfo ("Application promotion");
        Utils::CLIout ("  Name: {$application}");
        Utils::CLIout ("  From: {$this->server['origin']['weburl']}");
        Utils::CLIout ("  To:   {$this->server['destination']['weburl']}");
        
        Utils::CLIout("");
        Utils::CLIstart( "Promotion start time: " . date("d-m-Y h:i:s") );
        
        $app = new Application();
        
        // Download from origin
        if ( ! $app->setupServer( 'origin', $this->promote, true ) ) {
            Utils::CLIerror("Cannot connect to origin server.");
            return false;
        }
        
        // Get application list
        $result = $app->getList();
        if ( $result === false ) {
            Utils::CLIerror("Failed to retrieve list of applications from origin server");
            return false;
        }
        
        $appList = array();
        foreach ( $result as $item ) {
            array_push ( $appList, $item['name'] );
        }
        
        // Check if application exists
        if ( ! in_array ( $application, $appList ) ) {
            Utils::CLIerror("Application '{$application}' does not exist in origin server");
            if ( ! empty ( $appList ) ) {
                Utils::CLIout("List of available applications: ");
                foreach ( $appList as $item ) {
                    Utils::CLIout("  $item");
                }
            } else {
                Utils::CLIout("No application available in origin server");
            }
            return false;
        }
        
        Utils::CLIout("");
        Utils::CLIinfo ( "Retrieving application {$application} from origin server" );
        
        $downloadWatch =  time();
        StopWatch::getInstance()->start( $downloadWatch );
        
        $app->setReturn('file');
        $applicationFile = $app->get ( $application, true );
        
        StopWatch::getInstance()->stop( $downloadWatch );
        Utils::CLIout( "  Completed (" . StopWatch::getInstance()->get( $downloadWatch ) . ")\n" );
        
        if ( $applicationFile === false ) {
            Utils::CLIerror("Failed to retrieve application from origin server");
            return false;
        }
        
        // Import into destination
        if ( ! $app->setupServer( 'destination', $this->promote, true ) ) {
            Utils::CLIerror("Cannot connect to destination server.");
            return false;
        }
        $app->setReturn('json');
        
        // Clean environments
        if ( $clean ) {
            Utils::CLIinfo("Cleaning environments before import");
            $applicationArray = json_decode ( file_get_contents ( $applicationFile ), true );
            $applicationArray['environments'] = array();
            $result = Utils::createJsonFile ( $applicationArray, $applicationFile, true );
            if ( $result === false ) {
                // Error messages handled by Utils
                return false;
            }
        }
        
        Utils::CLIinfo ( "Promoting application {$application} into destination server" );
        
        $importWatch =  time();
        StopWatch::getInstance()->start( $importWatch );
        
        $result = $app->upgrade ( $applicationFile );
        
        StopWatch::getInstance()->stop( $importWatch );
        Utils::CLIout( "  Completed (" . StopWatch::getInstance()->get( $importWatch ) . ")\n" );
        
        if ( $result === false ) {
            Utils::CLIerror("Failed to promote application into destination server");
            return false;
        }
        
        StopWatch::getInstance()->stop( $runtimeWatch );
        Utils::CLIcompleted( "Overall execution time: " . StopWatch::getInstance()->get( $runtimeWatch ) );
        
        return true;
    }
    
    /**
     * 
     * Promote the entire UCD configuration
     *
     * @param boolean $clean
     * @return boolean
     */
    public function config ( $clean = false ) {
        
    }
    
    /**
     *
     * Gives information about destination server
     *
     * @return array
     */
    public function getDestination () {
        return $this->server['destination'];
        
    }
    
    /**
     * 
     * Gives information about origin server
     * 
     * @return array
     */
    public function getOrigin () {
        return $this->server['origin'];
    }
    
    /**
     * 
     * Promote a snapshot
     *
     * @param string $snapshot
     * @param string $config
     * @param string $artifacts
     * @param boolean $force
     * @return boolean
     */
    public function snapshot ( $snapshot, $force = null ) {
        $configFile = 'config/promote.config.php';
        include $configFile;
        
        $result = array();
        $snapshotFailures = 0;
        $appNameError = false;
        
        $snapshotObj = new Snapshot();
        
        // Check application name
        if ( $this->server['origin']['application'] != $this->server['destination']['application'] ) {
            Utils::CLIerror ("Origin and destination server have different application name");
            $appNameError = true;
        }
        if ( empty( $this->server['origin']['application'] ) ) {
            Utils::CLIerror ("Origin application name is missing");
            $appNameError = true;
        } 
        if ( empty( $this->server['destination']['application'] ) ) {
            Utils::CLIerror ("Destination application name is missing");
            $appNameError = true;
        }
        
        if ( $appNameError ) {
            return false;
        } else {
            // At this stage, destination and origin app name are the same
            $this->application = $this->server['origin']['application'];
        }
        
        // Setup connection to first server (origin)
        if (!$this->setupServer('origin', $config, true)) {
            Utils::CLIerror("Cannot connect to origin server.");
            return false;
        }
        
        // Retrieve snapshot from origin uDeploy server
        Utils::CLIinfo("Retrieving requested snapshot '{$snapshot}' from '{$this->application}' application on origin server");
        $output = $this->rest()->snapshot()->getSnapshotsInApplication($this->application);
        $appSnapshots = Utils::outputToArray($output);
        $snapshotId = null;
        foreach ($appSnapshots as $appSnapshot) {
            if ($appSnapshot['name'] == $snapshot) {
                $snapshotId = $appSnapshot['id'];
                break;
            }
        }
        if (empty($snapshotId)) {
            Utils::CLIerror("Requested snapshot '{$snapshot}' in '{$this->application}' application NOT FOUND on origin server.");
            return false;
        }
        
        // Get component and component version list, as defined in the snapshot
        $output = $this->rest()->component()->getComponentVersionsInSnapshot($snapshotId);
        $originComponents = Utils::outputToArray($output);
        
        
        // Verify that snapshot doesn't already exist on destination server
        Utils::CLIinfo("Verifying that snapshot '{$snapshot}' doesn't exists on destination server");
        if (!$this->setupServer('destination', $config, true)) {
            Utils::CLIerror("Cannot connect to destination server.");
            return false;
        }
        
        $output = $this->rest()->snapshot()->getSnapshotsInApplication($this->application);
        $checkSnapshots = Utils::outputToArray($output);
        $checkSnapshotId = null;
        foreach ($checkSnapshots as $checkSnapshot) {
            if ($checkSnapshot['name'] == $snapshot) {
                $checkSnapshotId = $checkSnapshot['id'];
                break;
            }
        }
        if (!empty($checkSnapshotId)) {
            Utils::CLIerror("Requested snapshot '{$snapshot}' in '{$this->application}' application ALREADY EXISTS on destination server.");
            return false;
        }
        
        // Compare components between origin and destination, verify if every component exists in destination. Create if missing
        Utils::CLIinfo("Verifying existing components in destination server");
        $appDestComponents = $this->udclient()->component()->getComponentsInApplication($this->application);
        if ($appDestComponents) {
            $appDestComponents = Utils::outputToArray($appDestComponents);
        } else {
            Utils::CLIerror("Error retrieving components for application '{$this->application}' on destination server.");
            return false;
        }
        
        // All components in destination server
        $allDestComponents = $this->udclient()->component()->getComponents();
        if ($allDestComponents) {
            $allDestComponents = Utils::outputToArray($allDestComponents);
        } else {
            Utils::CLIerror("Error retrieving all components from destination server.");
            return false;
        }
        
        $toCreateComponents = array();
        $toAddComponents = array();
        // Loop components required for snapshot
        foreach ($originComponents as $component) {
            $componentExists = false;
            $componentIsAdded = false;
            // Check if current component is available in destination server
            foreach ($appDestComponents as $appDestComponent) {
                if ($component['name'] == $appDestComponent['name']) {
                    $componentExists = true;
                    // Check if current component is added to the target application
                    foreach($allDestComponents as $destComponent) {
                        if ($component['name'] == $destComponent['name']) {
                            $componentIsAdded = true;
                            break;
                        }
                    }
                    break;
                }
            }
        
            if (!$componentExists) {
                // Component doesn't exist: need to create it
                $toCreateComponents[$component['id']] = $component['name'];
            }
            if (!$componentIsAdded) {
                // Component is not added to target applicaiton: need to add it
                $toAddComponents[$component['id']] = $component['name'];
            }
        } // END Loop components
        
        
        if (!empty($toCreateComponents)) {
            Utils::CLIwarning("The following components are missing and need to be created:");
            foreach ($toCreateComponents as $id => $name){
                Utils::CLIout("  $name");
            }
        
            return false;
        } else {
            Utils::CLIout("  All components are available on destination server");
        }
        
        if (!empty($toAddComponents)) {
            Utils::CLIwarning("The following components are missing from '{$this->application}' application:");
            foreach ($toAddComponents as $id => $name){
                Utils::CLIout("  $name");
            }
            return false;
        } else {
            Utils::CLIout("  All components are associated to '{$this->application}' application");
        }
        Utils::CLIout("");
        
        // Export artifacts
        Utils::CLIinfo("Exporting snapshot artifacts");
        if (!$this->setupServer('origin', $config, true)) {
            Utils::CLIerror("Cannot connect to origin server.");
            return false;
        }
        Utils::CLIout("  (This may take a while)", true);
        $this->setReturn('file');
        
        $backupOutput = $this->output;
        $cwd = getcwd();
        $this->setOutput("{$cwd}/tmp", true);
        $result['snapshotExport'] = $this->rest()->snapshot()->exportSnapshot($snapshotId, $this->application, $snapshot);
        if ($result['snapshotExport'] === false) {
            Utils::CLIerror("Cannot download snapshot zip file from origin server.");
            return false;
        }
        $snapshotZip = "{$this->output}/{$this->application}.{$snapshot}.zip";
        $this->log->info("Snapshot zip: {$snapshotZip}");
        
        $snapshotDir = "{$this->output}/" . time();
        Utils::CLIinfo("Extracting snapshot artifacts to {$snapshotDir}");
        $unzip = "unzip -q -o {$snapshotZip} -d {$snapshotDir}";
        Utils::CLIinfo("unzip command: ".$unzip);
        Utils::CLIinfo("");
        exec($unzip, $output, $return);
        
        $this->log->info(print_r($output,true));
        
        if ($return != 0) {
            Utils::CLIerror("Cannot unzip snapshot artifacts.");
            return false;
        }
        
        $snapshotSubDirs = scandir($snapshotDir);
        
        $this->setOutput($backupOutput, true);
        
        $this->setReturn('json');
        
        // Preparing new Snapshot data
        Utils::CLIinfo("Preparing snapshot for destination server");
        if (!$this->setupServer('destination',$config, true)) {
            Utils::CLIerror("Cannot connect to destination server.");
            return false;
        }
        
        $newSnapshot['snapshot']['name'] = $snapshot;
        $newSnapshot['application'] = $this->application;
        $newSnapshot['snapshot']['description'] = "{$this->application} snapshot for build '{$snapshot}'";
        $newSnapshot['components'] = array();
        
        Utils::CLIout("  Name: {$newSnapshot['snapshot']['name']}");
        Utils::CLIout("  Application: {$newSnapshot['application']}");
        Utils::CLIout("  Description: {$newSnapshot['snapshot']['description']}");
        $this->log->info("Components: ");
        
        // Compare component versions set in snapshot with destination's component versions. Create if missing
        foreach ($originComponents as $component) {
            $name = $component['name'];
            foreach ($component['desiredVersions'] as $ver) {
                $version = $ver['name'];
                $versions = $version;
                if (array_key_exists($name, $newSnapshot['components'])) {
                    if (is_array($newSnapshot['components'][$name])) {
                        $versions = $newSnapshot['components'][$name];
                        array_push($versions, $version);
                    } else {
                        $versions = array($newSnapshot['components'][$name], $version);
                    }
                }
                $newSnapshot['components'][$name] = $versions;
                $this->log->log("  {$name} ({$version})");
        
                // Verify if any artifact and add it
                foreach ($snapshotSubDirs as $dir) {
                    if (strpos($dir,"{$version}-{$ver['id']}") !== false) {
                        $base =  "{$snapshotDir}/{$dir}";
                        $newSnapshot['artifacts'][$name][$version] = array('base' => $base);
                        $this->log->log("    Artifacts: {$base}");
                        break;
                    }
                }
            }
        }
        Utils::CLIout("\n");
        
        if (!$snapshotObj->setupServer('destination', $config, true)) {
            Utils::CLIerror("Cannot connect to destination server.");
            return false;
        }
        $result['versionsAndSnapshot'] = $snapshotObj->createVersionsAndSnapshot($newSnapshot, $force);
        return $result;
    }
}