<?php
/****************************************************************
* IBM Confidential
*
* SFA100-Collaboration Source Materials
*
* (C) Copyright IBM Corp. 2015
*
* The source code for this program is not published or otherwise
* divested of its trade secrets, irrespective of what has been
* deposited with the U.S. Copyright Office
*
* **************************************************************/

require_once('include/EntryPoint.php');

/**
 * Provides tools for creating and managing Applicarions requests
 *
 * @class Application
 * @author Felice Geracitano.......feliecege@ie.ibm.com
 */
class Application extends EntryPoint {
    public function __construct(){
        parent::__construct();
        $this->setReturn('json');
    }
    
    /**
     * Create a Application from scratch or import from a file path
     *
     * @param array|string $application
     * @return array|string|boolean
     */
    public function create($application) {
        $file_name = null;
        // Check if arg is file path or array obj
        if (is_array( $application ) || is_object( $application )) {
            $file_name = Utils::createJsonFile( $application );
        } else {
            if (file_exists($application) === false) {
                Utils::CLIerror( "Json file does not exist" );
                Utils::CLIdebug( __CLASS__ . '::' . __FUNCTION__.": Json file '{$application}' does not exist" );
                return false;
            }
            if (Utils::validateJSON( file_get_contents( $application ) ) === false) {
                Utils::CLIerror( "Json file has wrong format" );
                Utils::CLIdebug( __CLASS__ . '::' . __FUNCTION__.": Json file '{$application}' has wrong format" );
                return false;
            }
            $file_name = $application;
        }
        if ($this->rest()->application()->importApplication( $file_name ) === false) {
            Utils::CLIerror( "Application already exist" );
            return false;
        }
        is_string( $application ) === false ? unlink( $file_name ) : null;
        return true;
    }
    
    /**
     * Delete a Application passing Id or Name
     *
     * @param string $application
     * @return boolean
     */
    public function delete($application) {
        if (empty($application)){
            Utils::CLIerror( "Application name/id is missing" );
            return false;
        }
        // Delete Applicatoin by Name or Id
        $result = $this->rest()->application()->deleteApplication($application);
        if ($result === false){
            Utils::CLIerror( "Application doesn't exist or the current user doesn't have enough rights to delete it" );
            return null;
        }
        return true;
    }
    
    /**
     * Get short or full description Application from Id or Name
     *
     * @param string $application
     * @param boolean $full_object
     * @return array|path|boolean
     */
    public function get($application, $full_object = false) {
        $result = null;
        if (empty( $application )) {
            Utils::CLIerror( "Application name/id is missing" );
            return false;
        }
        // getApplication provides small description, getApplicationRest
        // provides large description
        if (empty( $full_object )) {
            $result = $this->udclient()->application()->getApplication( $application );
        } else {
            $result = $this->rest()->application()->getApplicationRest( $application );
        }
        
        // If return is set to file, then the API Rest returns a file with an
        // error message
        // so if API Rest fails or returning a file is not a json -> error
        if ($result === false ||
                 ($this->return === 'file' &&
                 ! json_decode( file_get_contents( substr( $result, 1, - 1 ) ) ))) {
            Utils::CLIerror( "Application does not exist" );
            return null;
        }
        
        if ($this->return === 'file') {
            // delete quotes and return path
            return substr( $result, 1, - 1 );
        }
        return Utils::outputToArray( $result );
    }
    
    /**
     * Get list of all Applications from UCD server
     *
     * @return array|path|boolean
     */
    public function getList() {
        $result = $this->rest()->application()->getApplicationsRest();
        if ($result === false){
            Utils::CLIerror( "Failed to retrieve applications list" );
            return null;
        }
        if ($this->return === 'file'){
            // delete quotes and return path
            return substr($result,1,-1);
        }
        return Utils::outputToArray( $result );
    }
    
    /**
     *  upgrade an application passing array object or json file path
     *  allowed value for upgradeComponentTemplate and upgradeProcess
     ** USE_EXISTING_IF_EXISTS
     ** CREATE_NEW_IF_EXISTS
     ** FAIL_IF_EXISTS
     ** FAIL_IF_DOESNT_EXIST
     ** UPGRADE_IF_EXISTS
     *
     * @param array | string $application
     * @param string $processUpgrade
     * @param string $templateUpgrade
     * @return boolean
     */
    public function upgrade($application, $upgradeComponentTemplate = 'UPGRADE_IF_EXISTS', $upgradeProcess = 'UPGRADE_IF_EXISTS') {
        $file_name = null;
        // Check if arg is file path or array obj
        if (is_array( $application ) || is_object( $application )) {
            $file_name = Utils::createJsonFile( $application );
        } else {
            if (file_exists($application) === false) {
                Utils::CLIerror( "Json file does not exist" );
                Utils::CLIdebug( __CLASS__ . '::' . __FUNCTION__.": Json file '{$application}' does not exist" );
                return false;
            }
            if (Utils::validateJSON( file_get_contents( $application ) ) === false) {
                Utils::CLIerror( "Json file is invalid" );
                Utils::CLIdebug( __CLASS__ . '::' . __FUNCTION__.": Json file '{$application}' Json file is invalid" );
                return false;
            }
            $file_name = $application;
        }
        
        if (! in_array( $upgradeComponentTemplate, $this->upgrade_values ) && ! is_null( $upgradeComponentTemplate )) {
            Utils::CLIerror( "Component Template upgrade has wrong parameter" );
            Utils::CLIdebug( __CLASS__ . '::' . __FUNCTION__ .": Template upgrade has wrong parameter '{$upgradeComponentTemplate}'" );
            return false;
        }
        if (! in_array( $upgradeProcess, $this->upgrade_values ) && ! is_null( $upgradeProcess )) {
            Utils::CLIerror( "Process upgrade has wrong parameter" );
            Utils::CLIdebug( __CLASS__ . '::' . __FUNCTION__ .": Process upgrade has wrong parameter '{$upgradeProcess}'" );
            return false;
        }
        $this->setReturn( 'json' );
        if ($this->rest()->application()->upgradeApplication( $file_name, $upgradeComponentTemplate, $upgradeProcess ) === false) {
            Utils::CLIdebug( __CLASS__ . '::' . __FUNCTION__ .": Application upgrade could have failed for these reasons: Component template option '$upgradeComponentTemplate', process option '$upgradeProcess'");
            return false;
        }
        is_string( $application ) === false ? unlink( $file_name ) : null;
        return true;
    }
}
