<?php
/****************************************************************
 * IBM Confidential
*
* SFA100-Collaboration Source Materials
*
* (C) Copyright IBM Corp. 2015
*
* The source code for this program is not published or otherwise
* divested of its trade secrets, irrespective of what has been
* deposited with the U.S. Copyright Office
*
* **************************************************************/

require_once('include/EntryPoint.php');

/**
 * Provides tools for creating and managing components requests
 *
 * @class Component
 * @author Felice Geracitano
 */
class Component extends EntryPoint {
    public function __construct(){
        parent::__construct();
        $this->setReturn('json');
    }
    
    /**
     * Create a Component from scratch or import from a file path
     *
     * @param array|string $component
     * @return array|string|boolean
     */
    public function create($component) {
        $file_name = null;
        // Check if arg is file path or array obj
        if (is_array( $component ) || is_object( $component )) {
            $file_name = Utils::createJsonFile( $component );
        } else {
            if (file_exists($component) === false) {
                Utils::CLIerror( "Json file does not exist" );
                Utils::CLIdebug( __CLASS__ . '::' . __FUNCTION__.": Json file '{$component}' does not exist" );
                return false;
            }
            if (Utils::validateJSON( file_get_contents( $component ) ) === false) {
                Utils::CLIerror( "Json file has wrong format" );
                Utils::CLIdebug( __CLASS__ . '::' . __FUNCTION__.": Json file '{$component}' has wrong format" );
                return false;
            }
            $file_name = $component;
        }
        if ($this->rest()->component()->importComponent( $file_name, false ) === false) {
            Utils::CLIerror( "Component already exist" );
            return false;
        }
        is_string( $component ) === false ? unlink( $file_name ) : null;
        return true;
    }
    
    /**
     * Create new version of a component
     *
     * @param string $component
     * @param string $version
     * @return array|boolean
     */
    public function createComponentVersion($component, $version) {
        if (empty($component)){
            Utils::CLIerror( "Component name/id is missing" );
            return false;
        }
        if (empty($version)){
            Utils::CLIerror( "Version name is missing" );
            return false;
        }
        $result = $this->udclient()->component()->createVersion($component,$version);
        if ($result === false){
            Utils::CLIerror( "Failed to create component version" );
            Utils::CLIdebug( __CLASS__ . '::' . __FUNCTION__.": Failed to create version '$version' of component '$component'" );
            return false;
        }
        
        if ($this->return === 'file'){
            // delete quotes and return path
            return substr($result,1,-1);
        }
        return Utils::outputToArray( $result );
    }
    
    /**
     * Delete a component passing Id or Name
     *
     * @param string $component
     * @return boolean
     */
    public function delete($component) {
        if (empty($component)){
            Utils::CLIerror( "Component name/id is missing" );
            return false;
        }
        // Delete component by Name or Id
        $result = $this->rest()->component()->deleteComponent($component);
        if ($result === false){
            Utils::CLIerror( "Component does not exist" );
            return null;
        }
        return true;
    }
    
    /**
     * Get short or full description component from Id or Name
     *
     * @param string $component
     * @param boolean $full_object
     * @return array|path|boolean
     */
    public function get($component, $full_object = false) {
        $result = null;
        if (empty($component)){
            Utils::CLIerror( "Component name/id is missing" );
            return false;
        }
        // getComponent provides large description, getComponentRest provides small descripion
        $result = $full_object ? $this->udclient()->component()->getComponent($component) : $this->rest()->component()->getComponentRest($component);
        if ($result === false){
            Utils::CLIerror( "Component does not exist" );
            return null;
        }
        if ($this->return === 'file'){
            // delete quotes and return path
            return substr($result,1,-1);
        }
        return Utils::outputToArray( $result );
    }
    
    /**
     * Get list of all components from UCD server
     *
     * @return array|path|boolean
     */
    public function getList() {
        $result = $this->rest()->component()->getComponentsRest();
        if ($result === false){
            Utils::CLIerror( "Failed to retrieve components list" );
            return null;
        }
        if ($this->return === 'file'){
            // delete quotes and return path
            return substr($result,1,-1);
        }
        return Utils::outputToArray( $result );
    }    
    
    /**
     * Get specific version info of a component
     *
     * @param string $component
     * @return array|boolean
     */
    public function getVersion($component, $version) {
        $found = false;
        // Save parent request
        $parent_request = $this->return;
        $this->setReturn( 'array' );
        $versions = $this->listVersions( $component );
        // Allow version parameter as Name or ID 
        foreach ( $versions as $ver ) {
            if ($ver ['name'] == $version || $ver ['id'] == $version) {
                $found = true;
                $version = $ver ['id'];
                break;
            }
        }
        
        if ($found === false) {
            Utils::CLIerror( "Version does not exist" );
            return false;
        }
        // Come back to parent request
        $this->setReturn( $parent_request );
        $result = $this->rest()->component()->getVersionFromId( $version );
        if ($result === false) {
            Utils::CLIerror( "Failed to retrive version of component" );
            return false;
        }
        if ($this->return === 'file') {
            // delete quotes and return path
            return substr( $result, 1, - 1 );
        }
        return Utils::outputToArray( $result );
    }
    
    /**
     * Get versions list of a component 
     *
     * @param string Id or Name $component
     * @return array|boolean
     */
    public function listVersions($component) {
        if (empty($component)){
            Utils::CLIerror( "Component name/id is missing" );
            return false;
        }
        $result = $this->rest()->component()->getComponentVersions($component);
        if ($result === false){
            Utils::CLIerror( "Failed to retrieve the list of versions" );
            Utils::CLIdebug( __CLASS__ . '::' . __FUNCTION__.": Failed to retrieve the list versions for the component'{$component}'" );
            return false;
        }
        $tmp = Utils::outputToArray( $result );
        if (empty($tmp)){
            // Version list is empty
            return null;
        }
        if ($this->return === 'file'){
            // delete quotes and return path
            return substr($result,1,-1);
        }
        return Utils::outputToArray( $result );
    }
    
    /**
     *  upgrade a component passing array object or json file path
     *  allowed value for templateUpgrade and processUpgrade of the component
     ** USE_EXISTING_IF_EXISTS
     ** CREATE_NEW_IF_EXISTS
     ** FAIL_IF_EXISTS
     ** FAIL_IF_DOESNT_EXIST
     ** UPGRADE_IF_EXISTS
     *
     * @param array | string $component
     * @param string $processUpgrade
     * @param string $templateUpgrade
     * @return boolean
     */
    public function upgrade($component, $templateUpgrade = 'UPGRADE_IF_EXISTS', $processUpgrade = 'UPGRADE_IF_EXISTS') {
        $file_name = null;
        // Check if arg is file path or array obj
        if (is_array( $component ) || is_object( $component )) {
            $file_name = Utils::createJsonFile( $component );
        } else {
            if (file_exists($component) === false) {
                Utils::CLIerror( "Json file does not exist" );
                Utils::CLIdebug( __CLASS__ . '::' . __FUNCTION__.": Json file '{$component}' does not exist" );
                return false;
            }
            if (Utils::validateJSON( file_get_contents( $component ) ) === false) {
                Utils::CLIerror( "Json file is invalid" );
                Utils::CLIdebug( __CLASS__ . '::' . __FUNCTION__.": Json file '{$component}' Json file is invalid" );
                return false;
            }
            $file_name = $component;
        }
        if (! in_array( $templateUpgrade, $this->upgrade_values ) && ! is_null( $templateUpgrade )) {
            Utils::CLIerror( "Template upgrade has wrong parameter" );
            Utils::CLIdebug( __CLASS__ . '::' . __FUNCTION__ .": Template upgrade has wrong parameter '{$templateUpgrade}'" );
            return false;
        }
        if (! in_array( $processUpgrade, $this->upgrade_values ) && ! is_null( $processUpgrade )) {
            Utils::CLIerror( "Process upgrade has wrong parameter" );
            Utils::CLIdebug( __CLASS__ . '::' . __FUNCTION__ .": Process upgrade has wrong parameter '{$processUpgrade}'" );
            return false;
        }
        $this->setReturn( 'json' );
        if ($this->rest()->component()->upgradeComponent( $file_name, $templateUpgrade, $processUpgrade ) === false) {
            Utils::CLIdebug( __CLASS__ . '::' . __FUNCTION__ .": Component upgrade could have failed for these reasons: template option '$templateUpgrade', process option '$processUpgrade'");
            return false;
        }
        is_string( $component ) === false ? unlink( $file_name ) : null;
        return true;
    }
    
    /**
     * Check if the component version exist
     *
     * @param string $component
     * @param string $version
     * @return boolean
     */
    public function versionExists($component, $version) {
        $this->setReturn( 'json' );
        $versions = $this->rest()->component()->getComponentVersions($component);
        $versions = Utils::outputToArray($versions);
        foreach ($versions as $ver) {
            if ($ver['name'] == $version || $ver['id'] == $version) {
                return true;
            }
        }
        return false;
    }
    
}
