<?php
/****************************************************************
 * IBM Confidential
 *
 * SFA100-Collaboration Source Materials
 *
 * (C) Copyright IBM Corp. 2015
 *
 * The source code for this program is not published or otherwise
 * divested of its trade secrets, irrespective of what has been
 * deposited with the U.S. Copyright Office
 *
 * **************************************************************/

require_once ('include/EntryPoint.php');

/**
 * Provides tools for creating and managing Processes requests
 *
 * @class Process
 * @author Felice Geracitano
 */
class Process extends EntryPoint {
    
    public function __construct(){
        parent::__construct();
        $this->setReturn('json');
    }

    /**
     * Get the process info in an array or in a file path
     * 
     * @param string $processId
     * @return array|path|boolean
     */
    public function get($processId) {
        // Save user preference
        $return_type = $this->return;
        $return_value = null;
        if (! Utils::isUuid( $processId )) {
            Utils::CLIerror( "Parameter is not a process Id" );
            Utils::CLIdebug( __CLASS__ . '::' . __FUNCTION__.": parameter '{$processId}' is not a process Id" );
            return false;
        }
        // Failure check allowed only if return is json
        $this->setReturn( 'json' );
        $return_value = $this->rest()->process()->getProcessRest( $processId );
        if (empty( $return_value )) {
            Utils::CLIerror( "Process does not exist" );
            Utils::CLIdebug( __CLASS__ . '::' . __FUNCTION__.": process with id '{$processId}' does not exist" );
            return false;
        }
        $return_value = Utils::outputToArray( $return_value );
        // Return the right type
        if ($return_type === 'file'){
            return (Utils::createJsonFile( $return_value ));
        }
        return $return_value;
    }
    
    /**
     * Create a Process from scratch or import a Process from a file path
     * 
     * @param array|json_file_path $process
     * @return boolean
     */
    public function create($process) {
        $file_name = null;
        // Check if arg is file path or array obj
        if (is_array( $process ) || is_object( $process )) {
            $file_name = Utils::createJsonFile( $process );
        } else {
            if (file_exists($process) === false) {
                Utils::CLIerror( "Json file does not exist" );
                Utils::CLIdebug( __CLASS__ . '::' . __FUNCTION__.": Json file '{$process}' does not exist" );
                return false;
            }
            if (Utils::validateJSON( file_get_contents( $process ) ) === false) {
                Utils::CLIerror( "Json file has wrong format" );
                Utils::CLIdebug( __CLASS__ . '::' . __FUNCTION__.": Json file '{$process}' has wrong format" );
                return false;
            }
            $file_name = $process;
        }
        if ($this->rest()->process()->importProcess( $file_name, false ) === false) {
            Utils::CLIerror( "Process already exist" );
            return false;
        }
        is_string( $process ) === false ? unlink( $file_name ) : null;
        return true;
    }
    
    /**
     * Delete a process using processId
     *
     * @param string $processId
     * @return boolean
     */
    public function delete($processId) {
        if (! Utils::isUuid( $processId )) {
            Utils::CLIerror( "Parameter is not a process Id" );
            return false;
        }
        $response = $this->rest()->process()->deleteGenericProcess( $processId );
        if ( $response === false ) {
            Utils::CLIerror( "Process does not exist" );
            return false;
        }
        return true;
    }
    
    /**
     * Upgrade an existing Process
     *
     * @param string $json_file_path
     * @return boolean
     */
    public function upgrade($json_file_path) {
        if (Utils::validateJSON( file_get_contents( $json_file_path ) ) === false) {
            Utils::CLIerror( "Json file has wrong format" );
            Utils::CLIdebug( __CLASS__ . '::' . __FUNCTION__ .": Json file has wrong format" );
            return false;
        }
        if ($this->rest()->process()->importProcess( $json_file_path, true ) === false) {
            Utils::CLIerror( "Failure during upgrade of process" );
            Utils::CLIdebug( __CLASS__ . '::' . __FUNCTION__ .": Failure during upgrade of process described in {$json_file_path}" );
            return false;
        }
        return true;
    }
    
    /**
     * Get the Processes list in array value or file path
     *
     * @return array | path | boolean
     */
    public function getList() {
        $return = $this->rest()->process()->getProcesses();
        if ($return === false){
            Utils::CLIerror( "Failed to retrieve processes list" );
            return false;
        }
        if ($this->return == 'file') {
            $file_path = trim( $return, "'" );
            if (Utils::validateJSON( file_get_contents( $file_path ) ) === true){
                return $file_path; 
            }
        } else if (!empty( $return )) {
            return Utils::outputToArray( $return );
        } else {
            return null;
        }
        return false;
    }
}