<?php
/****************************************************************
* IBM Confidential
*
* SFA100-Collaboration Source Materials
*
* (C) Copyright IBM Corp. 2015
*
* The source code for this program is not published or otherwise
* divested of its trade secrets, irrespective of what has been
* deposited with the U.S. Copyright Office
*
* **************************************************************/

/**
 * API for component requests via REST
 *
 * @author Felice Geracitano.......feliecege@ie.ibm.com
 */
class ComponentRestApi extends AbstractApi {
    
    /**
     * Set up a uDeploy API Request for Creation of a componentTemplate,
     * based on a JSON formatted file.
     *
     * @param string $componentTemplate Path to the JSON File, to be used for ComponentTemplate creation
     *
     * @return boolean|string String, holding the uDeploy API action to be executed OR
     *                        Boolean FALSE, when $componentTemplate File is not provided
     */
    public function createComponentTemplate($componentTemplate = null) {
        $action = "-X PUT ";
        $action .= '-H "Content-Type: application/json" ';
        $action .= "{$this->weburl}/rest/deploy/componentTemplate -d @'{$componentTemplate}'";
    
        return $this->execRest($action);
    }
    
    /**
     *
     * Deletes a component using name or ID
     *
     * @param unknown $component
     */
    public function deleteComponent($component = null){
        $command = "-X DELETE ";
        $command .= "'{$this->weburl}/rest/deploy/component/{$component}'";
        return $this->execRest($command);
    }
    
    /**
     *
     * Deletes a componentProcess using ID
     *
     * @param unknown $component ID
     */
    public function deleteComponentProcess($componentProcessID = null){
        $command = "-X DELETE ";
        $command .= "'{$this->weburl}/rest/deploy/componentProcess/{$componentProcessID}'";
        return $this->execRest($command);
    }
    
    /**
     * Set a uDeploy DELETE action to be executed for
     * Deletion of a Component Template using its ID
     *
     * @param integer $componentTemplateID
     *
     * @return string String holding the uDeploy action for the Deletion of a Component Template
     */
    public function deleteComponentTemplate($componentTemplateID){
        $command = "-X DELETE ";
        $command .= "{$this->weburl}/rest/deploy/componentTemplate/{$componentTemplateID} ";
    
        return $this->execRest($command);
    }
    
    /**
     * Delete a component Version from UCD 
     *
     * @param string $versionID
     * @return string|boolean
     */
    public function deleteComponentVersion($versionID){
        $command = "-X DELETE ";
        $command .= "{$this->weburl}/rest/deploy/version/{$versionID} ";
        return $this->execRest($command);
    }
    
    /**
     *
     * @param string $environment
     * @return string
     *
     * Get component environment properties for an environment
     */
    public function getComponentEnvironment($environmentId = null) {
        if ($this->return == 'file') {
            $this->destination = " -o '{$this->output}/{$environmentId}.json'";
        }
        $action = "{$this->weburl}/rest/deploy/environment/{$environmentId}/componentProperties {$this->destination}";
        return $this->execRest( $action );
    }
    
    /**
     *
     * Export a component process to a file or returns json
     *
     * @param string $componentID
     * @param string $componentName
     * @return boolean|string
     */
    public function getComponentProcessRest($componentProcessID = null, $componentProcessName = null){
        if ($this->return == 'file') {
            $this->destination = " -o '{$this->output}/{$componentProcessName}.json'";
        }
        $action = "{$this->weburl}/rest/deploy/componentProcess/{$componentProcessID}/-1 {$this->destination}";
        return $this->execRest($action);
    }
    
    /**
     *
     * Export a component to a file or returns json
     *
     * @param string $componentID
     * @param string $componentName
     * @return boolean|string
     */
    public function getComponentRest($componentID = null, $componentName = null){
        $compomentLabel = $componentID;
        if (!empty($componentName)) {
            $compomentLabel = $componentName;
            $compomentLabel = str_replace(" ", "%20", $compomentLabel);
        }
        if ($this->return == 'file') {
            $this->destination = " -o '{$this->output}/{$compomentLabel}.json'";
        }
        $action = "{$this->weburl}/rest/deploy/component/{$compomentLabel}/export {$this->destination}";
        return $this->execRest($action);
    }
    
    /**
     *
     * Export all components to a file or returns a json
     *
     * @return boolean|string
     */
    public function getComponentsRest(){
        if ($this->return == 'file') {
            $this->destination = " -o '{$this->output}/components.json'";
        }
        $action = "{$this->weburl}/rest/deploy/component?active=true {$this->destination}";
        return $this->execRest($action);
    }
    
    /**
     * Set a uDeploy API request to obtain FULL data for a specified ComponentTemplate.
     * Note: ComponentTemplate - Details can be obtaine only by using ComponentTemplate ID.
     *
     * @param integer $template_id      Integer, holding the ID of the ComponentTemplate to be retrieved.
     *                                  This is required to build the API request.
     * @param string $template_name     String, holding the name of the ComponentTemplate to obtain.
     *                                  This is required for when we are saving the Template into a file.
     *
     * @return boolean|string           The uDeploy API request to be executed or
     *                                  FALSE in case of a problem with passed arguments.
     */
    public function getComponentTemplateDetails($template_id=null, $template_name=null) {
        if (empty($template_name)) {
            $template_name = $template_id;
        }
        if ($this->return == 'file') {
            $this->destination = " > '{$this->output}/{$template_name}.json'";
        }
        $action = $this->weburl . '/rest/deploy/componentTemplate/' . $template_id . $this->destination;
        return $this->execRest($action);
    }
    
    /**
     * Set a uDeploy API request-command to obtain, ALL available Component Templates.
     * NOTE: The returned list with templates will ONLY hold a partial (general) Template data,
     *       for each ComponentTemplate.
     *
     * @return string - The uDeploy API request to be executed
     */
    public function getComponentTemplates() {
        if ($this->return == 'file') {
            $this->destination = " > '{$this->output}/componentTemplates.json'";
        }
        $action = $this->weburl . '/rest/deploy/componentTemplate' . $this->destination;
        return $this->execRest($action);
    }
    
    /**
     * @param string    $environment
     * @return string
     *
     * Will return false if argument is incorrect
     */
    public function getComponentVersionPropDefs($component = null){
        if ($this->return == 'file') {
            $this->destination = " > '{$this->output}/{$component}.propDefs.json'";
        }
        $action = "getComponentVersionPropDefs -component '{$component}'{$this->destination}";
        return $this->execRest($action);
    }
    
    /**
     *
     * Returns a paginated list of component versions
     *
     * @param string $component
     * @param number $rowsPerPage
     * @param number $pageNumber
     * @param string $orderField
     * @param string $sortType
     * @return boolean|string
     */
    public function getComponentVersions($component = null, $rowsPerPage = 10, $pageNumber = 1, $orderField = 'dateCreated', $sortType = 'desc'){
        if ($this->return == 'file') {
            $this->destination = " > '{$this->output}/{$component}.versions.{$pageNumber}.json'";
        }
    
        /* DEPRECATED ***
         $action = "'{$this->weburl}/rest/deploy/component/{$component}/versions/paginated/false?"
        . "rowsPerPage={$rowsPerPage}"
        . "&pageNumber={$pageNumber}"
        . "&orderField={$orderField}"
        . "&sortType={$sortType}' "
        . "{$this->destination}";
        ***/
    
        $action = "'{$this->weburl}/rest/deploy/component/{$component}/versions/paginated' {$this->destination}";
    
        return $this->execRest($action);
    }
    
    /**
     *
     * Return componentVersions in a snapshot
     *
     * @param string $snapshotId
     * @return boolean|string
     */
    public function getComponentVersionsInSnapshot($snapshotId = null) {
        if ($this->return == 'file') {
            $this->destination = "> '{$this->output}/{$snapshotId}.componentVersions.json'";
        }
        $action = $this->weburl . "/rest/deploy/snapshot/{$snapshotId}/versions {$this->destination}";
        return $this->execRest($action);
    }
    
    /**
     *
     * Get all tags in a component
     *
     * @param string $component
     * @param string $componentName
     * @return boolean
     */
    public function getTagsOnComponent($component = null, $componentName = null) {
        if (empty($componentName)) {
            $componentName = $component;
        }
    
        if ($this->return == 'file') {
            $this->destination = " > '{$this->output}/{$componentName}.tags.json'";
        }
        $action = "getTagsOnComponent -component '{$component}'{$this->destination}";
        return $this->exec($action);
    }
    
    /* Create a STRING hoolding a POST curl request to be used for
    * IMPORTING or UPGRADE of a Component.
    *
    * This function will CHECK if the component to be Imported,
    * is based on a Component Template and
    * WILL ENSURE the corresponding Template is PRESENT on the Server,
    * before to prepare the IMPORT Component request string.
    *
    * @param string $componentFile [REQUIRED] Path to a Component JSON file
    *                              If not provided => $this->args[0] will be used.
    *                              If BOTH are not set => function will return FALSE
    * @param boolean $upgrade      [REQUIRED] Boolean TRUE/FALSE, used to set whether
    *                              to set an UPGRADE or IMPORT POST request correspondingly.
    *                              If not provided => $this->args[1] will be used.
    *                              If BOTH are not set => function will return false
    *
    * @return boolean|string   String holding a POST request for Component Import/Upgrade to be executed
    *                          or Boolean FALSE in case of either of the required parameters is NOT provided.
    */
    public function importComponent ($componentFile = null, $upgrade = null) {
        /*
         * Check if Component is based on a Template and
        * make sure the Template is present on server before to import the component
        */
        // STEP 1: Get contents of the Component to be imported and use it as Object
        $component = file_get_contents($componentFile);
        if ( $component === false ) {
            $this->log->error(__CLASS__ . '::' . __FUNCTION__
                    . ": There is a problem with reading Component backup file: \n" . $componentFile
                    . " Component IMPORT CANNOT be COMPLETED");
            return false;
        }

         
        // Prepare POST Request-action to be executed
        $action = "-X POST ";
        // NOTE the whole URL part MUST be in quotes. Otherwise it will brake on the & sign
        $action .= "'" . $this->weburl . '/rest/deploy/component/';
        // Import or Upgrade ComponentTemplate
        $action .= $upgrade ? 'upgrade?' : 'import?';
        // Use existent ComponentTemplate or Create it if it is Missing
        $action .= 'upgradeType=USE_EXISTING_IF_EXISTS';
        // Use existent Process or Create it if it is missing
        $action .= '&processUpgradeType=USE_EXISTING_IF_EXISTS\'';
        // Set parameters required for the POSt request
        $action .= ' -H "Content-Type: multipart/form-data"';
        $action .= " -F file=@'" . $componentFile . "'";
         
        // Return
        return $this->execRest($action);
    }
    
    /**
     * Create a STRING hoolding a POST curl request to be used for
     * IMPORTING or UPGRADE of a ComponentTemplate.
     *
     * @param string $componentTemplate [REQUIRED] Path to a ComponentTemplate JSON file
     *                                  If not provided => $this->args[0] will be used.
     *                                  If BOTH are not set => function will return FALSE
     * @param boolean $upgrade          [REQUIRED] Boolean TRUE/FALSE, used to set whether
     *                                  to set an UPGRADE or IMPORT POST request correspondingly.
     *                                  If not provided => $this->args[1] will be used.
     *                                  If BOTH are not set => function will return false
     *
     * @return boolean|string   String holding a POST request for ComponentTemplate Import/Upgrade to be executed
     *                          or Boolean FALSE in case of either of the required parameters is NOT provided.
     */
    public function importComponentTemplate ($componentTemplate = null, $upgrade = null) {
        // Prepare POST Request-action to be executed
        $action = "-X POST ";
        $action .= $this->weburl . '/rest/deploy/componentTemplate/';
        // Import or Upgrade ComponentTemplate
        $action .= $upgrade ? 'upgrade?' : 'import?';
        // Use existent Process or Create it if it is missing
        $action .= 'processUpgradeType=USE_EXISTING_IF_EXISTS';
        // Set parameters required for the POSt request
        $action .= ' -H "Content-Type: multipart/form-data"';
        $action .= " -F file=@'" . $componentTemplate . "'";
         
        // Return
        return $this->execRest($action);
    }
    
    /**
     *
     * Get Version by id
     *
     * @return boolean|string
     */
    public function getVersionFromId($versionId) {
        if ($this->return == 'file') {
            $this->destination = "> '{$this->output}/version.{$versionId}.json'";
        }
        $action = $this->weburl . "/rest/deploy/version/{$versionId} {$this->destination}";
        return $this->execRest($action);
    }
    
    
    
    /**
     *
     * Remove component from Application
     *
     */
    public function removeComponentFromApplication ($applicationId = null, $componentId = null) {
        if ($this->return == 'file') {
            $this->destination = " > '{$applicationId}/{$componentId}.json'";
        }
        $jsonData = '{"components":["'. $componentId . '"]}';
        
        $action = "-X PUT ";
        $action .= '-H "Content-Type: application/x-www-form-urlencoded" ';
        $action .= " --data '$jsonData' --compressed ";
        $action .= "'{$this->weburl}/rest/deploy/application/$applicationId/removeComponents' {$this->destination} ";
        return $this->execRest( $action );
    }
    
    
    /**
     *
     * Upgrade a component using a json file.
     * First argument is the path of the component file. uDeploy will then upgrade the component with the same name
     * If upgradeTemplate is set (by default, USE_EXISTING_IF_EXISTS) also component template are upgraded
     * If upgradeProcess is set (by default, USE_EXISTING_IF_EXISTS) also component processes are upgraded
     *
     * allowed value for upgradeTemplate and upgradeProcess
     *  USE_EXISTING_IF_EXISTS
     *  CREATE_NEW_IF_EXISTS
     *  FAIL_IF_EXISTS
     *  FAIL_IF_DOESNT_EXIST
     *  UPGRADE_IF_EXISTS
     *
     * @param string $component
     * @param string $upgradeProcess
     * @param string $upgradeProcess
     */
    public function upgradeComponent($component = null, $upgradeTemplate = 'USE_EXISTING_IF_EXISTS', $upgradeProcess = 'USE_EXISTING_IF_EXISTS') {
        $command = "-F file=@'{$component}' ";
        $command .= "'{$this->weburl}/rest/deploy/component/upgrade?";
        if (in_array( $upgradeTemplate, $this->upgrade_values )) {
            $command .= "upgradeType=$upgradeTemplate";
        } else {
            $command .= "upgradeType=USE_EXISTING_IF_EXISTS";
        }
    
        if (in_array( $upgradeProcess, $this->upgrade_values )) {
            $command .= "&processUpgradeType=$upgradeProcess'";
        } else {
            $command .= "&processUpgradeType=USE_EXISTING_IF_EXISTS'";
        }
        return $this->execRest($command);
    }
    
    /**
     * Set up a uDeploy API Request for UPDATING of a specified ComponentTemplate
     * from a JSON formatted file.
     *
     * @param string $componentTemplate Path to the JSON File, to be used for ComponentTemplate update
     * @param string $upgradeProcess    Boolean FLAG to set wheather to update ComponentTemplate Process(es) or Not
     *                                  DEFAULT value: TRUE
     *
     * @return boolean|string   String, holding the uDeploy API UPDATE action to be executed OR
     *                          Boolean FALSE, when $componentTemplate File is not provided
     */
    public function upgradeComponentTemplate($componentTemplate = null, $upgradeProcess = true) {
        $command = "-F file=@'{$componentTemplate}' ";
        $command .= "{$this->weburl}/rest/deploy/componentTemplate/upgrade?";
        if ($upgradeProcess) {
            $command .= "&processUpgradeType=UPGRADE_IF_EXISTS ";
        } else {
            $command .= "&processUpgradeType=USE_EXISTING_IF_EXISTS";
        }
    
        return $this->execRest($command);
    }

}