<?php
/****************************************************************
 * IBM Confidential
*
* SFA100-Collaboration Source Materials
*
* (C) Copyright IBM Corp. 2015
*
* The source code for this program is not published or otherwise
* divested of its trade secrets, irrespective of what has been
* deposited with the U.S. Copyright Office
*
* **************************************************************/

require_once('include/EntryPoint.php');

/**
 * Provides tools for creating and managing components requests
 *
 * @class Component
 * @author Felice Geracitano
 */
class Component extends EntryPoint {
    public function __construct($alias = null){
        parent::__construct($alias);
        $this->setReturn('json');
    }

    /**
     * Add component to application
     *
     * @param string $component
     * @return boolean
     */
    public function addToApplication($components, $application)
    {
        if (empty($components)) {
            Utils::CLIerror("Component Name/ID is missing.");
            return false;
        }
        if (empty($application)) {
            Utils::CLIerror("Application Name/ID is missing.");
            return false;
        }
        
        if (!is_array($components)) {
            $components = array( $components );
        }
        $return = array();
        foreach ($components as $comp) {
            // if is a complex array get the component name, otherwise leave the id string
            $comp = (isset($comp['name'])) ? $comp['name'] : $comp;
            $result = $this->udclient()->component()->addComponentToApplication($comp, $application);
            if ($result === false) {
                Utils::CLIerror("Failed adding '$comp' to the application");
                Utils::CLIdebug("Failed adding component '$comp' to Application $application");
                $return["$comp"] = false;
                continue;
            } else {
                Utils::CLIdebug("[ADDED] '$comp'.");
                $return["$comp"] = true;
            }
        }
        return $return;
    }

    /**
     * Add component to team
     *
     * @param string Name|ID $component
     * @param string Name $team
     * @return boolean
     */
    public function addToTeam($component, $team) {
        if (empty( $component)) {
            Utils::CLIerror( "Component Name/ID is missing." );
            return false;
        }
        if (empty($team)) {
            Utils::CLIerror( "Team is missing." );
            return false;
        }
        if (!$this->exists($component)){
            Utils::CLIerror( "Component does not exist." );
            return false;
        }
        // Connect to the current alias
        $teamLayer = $this->getClass('Team', $this->alias);
        if (!$teamLayer->exists($team)){
            Utils::CLIerror( "Team does not exist." );
            return false;
        }
        
        $result = $this->udclient()->component()->addComponentToTeam($component, $team);
        if ($result === false) {
            Utils::CLIerror( "Failed adding component to Team" );
            Utils::CLIdebug( "Failed adding component: $component to team: $team" );
            return false;
        }
        return true;
    }

    /**
     * Add files to component version
     * 
     * @param string $component Name
     * @param string $version Name
     * @param string $base path directory
     * @param string $include files to include
     * @param string $exclude files to exclude
     *
     * @return boolean
     */
    public function addFilesToVersion($component, $version, $base, $include = null, $exclude = null)
    {
        // Validate arguments
        if (empty($component)) {
            Utils::CLIerror("Component Name is missing.");
            return false;
        }
        if (empty($version)) {
            Utils::CLIerror("Version Name is missing.");
            return false;
        }
        if (empty($base) || !is_dir($base)) {
            Utils::CLIerror("Directory path is invalid.");
            return false;
        }
        // Validate component version
        $this->setReturn('array');
        $versionObj = $this->getVersion($component, $version);
        if ($versionObj === false) {
            Utils::CLIerror("Version '$version' does not exist for component '$component'.");
            return false;
        }
        // Trasform comma separated list into array
        if (! empty($include)) {
            $include = explode(',', $include);
        }
        if (! empty($exclude)) {
            $exclude = explode(',', $exclude);
        }
        $result = $this->udclient()->component()->addVersionFiles($component, $version, $base, $include, $exclude);
        if ($result === false) {
            Utils::CLIerror("Failed to upload the files to component '$component' at version '$version'.");
            return false;
        }
        return true;
    }
    
    /**
     * Create a Component from scratch or import from a file path
     *
     * @param array|string $component
     * @return array|string|boolean
     */
    public function create($component) {

        if (empty($component)){
            Utils::CLIerror( "Empty component parameter." );
            return false;
        }

        $file_name = null;
        // Check if arg is file path or array obj
        if (is_array( $component ) || is_object( $component )) {
            $file_name = Utils::createJsonFile( $component );
        } else {
            if (file_exists($component) === false) {
                Utils::CLIerror( "Json file does not exist" );
                Utils::CLIdebug( __CLASS__ . '::' . __FUNCTION__.": Json file '{$component}' does not exist" );
                return false;
            }
            if (Utils::validateJSON( file_get_contents( $component ) ) === false) {
                Utils::CLIerror( "Json file has wrong format" );
                Utils::CLIdebug( __CLASS__ . '::' . __FUNCTION__.": Json file '{$component}' has wrong format" );
                return false;
            }
            $file_name = $component;
        }
        if ($this->rest()->component()->importComponent( $file_name, false ) === false) {
            Utils::CLIerror( "Component already exist" );
            return false;
        }
        is_string( $component ) === false ? unlink( $file_name ) : null;
        return true;
    }
    
    /**
     * Create a Component Process from scratch or import a from a file path
     *
     * @param array|json_file_path $process
     * @return boolean
     */
    public function createProcess($componentId, $process) {
        $file_name = null;
        if (empty ($componentId) || empty ($process)){
            Utils::CLIerror( "Invalid null parameter/s." );
            Utils::CLIdebug( __CLASS__ . '::' . __FUNCTION__.": Invalid null parameter/s." );
            return false;
        }
    
        // Check if arg is file path or array obj
        if (is_array( $process ) || is_object( $process )) {
            $file_name = Utils::createJsonFile( $process );
        } else {
            if (file_exists($process) === false) {
                Utils::CLIerror( "Json file does not exist" );
                Utils::CLIdebug( __CLASS__ . '::' . __FUNCTION__.": Json file '{$process}' does not exist" );
                return false;
            }
            if (Utils::validateJSON( file_get_contents( $process ) ) === false) {
                Utils::CLIerror( "Json file has wrong format" );
                Utils::CLIdebug( __CLASS__ . '::' . __FUNCTION__.": Json file '{$process}' has wrong format" );
                return false;
            }
            $file_name = $process;
        }
        if ($this->udclient()->component()->createComponentProcess( $componentId,$file_name ) === false) {
            Utils::CLIerror( "Failed to create component process." );
            Utils::CLIdebug( __CLASS__ . '::' . __FUNCTION__.": Failed to create component process, for component: $componentId" );
            return false;
        }
        is_string( $process ) === false ? unlink( $file_name ) : null;
        return true;
    }

    /**
     * Create a Component Template from scratch or import a from a file path
     *
     * @param array|json_file_path $componentTemplate
     * @return boolean
     */
    public function createTemplate($componentTemplate , $upgrade = false) {
        $file_name = null;
        // Check if arg is file path or array obj
        if (is_array( $componentTemplate ) || is_object( $componentTemplate )) {
            $file_name = Utils::createJsonFile( $componentTemplate );
        } else {
            if (file_exists($componentTemplate) === false) {
                Utils::CLIerror( "Json file does not exist" );
                Utils::CLIdebug( __CLASS__ . '::' . __FUNCTION__.": Json file '{$componentTemplate}' does not exist." );
                return false;
            }
            if (Utils::validateJSON( file_get_contents( $componentTemplate ) ) === false) {
                Utils::CLIerror( "Json file has wrong format" );
                Utils::CLIdebug( __CLASS__ . '::' . __FUNCTION__.": Json file '{$componentTemplate}' has wrong format." );
                return false;
            }
            $file_name = $componentTemplate;
        }
        if ($this->rest()->component()->importComponentTemplate( $file_name, $upgrade ) === false) {
            Utils::CLIerror( "Failed to create Component Template." );
            return false;
        }
        is_string( $componentTemplate ) === false ? unlink( $file_name ) : null;
        return true;
    }
    
    /**
     * Create new version of a component
     *
     * @param string $component
     * @param string $version
     * @return array|boolean
     */
    public function createComponentVersion($component, $version, $description = null) {
        if (empty($component)){
            Utils::CLIerror( "Component name/id is missing" );
            return false;
        }
        if (empty($version)){
            Utils::CLIerror( "Version name is missing" );
            return false;
        }
        $result = $this->udclient()->component()->createVersion($component, $version, $description);
        if ($result === false){
            Utils::CLIerror( "Failed to create component version" );
            Utils::CLIdebug( __CLASS__ . '::' . __FUNCTION__.": Failed to create version '$version' of component '$component'" );
            return false;
        }
        
        if ($this->return === 'file'){
            // delete quotes and return path
            return substr($result,1,-1);
        }
        return Utils::outputToArray( $result );
    }
    
    /**
     * Delete a component passing Id or Name
     *
     * @param string $component
     * @return boolean
     */
    public function delete($component) {
        if (empty($component)){
            Utils::CLIerror( "Component name/id is missing" );
            return false;
        }
        // Delete component by Name or Id
        $result = $this->rest()->component()->deleteComponent($component);
        if ($result === false){
            Utils::CLIerror( "Component does not exist" );
            return null;
        }
        return true;
    }
    
    /**
     * Delete a component process passing Id
     *
     * @param string $component
     * @return boolean
     */
    public function deleteProcess($componentId) {
        if (empty($componentId)){
            Utils::CLIerror( "Component Process id is missing" );
            return false;
        }
        // Delete component process by Id
        $result = $this->rest()->component()->deleteComponentProcess($componentId);
        if ($result === false){
            Utils::CLIerror( "Component process does not exist." );
            return null;
        }
        return true;
    }
    
    /**
     * Delete a component Version
     *
     * @param string $version
     * @return boolean
     */
    public function deleteVersion($versionId) {
        if (empty($versionId)){
            Utils::CLIerror( "Missing version ID." );
            return false;
        }
        // Delete component by Name or Id
        $result = $this->rest()->component()->deleteComponentVersion($versionId);
        if ($result === false){
            Utils::CLIerror( "Failed to remove the Component Version." );
            return false;
        }
        $result = Utils::outputToArray( $result );
        if ($result['passed'] != true){
            Utils::CLIerror( "Failed to remove the Component Version." );
            return false;
        }
        return true;
    }
    
    /**
     * Check if Component exists
     *
     * @param string $comp
     * @return boolean
     */
    public function exists($comp)
    {
        $return_type = $this->return;
        if (empty($comp) || !is_string($comp)) {
            Utils::CLIerror("Invalid Component parameter.");
            return false;
        }
        $this->setReturn('json');
        $comps = $this->getList();
    
        $result = false;
        foreach ($comps as $item) {
            if ($item['name'] == $comp || $item['id'] == $comp) {
                $result = true;
            }
        }
        $this->setReturn($return_type);
        return $result;
    }
    
    /**
     * Get short or full description component from Id or Name
     *
     * @param string $component
     * @param boolean $full_object
     * @return array|path|boolean
     */
    public function get($component, $full_object = false) {
        $result = null;
        $component_name = null;
        if (empty($component)){
            Utils::CLIerror( "Component name/id is missing" );
            return false;
        }
        if (! Utils::isUuid($component)){
            $component_name = $component;
        }
        // getComponent provides large description, getComponentRest provides
        // small descripion
        if (empty( $full_object )) {
            $result = $this->rest()->component()->getComponentRest( $component, $component_name );
        } else {
            $result = $this->udclient()->component()->getComponent( $component );
        }
        if ($result === false) {
            Utils::CLIerror( "Component does not exist" );
            return null;
        }
        if ($this->return === 'file'){
            // delete quotes and return path
            return substr($result,1,-1);
        }
        return Utils::outputToArray( $result );
    }
    
    /**
     * Get list of all components from UCD server
     * @param string 'application name'
     * @return array|path|boolean
     */
    public function getList($application = null) {
        if (empty($application)){
            $result = $this->rest()->component()->getComponentsRest();
        }
        else{
            $result = $this->udclient()->component()->getComponentsInApplication( $application );
        }
        if ($result === false){
            Utils::CLIerror( "Failed to retrieve components list" );
            return null;
        }
        if ($this->return === 'file'){
            // delete quotes and return path
            return substr($result,1,-1);
        }
        return Utils::outputToArray( $result );
    }
    
    /**
     * Get tag from component
     * @param string 'component'
     * @return array path boolean
     */
    public function getTags($component, $componentName = null) {
        if (empty( $component ) && empty( $componentName )) {
            Utils::CLIerror( "Component name/id is missing" );
            return false;
        }
        $result = $this->rest()->component()->getTagsOnComponent( $component, $componentName );
        if ($result === false) {
            Utils::CLIerror( "Failed to retrieve tag list" );
            return null;
        }
        if ($this->return === 'file') {
            // delete quotes and return path
            return substr( $result, 1, - 1 );
        }
        return Utils::outputToArray( $result );
    }
    
    /**
     * Get specific version info of a component
     *
     * @param string $component
     * @return array|boolean
     */
    public function getVersion($component, $version) {
        $found = false;
        // Save parent request
        $parent_request = $this->return;
        $this->setReturn( 'array' );
        $versions = $this->listVersions( $component );
        // Allow version parameter as Name or ID 
        foreach ( $versions as $ver ) {
            if ($ver ['name'] == $version || $ver ['id'] == $version) {
                $found = true;
                $version = $ver ['id'];
                break;
            }
        }
        
        if ($found === false) {
            Utils::CLIerror( "Version does not exist" );
            return false;
        }
        // Come back to parent request
        $this->setReturn( $parent_request );
        $result = $this->rest()->component()->getVersionFromId( $version );
        if ($result === false) {
            Utils::CLIerror( "Failed to retrive version of component" );
            return false;
        }
        if ($this->return === 'file') {
            // delete quotes and return path
            return substr( $result, 1, - 1 );
        }
        return Utils::outputToArray( $result );
    }
    
    
    /**
     * Get Versions in a specific spapshot
     *
     * @param string $snapshot
     * @return array|string|boolean
     */
    public function getVersionsInSnapshot($snapshotId) {
        
        if (empty( $snapshotId )) {
            Utils::CLIerror( "Snapshot id is missing." );
            return false;
        }
        $result = $this->rest()->component()->getComponentVersionsInSnapshot($snapshotId);
        if ($result === false) {
            Utils::CLIerror( "Failed to retrive components versions." );
            return false;
        }
        if ($this->return === 'file') {
            // delete quotes and return path
            return substr( $result, 1, - 1 );
        }
        return Utils::outputToArray( $result );
    }
    
    
    /**
     * Get versions list of a component 
     *
     * @param string Id or Name $component
     * @return array|boolean
     */
    public function listVersions($component) {
        if (empty($component)){
            Utils::CLIerror( "Component name/id is missing" );
            return false;
        }
        $result = $this->rest()->component()->getComponentVersions($component);
        if ($result === false){
            Utils::CLIerror( "Failed to retrieve the list of versions" );
            Utils::CLIdebug( __CLASS__ . '::' . __FUNCTION__.": Failed to retrieve the list versions for the component'{$component}'" );
            return false;
        }
        $tmp = Utils::outputToArray( $result );
        if (empty($tmp)){
            // Version list is empty
            return null;
        }
        if ($this->return === 'file'){
            // delete quotes and return path
            return substr($result,1,-1);
        }
        return Utils::outputToArray( $result );
    }
    
    /**
     *  upgrade a component passing array object or json file path
     *  allowed value for templateUpgrade and processUpgrade of the component
     ** USE_EXISTING_IF_EXISTS
     ** CREATE_NEW_IF_EXISTS
     ** FAIL_IF_EXISTS
     ** FAIL_IF_DOESNT_EXIST
     ** UPGRADE_IF_EXISTS
     *
     * @param array | string $component
     * @param string $processUpgrade
     * @param string $templateUpgrade
     * @return boolean
     */
    public function upgrade($component, $templateUpgrade = 'UPGRADE_IF_EXISTS', $processUpgrade = 'UPGRADE_IF_EXISTS') {
        $file_name = null;
        // Check if arg is file path or array obj
        if (is_array( $component ) || is_object( $component )) {
            $file_name = Utils::createJsonFile( $component );
        } else {
            if (file_exists($component) === false) {
                Utils::CLIerror( "Json file does not exist" );
                Utils::CLIdebug( __CLASS__ . '::' . __FUNCTION__.": Json file '{$component}' does not exist" );
                return false;
            }
            if (Utils::validateJSON( file_get_contents( $component ) ) === false) {
                Utils::CLIerror( "Json file is invalid" );
                Utils::CLIdebug( __CLASS__ . '::' . __FUNCTION__.": Json file '{$component}' Json file is invalid" );
                return false;
            }
            $file_name = $component;
        }
        if (! in_array( $templateUpgrade, $this->upgrade_values ) && ! is_null( $templateUpgrade )) {
            Utils::CLIerror( "Template upgrade has wrong parameter" );
            Utils::CLIdebug( __CLASS__ . '::' . __FUNCTION__ .": Template upgrade has wrong parameter '{$templateUpgrade}'" );
            return false;
        }
        if (! in_array( $processUpgrade, $this->upgrade_values ) && ! is_null( $processUpgrade )) {
            Utils::CLIerror( "Process upgrade has wrong parameter" );
            Utils::CLIdebug( __CLASS__ . '::' . __FUNCTION__ .": Process upgrade has wrong parameter '{$processUpgrade}'" );
            return false;
        }
        $this->setReturn( 'json' );
        if ($this->rest()->component()->upgradeComponent( $file_name, $templateUpgrade, $processUpgrade ) === false) {
            Utils::CLIdebug( __CLASS__ . '::' . __FUNCTION__ .": Component upgrade could have failed for these reasons: template option '$templateUpgrade', process option '$processUpgrade'");
            return false;
        }
        is_string( $component ) === false ? unlink( $file_name ) : null;
        return true;
    }


    /** 
     * Upgrade a Component Template
     * @param array | string $componentTemplate
     * @param boolean $upgradeProcess
     * @return boolean
     */
    public function upgradeTemplate( $componentTemplate ) {
        $file_name = null;
        // Check if arg is file path or array obj
        if (is_array( $componentTemplate ) || is_object( $componentTemplate )) {
            $file_name = Utils::createJsonFile( $componentTemplate );
        } else {
            if (file_exists($componentTemplate) === false) {
                Utils::CLIerror( "Json file does not exist" );
                Utils::CLIdebug( __CLASS__ . '::' . __FUNCTION__.": Json file '{$componentTemplate}' does not exist." );
                return false;
            }
            if (Utils::validateJSON( file_get_contents( $componentTemplate ) ) === false) {
                Utils::CLIerror( "Json file has wrong format" );
                Utils::CLIdebug( __CLASS__ . '::' . __FUNCTION__.": Json file '{$componentTemplate}' has wrong format." );
                return false;
            }
            $file_name = $componentTemplate;
        }
        if ($this->rest()->component()->upgradeComponentTemplate($componentTemplate) === false) {
            Utils::CLIerror( "Failed to update Component Template." );
            return false;
        }
        is_string( $componentTemplate ) === false ? unlink( $file_name ) : null;
        return true;
    }


    
    /**
     * Get Component Templates details
     * @return array path boolean
     */
    public function getTemplate($template_id, $template_name = null) {
        if (empty($template_id)){
            Utils::CLIerror( "Template id empty." );
            Utils::CLIdebug( __CLASS__ . '::' . __FUNCTION__ .": Template id empty." );
            return false;
        }
        $result = $this->rest()->component()->getComponentTemplateDetails($template_id, $template_name);
        if (empty($result)){
            Utils::CLIerror( "Failed to retrive template details." );
            Utils::CLIdebug( __CLASS__ . '::' . __FUNCTION__ .": Failed to retrive template details for id: {$template_id}, name: {$template_name} " );
            return false;
        }
        if ($this->return === 'file') {
            // delete quotes and return path
            return substr( $result, 1, - 1 );
        }
        return Utils::outputToArray( $result );
    }
    
    /**
     * Get Component Templates
     * @return array path boolean
     */
    public function listTemplates() {
        $result = $this->rest()->component()->getComponentTemplates();
        if (empty($result)){
            Utils::CLIerror( "Failed to retrive template list." );
            Utils::CLIdebug( __CLASS__ . '::' . __FUNCTION__ .": Failed to retrive template list." );
            return false;
        }
        if ($this->return === 'file') {
            // delete quotes and return path
            return substr( $result, 1, - 1 );
        }
        return Utils::outputToArray( $result );
    }

    /**
     * Remove Component from application
     *
     * @param string $component Id|Name
     * @return boolean
     */
    public function removeComponentFromApplications ($component)
    {
        // Check Parameter
        if (empty($component) || ! is_string($component)) {
            Utils::CLIerror("Invalid Component parameter.");
            return false;
        }
        // Save Return
        $return_old = $this->return;
        
        // Removing the comp from all the applications
        $compLayer = new Component();
        $compObj = $compLayer->get($component, true);
        if (empty($compObj['applications'])) {
            return true; // Not linked
        }
        $errors = false;
        foreach ($compObj['applications'] as $application) {
            // Run Api
            $result = $this->rest()->component()->removeComponentFromApplication($application['id'], $compObj['id']);
            if ($result == false) {
                Utils::CLIerror("Failed to remove component '${compObj['name']}' from application '${application['name']}'.");
                $errors = true;
            }
        }
        $this->setReturn($return_old);
        return ! $errors;
    }
    
    /**
     * Check if the component version exist
     *
     * @param string $component
     * @param string $version
     * @return boolean
     */
    public function versionExists($component, $version) {
        $this->setReturn( 'json' );
        $versions = $this->rest()->component()->getComponentVersions($component);
        $versions = Utils::outputToArray($versions);
        foreach ($versions as $ver) {
            if ($ver['name'] == $version || $ver['id'] == $version) {
                return true;
            }
        }
        return false;
    }
    
}