<?php
/**
 * **************************************************************
 * IBM Confidential
 *
 * SFA100-Collaboration Source Materials
 *
 * (C) Copyright IBM Corp. 2015
 *
 * The source code for this program is not published or otherwise
 * divested of its trade secrets, irrespective of what has been
 * deposited with the U.S. Copyright Office
 *
 * *************************************************************
 */
require_once ('include/EntryPoint.php');

/**
 * Environment class would be used to retrieve environments list from
 * application, using udclient's APIs
 *
 * @class Environment
 *
 * @author Kun Niu kunniu@ie.ibm.com
 * @author Moises Dominguez moisesdg@ie.ibm.com
 * @author Kostadin Ivanov kostadii@ie.ibm.com
 *        
 */
class Environment extends EntryPoint {
    public $environments = array ();
    public function __construct($alias = null) {
        parent::__construct($alias);
        $this->setReturn( 'json' );
    }
    
    /**
     * Create environment from a json file
     *
     * @param string $file            
     * @return boolean
     */
    public function create($file) {
        // Check if file exists and is valid
        if (! is_file( $file ) || ! Utils::validateJSON( 
                file_get_contents( $file ) )) {
            Utils::CLIerror( "'{$file}' is not a valid JSON file." );
            return false;
        }
        
        // Check for errors
        $newEnvironment = json_decode( file_get_contents( $file ), true );
        $error = false;
        if (empty( $newEnvironment ['name'] )) {
            Utils::CLIerror( "Environment name is missing." );
            $error = true;
        }
        
        if (empty( $newEnvironment ['application'] )) {
            Utils::CLIerror( "Application name is missing" );
            $error = true;
        } else {
            Utils::CLIinfo( "Retrieving existing environments" );
            $getEnvResult = self::getList( 
                    $newEnvironment ['application'] );
            if ($getEnvResult === false) {
                return false;
            } else {
                $this->environments = $getEnvResult;
            }
        }
        
        foreach ( $this->environments as $envName => $envId ) {
            if ($envName == $newEnvironment ['name']) {
                Utils::CLIerror( 
                        "Environment {$newEnvironment['name']} already exists in application {$newEnvironment['application']}." );
                $error = true;
            }
        }
        
        if (empty( $newEnvironment ['baseResource'] )) {
            Utils::CLIerror( "Base resource is missing." );
            $error = true;
        }
        if (empty( $newEnvironment ['blueprint'] )) {
            Utils::CLIerror( "Blueprint is missing." );
            $error = true;
        }
        
        //  An error occured
        if ($error) {
            return false;
        }
        
        // All right, create environment
        Utils::CLIinfo( 
                "Creating environment '{$newEnvironment['name']}' " . 
                "for application '{$newEnvironment['application']}'" );
        $return = $this->rest()->environment()->provisionEnvironment( $file );
        
        if ($return == false) {
            Utils::CLIerror( 
                    "An error occurred while creating a new environment." );
            return false;
        } else {
            Utils::CLIout( "Environment successfully created.", true, 
                    'success' );
        }
        return true;
    }
    
    /**
     * Delete environment passing id or name env name|id apprlication
     *
     *
     * @param string $environment            
     * @param string $application            
     * @param bool $deleteResources            
     * @param bool $deleteCloud            
     * @return boolean
     */
    public function delete($environment, $application , $deleteResources = null, $deleteCloud = null) {
        $errors = false;
        $result = false;
        if (empty( $environment )) {
            Utils::CLIerror( "Missing environment." );
            $errors=true;
        }
        if (empty( $application )) {
            Utils::CLIerror( "Missing application." );
            $errors=true;
        }
        if ($errors===true){
            return false;
        }
        $result = $this->udclient()->environment()->deleteEnvironment( $environment , $application , $deleteResources, $deleteCloud );
        if ($result === false){
            Utils::CLIerror("Unable to delete '$environment' enviroment in '$application' application." );
            return false;
        }
        return true;
    }
    
    /**
     * Export environment properties to a json file based on environment id or
     * environment name
     *
     * @param string $environment            
     * @param string $application            
     * @return boolean
     */
    public function exportProperties($environment, $application) {

        // if $environment is environment id
        if (Utils::isUuid( $environment )) {
            $environmentId = $environment;
            if (isset( $application )) {
                $this->getList( $application );
                // array_search retuns false if it coudn't find the $environment
                // in the environments array
                $environmentName = array_search( $environmentId, 
                        $this->environments );
                if ($environmentName !== false) {
                    Utils::CLIinfo( 'Environment Name: "' . $environmentName . '" ' );
                    Utils::CLIinfo( 'Environment ID: "' . $environment . '" ' );
                    Utils::CLIinfo( 'Environment ID and environment name ' .'exist in the system, parsing the input file' );
                } else {
                    Utils::CLIerror( 'The environment ID: ' . $environmentId ." doesn't exist in the system: '$application' " );
                    return false;
                }
            } else {
                Utils::CLIerror( "The application name $application in the configuration is invalid" );
                return false;
            }
            /*
             * if the content of $environment is not the environment id get the
             * id
             */
        } else {
            $environmentName = $environment;
            Utils::CLIinfo( 'Environment Name: "' . $environmentName . '" ' );
            
            if (isset( $application )) {
                $environmentId = $this->getId( $environmentName, $application );
                if (isset( $environmentId )) {
                    Utils::CLIinfo( 'Environment ID: "' . $environmentId . '" ' );
                    Utils::CLIinfo( 'Environment ID and environment name exist in the system, parsing the input file' );
                } else {
                    Utils::CLIerror( "The environment Name or ID  doesn't exist in the system " );
                    return false;
                }
            } else {
                Utils::CLIerror( 'The application name in the configuration is invalid' );
                return false;
            }
        }
        
        Utils::CLIinfo( 'Getting all of the environments in the application ' .$application );
        Utils::CLIinfo( 'Getting all the environment properties from the server. This process takes some time' );
        
        // store all the components Environment in $componentsEnvironment
        $componentsEnvironment = $this->rest()->component()->getComponentEnvironment( $environmentId );
        Utils::CLIinfo( 'Parsing the environment properties.' );
        
        if (isset( $componentsEnvironment )) {
            $envPropsArray = Utils::outputToArray( $componentsEnvironment );
        } else {
            Utils::CLIerror( 'Problem with the server to get the components of the environment. The server might be down' );
        }
        
        $envProps = array ();
        $outputArray = array ();
        
        // loop all the properties in the array
        for($i = 0; $i < count( $envPropsArray ); $i++) {
            $groupComponents = array ();
            
            // loop all the components of the property $i and group them by
            // value
            for($j = 0; $j < count( $envPropsArray [$i] ['components'] ); $j++) {
                $groupComponents [$envPropsArray [$i] ['components'] [$j] ['propValue'] ['value']] [] = $envPropsArray [$i] ['components'] [$j] ['name'];
            }
            
            // if there is only one group of components for the property $i
            if (count( $groupComponents ) == 1) {
                // generate the $subContent array
                $subContent ['propertyName'] = $envPropsArray [$i] ['name'];
                $subContent ['value'] = $envPropsArray [$i] ['components'] [0] ['propValue'] ['value'];
                $subContent ['secure'] = ($envPropsArray [$i] ['components'] [0] ['propValue'] ['secure']) ? true : false;
                
                // add all the components to the $subContent array
                foreach ( $groupComponents as $componentValue ) {
                    foreach ( $componentValue as $component ) {
                        $subContent  ['components'] [] = $component;
                    }
                }
                
                // store the $subContent into $outputArray
                array_push( $outputArray, $subContent );
                $subContent = null;
                
              // if there are multiple groups of components for the property $i
            } else if (count( $groupComponents ) > 1) {
                // generate a $subContent array for each group components
                foreach ( $groupComponents as $key => $val ) {
                    $subContent ['propertyName'] = $envPropsArray [$i] ['name'];
                    //the value is the key of the array $groupComponents
                    $subContent ['value'] = $key;
                    $subContent ['secure'] = ($envPropsArray [$i] ['components'] [0] ['propValue'] ['secure']) ? true : false;
                    $subContent ['components'] = $val;
                    
                    // store the $subContent into $outputArray
                    array_push( $outputArray, $subContent );
                    $subContent = null;
                }
            }
            $groupComponents = null;
        }
        
        $outputfile = json_encode( $outputArray, true );
        Utils::CLIinfo( 
                'The environment properties back up is in : ' . $this->output .
                         '/' . $environmentId . '.json' );
        $return = Utils::writeToFile( "{$this->output}/{$environmentId}.json", 
                $outputfile, false );
        
        return $return;
    }
    
    public function getId($environmentName, $application) {
        if (empty( $environmentName )) {
            Utils::CLIerror( 'The Environment parameter is empty.' );
            return false;
        }
        
        // get the environmentId from the environments array
        if (isset( $this->environments [$environmentName] )) {
            return $this->environments [$environmentName];
            // if the environmentId is not in the environments array the
            // getEnvironments method will populate the array
        } else {
            if (empty( $application )) {
                Utils::CLIerror( 'The Application parameter is empty.' );
                return false;
            }
            $this->getList( $application );
            $environmentId = $this->environments [$environmentName];
            // check if the environmentId was set in the variable
            if (isset( $this->environments [$environmentName] )) {
                return $this->environments [$environmentName];
            } else {
                Utils::CLIerror( 
                        'There was a problem with retrieving the Environment Id' );
                return null;
            }
        }
    }
    public function getList($application) {
        // Get Environment list
        $this->setReturn( 'json' );
        if (empty( $application )) {
            Utils::CLIerror( 'The Application parameter is empty.' );
            return false;
        }
        $apps = $this->udclient()->application()->getApplications();
        
        if ($apps === false) {
            Utils::CLIerror( 
                    'There was a problem with retrieving the list of applications.' );
            return false;
        }
        $apps = Utils::outputToArray( $apps );
        if (count( $apps ) == 0) {
            Utils::CLIerror( 'There is no Application on the system.' );
            return false;
        }
        $exists = false;
        foreach ( $apps as $app ) {
            if ($application == $app ['name'] || $application == $app ['id']) {
                $exists = true;
                break;
            }
        }
        if ($exists == false) {
            Utils::CLIerror( 
                    'The Application "' . $application . '" does not exist.' );
            return false;
        }
        
        $environmentList = $this->udclient()->environment()->getEnvironmentsInApplication( $application );
        if ($environmentList === false) {
            Utils::CLIwarning( 
                    'There was a problem with retrieving the environments' .
                             ' for the Application: "' . $application . '".' );
            return false;
        }
        
        $environmentList = Utils::outputToArray( $environmentList );
        
        if (count( $environmentList ) == 0) {
            Utils::CLIwarning( 
                    'There is no Environment in Application: "' . $application .
                             '".' );
            return null;
        }
        
        foreach ( $environmentList as $environment ) {
            $name = $environment ['name'];
            $this->environments [$name] = $environment ['id'];
        }
        // Restore default return to json
        $this->setReturn( 'json' );
        return $this->environments;
    }
    
    
    /**
     * Parse an environment properties file and 
     * generates a JSON formatted string, 
     * compatible with the ucd Update Environment Properties webservice.
     *
     * @param string $file 
     *              - Path to Environment Properties JSON file           
     *
     * @return mixed Boolean FALSE in case of an Error 
     *               JSON formatted String ($parsedProperties) 
     *               with Environment Properties to be updated
     */
    private function parseEnvironmentPropertiesFile($file = null) {

        if (empty( $file ) || trim( $file ) == false) {
            Utils::CLIerror( 
                    'Path to Environment Properties JSON file is REQUIRED!' );
            return false;
        }
        
        // Read and Validate contents of the specified File
        $envProps = file_get_contents( $file );
        if ($envProps === false || Utils::validateJSON( $envProps ) === false) {
            Utils::CLIout(' ');
            Utils::CLIerror( 
                    'Provided Properties file: "' . $file .
                             '" was EMPTY or not in a correct JSON Format!' );
            return false;
        }
        
        // Set Environment Properties to process into Array
        $envPropsArray = json_decode( $envProps, true );
        
        // Set Array for the Properties to Parse to JSON
        $propsToParse = array ();
        
        // Loop through $envPropsArray
        foreach ( $envPropsArray as $propDetails ) {
            
            // check if the Environment property is secure
            // and has secured value, i.e. '****'
            if ($propDetails ['secure'] === true) {
                if ($propDetails ['value'] == '****') {
                    if (! isset( $problematicItems )) {
                        $problematicItems = $propDetails ['propertyName'] .PHP_EOL;
                    } else {
                        $problematicItems .= $propDetails ['propertyName'] . ' ' .PHP_EOL;
                    }
                    /*
                     * Move to next Property, as we don't want to add it to the
                     * $propsToParse
                     */
                    continue;
                }
                // else continue as usual
            }
            
            // Raise Warning and Skip property for update
            if (! is_array( $propDetails ['components'] ) ||
                     count( $propDetails ['components'] ) == 0) {
                Utils::CLIwarning( 
                        'Skipping property: "' . $propDetails ['propertyName'] .
                         '"' );
                Utils::CLIout( 'Missing Property Components' . PHP_EOL, false, 
                        'reason' );
            } else if ($propDetails ['components'] [0] == '*') {
                // Update Property to ALL Components
                $propKey = $propDetails ['propertyName'];
                $propsToParse [$propKey] = $propDetails ['value'];
            } else {
                // Update Property for Specified Components
                foreach ( $propDetails ['components'] as $compName ) {
                    $propKey = $compName . '/' . $propDetails ['propertyName'];
                    $propsToParse [$propKey] = $propDetails ['value'];
                }
            }
        }
        
        // If there were Properties with Secured values => Raise an ERROR
        if (isset( $problematicItems )) {
            Utils::CLIerror( 
                    'Aborting the operation. ' . PHP_EOL .
                             '        There is ENCRYPTED values in ' .
                             'the input file: ' . PHP_EOL . '        "' . $file .
                             '"' );
            Utils::CLIerror( 
                    'Please check the values of these properties: '  .PHP_EOL .
                             $problematicItems  );
            return false;
        } else {
            // Parse $propsToParse to JSON Format and return
            $parsedProperties = json_encode( $propsToParse );
            $parsedProperties = str_replace( '\/', '/', $parsedProperties );
            return $parsedProperties;
        }
    }
    
    
    /**
     * Import the environment properties of an application
     *
     * @param string $file
     *            - Path to Environment Properties JSON file
     * @param string $environment
     *            - Environment Id or Name
     *            
     * @return mixed Boolean FALSE in case of an Error or
     *         Result from call to function: $this->importEnvironmentProperties
     */
    public function importProperties($file = null, $environment = null, $application = null ) {
        
        if (! (file_exists( $file ))) {
            Utils::CLIerror( 'The file "' . $file . '" does not exist.' );
            return false;
        }
        $envProps = file_get_contents( $file );
        if ($envProps === false || Utils::validateJSON( $envProps ) === false) {
            Utils::CLIerror('Environment Properties JSON file is REQUIRED!' );
            return false;
        }
        
        if (Utils::isUuid($environment)) {
            $environmentId = $environment;
            if (isset( $application )) {
                $this->getList( $application );
                // array_search retuns false if it coudn't find the $environment
                // in the environments array
                $environmentName = array_search( $environmentId, 
                        $this->environments );
                if ($environmentName !== false) {
                    Utils::CLIinfo( 'Environment Name: "' . $environmentName . '" ' );
                    Utils::CLIinfo( 'Environment ID: "' . $environment . '" ' );
                    Utils::CLIinfo( 'Environment ID and environment name ' . 'exist in the system, parsing the input file' );
                } else {
                    Utils::CLIerror( 'The environment ID: ' . $environmentId ." doesn't exist in the system: '$application' " );
                    return false;
                }
            } else {
                Utils::CLIerror( 
                        'The application name in the configuration is invalid' );
                return false;
            }
            /*
             * if the content of $environment is not the environment id 
             * get the id
             */
        } else {
            $environmentName = $environment;
            Utils::CLIinfo( 'Environment Name: "' . $environmentName . '" ' );
            
            if (isset( $application )) {
                $environmentId = $this->getId( $environmentName, $application );
                if (isset( $environmentId )) {
                    Utils::CLIinfo( 'Environment ID: "' . $environmentId . '" ' );
                    Utils::CLIinfo( 'Environment ID and environment name exist in the system, parsing the input file' );
                } else {
                    Utils::CLIerror( "The environment Name or ID  doesn't exist in the system " );
                    return false;
                }
            } else {
                Utils::CLIerror( "The application name $application in the configuration is invalid" );
                return false;
            }
        }
        
        $fileParsed = $this->parseEnvironmentPropertiesFile( $file );
        if ($fileParsed === false) {
            // Error messages are handled in called function
            return false;
        } else {
            return $this->rest()->environment()->importEnvironmentProperties( $fileParsed, $environmentId );
        }
    }
}
