<?php
/****************************************************************
 * IBM Confidential
 *
 * SFA100-Collaboration Source Materials
 *
 * (C) Copyright IBM Corp. 2015
 *
 * The source code for this program is not published or otherwise
 * divested of its trade secrets, irrespective of what has been
 * deposited with the U.S. Copyright Office
 *
 * **************************************************************/

require_once ('include/EntryPoint.php');

/**
 * Provides tools for creating and managing Processes requests
 *
 * @class Team
 * @author Marco Bassi      marcobas@ie.ibm.com
 */
class Team extends EntryPoint {
    protected $resourceType = array(
        'agent' => 'Agent',
        'agentPool' => 'Agent%20Pool',
        'application' => 'Application',
        'cloudConnection' => 'Cloud%20Connection',
        'component' => 'Component',
        'componentTemplate' => 'Component%20Template',
        'environment' => 'Environment',
        'process' => 'Process',
        'resource' => 'Resource',
        'resourceTemplate' => 'Resource%20Template'
    );
    
    public function __construct($alias = null){
        parent::__construct($alias);
        $this->setReturn('json');
    }

    /**
     *
     *
     * Add one or more teams to components, component templates and processes linked to a specific application or list of applications
     *
     * @param string|array $application
     * @param string|array $team
     *
     * @return boolean array
     */
    public function addToApplication($application, $team)
    {
        
        if (empty($team) || empty($application)) {
            Utils::CLIerror("Invalid parameters.");
            Utils::CLIdebug(__CLASS__ . '::' . __FUNCTION__ . ": Invalid parameters, team: $team, application: $application");
            return false;
        }
        
        $team = is_array($team) ? $team : array( $team );
        $application = is_array($application) ? $application : array( $application );
        
        $return = array();
        $appObj = new Application();
        $compObj = new Component();
        

        foreach ($application as $app) {
            // Check if application is valid
            if (! $appObj->exists($app)) {
                Utils::CLIerror("Application '$app' does not exist.");
                return false;
            }
            
            foreach ($team as $single) {
                
                if (! $this->exists($single)) {
                    Utils::CLIerror("Team '$single' does not exist.");
                    $return[$app][$single]=false;
                    continue;
                }
                
                Utils::CLIinfo("Adding team '$single' to application '$app'...");
                
                $return['application'][$single] = $this->udclient()
                    ->application()
                    ->addApplicationToTeam($app, $single);
                if ($return['application'][$single] === false) {
                    Utils::CLIerror("Failed to set team '{$single}' to application '{$app}'");
                }
                
                // Set team to components
                Utils::CLIinfo("    Adding team '$single' to components...");
                $compInApp = $compObj->getList($app);
                $compList = array();
                foreach ($compInApp as $component) {
                    array_push($compList, $component['name']);
                }
                $return['components'] = $this->addToComponent($team, $compList);
                if ($return['components'] === false) {
                    Utils::CLIerror("Failed to set team(s) to components in application '{$app}'");
                }
                
                // Set team to processes
                Utils::CLIinfo("    Adding team '$single' to processes...");
                $procInApp = $appObj->getProcesses($app);
                $procList = array();
                foreach ($procInApp as $process) {
                    array_push($procList, $process['name']);
                }
                
                $return['processes'] = $this->addToProcess($team, $procList);
                if ($return['processes'] === false) {
                    Utils::CLIerror("Failed to set team(s) to processes in application '{$app}'");
                }
            }
        }
        
        return $return;
    }
    
    /**
     *
     * Add one or more teams to one or more components
     *
     * @param string|array $team
     * @param string|array $component
     * 
     * @return boolean|array
     */
    public function addToComponent ( $team, $component ) {
        if ( empty ( $team ) ) {
            Utils::CLIerror("Missing team name");
            return false;
        } else if ( !is_array( $team ) ) {
            $team = array( $team );
        }
        
        if ( empty ( $component ) ) {
            Utils::CLIerror ("Missing component name or ID");
            return false;
        } else if ( !is_array( $component ) ) {
            $component  = array( $component );
        }
        
        $return = array();
        foreach ( $component as $comp ) {
            foreach ( $team as $singleTeam ) {
                $result = $this->udclient()->component()->addComponentToTeam( $comp, $singleTeam );
                if ( $result === false ) {
                    $return[$comp][$singleTeam] = false;
                } else {
                    $return[$comp][$singleTeam] = true;
                }
            }
        }
        
        return $return;
    }
    
    /**
     *
     * Add one or more teams to one or more component templates
     *
     * @param string|array $team
     * @param string|array $template
     * 
     * @return boolean|array
     */
    public function addToComponentTemplate ( $team, $template ) {
        foreach ( $team as $singleTeam ) {
            $unmappedTemplates = $this->getUnmapped( $singleTeam, 'componentTemplate');
            $idList['resources'] = array();
            $idList['resourceRole'] = null;
            
            if ( !is_array( $template ) ) {
                $template = array( $template );
            }
            foreach ( $template as $single) {
                $id = null;
                foreach ( $unmappedTemplates as $unmapped ) {
                    if ( $unmapped['name'] == $single ) {
                        $id = $unmapped['id'];
                        break;
                    }
                }
                
                if ( ! empty( $id ) ) {
                    array_push( $idList['resources'], $id );
                }
            }
            $file = Utils::createJsonFile( $idList );
            $return[$singleTeam] = $this->rest()->system()->batchResourceMap( $singleTeam, $file );
            if ( $return[$singleTeam] !== false ) {
                unlink( $file );
            }
        }
        
        return $return;
    }
    
    /**
     * 
     * Add a team to all the unmapped resources
     *
     * @param unknown $team
     */
    public function addToUnmapped ( $team ) {
        if ( empty ( $team ) ) {
            Utils::CLIerror( "Please set a valid team" );
            return false;
        }
        if ( !is_array( $team ) ) {
            $team = array($team);
        }
        
        $return = array();
        foreach ( $team as $singleTeam ) {
            
            if (!$this->exists($singleTeam)){
                Utils::CLIerror( "Team '$singleTeam' does not exist." );
                $return[$singleTeam] = false;
                continue;
            }

            foreach ( $this->resourceType as $key => $value ) {
                
                Utils::CLIinfo("Adding team '$singleTeam' to resource type '$key'. ");
                $unmapped = $this->getUnmapped( $singleTeam, $key);
                if ($unmapped === false){
                    Utils::CLIerror("Failed to get unmapped resources.");
                    return false;
                }
                $idList['resources'] = array();
                $idList['resourceRole'] = null;
                
                foreach ( $unmapped as $singleUnmapped ) {
                    array_push( $idList['resources'], $singleUnmapped['id'] );
                }
                
                $file = Utils::createJsonFile( $idList );
                $return[$singleTeam][$key] = $this->rest()->system()->batchResourceMap( $singleTeam, $file );
                if ( $return[$singleTeam][$key] !== false ) {
                    unlink( $file );
                }
            }
        }
        
        return $return;
    }
    
    /**
     *
     * Add one or more teams to one or more processes
     *
     * @param string|array $team
     * @param string|array $process
     * 
     * @return boolean|array
     */
    public function addToProcess ( $team, $process ) {
        foreach ( $team as $singleTeam ) {
            $unmappedTemplates = $this->getUnmapped( $singleTeam, 'process');
            $idList['resources'] = array();
            $idList['resourceRole'] = null;
        
            if ( !is_array( $process ) ) {
                $process = array( $process );
            }
            foreach ( $process as $single) {
                $id = null;
                foreach ( $unmappedTemplates as $unmapped ) {
                    if ( $unmapped['name'] == $single ) {
                        $id = $unmapped['id'];
                        break;
                    }
                }
        
                if ( ! empty( $id ) ) {
                    array_push( $idList['resources'], $id );
                }
            }
            $file = Utils::createJsonFile( $idList );
            $return[$singleTeam] = $this->rest()->system()->batchResourceMap( $singleTeam, $file );
            if ( $return[$singleTeam] !== false ) {
                unlink( $file );
            }
        }
        
        return $return;
        
    }
    
    /**
     *
     * Create a new team and returns the id if successful
     *
     * @param string $team
     * 
     * @return boolean|string
     */
    public function create ( $team ) {
        $result = $this->rest()->system()->createTeam( $team );
        if ( $result === false ) {
            return false;
        } else {
            $result = Utils::outputToArray( $result );
            return $result['id'];
        }
    }
    
    /**
     *
     * Delete a team
     *
     * @param string $team
     * 
     * @return boolean
     */
    public function delete ( $team ) {
        $list = $this->getList();
        if ( $list === false ) {
            return false;
        }
        $id = null; 
        foreach ( $list as $single ) {
            if ( $single['id'] == $team || $single['name'] == $team ) {
                $id = $single['id'];
                break;
            }
        }
        
        if ( empty  ($id ) ) {
            Utils::CLIerror( "No team with name/ID {$team}");
            return false;
        }
        
        $result = $this->rest()->system()->deleteTeam( $id );
        if ( $result === false ) {
            return false;
        } else {
            return true;
        }
    }
   
    /**
     * Check if Team exist
     *
     * @param string $team
     * @return boolean
     */
    public function exists($team)
    {
        $return_type = $this->return;
        if (empty($team) || ! is_string($team)) {
            Utils::CLIerror("Invalid team parameter.");
            return false;
        }
        $this->setReturn('json');
        $teams = $this->getList();
        $result = false;
        foreach ($teams as $item) {
            if ($item['name'] == $team || $item['id'] == $team) {
                $result = true;
            }
        }
        $this->setReturn($return_type);
        return $result;
    }
    
    /**
     *
     * Get the list of components that are associated to a particular team
     *
     * @param string $team
     * 
     * @return boolean|array
     */
    public function getComponents ( $team ) {
        
    }
    
    /**
     * 
     * Get list of teams
     * 
     * @return boolean|array
     */
    public function getList () {
        $result = $this->rest()->system()->getTeams();
        if ( $result === false ) {
            Utils::CLIerror( "Failed to retrieve list of teams" );
            return false;
        }
        if ( $this->return === 'file' ){
            return substr($result,1,-1);
        }
        return Utils::outputToArray( $result );
    }
    
    /**
     * 
     * Returns list of unmapped resources for the defined type.
     * If no type is defined, return them all
     *
     * @param string $team
     * @param string $type
     * 
     * @return array|boolean
     */
    public function getUnmapped ( $team, $type = null ) {
        if ( empty( $team) ) {
            Utils::CLIerror( "Team is required" );
            return false;
        }
        // Type is defined
        if ( !empty( $type ) ) {
            if ( ! array_key_exists( $type, $this->resourceType ) ) {
                Utils::CLIerror( "Invalid resource type '{$type}'." );
                return false;
            } else {
                $result = $this->rest()->system()->getUnmappedResources( $team, $this->resourceType[$type] );
                if ( $result === false ) {
                    Utils::CLIerror( "Failed to retrieve list of unmapped resources for type '{$type}'");
                    return false;
                }
                if ( $this->return === 'file' ){
                    return substr($result,1,-1);
                }
                return Utils::outputToArray( $result );
            }
        }
        
        // Type not defined, return them all
        foreach ( $this->resourceType as $resourceType => $url ) {
            $result = $this->rest()->system()->getUnmappedResources( $team, $url );
            if ( $result === false ) {
                Utils::CLIerror( "Failed to retrieve list of unmapped resources for type '{$resourceType}'");
                return false;
            } else {
                $return[$resourceType] = Utils::outputToArray( $result );
            }
        }
        if ( $this->return === 'file' ){
            return Utils::createJsonFile( $return );
        }
        return $return;
    }
}