<?php

/****************************************************************
 * IBM Confidential
 *
 * SFA100-Collaboration Source Materials
 *
 * (C) Copyright IBM Corp. 2015
 *
 * The source code for this program is not published or otherwise
 * divested of its trade secrets, irrespective of what has been
 * deposited with the U.S. Copyright Office
 *
 ****************************************************************/

/**
 *
 * @author Marco Bassi marcobas@ie.ibm.com
 *        
 *         This class is providing actions to manage aliases
 */
class AliasHandler
{
    private $alias_dir;
    private $config_dir;
    private $promote_config;
    private $ucd_config;
    
    public function __construct() {
        $this->config_dir = getcwd() . '/config';
        $this->alias_dir = "{$this->config_dir}/servers";
        $this->ucd_config = "{$this->config_dir}/ucd.config.php";
        $this->promote_config = "{$this->config_dir}/promote.config.php";
    }

    /**
     *
     *
     * Returns if an alias is valid or not
     *
     * @param string $alias            
     * @return boolean
     */
    public function checkAlias($alias)
    {
        $error = false;
        if (empty($alias)) {
            Utils::CLIerror("Alias cannot be empty.");
            $error = true;
        } else 
            if (! $this->getAlias($alias)) {
                $error = true;
            }
        
        if ($error) {
            $this->printList();
            return false;
        }
        
        return true;
    }
    
    /**
     *
     *
     * Given the alias name, return the information about the server config
     *
     * @param string $alias            
     */
    public function getAlias($alias)
    {
        if (! is_string($alias)) {
            Utils::CLIerror("Invalid argument passed as alias.");
            return false;
        }
        $alias_file = "{$this->alias_dir}/{$alias}.server.php";
        if (is_file($alias_file) && ! is_dir($alias_file)) {
            include $alias_file;
            return $config;
        }
        // Alias is invalid
        Utils::CLIerror("Invalid alias name '{$alias}'.");
        return false;
    }

    /**
     * Return a list of available aliases
     */
    public function getList()
    {
        if (! is_dir($this->alias_dir)) {
            Utils::CLIerror("Server config directory '{$this->alias_dir}' is missing.");
            return false;
        }
        $list = array();
        $file_list = scandir($this->alias_dir);
        if ($file_list === false) {
            Utils::CLIerror("Failed to retrieve list of available server config files.");
            return false;
        }
        foreach ($file_list as $file) {
            if ($file == '.' || $file == '..') {
                continue;
            }
            $strlen = strlen($file);
            // Position where .server.php starts
            $pos = $strlen - 11;
            
            if (strpos($file, '.server.php') != $pos) {
                continue;
            }
            $file = str_replace('.server.php', '', $file);
            array_push($list, $file);
        }
        return $list;
    }
    
    /**
     * 
     * Given the alias name, print on the screen the information about the server config
     *
     * @param string $alias
     */
    public function printAlias ( $alias ) {
        $config = $this->getAlias($alias);
        if ($config===false) {
            return false;
        }
        
        Utils::CLIbold("Alias:        ");
        Utils::CLIout($config['alias'], false);
        
        Utils::CLIout("");
        
        Utils::CLIbold("Weburl:       ");
        Utils::CLIout($config['weburl'], false);
        
        Utils::CLIbold("Username:     ");
        Utils::CLIout($config['username'], false);
        
        Utils::CLIbold("Authtoken:    ");
        Utils::CLIout($config['authtoken'], false);
        
        Utils::CLIbold("Application:  ");
        Utils::CLIout($config['application'], false);
        
        Utils::CLIbold("Environment:  ");
        Utils::CLIout($config['environment'], false);
        
        Utils::CLIout("");
        
        return true;
    }
    
    /**
     *
     * Print a list of available aliases
     *
     */
    public function printList () {
        $list = $this->getList();
        if ( $list == false) {
            return false;
        }
        Utils::CLIinfo ("List of availble alias");
        foreach ( $list as $alias ) {
            $config = array();
            $config = $this->getAlias( $alias );
            Utils::CLIbold("\t{$alias}");
            Utils::CLIout(" ({$config['weburl']})", false);
        }
        
        Utils::CLIout("");
        return true;
    }

    /**
     *
     *
     *
     * Given the alias name, set it as alias in ucd.config.php
     *
     * @param string $alias            Alias name
     * @param string $type            alias, destination, origin
     * @param strig $config_file            Path of the config file
     */
    public function setAlias($alias, $type = 'alias', $config_file = null)
    {
        if ($this->checkAlias($alias) == false) {
            return false;
        }
        
        if (empty($config_file)) {
            $config_file = $this->ucd_config;
        }
        
        $edit = "cat {$config_file} | grep -v '{$type}' > {$config_file}.tmp";
        $move = "mv {$config_file}.tmp {$config_file}";
        $update = "\$config['{$type}'] = '{$alias}';";
        
        // Edit file
        exec($edit, $out, $return);
        if ($return != 0) {
            Utils::CLIerror("Failed to edit {$type} in '{$config_file}'.");
            return false;
        }
        
        // Replace file
        exec($move, $out, $return);
        if ($return != 0) {
            Utils::CLIerror("Failed to move {$type} file in '{$config_file}'.");
            return false;
        }
        
        // Update file
        $result = file_put_contents($config_file, $update, FILE_APPEND);
        if ($result === false) {
            Utils::CLIerror("Failed to update {$type} in '{$config_file}'. Please check for file corruption.");
            return false;
        }
        
        return true;
    }
    
    /**
     *
     *
     * Given the alias list in the form of array, set each alias in the correspondent config file
     *
     * @param array $aliasList
     */
    public function setAll($aliasList)
    {
        $result = true;
        $aliasTypeList = array(
            'alias',
            'destination',
            'origin'
        );
        foreach ($aliasTypeList as $type) {
            if (array_key_exists($type, $aliasList)) {
                $method = 'set' . ucfirst($type);
                $success = $this->$method($aliasList[$type]);
                if ($success) {
                    Utils::CLIinfo("Alias '{$aliasList[$type]}' set as '{$type}'.");
                } else {
                    $result = false;
                }
            }
        }
        
        return $result;
    }
    
    /**
     *
     *
     * Given the alias name, set it as alias for 'destination' in promote.config.php
     *
     * @param string $alias            
     */
    public function setDestination($alias)
    {
        return $this->setAlias($alias, 'destination_alias', $this->promote_config);
    }

    /**
     *
     *
     * Given the alias name, set it as alias for 'origin' in promote.config.php
     *
     * @param string $alias            
     */
    public function setOrigin($alias)
    {
        return $this->setAlias($alias, 'origin_alias', $this->promote_config);
    }
}