<?php

/****************************************************************
 * IBM Confidential
 *
 * SFA100-Collaboration Source Materials
 *
 * (C) Copyright IBM Corp. 2015
 *
 * The source code for this program is not published or otherwise
 * divested of its trade secrets, irrespective of what has been
 * deposited with the U.S. Copyright Office
 *
 ****************************************************************/

/**
 *
 * @author Felice Geracitano felice.geracitano@ie.ibm.com
 *        
 *         This class is providing argument support to the ucommand cli
 */
class ArgumentsHandler extends AbstractAPI
{
    private $args_options;
    private $argv;
    private $valid_actions;
    
    // Costructor
    public function __construct($argv)
    {
        parent::__construct(); // set up connection and check alias.server.php
        // Load config
        $this->argv = $argv;
        $errors = false;
        
        // Check options
        if (! file_exists('config/arguments.config.php')) {
            $errors = true;
            Utils::CLIerror("Missing " . getcwd() . "/config/arguments.config.php");
        }
        // Check actions
        if (! file_exists('config/actions.config.php')) {
            $errors = true;
            Utils::CLIerror("Missing " . getcwd() . "config/actions.config.php");
        }
        
        if ($errors) {
            exit(1);
        }
        
        include ('config/arguments.config.php');
        $this->args_options = getopt($shortopts, $longopts);
        include ('config/actions.config.php');
        $this->valid_actions = $valid_actions;
    }
    
    
    // From old CLI to new CLI
    public function fromOldToNew()
    {
        // Load config
        $valid_actions = $this->valid_actions;
        $argv = $this->argv;
        // Old action into argv 1
        $return['action'] = isset($argv[1])?$argv[1] : null;

        // if is not exist return the argv, general message will be prompt from ucommand.php
        if (!array_key_exists($return['action'],$valid_actions)){
            $this->helper();
            exit(1);
        }
        
        
        // Detected old way, print suggestion
        Utils::CLIwarning("Deprecated, this action should be called as: ");
        $this->helper($return['action']);
        echo PHP_EOL;
        // Load all the action info from valid actions array
        $action_info = $valid_actions[$return['action']];
        $requires = $action_info['require'];
        $optionals = $action_info['optional'];
        
        
        // argv sturcture
        // [0] => "ucommand.php"
        // [1] => "action"
        // ...
        // [x] => "requires"
        // [x+1] => "optional"
        
        // We start read from index 2 of argv
        $count = 2;
        // requires loop
        foreach ($requires as $key => $value) {
            // So if we already get the last element, something is missing, throw error
            if ($count == count($argv)){
                if ($key === 'environment' || $key === 'application'){
                    if (!empty($this->$key)){
                        $return[$key] = $this->$key; // this->key is set into abstractapi
                        continue; // Continue to next arg
                    }
                    // Continue e throw error
                }
                Utils::CLIerror("Missing '$key' parameter for action: {$return['action']}");
                exit(1);
            }
            $return[$key] = $argv[$count++];
        }
        // optionals loop (should be at most 1)
        foreach ($optionals as $key => $value) {
            // So if we already get the last element return
            if ($count == count($argv)){
                break;
            }
            $return[$key] = $argv[$count++];
        }
        return $return;
    }

    /**
     * Return $arguments array for the ucommand switch
     *
     * @return array
     */
    public function getArguments()
    {
        //Load config, checked into costructor
        $args_options = $this->args_options;
        $valid_actions = $this->valid_actions;
        
        if (!$this->isNewCLI()) {
            return $this->fromOldToNew($this->argv);
        }
        
        $arguments = array(); // return value
        // Getting the action, already validated
        $action_key = isset($args_options['action'])? $args_options['action'] : null ;
        $action = isset($valid_actions[$args_options['action']]) ? $valid_actions[$args_options['action']] : null;
        $requires = isset($action['require']) ? $action['require'] : array();
        $optionals = isset($action['optional']) ? $action['optional'] : array();
        
        // Build new CLI arguments array
        $arguments['action'] = $action_key;
        
        // Fails is some requires are missing
        // Do we want to hanlde here on the ucommand switch ???
        $req_err = false;
        foreach ($requires as $label => $description) {
            // if is not in the array
            if (!array_key_exists($label, $args_options)) {
                // Before throw error check into config file
                if ($label === 'environment' || $label === 'application'){
                    if (!empty($this->$label)){
                        $arguments[$label] = $this->$label; //this->label is set into abstractapi 
                        continue; // Continue to next arg
                    }
                    // Continue e throw error
                }
                $req_err = true;
                Utils::CLIerror("Missing '$label' parameter for action: $action_key");
            }else {
                $arguments[$label] = $args_options[$label];
            }
        }
        // Exit if message was printed
        if ($req_err){
            $this->helper($action_key);
            exit(1);
        }
        
        // Optional arguments loop
        foreach ($optionals as $label => $description) {
            if (array_key_exists($label, $args_options)) {
                $arguments[$label] = $args_options[$label];
            }
        }
        
        // Special action : 'config'
        // At least one optional parameter
        if ($arguments['action']=='config' && count($arguments)==1){
            Utils::CLIerror("Please specify at least one optional parameter for 'config' action.");
            $this->helper($arguments['action']);
            exit(1);
        }
        
        // If action exist and there is the helper flag, print,
        // Otherwise go to default switch
        if (array_key_exists('help', $args_options) && array_key_exists($action_key, $valid_actions)) {
            $this->helper($action_key);
            exit(1);
        }
        
        return $arguments;
    }

    /**
     * Print helper
     * TODO create file for the new CLI
     *
     * @return bool
     */
    public function helper($key_action = null)
    {
        if (empty($key_action)) {
            // General Helper only the actions list
            Utils::CLIinfo("Please provide a valid Action as Argument.");
            Utils::CLIout("Usage:");
            Utils::CLIout("\tphp ucommand.php");
            Utils::CLIbold(" --action='action'", false);
            
            Utils::CLIout("Actions:");
            foreach ($this->valid_actions as $action => $actionArray) {
                Utils::CLIbold("\t{$action}");
                Utils::CLIout(" - {$actionArray['description']}", false);
            }
            Utils::CLIout("\n", true);
            return true;
        }
        
        // Specific function helper
        // Follow the man help pattern
        $action = $this->valid_actions[$key_action];
        Utils::CLIbold("\tNAME");
        Utils::CLIbold("\t\t$key_action", true);
        Utils::CLIout(" - ${action['description']}", false);
        Utils::CLIbold("\n\tSYNOPSIS");
        Utils::CLIout("\t\tphp ucommand.php --action=$key_action ");
        
        // Print synopsis example -REQUIRED
        foreach ($action['require'] as $key => $value) {
            Utils::CLIout("--$key='value' ", false);
        }
        // Print synopsis example -OPTIONAL
        foreach ($action['optional'] as $key => $value) {
            Utils::CLIout("--$key='value' ", false);
        }
        
        // Print REQUIRED List
        if (! empty($action['require'])) {
            Utils::CLIbold("\n\tREQUIRED", true, 2);
            foreach ($action['require'] as $key => $value) {
                Utils::CLIbold("\t\t$key", true);
                Utils::CLIout(" - $value", false);
            }
        }
        
        // Print OPTIONAL List
        if (! empty($action['optional'])) {
            Utils::CLIbold("\n\tOPTIONAL", true, 2);
            foreach ($action['optional'] as $key => $value) {
                Utils::CLIbold("\t\t$key", true);
                Utils::CLIout(" - $value", false);
            }
        }
        Utils::CLIout("\n", true);
        return true;
    }

    /**
     * Check if command is a valid function and is not dupliated
     *
     * @return bool
     */
    private function isNewCLI()
    {
        $args_options = $this->args_options;
        $valid_actions = $this->valid_actions;
        
        // If empty fails - old CLI
        if (empty($args_options) || empty($valid_actions)) {
            return false;
        }
        
        $long_action = array_key_exists("action", $args_options);
        // if not actioon found - old CLI
        if (empty($long_action)) {
            return false;
        }
        
        // Check for double actions
        foreach ($args_options as $label) {
            if (is_array($label)) {
                Utils::CLIerror("Commands can only have one value for each param.");
                $this->helper();
                exit(1);
            }
        }
        return true;
    }
}
