<?php
/**
 * **************************************************************
 * IBM Confidential
 *
 * Collaboration Source Materials
 *
 * (C) Copyright IBM Corp. 2014-2015
 *
 * The source code for this program is not published or otherwise
 * divested of its trade secrets, irrespective of what has been
 * deposited with the U.S. Copyright Office
 *
 * *************************************************************
 */

/**
 * APIs
 *
 * @author Felice Geracitano.......feliecege@ie.ibm.com
 *        
 */
require_once 'ApiInterface.php';

abstract class AbstractApi implements ApiInterface{
    protected $actions;
    public $alias;
    public $application;
    protected $args;
    protected $authtoken;
    protected $exp_date;
    protected $b64_login;
    protected $certificate;
    protected $config;
    protected $connected;
    protected $destination;
    public $environment;
    protected $error_log;
    protected $insecure;
    protected $impersonation_config;
    protected $java_home;
    protected $log;
    protected $logout;
    public $output;
    protected $restActions;
    protected $return;
    protected $server;
    public $udcpath;
    protected $upgrade_values = array (
        "USE_EXISTING_IF_EXISTS",
        "CREATE_NEW_IF_EXISTS",
        "FAIL_IF_EXISTS",
        "FAIL_IF_DOESNT_EXIST",
        "UPGRADE_IF_EXISTS" 
    );
    protected $username;
    public $weburl;
        
    public function __construct($alias = null){
        $this->log = new Logger();
        $this->error_log = new Logger();
        $this->setup($alias);
        $this->setReturn();
    }
    
    /**
     * Verify that udclient files are in place.
     */
    protected function checkUdclient(){
        // Check if udclient files are valid
        $udclient_jar = $this->udcpath . '/udclient.jar';
        $udclient_jar = file_exists($udclient_jar);
        $udclient_exe = $this->udcpath . '/udclient';
        $udclient_exe = file_exists($udclient_exe);
        $udclient_cmd = $this->udcpath . '/udclient.cmd';
        $udclient_cmd = file_exists($udclient_cmd);
    
        // If any of the files is missing, check and extract zip
        if ( !$udclient_jar || !$udclient_exe || !$udclient_cmd ) {
            Utils::CLIdebug("Udclient files are missing: checking for zip");
            // Check if udclient zip exists and extract it
            $udclient_zip = 'udclient.zip';
            if ( !file_exists($udclient_zip) ) {
                Utils::CLIinfo("Udclient zip file {$udclient_zip} was not found. Downloading...");
                $download = $this->downloadUdclient();
                if ($download === false) {
                    Utils::CLIerror("Cannot download udclient.zip form UrbanCode Deploy server. Try to re-configure uCommand running ./setup");
                    return false;
                }
            }
            Utils::CLIdebug("Unzipping '{$udclient_zip}' into '{$this->udcpath}' ");
            $command = "unzip -q -o {$udclient_zip} -d {$this->udcpath}";
            exec($command, $output, $result);
            $this->udcpath = "{$this->udcpath}/udclient";
            if ($result === 0) {
                // Check if extracted files are valid
                return $this->checkUdclient();
            } else {
                Utils::CLIerror("An error occured while extracting {$udclient_zip}");
                return false;
            }
        }
        return true;
    }
    
    /**
     * Download Udclient from $this->weburl
     */
    public function downloadUdclient() {
        $action = "{$this->weburl}/tools/udclient.zip -o udclient.zip";
        return $this->execRest($action);
    }
    
    /**
     * @param string    $action
     *
     * @return bool
     */
    public function exec($action){
        // Prepare the command to be executed
        if (!empty($this->authtoken) && ! empty($this->username)) {
            // Access with auth-token doesn't need an existing connection
            $command = "export JAVA_HOME={$this->java_home}; ";
            $command .= "{$this->udcpath}/udclient -weburl {$this->weburl} -username {$this->username} -authtoken {$this->authtoken} {$action}";
        } else {
            Utils::CLIerror("Login with username/password has been deprecated, authtoken is required.");
            exit (1);
        }
        // Log details for the command to be executed
        $command_info = 
            'Execute command: ' . PHP_EOL
            . '    Type  : weburl' . PHP_EOL
            . '    Path  : ' . $this->udcpath . '/udclient' . PHP_EOL
            . '    WebURL: ' . $this->weburl . PHP_EOL
            . '    Action: ' . $action;
        $this->log->info( $command_info );
    
        // Enable Debug
        if ($this->log->isDebug()){
            $debug = ' --verbose';
        } else {
            $debug = '';
        }
    
        // Process $command
        $processResult = self::processServerRequest($command.$debug);
    
        /*
         * Log success / fail to process request and return result
        * NOTE: error logs and messages to console are handled in: processServerRequest()
        */
        if ( $processResult === false ) {
            $this->log->error('Command processed with ERRORS');
            
            // Log command info into error log
            $this->error_log->error('Command processed with ERRORS');
            $this->error_log->error( $command_info );
        } else {
            $this->log->info('Command executed - SUCCESS');
        }
    
        // Return result
        return $processResult;
    }
    
    /**
     *
     * Executes curl commands using REST APIs
     *
     * @param string $action
     */
    public function execRest($action){
        // Cookies
        if (!empty($this->cookie_file) && is_file($this->cookie_file) && !is_dir($this->cookie_file)) {
            $cookies = " -b '{$this->cookie}' ";
        } else {
            $cookies = '';
        }
    
        // Username:Password | Base_64 login
        if (!empty($this->b64_login)) {
            $login = "-H \"Authorization: Basic {$this->b64_login}\" ";
        } else {
            Utils::CLIerror("Login with username/password has been deprecated, authtoken is required.");
            exit (1);
        }
    
        // Secure login
        if ($this->insecure) {
            $ssl = "-k";
        } else {
            if (!empty($this->certificate) && file_exists($this->certificate)){
                $ssl = "--cacert {$this->certificate}";
            } else {
                Utils::CLIerror("Certificate is missing or cannot be read.");
                return false;
            }
        }
    
        // Enable Debug
        if ($this->log->isDebug()){
            $debug = '--verbose';
        } else {
            $debug = '-s';
        }
    
        $command = "curl {$cookies} {$login} {$ssl} {$action} {$debug}";
    
        // Log details for the command to be executed
        $command_info = 
            'Execute command: ' . PHP_EOL
            . '    Type  : curl' . PHP_EOL
            . '    SSL   : ' . $ssl . PHP_EOL
            . '    Cookie: ' . $cookies . PHP_EOL
            . '    Action: ' . $action;
        $this->log->info( $command_info );
    
        // Process $command
        $processResult = self::processServerRequest($command);
    
        /*
         * Log success / fail to process request
        * NOTE: error logs and messages to console are handled in: processServerRequest()
        */
        if ( $processResult === false ) {
            $this->log->error('Command processed with ERRORS');
            
            // Log command info into error log
            $this->error_log->error('Command processed with ERRORS');
            $this->error_log->error( $command_info );
        } else {
            $this->log->info('Command executed - SUCCESS');
        }
    
        // Return result
        return $processResult;
    }
    

    
    /**
     * Get the output dir
     */
    public function getOutput(){
        return $this->output;
    }
    
    /**
     * Get conf for setupServer
     */
    public function getPromoteConfig() {
        include ("config/promote.config.php");
        return $config;
    }
    
    /**
     * Get the weburl
     */
    public function getWeburl(){
        return $this->weburl;
    }
    
    
    /**
     * A helper function, which is used to support the $this->exec() and$this->execRest() functions.
     * Will process a specified command ($commandToExecute),
     * which was generated for the specified: $requestedAction, using PHP: exec() command.
     * In case of an error in either: exec() command execution using or
     * if the Server returns an error message, this function will log corresponding message
     * and will return Boolean FALSE.
     * Otherwise it will return the result from the executed command
     * as defined in the: '$this->return' variable.
     *
     * @param string $requestedAction   The name of the requested Action, for which
     *                                  corresponding $commandToExecute, has been generated
     * @param string $commandToExecute  The command to be executed
     *
     * @return boolean|string|mixed Boolean FALSE - in case of an Error with execution
     *                              of the specified command, or when Server request returns with error message.
     *                              String holding the path to a file, when "$this->return" is set to: "file"
     *                              or the raw result from the executed command, otherwise.
     */
    private function processServerRequest($commandToExecute) {
        // Execute specified command
        exec($commandToExecute, $requestResult, $requestError);
    
        // Debug
        if ($this->log->isDebug()){
            $this->log->debug("Request exit code: {$requestError}");
            $this->log->debug("Request result: ");
            $this->log->debug(print_r($requestResult,true));
        }
    
        /*
         * Get destination and clean it up for next function call before to procede
        * NOTE: depending on ServerRequest destinations might begin with:
        *       ' > ', '> ', ' -o ' or '-o'
        *       We need to clean these in order to avoid any confusion in results
        */
        $fileDestination   = str_replace(
                array(' > ', '> ', ' -o ', '-o '),
                '',
                $this->destination);
        $this->destination = '';
    
        /*
         * Validate execution - result, log errors and return correspondingly.
        * NOTE: $command details have already been logged by the caller to this function
        */
    
        /* 1. Check for errors in the request execution,
         *    i.e. there was a problem with execution of the system command
        */
        if ( $requestError ){
            // Log errors into UCOMMAND/log/error.log
            $this->error_log->error("Request exit code: {$requestError}");
            $this->error_log->error("Request result: ");
            $this->error_log->error(print_r($requestResult,true));
            
            // Print error message to the user
            Utils::CLIerror(
            'Command: "' . $commandToExecute . '" execution exited with error code: ' . $requestError,
            true, 0, 'system');
            Utils::CLIinfo('For more details, please check: "' . $this->log->getLogFile() . '" file');
    
            /*
             * Shell command errors are printed on the console
            * just log the error code
            */
            $this->log->error('System Error code: ' . $requestError . ' :: There was an ERROR in specified command string, '
                    .'or Request exited with a message "Broken Pipe".');
            return false;
        } // else request was processed OK -=> validate request result and return
    
        /* 2. Check for an error response from the Server
         *    command execution went OK -=> validate command result.
        *    Get $requestResult as a string using: Utils::outputToJson() and validate it.
        * NOTE: 1) in case when $command result was written to a file
        *          $output will be empty array which will be validated to TRUE
        *          by: Utils::validateJSON(Utils::outputToJson($output))
        *       2) $output might be in the form:
        *          - just a string with error message or
        *          - string in the form:
        *              '<textarea>{"status":"ok"...}</textarea>' - for Success
        *              '<textarea>{"status":"failed","error":"..."}</textarea>'
        */
        $resultToValidate = Utils::outputToJson($requestResult);
        // Server returned failure to process request
        if (
        !Utils::validateJSON($resultToValidate) &&
        strpos( $resultToValidate, '"status":"ok"' ) === false &&
        strpos( $resultToValidate, '"status": "ok"' ) === false &&
        strpos( $resultToValidate, 'Operation succeeded') === false
        ) {
            Utils::CLIerror('Command: "' . $commandToExecute . '" completed with an ERROR message.');
            Utils::CLIinfo('For more details, please check: "' . $this->log->getLogFile() . '" file');
            $this->log->warning('Command execution - COMPLETED with ERROR response:');
            $this->log->error('Server response :: ' . var_export($requestResult,true));
            return false;
    
            /*
             * Request was executed OK and Server returned success result
            * -=> return as usual based on the $this->return type
            */
        } else {
            // $output has been set to be saved to a file
            if ($this->return == 'file'){
                // return $output - file destination
                return $fileDestination;
    
                // return $output as it is
            } else {
                return $requestResult;
            }
        } // END of validating result ($output) from $actionToExec
    }
    

    
    public function setOutput($dir, $message = false) {
        // Chek if path for folder ends with '/'
        if ( substr($dir, -1) == '/' ){
            $dir = substr($dir, 0, -1);
        }
    
        // Default output to current working directory
        if ( trim( $dir) == '' ) {
            $dir = getcwd() . '/backups';
            if ($message){
                Utils::CLIinfo('Default "output" to: ' . $dir);
            }
        }
    
        if (is_dir($dir)) {
            if ($message){
                Utils::CLIinfo('"Output" to: ' . $dir);
            }
            $this->output = $dir;
            return true;
        } else if (is_file($dir)) {
            return false;
        } else {
            if (mkdir($dir,0744,true)) {
                $this->output = $dir;
                return true;
            } else {
                return false;
            }
        }
    }
    
    /**
     * @param string    $type
     * Set if the API are returning the json value or echoing the output
     */
    public function setReturn($type = null){
        if (empty($type) || (strtolower($type) != 'json' && strtolower($type) != 'file' )){
            $this->return = 'json';
        } else {
            $this->return = $type;
        }
    }
    
    /**
     * Setup the basic config values that will be used by this class
     */
    public function setup( $alias_name = null ){
        include 'config/ucd.config.php';
        // Check Alias server file
        $aliasPath = ($alias_name === null) ? $this->setupAlias($config['alias']) : $this->setupAlias($alias_name);
        if ( $aliasPath === false ) {
            Utils::CLIout("  Please set a valid alias for the server configuration you want to use.");
            Utils::CLIout("  If you do not have a server configuration yet, please run ./setup");
            exit (1);
        } else {
            include $aliasPath;
        }
        
        // Assign config values
        $this->username = $config['username'];
        $this->authtoken = $config['authtoken'];
        if (!empty($this->authtoken)) {
            $this->b64_login = base64_encode('PasswordIsAuthToken:{"token":"' . $this->authtoken . '"}');
        } else {
            $this->b64_login = null;
        }
        
        if ( array_key_exists ( 'exp_date', $config ) && ! empty( $config['exp_date'] ) ) {
            if ( $config['exp_date'] > date ("Y-m-d") ) {
                $this->exp_date = $config['exp_date'];
            } else {
                Utils::CLIerror("Your token for alias '{$config['alias']}' has expired");
                Utils::CLIinfo( "Please run ./setup to request a new token");
                exit (1);
            }
        }
        
        $this->weburl = $config['weburl'];
        $this->insecure = $config['insecure'];
        $this->certificate = $config['certificate'];
        // application and environment are optional parameters
        $this->application = array_key_exists('application', $config) ? $config['application'] : null;
        $this->environment = array_key_exists('environment', $config) ? $config['environment'] : null;
        $this->udcpath = getcwd();
        
        
        // Check udclient
        $checkUdclient = $this->checkUdclient();
        if ( !$checkUdclient ){
            Utils::CLIerror("Failed to download and extract udclient binaries.");
            exit (1);
        }
        $this->java_home = $config['java_home'];
        $this->destination = '';
        
        if ( !$this->setOutput($config['output']) ) {
            Utils::CLIerror('There was a problem with setting up of "output" directory.');
            Utils::CLIinfo('For more information please check "ucommand/README" file', false, 1);
            exit (1);
        }
        
        // Set silent mode for outputs
        Utils::setSilent( $config['silent'] );
        // Set JAVA_HOME
        $result = Utils::setJavaHome( $this->java_home );
        if ( $result === false ){
            Utils::CLIerror("Path for JAVA_HOME was not set correctly or you don't have permission on it.");
            Utils::CLIout("  Please check in 'config/ucd.config.php' that ");
            Utils::CLIout("    - the parameter 'java_home' is set correctly, or");
            Utils::CLIout("    - if empty, that the CLI environment variable 'JAVA_HOME' is correct");
            exit (1);
        } else {
            $this->java_home = $result;
            Utils::CLIdebug("JAVA_HOME set to '{$this->java_home}' ");
        }
        
        // Set Logger, if config file exists
        if (file_exists('config/logger.config.php')) {
            include('config/logger.config.php');
            $this->log->setLogFile($config['log']);
            $this->log->setDebug($config['debug']);
        }
    }
    
    /**
     * 
     * Given an alias, return the server config file, if exists
     *
     * @return boolean|string
     */
    public function setupAlias($alias) {
        if (empty( $alias )) {
            Utils::CLIerror( "No alias set in 'config/ucd.config.php' file." );
            return false;
        } else {
            $this->alias = $alias;
            $serverSetup = "config/servers/{$this->alias}.server.php";
            if (is_file( $serverSetup )) {
                return $serverSetup;
            } else {
                Utils::CLIerror( "No setup file for alias '{$this->alias}'." );
                return false;
            }
        }
    }

    /**
     *
     * Logout from existing connection and setup a new connection to a new server
     *
     * @param string $server
     * @param array $config
     * @param boolean $message   // displays connecting message
     * @param boolean $logout
     * @return boolean
     */
    public function setupServer($server, $config, $message = false, $logout = true) {
        $this->config = $config;
        
        if (empty($server)) {
            Utils::CLIerror("No server set");
            return false;
        }
        if (empty($config)) {
            Utils::CLIerror("No configuration set");
            return false;
        }
        if (empty($config[$server.'_alias'])) {
            Utils::CLIerror("No server alias is set");
            return false;
        }
    
        // Logout from an existing connection
        if ($logout) {
            if ($this->connected) {
                Utils::CLIout("  Disconnecting from server: {$this->weburl}");
                $this->logout();
            }
        }
    
        // Check files existing
        $serverConfig = $this->setupAlias($config[$server.'_alias']);
        if ($serverConfig === false){
            return false;
        }else{
            include $serverConfig;
        }
        
        if ( array_key_exists ( 'exp_date', $config ) ) {
            if ( $config['exp_date'] > date ("Y-m-d") ) {
                $this->exp_date = $config['exp_date'];
            } else {
                Utils::CLIerror("Your token for alias '{$config['alias']}' has expired.");
                Utils::CLIinfo( "Please run ./setup to request a new token");
                exit (1);
            }
        }
        
        $this->weburl = $config['weburl'];
        $this->username = $config['username'];
        $this->authtoken = $config['authtoken'];
        $this->application = $config['application'];
        $this->environment = $config['environment'];
        if ($message){
            Utils::CLIout("  Connecting to '{$server}' server: {$this->weburl}");
        }
        if (!empty($this->authtoken)) {
            $this->b64_login = base64_encode('PasswordIsAuthToken:{"token":"' . $this->authtoken . '"}');
        } else {
            $this->b64_login = null;
            Utils::CLIerror("Login with username/password has been deprecated, authtoken is required.");
            exit (1);
        }
        // If $this->serveris setted so children api should be updated when is created
        $this->server = $server;
        return true;
    }
    
    
    /**
     *
     * Validate server configuration used by promoteSnapshot()
     *
     * @param array $config
     * @return boolean
     */
    public function validateServerConfig($config) {
        if (empty($config)) {
            Utils::CLIerror("Configuration is empty. Please provide a valid configuration");
            return false;
        }
        $return = true;
        if (empty($config['origin_alias'])) {
            Utils::CLIerror("No origin_alias set for origin server");
            $return = false;
        }
        if (empty($config['destination_alias'])) {
            Utils::CLIwarning("No destination_alias for destination server");
            $return = false;
        }
        return $return;
    }
    
}