<?php
/****************************************************************
 * IBM Confidential
 *
 * SFA100-Collaboration Source Materials
 *
 * (C) Copyright IBM Corp. 2014
 *
 * The source code for this program is not published or otherwise
 * divested of its trade secrets, irrespective of what has been
 * deposited with the U.S. Copyright Office
 *
 ***************************************************************/

require_once('include/EntryPoint.php');
require_once('include/Layers/Application.php');
require_once('include/Layers/Component.php');
require_once('include/Layers/Environment.php');
require_once('include/Layers/Resource.php');
require_once('include/Layers/Team.php');
/**
 * Restore uDeploy from files created by BackupUCD class
 * 
 * @class   RestoreUCD
 * @author  Marco Bassi         marcobas@ie.ibm.com
 * @author  Kostadin Ivanov     kostadii@ie.ibm.com
 */
class RestoreUCD extends EntryPoint {
    
    protected $source = '';
    protected $resource = null;
    
    public function __construct($alias = null){
        parent::__construct($alias);
        $this->setReturn('json');
        $this->resource = new Resource($alias);
    }
    
    /**
     *
     * Assing a team to all components in uDeploy
     *
     * @param string $team
     * @return boolean
     */
    public function addTeamToAllComponents($team) {
        $componentAPIs = new Component($this->alias);
        $teamAPIs = new Team($this->alias);

        if (!$teamAPIs->exists($team)) {
            Utils::CLIerror("Team {$team} doesn't exist");
            return false;
        }
        
        Utils::CLIinfo("Adding team '{$team}' to all available components...");
        
        $componentList = $componentAPIs->getList();
    
        $result = true;
        foreach ($componentList as $component) {
            $success = $componentAPIs->addToTeam($component['id'], $team);
            if ($success === false) {
                Utils::CLIout("  [FAILURE] {$component['name']}");
                $result = false;
            } else {
                Utils::CLIout("  [SUCCESS] {$component['name']}");
            }
        }
        
        return $result;
    }

    /**
     *
     * Assing a team to all resources in uDeploy
     *
     * @param string $team
     * @return boolean
     */
    public function addTeamToAllResources($team)
    {
        $teamAPIs = new Team($this->alias);
        $resourceAPIs = new Resource($this->alias);
                
        if (!$teamAPIs->exists($team)) {
            Utils::CLIerror("Team {$team} doesn't exist");
            return false;
        }
        
        // Getting all the resources, getList is returnig resources into resutl['records']
        $resources = $resourceAPIs->getList();
        $resources = $resources['records'];
        
        $result = true;
        foreach ($resources as $resource) {
            $success = $resourceAPIs->addToTeam($resource['id'], $team);
            if ($success === false) {
                Utils::CLIout("  [FAILURE] {$resource['name']}");
                $result = false;
            } else {
                Utils::CLIout("  [SUCCESS] {$resource['name']}");
            }
        }
        
        return $result;
    }

    /**
     * Giving a folder path, will look for all available applications and restore them all, components included.
     * Folder should be in the format BackupUCD is creating it
     * OUTPUT_DIRECTORY
     * - APPLICATION_NAME
     * - APPLICATION_NAME.json
     * - blueprints
     * -BLUEPRINT_1.json
     * -BLUEPRINT_1.json
     *
     * @param string $applicationsBackupsDir
     *            FULL PATH to a directory with UrbanCode Applications backups
     * @param boolean $upgrade
     *            Boolean FLAG, used to define, whether the specified ComponentTemplate
     *            will be CREATED or UPGRADED.
     *            DEFAULT value: FALSE (RESTORE (create))
     * @param boolean $full
     *            restore with components and blueprints
     *            
     * @param boolean $environments
     *            restore with environments
     *            
     * @return array | boolean Array with the result from Restore/Upgrade result
     *         for each Application OR
     *         FALSE in case of a problem with specified $applicationsBackupsDir
     */
    public function applications($applicationsBackupsDir, $upgrade = false, $restoreEnvironments = false)
    {
        $applicationAPIs = new Application($this->alias);
        
        // Set Action: RESTORE (create) or UPGRADE
        $process = $upgrade ? 'UPGRADE' : 'RESTORE (create)';
        Utils::CLIinfo($process . ' UrbanCode Applications from:', true);
        Utils::CLIdest($applicationsBackupsDir, false);
        // Pre-check 1
        if (! is_dir($applicationsBackupsDir)) {
            Utils::CLIerror('Problem with Applications backup folder: ', true);
            Utils::CLIdest($applicationsBackupsDir, false);
            Utils::CLIerror('Invalid Backup Folder.', false);
            return false;
        }
        // Pre-check 2
        Utils::CLIout("  Checking source directory");
        if (! Utils::validateDirectoryJSON($applicationsBackupsDir)) {
            Utils::CLIerror("Source directory didn't pass json validation check");
            return false;
        }
        
        // Getting all the existing applications
        $existingApplictions = array();
        $applications = $applicationAPIs->getList();
        foreach ($applications as $application) {
            array_push($existingApplictions, $application['name']);
        }
        
        $return = array();
        // scan of the 'applications' dir
        $appsDirs = scandir($applicationsBackupsDir);
        foreach ($appsDirs as $subdir) {
            // Skip: '.' and '..' pointers, get app folder
            if (! in_array($subdir, array(
                '.',
                '..'
            )) && is_dir("{$applicationsBackupsDir}/{$subdir}")) {
                
                // import is stopped if app exist
                if (in_array($subdir, $existingApplictions) && ! $upgrade) {
                    Utils::Cliout(" [SKIPPING] Application {$subdir} already exists");
                    $return["${subdir}"] = false;
                    continue;
                }
                
                // In the folder we can have, name.json/id.json
                // since we cant know the id of the exported application
                // listFiles will return the list of json files
                $jsons_app = Utils::listFiles("{$applicationsBackupsDir}/{$subdir}", 'json');
                if (count($jsons_app) != 1) {
                    Utils::CLIerror("Invalid foder for {$subdir} application: too many json files. Check {$applicationsBackupsDir}/{$subdir}.");
                    Utils::Cliout(" [SKIPPING] Application '{$subdir}'.");
                    $return["${subdir}"] = false;
                    continue;
                }
                // Application object
                $json_file = $jsons_app[0];
                $app_array = file_get_contents($json_file);
                $app_array = json_decode($app_array, true);
                
                // Clean envs
                $app_array = $restoreEnvironments ? $app_array : $applicationAPIs->cleanEnvironment($app_array);
                Utils::CLIinfo("Restoring '$subdir' application.");
                Utils::CLIinfo("This can take a while...", false, 3);
                // Import in Application layer will handle import or upgrade
                // upgrade never stopped
                $result = $applicationAPIs->import($json_file, in_array($subdir, $existingApplictions));
                if ($result === false) {
                    Utils::CLIerror("Error importing '$subdir' application;");
                    $return["${subdir}"] = false;
                    continue;
                }
                $return["${subdir}"] = true;
                // TODO
                // Restore blueprints/resources in another step of restoreAll
                Utils::Cliout("[SUCCESS] $subdir restored successfully.\n");
            }
        }
        return $return;
    }

    /**
     * Restoring blueprints, resourcetemplate for each application
     *
     * @param string $applicationDir
     */
    public function blueprints($applicationDir)
    {
        $applicationAPIs = new Application($this->alias);
        $resourceAPIs = new Resource($this->alias);
        $applicationAPIs->setOutput($this->output);
        $resourceAPIs->setOutput($this->output);
        
        if (! is_dir($applicationDir)) {
            Utils::CLIerror("'$applicationDir' invalid path.");
            return false;
        }
        
        $result = array();
        // Scar application directory
        $iteratorApp = new FilesystemIterator($applicationDir, FilesystemIterator::SKIP_DOTS);
        $filelist = array();
        
        // Loop for applications
        foreach ($iteratorApp as $entry) {
            if (! $entry->isDir()) {
                continue;
            }
            
            // Retrieve existing app blueprints
            $destBlueprints = array();
            $result = $resourceAPIs->getBlueprintList($entry->getFilename());
            if (! empty($result)) {
                foreach ($result as $key => $value) {
                    array_push($destBlueprints, $value['name']);
                }
            }
            
            $files = Utils::listFiles("{$entry->getRealPath()}/blueprints", "json");
            // loop for each blueprint
            foreach ($files as $blueprint) {
                $blueprint = file_get_contents($blueprint); // json string
                $blueprint = json_decode($blueprint, true); // array
                $destApp = $applicationAPIs->get($entry->getFilename(), false);
                
                foreach ($blueprint as $bp) {
                    Utils::CLIout("   [TEMPLATE] {$bp['resourceTemplate']['name']} ");
                    $result = $resourceAPIs->upgradeTemplate($bp['resourceTemplate']);
                    if ($result === false) {
                        Utils::CLIwarning("Failed to restore resource template '{$bp['resourceTemplate']['name']}' ");
                        continue;
                    }
                    $templateList = $resourceAPIs->getTemplateList();
                    $templateId = null;
                    foreach ($templateList as $single) {
                        if ($single['id'] == $bp['resourceTemplate']['name'] || $single['name'] == $bp['resourceTemplate']['name']) {
                            $templateId = $single['id'];
                            break;
                        }
                    }
                    if (! in_array($bp['name'], $destBlueprints) && ! empty($templateId)) {
                        $data['applicationId'] = $destApp['id'];
                        $data['description'] = $bp['description'];
                        $data['name'] = $bp['name'];
                        $data['parentId'] = $templateId;
                        Utils::CLIout("  [BLUEPRINT] {$bp['name']} ");
                        $result = $resourceAPIs->createBlueprint($data);
                    }
                }
            }
        }
        return $result;
    }

    /**
     * Import / Upgrade multiple Components.
     * Will scan the provided components directory and will attempt to
     * Import / Upgrade each available Component JSON file by calling
     * $componentAPIs->create|upgrade
     *
     * @param string $componentsBackupsDir
     *            Full Path to a Directory, which holds one or more
     *            Component JSON formatted files.
     * @param boolean $upgrade
     *            Boolean TRUE / FALSE - flag to set wether to run: UPGRADE or IMPORT process
     *            
     * @return boolean array FALSE in case of a problem with the specified $componentsBackupsDir or
     *         Array holding the results from $componentAPIs->create|upgrade calls for each
     *         available in the specified $componentsBackupsDir Componente JSON file
     */
    public function components($componentsBackupsDir, $upgrade = false)
    {
        $componentAPIs = new Component($this->alias);
        
        $result = array();
        // Set Action: IMPORT or UPGRADE
        $process = $upgrade ? 'upgrade' : 'create';
        
        Utils::CLIinfo($process . ' UrbanCode Components from:', true);
        Utils::CLIdest($componentsBackupsDir, false);
        
        Utils::CLIout("  Checking source directory");
        $validJson = Utils::validateDirectoryJSON($componentsBackupsDir);
        
        if (! $validJson) {
            Utils::CLIerror("Source directory didn't pass json validation check");
            return false;
        }
        
        if (is_dir($componentsBackupsDir)) {
            $componentsBackups = scandir($componentsBackupsDir);
            
            $existingComponents = array();
            $components = $componentAPIs->getList();
            
            foreach ($components as $component) {
                array_push($existingComponents, $component['name']);
            }
            
            foreach ($componentsBackups as $componentFile) {
                // Process ONLY 'ComponentTemplate.json' files
                if (strpos($componentFile, '.json') !== false && $componentFile != '.json') {
                    // Set component filepath
                    $componentFilePath = $componentsBackupsDir . '/' . $componentFile;
                    // Get component Name
                    $componentName = end(explode('/', $componentFile));
                    // Remove the .json part from the component name
                    $componentName = str_ireplace('.json', '', $componentName);
                    
                    if (in_array($componentName, $existingComponents) && ! $upgrade) {
                        Utils::CLIout("  [SKIPPING] {$componentName}");
                        continue;
                    }
                    // process = upgrade | create
                    $result[$componentName] = $componentAPIs->$process($componentFilePath);
                    
                    if ($result[$componentName] === false) {
                        Utils::CLIout("  [FAILURE] {$componentName}");
                    } else {
                        Utils::CLIout("  [SUCCESS] {$componentName}");
                    }
                } // else skip this file, as it is not a proper json file
            }
        } else {
            Utils::CLIerror('Problem with Components backup folder: ', true);
            Utils::CLIdest($componentsBackupsDir, false);
            Utils::CLIerror('Backup Folder DOES NOT exist or specified PATH is wrong.', false);
            
            $result = false;
        }
        
        return $result;
    }

    /**
     * Import / Upgrade multiple ComponentTemplates.
     * Will scan the provided templates directory and will attempt to
     * Import / Upgrade each available Template JSON file 
     *
     * @param string  $templatesDir Full Path to a Directory, which holds one or more
     *                              ComponentTemplates JSON formatted files.
     * @param boolean $upgrade      Boolean TRUE / FALSE - flag to set wether to run: UPGRADE or IMPORT process
     *
     * @return boolean|array
     */
    public function componentTemplates ($templatesDir, $upgrade = false) {
        $componentAPIs = new Component($this->alias);
        
        $result = array();
        // Set Action: IMPORT or UPGRADE
        $process = $upgrade ? 'UPGRADE' : 'IMPORT';
    
        Utils::CLIinfo( $process . ' UrbanCode Component Templates from:', true);
        Utils::CLIdest( $templatesDir, false);
    
        Utils::CLIout("  Checking source directory");
        $validJson = Utils::validateDirectoryJSON( $templatesDir );
    
        if (!$validJson) {
            Utils::CLIerror("Source directory didn't pass json validation check");
            return false;
        }
    
        if ( is_dir( $templatesDir ) ) {
            // Get existing component templates
            $templates = $componentAPIs->listTemplates();
            $existingTemplates = array();
    
            foreach ($templates as $template) {
                array_push($existingTemplates, $template['name']);
            }
    
            $templatesFiles = scandir($templatesDir);
    
            foreach ($templatesFiles as $templateFile) {
                // Process ONLY 'ComponentTemplate.json' files
                if ( strpos($templateFile, '.json') !== false && $templateFile != '.json' ) {
                    $templateBackupFile   = $templatesDir . '/' . $templateFile;
    
                    // Check if component template exists
                    $currentTemplate = json_decode(file_get_contents($templateBackupFile), true);
                    if (in_array($currentTemplate['name'], $existingTemplates) && !$upgrade) {
                        Utils::CLIout("  [SKIPPING] {$currentTemplate['name']}");
                        continue;
                    }
                    $result[$templateFile] = $componentAPIs->createTemplate($templateBackupFile, $upgrade);
                } // else skip this file, as it is not a proper json file
            }
        } else {
            Utils::CLIerror( 'Problem with Component Templates backup folder: ', true );
            Utils::CLIdest( $templatesDir, false );
            Utils::CLIerror( 'Component Templates Backup Folder DOES NOT exist or specified PATH is wrong.', false);
            
            $result = false;
        }
        
        return $result;
    }

    /**
     *
     * restores all post processing scripts from exported file
     *
     * @param unknown $file
     * @return boolean array
     */
    public function postProcessingScripts($settingsDir, $upgrade)
    {
        $processAPIs = new Process($this->alias);
        
        
        Utils::CLIinfo("Restoring Post Processing Scritps from {$settingsDir}");
        $result = array();
        
        // Check directory
        if (is_dir($settingsDir)) {
            Utils::CLIout("  Checking source directory");
            $validJson = Utils::validateDirectoryJSON($settingsDir);
            
            if (! $validJson) {
                Utils::CLIerror("Source directory didn't pass json validation check");
                return false;
            }
            
            // Get existing post-processing scripts
            $pps = $processAPIs->getPostProcessingScripts();
            $existingScripts = array();
            foreach ($pps as $ppscript) {
                array_push($existingScripts, $ppscript['name']);
            }
            
            $files = scandir($settingsDir);
            // Scan directory for files
            foreach ($files as $file) {
                if ($file == '.' || $file == '..') {
                    continue;
                }
                $file = "{$settingsDir}/{$file}";
                if (is_file($file)) {
                    // Get file contents
                    $fileContents = file_get_contents($file);
                    
                    if (Utils::validateJSON($fileContents)) {
                        $scripts = json_decode($fileContents, true);
                        foreach ($scripts as $script) {
                            // Check if script already exists
                            if (in_array($script['name'], $existingScripts)) {
                                Utils::CLIout("  [SKIPPING] {$script['name']}");
                                continue;
                            }
                            
                            // Restore scripts
                            $output = $processAPIs->createPostProcessingScript($script);
                            if ($output === false) {
                                Utils::CLIout("  [FAILURE] {$script['name']}");
                                $result[$file][$script['name']] = false;
                            } else {
                                Utils::CLIout("  [SUCCESS] {$script['name']}");
                                $result[$file][$script['name']] = $output;
                            }
                        }
                    } else {
                        Utils::CLIwarning("File {$file} is not a valid JSON file. Skipping.", true);
                    }
                } else {
                    Utils::CLIerror('Specified Post Processing Scripts File is not a valid file.', true);
                    $this->log->error('Specified Post Processing Scripts file: "' . $file . '" is NOT a valid file');
                    $result[$file] = false;
                }
            }
        } else {
            Utils::CLIerror('Specified settings directory is not valid.', true);
            $this->log->error('Specified settings directory is not valid: "' . $$settingsDir . '" is NOT a valid file');
            $result = false;
        }
        
        return $result;
    }

    /**
     *
     * restores all processes
     *
     * @param string $dir
     * @param bool $upgrade
     * @return mixed 
     */
    public function processes($dir, $upgrade = false)
    {
        $processAPIs = new Process($this->alias);
        Utils::CLIinfo("Restoring processes from {$dir}");
        
        $result = array();
        $files = scandir($dir);
        
        // Retrieve existing processes
        $processes = $processAPIs->getList();
        $existingProcesses = array();
        foreach ($processes as $process) {
            array_push($existingProcesses, $process['name']);
        }
        
        // Files loop
        foreach ($files as $file) {
            if ($file != '.' && $file != '..' && ! is_dir("{$dir}/{$file}")) {
                $filePath = "{$dir}/{$file}";
                $json = file_get_contents($filePath);
                if (! Utils::validateJSON($json)) {
                    Utils::CLIwarning("File {$filePath} is not a valid JSON file");
                    continue;
                }
                
                $process = json_decode($json, true);
                $exist = in_array($process['name'], $existingProcesses);
                
                // If exist and create -> skip
                if ($exist && !$upgrade) {
                    Utils::CLIout("  [SKIPPING] {$process['name']}");
                    continue;
                }
                // Create if not exist
                if (!$exist){
                    $result[$process['name']] = $processAPIs->create($filePath);
                }else {
                    $result[$process['name']] = $processAPIs->upgrade($filePath);
                }
                
                if ($result[$process['name']] !== false) {
                    Utils::CLIout("  [SUCCESS] {$process['name']}");
                } else {
                    Utils::CLIout("  [FAILURE] {$process['name']}");
                }
            }
        }
        
        return $result;
    }

    /**
     *
     * restore all resource templates from a directory
     *
     * @param string $templatesDir
     *
     * @return boolean array
     */
    public function resourceTemplates($templatesDir)
    {
        $resourceAPIs = new Resource($this->alias);
        $resourceAPIs->setOutput($this->output);
        
        Utils::CLIinfo("Restoring Resource Templates from {$templatesDir}");
        
        if (is_dir($templatesDir)) {
            Utils::CLIout("  Checking source directory");
            $validJson = Utils::validateDirectoryJSON($templatesDir);
            
            if (! $validJson) {
                Utils::CLIerror("Source directory didn't pass json validation check");
                return false;
            }
            
            $result = array();
            $templatesFiles = scandir($templatesDir);
            
            foreach ($templatesFiles as $file) {
                if ($file == '.' || $file == '..' || is_dir("{$templatesDir}/{$file}")) {
                    continue;
                }
                
                $filePath = "{$templatesDir}/{$file}";
                $result[$file] = $resourceAPIs->upgradeTemplate($filePath);
            }
        } else {
            Utils::CLIerror('Problem with Resource Templates backup folder: ', true);
            Utils::CLIdest($templatesDir, false);
            Utils::CLIerror('Resource Templates Backup Folder DOES NOT exist or specified PATH is wrong.', false);
            
            $result = false;
        }
        
        return $result;
    }

    /**
     * Run a FULL UCD RESTORE process from a complete UCD backup.
     * This includes:
     *      - restoring of UCD Applications and their corresponding Components,
     *      - restoring of UCD component Templates.
     *
     * @param string  $backup_dir   String, holding the FULL PATH to a UCD backup directory
     * @param boolean $upgrade      Boolean FLAG, used to define, whether the specified ComponentTemplate
     *                              will be CREATED or UPGRADED.
     *                              DEFAULT value: FALSE (RESTORE (create))
     *
     * @return array    Array, holding the results from 'restore' processes of UCD Applications and
     *                  Component Templates
     */
    public function restoreAll ( $backup_dir, $upgrade = false, $restoreEnvironments = false ) {
        $applicationAPIs = new Application($this->alias);
        $environmentAPIs = new Environment($this->alias);
        
        $result = array();
        $this->source = $backup_dir;
        
        // Set Action: RESTORE (create) or UPGRADE
        $process = $upgrade ? 'UPGRADE' : 'RESTORE (create)';
        
        $restore_watch =  time();
        StopWatch::getInstance()->start($restore_watch);
        // Prompt message for Backup Process
        Utils::CLIinfo( '### START of UrbanCode ' . $process . ' process ###' );
        Utils::CLIstart( $process.' process: '. date("d-m-Y h:i:s"));
        Utils::CLIinfo( 'Server to ' . $process . ':', false, 4 );
        Utils::CLIdest( $this->weburl, false, 6 );
        Utils::CLIinfo( $process . ' from backup folder:', false, 4 );
        Utils::CLIdest( $backup_dir, false, 6 );
        // Get full path of backupdir
        $backup_dir = realpath($backup_dir);
        
        Utils::CLIout("  Checking source directory");
        $validJson = Utils::validateDirectoryJSON( $backup_dir );
        
        if (!$validJson) {
            Utils::CLIerror("Source directory didn't pass json validation check");
            return false;
        }
        
        // Restore tags
        $restoreTags_watch =  time();
        StopWatch::getInstance()->start($restoreTags_watch);
        $result['restore_tags'] = $this->tags( $backup_dir . '/tags' );
        StopWatch::getInstance()->stop($restoreTags_watch);
        Utils::CLIcompleted( 'Restore tags: '. StopWatch::getInstance()->get($restoreTags_watch));
        
        // Restore all post processing scripts
        $postProcessingScripts_watch =  time();
        StopWatch::getInstance()->start($postProcessingScripts_watch);
        $result['post_processing_scripts'] = $this->postProcessingScripts( $backup_dir . '/settings', $upgrade);
        StopWatch::getInstance()->stop($postProcessingScripts_watch);
        Utils::CLIcompleted( 'Post Processing Scripts: '. StopWatch::getInstance()->get($postProcessingScripts_watch));
        
        // Restore all general processes
        $generalProcesses_watch =  time();
        StopWatch::getInstance()->start($generalProcesses_watch);
        $result['processes'] = $this->processes( $backup_dir . '/processes', $upgrade );
        StopWatch::getInstance()->stop($generalProcesses_watch);
        Utils::CLIcompleted( 'General Processes: '. StopWatch::getInstance()->get($generalProcesses_watch));
        
        // Restore ALL ComponentTemplates
        $restoreComponentTemplates_watch =  time();
        StopWatch::getInstance()->start($restoreComponentTemplates_watch);
        $result['restore_component_templates'] = $this->componentTemplates( $backup_dir . '/componentTemplates', $upgrade );
        StopWatch::getInstance()->stop($restoreComponentTemplates_watch);
        Utils::CLIcompleted( 'Restore Component Templates: '. StopWatch::getInstance()->get($restoreComponentTemplates_watch));
        
        // Restore ALL Components
        $restoreComponents_watch =  time();
        StopWatch::getInstance()->start($restoreComponents_watch);
        $result['restore_components'] = $this->components( $backup_dir . '/components', $upgrade );
        StopWatch::getInstance()->stop($restoreComponents_watch);
        Utils::CLIcompleted( 'Restore Components: '. StopWatch::getInstance()->get($restoreComponents_watch));
        
        // Restore ALL Applications (includes All APPS + underlying App components, blueprints)
        $restoreAllApplications_watch =  time();
        StopWatch::getInstance()->start($restoreAllApplications_watch);
        $result['restore_applications'] = $this->applications( $backup_dir . '/applications', $upgrade, $restoreEnvironments );
        StopWatch::getInstance()->stop($restoreAllApplications_watch);
        Utils::CLIcompleted( 'Restore Applications: '. StopWatch::getInstance()->get($restoreAllApplications_watch));
        
        // Restore resource templates
        $restoreResourceTemplates_watch =  time();
        StopWatch::getInstance()->start($restoreResourceTemplates_watch);
        $result['restore_resource_templates'] = $this->resourceTemplates( $backup_dir . '/resourceTemplates' );
        StopWatch::getInstance()->stop($restoreResourceTemplates_watch);
        Utils::CLIcompleted( 'Restore Resource Templates: '. StopWatch::getInstance()->get($restoreResourceTemplates_watch));
        
        // Restore ALL Blueprints
        // The bluePrints are stored inside each application folder
        $restoreBlueprints =  time();
        StopWatch::getInstance()->start($restoreBlueprints);
        $result['restore_blueprints'] = $this->blueprints( $backup_dir . '/applications' );
        StopWatch::getInstance()->stop($restoreBlueprints);
        Utils::CLIcompleted( 'Restore Blueprints: '. StopWatch::getInstance()->get($restoreBlueprints));
        
        StopWatch::getInstance()->stop($restore_watch);
        Utils::CLIcompleted( $process .' process: '. StopWatch::getInstance()->get($restore_watch));
        // Prompt message for end of process and return
        Utils::CLIinfo( '### END of UrbanCode ' . $process . ' process ###' );
        return $result;
    }

    /**
     * // used in Fork.php
     * Creates a new blueprint associated to the application set as argument
     *
     * @param string $applicationId
     * @param string $blueprintName
     * @param string $templateId
     * @param string $description
     * @return boolean
     */
    public function restoreBlueprint($applicationId, $blueprintName, $templateId, $description = '') {
        $resourceAPIs = new Resource($this->alias);
        $error = false;
        if (empty($applicationId)) {
            Utils::CLIerror("Application ID required");
            $error = true;
        }
        if (empty($blueprintName)) {
            Utils::CLIerror("Blueprint name required");
            $error = true;
        }
        if (empty($templateId)) {
            Utils::CLIerror("Template ID required");
            $error = true;
        }
        if ($error) {
            return false;
        }
        $json = array(
            'applicationId' => $applicationId,
            'name' => $blueprintName,
            'resourceTemplateId' => $templateId,
            'description' => $description,
        );
        $result = $resourceAPIs->createBlueprint($json);
        if ($result !== false) {
            Utils::CLIinfo("Creation of blueprint '{$blueprintName}' - SUCCESS");
        } else {
            Utils::CLIinfo("Creation of blueprint '{$blueprintName}' faliled");
        }
        return $result;
    }

    /**
     *
     * restore all tags from a directory
     *
     * @param string $dir
     *
     * @return mixed
     */
    public function tags( $dir ) {
        $resourceAPIs = new Resource($this->alias);
        
        Utils::CLIinfo( "Restoring tags from {$dir}" );
        
        $result = array();
        $files = scandir($dir);
        
        // Retrieve existing tags
        $resourceAPIs->setReturn('array');
        $tagsComponent = $resourceAPIs->getTag('component');
        $tagsResource = $resourceAPIs->getTag('resource');
        
        $existingTags['component'] = array();
        $existingTags['resource'] = array();
        
        foreach ($tagsComponent as $tag) {
            array_push($existingTags['component'], $tag['name']);
        }
        
        foreach ($tagsResource as $tag) {
            array_push($existingTags['resource'], $tag['name']);
        }
        
        foreach ($files as $file) {
            if ($file != '.' && $file != '..') {
                $filePath = "{$dir}/{$file}";
                $json = file_get_contents($filePath);
                if (! Utils::validateJSON($json)) {
                    Utils::CLIwarning("File {$filePath} is not a valid JSON file");
                    continue;
                }
                $tags = json_decode($json, true);
                foreach ($tags as $tag) {
                    // Check if tag already exist
                    $type = strtolower($tag['objectType']);
                    if (in_array ($tag['name'], $existingTags[$type])) {
                        Utils::CLIout("  [SKIPPING] {$tag['name']}");
                        continue;
                    }
                    
                    $tag['url'] = "/rest/tag/{$tag['objectType']}";
                    $tag['ids'] = array();
                    // Create tmp file into current output directory
                    $tag_path = $this->output."tag_".microtime(true).".json";
                    $tagFile = Utils::createJsonFile($tag,$tag_path);
                    $result[$tag['name']] = $this->rest()->system()->createTag($tagFile);
                    
                    if ($result[$tag['name']] !== false) {
                        Utils::CLIout("  [SUCCESS] {$tag['name']}");
                    } else {
                        Utils::CLIout("  [FAILURE] {$tag['name']}");
                    }
                    
                    unlink($tagFile);
                }
                
                Utils::CLIout("");
            }
        }
        return $result;
    }

    /**
     * 
     * Upgrade all the application
     *
     * @param string $applicationsBackupsDir
     * @return boolean
     */
    public function upgradeAllApplications( $applicationsBackupsDir ) {
        $applicationAPIs = new Application($this->alias);
        Utils::CLIinfo( 'Upgrade UrbanCode Applications from:', true );
        Utils::CLIdest( $applicationsBackupsDir, false );
        
        if ( !is_dir( $applicationsBackupsDir ) ) {
            Utils::CLIerror( 'Problem with Applications backup folder: ', true );
            Utils::CLIdest( $applicationsBackupsDir, false );
            Utils::CLIerror( 'Backup Folder DOES NOT exist or specified PATH is wrong.', false );
            $result =  false;
        }
        
        Utils::CLIout("  Checking source directory");
        $validJson = Utils::validateDirectoryJSON( $applicationsBackupsDir );
        
        if (!$validJson) {
            Utils::CLIerror("Source directory didn't pass json validation check");
            return false;
        }
        
        $result = array();
        //get applications from Server
        $appOnServer = $applicationAPIs->getList();
        // populating existing app array
        $existingApps = array();
        foreach ($appOnServer as $app) {
            array_push($existingApps, $app['name']);
        }
        
        $appsDirs = scandir($applicationsBackupsDir);
        foreach ($appsDirs as $subdir) {
            // Skip: '.' and '..' pointers
            if ( !in_array($subdir, array('.', '..')) ) {
                // Get files from subdir
                $jsons_app = Utils::listFiles("{$applicationsBackupsDir}/{$subdir}", 'json');
                if (count($jsons_app) != 1) {
                    Utils::CLIerror("Invalid foder for {$subdir} application: too many json files. Check {$applicationsBackupsDir}/{$subdir}.");
                    Utils::Cliout(" [SKIPPING] Application '{$subdir}'.");
                    $return["${subdir}"] = false;
                    continue;
                }
                
                // decode json file to get the app name
                $appArray = file_get_contents($jsons_app[0]);
                $appArray = json_decode($appArray, true);
                // check for upgrade
                $upgrade = in_array($appArray['name'], $existingApps);
                $result[$subdir] = $applicationAPIs->import($jsons_app[0], $upgrade);
            }
        }
        
        return $result;
    }
}