<?php
/****************************************************************
* IBM Confidential
*
* SFA100-Collaboration Source Materials
*
* (C) Copyright IBM Corp. 2015
*
* The source code for this program is not published or otherwise
* divested of its trade secrets, irrespective of what has been
* deposited with the U.S. Copyright Office
*
* **************************************************************/

require_once('include/EntryPoint.php');

/**
 * Provides tools for creating and managing Applicarions requests
 *
 * @class Application
 * @author Felice Geracitano.......feliecege@ie.ibm.com
 */
class Application extends EntryPoint {
    public function __construct($alias = null){
        parent::__construct($alias);
        $this->setReturn('json');
    }

    /**
     * Validate json|object application and clean the environments inside
     *
     * @param   object|string application
     * @return  obj|bool
     */
    public function cleanEnvironment($application)
    {
        if (!is_string($application) && !is_array($application) && !is_object($application)){
            return false;
        }
        // Save parameter
        $input_param = $application;
        // Validate json if is a file path
        if (is_string($application)) {
            if (file_exists($application) === false) {
                Utils::CLIerror("Json file does not exist");
                Utils::CLIdebug(__CLASS__ . '::' . __FUNCTION__ . ": Json file '{$application}' does not exist");
                return false;
            }
            if (Utils::validateJSON(file_get_contents($application)) === false) {
                Utils::CLIerror("Json file has wrong format");
                Utils::CLIdebug(__CLASS__ . '::' . __FUNCTION__ . ": Json file '{$application}' has wrong format");
                return false;
            }
            // Get obj from json
            $application = json_decode(file_get_contents($application),true);
        }
        
        // Here is an array/object
        if (array_key_exists('environments', $application)) {
            Utils::CLIinfo("Cleaning environments.");
            $application['environments'] = array();
        }
        
        // Return file path if was a file path
        if (is_string($input_param)) {
            if (Utils::createJsonFile($application, $input_param, true) === false) {
                Utils::CLIerror("Failed to save the file.");
                return false;
            }
            return $input_param;
        }
        
        // Return an array if was an array
        return $application;
    }

    /**
     * Create a ApplicationProcess from scratch or import from a file path
     *
     * @param array|string $ApplicationProcess
     * @return array|string|boolean
     */
    public function createProcess( $process ) {
        $file_name = null;
        // Check if arg is file path or array obj
        if (is_array( $process ) || is_object( $process )) {
            $file_name = Utils::createJsonFile( $process );
        } else {
            if (file_exists($process) === false) {
                Utils::CLIerror( "Json file does not exist." );
                Utils::CLIdebug( __CLASS__ . '::' . __FUNCTION__.": Json file '{$process}' does not exist." );
                return false;
            }
            if (Utils::validateJSON( file_get_contents( $process ) ) === false) {
                Utils::CLIerror( "Json file has wrong format." );
                Utils::CLIdebug( __CLASS__ . '::' . __FUNCTION__.": Json file '{$process}' has wrong format," );
                return false;
            }
            $file_name = $process;
        }
        if ($this->udclient()->application()->createApplicationProcess( $file_name ) === false) {
            Utils::CLIerror( "Application Process already exist." );
            return false;
        }
        is_string( $process ) === false ? unlink( $file_name ) : null;
        return true;
    }
    
    /**
     * Delete a Application passing Id or Name
     *
     * @param string $application
     * @return boolean
     */
    public function delete($application) {
        if (empty($application) || !is_string($application) ){
            Utils::CLIerror( "Invalid 'application' parameter." );
            return false;
        }
        // Delete Applicatoin by Name or Id
        $result = $this->rest()->application()->deleteApplication($application);
        if ($result === false){
            Utils::CLIerror( "Application doesn't exist or the current user doesn't have enough rights to delete it" );
            return false;
        }
        return true;
    }
    
    /**
     * Delete a Application Processes passing Id or Name
     *
     * @param string $id
     * @return boolean
     */
    public function deleteProcess($processId) {
        if (empty($processId) || !is_string($processId)){
            Utils::CLIerror( "Invalid 'processId' parameter." );
            return false;
        }
        // Delete Applicatoin Proccess by id
        $result = $this->rest()->application()->deleteApplicationProcess($processId);
        if ($result === false){
            Utils::CLIerror( "Application Process doesn't exist or the current user doesn't have enough rights to delete it." );
            return false;
        }
        return true;
    }
    
    /**
     * Check if App exists
     *
     * @param string $app
     * @return boolean
     */
    public function exists($app)
    {
        $return_type = $this->return;
        if (empty($app) || !is_string($app)) {
            Utils::CLIerror("Invalid Application parameter.");
            return false;
        }
        $this->setReturn('json');
        $apps = $this->getList();
    
        $result = false;
        foreach ($apps as $item) {
            if ($item['name'] == $app || $item['id'] == $app) {
                $result = true;
            }
        }
        $this->setReturn($return_type);
        return $result;
    }

    /**
     * Get short or full description Application from Id or Name
     *
     * @param string $application
     * @param boolean $full_object
     * @return array|path|boolean
     */
    public function get($application, $full_object = false) {
        $result = null;
        if (empty( $application ) || !is_string($application)) {
            Utils::CLIerror( "Invalid Application name/id parameter." );
            return false;
        }
        // getApplication provides small description, getApplicationRest
        // provides large description
        if (empty( $full_object )) {
            $result = $this->udclient()->application()->getApplication( $application );
        } else {
            $result = $this->rest()->application()->getApplicationRest( $application );
        }
        
        // If return is set to file, then the API Rest returns a file with an
        // error message
        // so if API Rest fails or returning a file is not a json -> error
        if ($result === false ||
                 ($this->return === 'file' &&
                 ! json_decode( file_get_contents( str_replace("'","",$result) ) ))) {
            Utils::CLIerror( "Application does not exist" );
            return false;
        }
        
        if ($this->return === 'file') {
            // delete quotes and return path
            return str_replace("'","",$result);
        }
        return Utils::outputToArray( $result );
    }

    
    /**
     * Get list of all Applications from UCD server
     *
     * @param string $application
     * @return array path boolean
     */
    public function getProcesses($application)
    {
        if (empty($application) || !is_string($application)) {
            Utils::CLIerror("Invalid Application name/id.");
            return false;
        }
        $result = $this->rest()->application()->getApplicationProcesses($application);
        if ($result === false) {
            Utils::CLIerror("Failed to retrieve application processes list");
            return false;
        }
        if ($this->return === 'file') {
            // delete quotes and return path
            return str_replace("'","",$result);
        }
        return Utils::outputToArray($result);
    }
    
    /**
     * Get list of all Snapshots
     *
     * @param string $application
     * @return array path boolean
     */
    public function getSnapshots($application)
    {
        if (empty($application) || !is_string($application)) {
            Utils::CLIerror("Invalid Application name/id.");
            return false;
        }
        $result = $this->rest()->snapshot()->getSnapshotsInApplication($application);
        if ($result === false) {
            Utils::CLIerror("Failed to retrieve application snapshot list");
            return false;
        }
        if ($this->return === 'file') {
            // delete quotes and return path
            return str_replace("'","",$result);
        }
        return Utils::outputToArray($result);
    }
    
    /**
     * Get processes from application
     *
     * @return array|path|boolean
     */
    public function getList() {
        $result = $this->rest()->application()->getApplicationsRest();
        if ($result === false){
            Utils::CLIerror( "Failed to retrieve applications list" );
            return false;
        }
        if ($this->return === 'file'){
            // delete quotes and return path
            return str_replace("'","",$result);
        }
        return Utils::outputToArray( $result );
    }
    /**
     * import application is a wrapper of import/upgrade application
     * - the json/object is exaclty the same for import and upgrade
     * - the json/object is different of what you need to create an application from scratch, see 'create' method
     * @param string $application
     * @param bool $upgrade
     * @return boolean
     */
    public function import($application, $upgrade = null)
    {
        if (empty($application)) {
            Utils::CLIerror("Missing Application parameter.");
            return false;
        }
        
        if ($upgrade) {
            return $this->upgrade($application);
        }
    
        $file_name = null;
        // Check if arg is file path or array obj
        if (is_array( $application ) || is_object( $application )) {
            $file_name = Utils::createJsonFile( $application );
        } else {
            if (file_exists($application) === false) {
                Utils::CLIerror( "Json file does not exist" );
                Utils::CLIdebug( __CLASS__ . '::' . __FUNCTION__.": Json file '{$application}' does not exist" );
                return false;
            }
            if (Utils::validateJSON( file_get_contents( $application ) ) === false) {
                Utils::CLIerror( "Json file has wrong format" );
                Utils::CLIdebug( __CLASS__ . '::' . __FUNCTION__.": Json file '{$application}' has wrong format" );
                return false;
            }
            $file_name = $application;
        }
        if ($this->rest()->application()->importApplication( $file_name ) === false) {
            Utils::CLIerror( "Application already exist" );
            return false;
        }
        is_string( $application ) === false ? unlink( $file_name ) : null;
        return true;
    }
        
    /**
     *  upgrade an application passing array object or json file path
     *  allowed value for upgradeComponentTemplate and upgradeProcess
     ** USE_EXISTING_IF_EXISTS
     ** CREATE_NEW_IF_EXISTS
     ** FAIL_IF_EXISTS
     ** FAIL_IF_DOESNT_EXIST
     ** UPGRADE_IF_EXISTS
     *
     * @param array | string $application
     * @param string $processUpgrade
     * @param string $templateUpgrade
     * @return boolean
     */
    public function upgrade($application, $upgradeComponentTemplate = 'UPGRADE_IF_EXISTS', $upgradeProcess = 'UPGRADE_IF_EXISTS') {
        
        if (empty($application)) {
            Utils::CLIerror("Missing Application parameter.");
            return false;
        }
        
        $file_name = null;
        // Check if arg is file path or array obj
        if (is_array( $application ) || is_object( $application )) {
            $file_name = Utils::createJsonFile( $application );
        } else {
            if (file_exists($application) === false) {
                Utils::CLIerror( "Json file does not exist" );
                Utils::CLIdebug( __CLASS__ . '::' . __FUNCTION__.": Json file '{$application}' does not exist" );
                return false;
            }
            if (Utils::validateJSON( file_get_contents( $application ) ) === false) {
                Utils::CLIerror( "Json file is invalid" );
                Utils::CLIdebug( __CLASS__ . '::' . __FUNCTION__.": Json file '{$application}' Json file is invalid" );
                return false;
            }
            $file_name = $application;
        }
        
        if (! in_array( $upgradeComponentTemplate, $this->upgrade_values ) && ! is_null( $upgradeComponentTemplate )) {
            Utils::CLIerror( "Component Template upgrade has wrong parameter" );
            Utils::CLIdebug( __CLASS__ . '::' . __FUNCTION__ .": Template upgrade has wrong parameter '{$upgradeComponentTemplate}'" );
            return false;
        }
        if (! in_array( $upgradeProcess, $this->upgrade_values ) && ! is_null( $upgradeProcess )) {
            Utils::CLIerror( "Process upgrade has wrong parameter" );
            Utils::CLIdebug( __CLASS__ . '::' . __FUNCTION__ .": Process upgrade has wrong parameter '{$upgradeProcess}'" );
            return false;
        }
        $this->setReturn( 'json' );
        if ($this->rest()->application()->upgradeApplication( $file_name, $upgradeComponentTemplate, $upgradeProcess ) === false) {
            Utils::CLIdebug( __CLASS__ . '::' . __FUNCTION__ .": Application upgrade could have failed for these reasons: Component template option '$upgradeComponentTemplate', process option '$upgradeProcess'");
            Utils::CLIdebug( __CLASS__ . '::' . __FUNCTION__ .": Application doesn't exist.");
            return false;
        }
        is_string( $application ) === false ? unlink( $file_name ) : null;
        return true;
    }
}