<?php
/****************************************************************
 * IBM Confidential
 *
 * SFA100-Collaboration Source Materials
 *
 * (C) Copyright IBM Corp. 2015
 *
 * The source code for this program is not published or otherwise
 * divested of its trade secrets, irrespective of what has been
 * deposited with the U.S. Copyright Office
 *
 * **************************************************************/

require_once ('include/EntryPoint.php');

/**
 * Provides tools for creating and managing Resource requests
 *
 * @class Resource
 * @author Marco Bassi          marcobas@ie.ibm.com
 */
class Resource extends EntryPoint {
    
    public function __construct($alias = null){
        parent::__construct($alias);
        $this->setReturn('json');
    }
    
    /**
     * 
     * Add a tag to one or more resources. If the tag is missing, create it
     *
     * @param string $tag
     * @param string|array $resource
     * 
     * @return boolean
     */
    public function addTag ( $tag, $resource ) {
        if ( empty ( $tag )) {
            return false;
        }
        
        if ( !is_array( $resource ) ) {
            $tagCreate['ids'] = array( $resource );
        } else {
            $tagCreate['ids'] = $resource;
        }
        $tagCreate['name'] = $tag;
        $tagCreate['url'] = "/rest/tag/resource";
        $tagCreate['objectType'] = "Resource";
        $tagFile = Utils::createJsonFile($tagCreate);
        $result = $this->rest()->system()->createTag($tagFile);
        
        if ( $result !== false ) {
            unlink( $tagFile );
            return true;
        }
        return false;
    }

    /**
     * 
     * Add a Resource to Team
     *
     * @param string $resource
     * @param string $team
     * 
     * @return boolean
     */
    public function addToTeam ( $resource, $team ) {
        if (empty($resource)){
                Utils::CLIerror( "Missing Resource Id parameter." );
                Utils::CLIdebug( __CLASS__ . '::' . __FUNCTION__.": Missing Resource Id parameter." );
                return false;
        }

        if (empty($team)){
                Utils::CLIerror( "Missing Team parameter." );
                Utils::CLIdebug( __CLASS__ . '::' . __FUNCTION__.": Missing Team parameter." );
                return false;
        }
        
        if (!$this->exists($resource)){
            Utils::CLIerror( "Resource '$resource' does not exists." );
            return false;
        }
        
        // Connect to the current alias
        $teamLayer = new Team($this->alias);
        if (!$teamLayer->exists($team)){
            Utils::CLIerror( "Team '$team' does not exists." );
            return false;
        }
        
        // check if API fails
        // get Res ID
        $resObj=$this->get($resource);
        $resourceId = $resObj['id'];
        if ($this->udclient()->resource()->addResourceToTeam( $resourceId, $team ) === false){
            Utils::CLIerror( "Failed to add the resource to the team." );
            Utils::CLIdebug( __CLASS__ . '::' . __FUNCTION__.": Failed to add the resource: $resource to the team: $team." );
            return false;
        }
        return true;
    }


    /**
     * Create a resource from scratch or import a resource from a file path
     * 
     * @param array|json_file_path $resource
     * @return boolean
     */
    public function create( $resource ) {
        $file_name = null;
        // Check if arg is file path or array obj
        if ( is_array( $resource ) || is_object( $resource )) {
            $file_name = Utils::createJsonFile( $resource );
        } else {
            if ( file_exists($resource) === false ) {
                Utils::CLIerror( "Json file does not exist" );
                Utils::CLIdebug( __CLASS__ . '::' . __FUNCTION__.": Json file '{$resource}' does not exist" );
                return false;
            }
            if (Utils::validateJSON( file_get_contents( $resource ) ) === false) {
                Utils::CLIerror( "Json file has wrong format" );
                Utils::CLIdebug( __CLASS__ . '::' . __FUNCTION__.": Json file '{$resource}' has wrong format" );
                return false;
            }
            $file_name = $resource;
        }
        
        if ($this->udclient()->resource()->createResource( $file_name ) === false) {
            Utils::CLIerror( "Failed to create resource" );
            Utils::CLIdebug( "Source file located in '{$file_name}'");
            return false;
        }
        is_string( $process ) === false ? unlink( $file_name ) : null;
        return true;
    }
    
    /**
     * Create a blueprint from scratch or import a resource from a file path
     *
     * @param array|json_file_path $blueprint
     * @return boolean
     */
    public function createBlueprint ( $blueprint ) {
        $file_name = null;
        // Check if arg is file path or array obj
        if ( is_array( $blueprint ) || is_object( $blueprint )) {
            $file_name = Utils::createJsonFile( $blueprint );
        } else {
            if ( file_exists($blueprint) === false ) {
                Utils::CLIerror( "Json file does not exist" );
                Utils::CLIdebug( __CLASS__ . '::' . __FUNCTION__.": Json file '{$blueprint}' does not exist" );
                return false;
            }
            if (Utils::validateJSON( file_get_contents( $blueprint ) ) === false) {
                Utils::CLIerror( "Json file has wrong format" );
                Utils::CLIdebug( __CLASS__ . '::' . __FUNCTION__.": Json file '{$blueprint}' has wrong format" );
                return false;
            }
            $file_name = $blueprint;
        }
        
        $result = $this->rest()->resource()->createResourceTemplate( $file_name );
        if ( $result === false ) {
            Utils::CLIerror( "Failed to create blueprint" );
            return false;
        }
        
        if ( is_array ( $blueprint ) ) {
            unlink( $file_name );
        }
        return Utils::outputToArray( $result );
    }

        /**
     * Create a blueprint Resource from scratch or import a resource from a file path
     *
     * @param array|json_file_path $resource
     * @return boolean
     */
    public function createBlueprintResource ( $resource ) {
        $file_name = null;
        // Check if arg is file path or array obj
        if ( is_array( $resource ) || is_object( $resource )) {
            $file_name = Utils::createJsonFile( $resource );
        } else {
            if ( file_exists($resource) === false ) {
                Utils::CLIerror( "Json file does not exist" );
                Utils::CLIdebug( __CLASS__ . '::' . __FUNCTION__.": Json file '{$resource}' does not exist" );
                return false;
            }
            if (Utils::validateJSON( file_get_contents( $resource ) ) === false) {
                Utils::CLIerror( "Json file has wrong format" );
                Utils::CLIdebug( __CLASS__ . '::' . __FUNCTION__.": Json file '{$resource}' has wrong format" );
                return false;
            }
            $file_name = $resource;
        }
        
        $result = $this->rest()->resource()->createBlueprintResource( $file_name );
        if ( $result === false ) {
            Utils::CLIerror( "Failed to create blueprint" );
            return false;
        }
        
        if ( is_array ( $resource ) ) {
            unlink( $file_name );
        }
        return Utils::outputToArray( $result );
    }
    
    /**
     * Create a resource template from scratch or import a resource from a file path
     *
     * @param array|json_file_path $template
     * @return boolean|array
     */
    public function createTemplate ( $template ) {
        $file_name = null;
        $return = array();
        
        // Check if arg is file path or array obj
        if ( ! is_array( $template ) && ! is_object( $template )) {
            if ( file_exists( $template ) === false ) {
                Utils::CLIerror( "Json file does not exist" );
                Utils::CLIdebug( __CLASS__ . '::' . __FUNCTION__.": Json file '{$template}' does not exist" );
                return false;
            }
            if (Utils::validateJSON( file_get_contents( $template ) ) === false) {
                Utils::CLIerror( "Json file has wrong format" );
                Utils::CLIdebug( __CLASS__ . '::' . __FUNCTION__.": Json file '{$template}' has wrong format" );
                return false;
            }
            $file_name = $template;
            unset( $template );
            $template = json_decode( file_get_contents( $file_name ), true );
        }
        
        // Request is the array that contains the request data
        $request['deleted'] = false;
        $request['name'] = $template['name'];
        $request['description'] = $template['resourceTemplate']['description'];
        
        $requestFile = Utils::createJsonFile( $request );
        
        $result = $this->rest()->resource()->createResourceTemplate ( $requestFile );
        if ( $result === false) {
            Utils::CLIerror( "Failed to create resource template" );
            Utils::CLIdebug( "Source file located in '{$file_name}'");
            return false;
        }
        
        // Template id
        $result = Utils::outputToArray( $result );
        $templateId = $result['id'];
        $return['id'] = $result['id'];
        $return['name'] = $result['name'];
        
        // Template resource id
        $result = $this->getTemplate ( $templateId );
        $parentId = $result['id'];
        $return['resources'][0]['id'] = $result['id'];
        $return['resources'][0]['name'] = $result['name'];
        
        // Assign tags
        if ( ! empty ( $template['tags'] ) ) {
            foreach ( $template['tags'] as $tag ) {
                $result[$key]['tags'][$tag['name']] = $this->addTag( $tag['name'], $result['id'] );
            }
        }
        
        // Recreate children
        $return['resources'][0]['children'] = array();
        if ( !empty( $template['children'] )) {
            $result = $this->createTemplateChildren( $templateId, $parentId, $template['children'] );
            $return['resources'][0]['children'] = $result;
            if ( $result === false ) {
                Utils::CLIerror( "Failed to recreate resource template children.");
                return false;
            }
        }
        
        unlink( $requestFile );
        //return true;
        return $return;
    }
    
    /**
     * Create a resource template children
     *
     * @param string $parentId
     * @param array $children
     * @return array
     */
    protected function createTemplateChildren ( $templateId, $parentId, $children ) {
        foreach ( $children as $key => $child ) {
            $request['name'] = $child['name'];
            $request['templateId'] = $templateId;
            $request['parentId'] = $parentId;
            
            switch ( $child['role'] ) {
                case "AGENT_PLACEHOLDER":
                    $request['agentPlaceholder'] = true;
                    break;
                case "COMPONENT":
                    $componentResource = $this->rest()->resource()->getResourceRole( $child['name'] );
                    if ( $componentResource === false ) {
                        Utils::CLIwarning( "Cannot find resource for component '{$child['name']}'. Skipping child creation.");
                        $return[$key] = false;
                        continue 2;
                    }
                    $componentResource = Utils::outputToArray( $componentResource );
                    $request['roleId'] = "";
                    foreach ( $componentResource as $compRes ) {
                        if ( $compRes['name'] == $child['name'] ) {
                            $request['roleId'] = $compRes['id'];
                            break;
                        }
                    }
                    
                    if ( empty( $request['roleId'] ) ) {
                        Utils::CLIwarning( "Cannot find resource for component '{$child['name']}'. Skipping child creation.");
                        $return[$key] = false;
                        continue 2;
                    }
                    break;
                default :
                    // Do nothing
                    break;
            }
            
            $requestFile = Utils::createJsonFile( $request );
            $result = $this->rest()->resource()->createResourceTemplateChild( $requestFile );
            
            if ( $result === false ) {
                Utils::CLIwarning( "Failed to create child '{$child['name']} its children.");
                $return[$key] = false;
                continue;
            } else {
                $result = Utils::outputToArray ( $result );
                $return[$key]['id'] = $result['id'];
                $return[$key]['name'] = $result['name'];
                $return[$key]['role'] = $child['role'];
                $return[$key]['children'] = array();
                
                // Assign tags
                if ( ! empty ( $child['tags'] ) ) {
                    foreach ( $child['tags'] as $tag ) {
                        $result[$key]['tags'][$tag['name']] = $this->addTag( $tag['name'], $result['id'] );
                    }
                }
            }
            
            if ( !empty( $child['children'] ) ) {
                $result = $this->createTemplateChildren ( $templateId, $result['id'], $child['children'] );
                $return[$key]['children'] = $result;
            }
        }
        
        return $return;
    }
      
    /**
     * Delete a resource using resource ID or path
     *
     * @param string $resource
     * @return boolean
     */
    public function delete( $resource ) {
        $result = $this->udclient()->resource()->deleteResource( $resource );
        if ( $result === false ) {
            Utils::CLIerror( "Failed to delete resource '{$resource}'");
            return false;
        }
        return true;
    }
    
    /**
     * Delete a resource using resource template ID 
     *
     * @param string $template
     * @return boolean
     */
    public function deleteTemplate( $template ) {
        $result = $this->rest()->resource()->deleteResourceTemplate( $template );
        if ( $result === false ) {
            Utils::CLIerror( "Failed to delete resource '{$template}'");
            return false;
        }
        return true;
    }
    
    /**
     * Check if Resource exists
     *
     * @param string $resource
     * @return boolean
     */
    public function exists($resource)
    {
        $return_type = $this->return;
        if (empty($resource) || ! is_string($resource)) {
            Utils::CLIerror("Invalid resource parameter.");
            return false;
        }
        $this->setReturn('json');
        $resourceList = $this->getList();
        $resourceList = $resourceList['records'];
        $result = false;
        foreach ($resourceList as $item) {
            $itemName = isset($item['name']) ? $item['name'] : null;
            $itemId = isset($item['id']) ? $item['id'] : null;
            if ($itemName == $resource || $itemId == $resource) {
                $result = true;
            }
        }
        $this->setReturn($return_type);
        return $result;
    }
    
    /**
     *
     * Export a blueprint giving information about application, template and blueprint itself.
     *
     * @param string $blueprintId
     * @param string $blueprintName
     * @return array|path|boolean
     */
    public function getBlueprint($blueprintId, $blueprintName = null) {
        if (empty($blueprintId) && empty($blueprintName)){
            Utils::CLIerror( "Blueprint Id/name is missing." );
            return false;
        }
        $result = $this->rest()->resource()->getBlueprint( $blueprintId, $blueprintName );
        if ( $result === false ) {
            Utils::CLIerror( "Failed to get Blueprint.");
            return false;
        }
        if ( $this->return == 'file' ){
            $destination_path = $this->output."/Blueprint_".$blueprintId."_".microtime(true).".json";
            //path from api has quotes
            $file_path = trim( $result, "'" );
            rename($file_path, $destination_path );
            return ($destination_path);
            
        }
        return Utils::outputToArray ( $result );
    }
    
    /**
     *
     * Get Blueprint Resource
     *
     * @param string $blueprintId
     * @param string $blueprintName
     * @return array|path|boolean
     */
    public function getBlueprintResources($blueprintId, $blueprintName) {
        if (empty($blueprintId)){
            Utils::CLIerror( "Missing Blueprint ID." );
            return false;
        }
        $result = $this->rest()->resource()->getBlueprintResources( $blueprintId, $blueprintName );
        if ( $result === false ) {
            Utils::CLIerror( "Failed to get Blueprint Resources.");
            Utils::CLIdebug( __CLASS__ . '::' . __FUNCTION__.": Failed to get Resources for Bluepront with id:$blueprintId, name: $blueprintName" );
            return false;
        }
        if ( $returnType == 'file' ){
            //path from api has quotes
            $file_path = trim( $result, "'" );
            return ($file_path);
        }
        return Utils::outputToArray ( $result );
    }

    
    /**
     * Get the blueprint list in an array or in a file path
     *
     * @param string $application
     * @param bool $resources
     * @return array|path|boolean
     */
    public function getBlueprintList ( $application, $resources = false ) {
        $return_type = $this->return;
        $this->setReturn('json');
        $bluePrintsJSON = $this->rest()->resource()->getBlueprintsInApplication( $application );
        if ( $bluePrintsJSON === false ) {
            Utils::CLIerror( "Failed to retrieve blueprint list for application '{$application}'");
            Utils::CLIdebug( __CLASS__ . '::' . __FUNCTION__.": Failed to retrieve blueprint list for application '{$application}'");
            return false;
        }

        # If resources==false we only want blueprint list, faster
        $blueprintsList = Utils::outputToArray( $bluePrintsJSON );
        if ( $resources === false ){
            $this->setReturn( $return_type );
            if ($this->return == "file"){
                $destination_path = $this->output."/".$application."_blueprint_". date('YmdHis').".json";
                return Utils::createJsonFile( $blueprintsList , $destination_path );
            }
        }
        
        foreach ( $blueprintsList as $key => $blueprint ) {
            Utils::CLIinfo( "Getting Bluerint templates and resources for {$blueprint['name']}, this could take a while..", false, 3 );
            # Step 1 = Get the BPtemplate
            $templateJSON = $this->rest()->resource()->getResourceTemplate( $blueprint['id'] );
            if ( $templateJSON === false ) {
                Utils::CLIwarning( "Cannot retrieve resources for blueprint '{$blueprint['name']}' ({$blueprint['id']})" );
                continue;
            }
            $templateArray = Utils::outputToArray ( $templateJSON );
            $blueprintsList[$key]['resources']['id'] = $templateArray['records'][0]['id'];
            $blueprintsList[$key]['resources']['name'] = $templateArray['records'][0]['name'];
            $blueprintsList[$key]['resources']['resource'] = $templateArray['records'][0]['resource'];
            $blueprintsList[$key]['resources']['resourceTemplate'] = $templateArray['records'][0]['resourceTemplate'];
            
            # Step 2 = Get the resources of BPtemplate
            $resourcesTemplateArray = $this->getTemplate($blueprintsList[$key]['resources']['resourceTemplate']['id']);
            if ( $resourcesTemplateArray === false ) {
                Utils::CLIwarning( "Cannot retrieve resources for Template '{$blueprintsList[$key]['resources']['resourceTemplate']['name']}'" );
                continue;
            }
            $blueprintsList[$key]['resourceTemplate'] = $resourcesTemplateArray;
        }
        
        $this->setReturn( $return_type );
            // Return file path if return is 'file'
        if ($this->return == 'file') {
            $destination_path = $this->output."/".$application."_blueprint_". date('YmdHis').".json";
            return Utils::createJsonFile( $blueprintsList , $destination_path );
        }
        return $blueprintsList;
    }

    /**
     * Get Resource by Role
     *
     * @return array | path | boolean
     */
    public function getByRole( $name ) {
        if (empty($name)){
            Utils::CLIerror( "Missing role name parameter." );
            return false;
        }
        $result = $this->rest()->resource()->getResourceRole( $name );
        if ( $result === false ) {
            Utils::CLIerror( "Failed to retrieve resource." );
            Utils::CLIdebug( "Failed to retrieve resource from $name." );
            return false;
        }
        if ( $returnType == 'file' ) {
            //path from api has quotes
            $file_path = trim( $result, "'" );
            return ($file_path);
        }
        return Utils::outputToArray( $result );
    }
    
    /**
     * Get the resources list in array value or file path
     *
     * @return array | path | boolean
     */
    public function getList() {
        $returnType = $this->return;
        $this->setReturn('json');
        $result = $this->rest()->resource()->getResourceTree();
        if ( $result === false ) {
            Utils::CLIerror( "Failed to retrieve list of resources." );
            return false;
        }
        $result = Utils::outputToArray( $result );
        $this->setReturn( $returnType );
        if ( $returnType == 'file' ) {
            return Utils::createJsonFile( $result );
        }
        return $result;
    }
    
    /**
     *
     * Get resource
     * @param string
     *
     * @return array | path | boolean
     */
    public function get ( $resource ) {
        if ( empty ( $resource ) ) {
            Utils::CLIerror( "Missing Name/Id." );
            return false;
        }
        
        // Resources api works only with id or path
        // try to traslate the name to the first path
        // name SalesConnect
        // path /SalesConnect
        if (!Utils::isUuid($resource) && $resource[0] !== '/'){
            $resource = '/'.$resource;
        }
        
        $result = $this->udclient()->resource()->getResource( $resource );
        if ( empty ( $result ) ) {
            Utils::CLIerror( "Failed to retrive." );
            Utils::CLIdebug( __CLASS__ . '::' . __FUNCTION__ .": Failed to retrive template. Name/id: {$result}" );
            return false;
        }
        if ($this->return === 'file') {
            // delete quotes and return path
            return substr( $result, 1, - 1 );
        }
        return Utils::outputToArray( $result );
    }
    
    /**
     *
     * Get tags for a specific type
     * @param string 
     *
     * @return array | path | boolean
     */
    public function getTag ( $type ) {
        if ( empty ( $type ) ) {
            Utils::CLIerror( "Type name is missing." );
            return false;
        }
        $result = $this->rest()->system()->getTag( $type );
        if ( $result === false ) {
            Utils::CLIerror( "Failed to retrieve list of Tag." );
            return false;
        }
        if ( $this->return == 'file' ){
            $destination_path = $this->output."/tag_".$type."_".microtime(true).".json";
            //path from api has quotes
            $file_path = trim( $result, "'" );
            rename($file_path, $destination_path );
            return ($destination_path);
            
        }
        return Utils::outputToArray( $result );
    }
    
    
    /**
     * Get the resource template info in an array or in a file path
     *
     * @param string $templateId
     * @return array|path|boolean
     */
    public function getTemplate ( $templateId ) {
        $date = new DateTime();
        $returnType = $this->return;
        $this->setReturn('json');
        $templateArray = array();
        
        $template = $this->rest()->resource()->getResourceTemplate( $templateId );
        if ( $template === false ) {
            Utils::CLIerror( "Failed to retrive resource template ({$templateId})" );
            return false;
        } else {
            $template = Utils::outputToArray( $template );
        }
        $templateArray['id'] = $template['records'][0]['id'];
        $templateArray['name'] = $template['records'][0]['name'];
        $templateArray['resource']['id'] = $template['records'][0]['resource']['id'];
        $templateArray['resourceTemplate']['id'] = $template['records'][0]['resourceTemplate']['id'];
        $templateArray['resourceTemplate']['description'] = $template['records'][0]['resourceTemplate']['description'];
        
        $resource = $this->getTemplateChildren( $templateArray['resourceTemplate']['id'], $templateArray['id'] );
        if ( $resource === false ) {
            Utils::CLIwarning( "There were some errors while retrieving resource template children" );
            $templateArray['children'] = array();
        } else {
            $templateArray['children'] = $resource;
        }
        
        $this->setReturn( $returnType );
        
        if ( $returnType == "file" ) {
            // force file location
            return Utils::createJsonFile ( $templateArray, "{$this->output}/{$date->getTimestamp()}.json" );
        } else {
            return $templateArray;
        }
    }
    
    protected function getTemplateChildren ( $parentResourceTemplateId, $parentResourceId ) {
        $resource = array();
        $result = $this->rest()->resource()->getResourceTemplateChildren( $parentResourceTemplateId, $parentResourceId );
        if ( $result === false ) {
            Utils::CLIwarning( "Cannot retrieve children for '{$parentResourceId}'");
            return false;
        }
        
        $result = Utils::outputToArray ( $result );
        foreach ( $result as $key => $child ) {
            $resource[$key]['id'] = $child['id'];
            $resource[$key]['name'] = $child['name'];
            $resource[$key]['tags'] = array();
            if ( array_key_exists ( 'tags', $child ) ) {
                $resource[$key]['tags'] = $child['tags'];
            }
            if ( array_key_exists ( 'role', $child ) &&
                 array_key_exists ( 'specialType', $child['role'] ) ) { 
                $resource[$key]['role'] = $child['role']['specialType'];
            } else {
                // For better import, 'GROUP' will be set as role value for folders
                $resource[$key]['role'] = "GROUP";
            }
            if ( $child['hasChildren'] == true ) {
                $children = $this->getTemplateChildren ( $parentResourceTemplateId, $child['id'] );
                if ( $children === false ) {
                    $resource[$key]['children'] = array();
                } else {
                    $resource[$key]['children'] = $children;
                }
            } else {
                $resource[$key]['children'] = array();
            }
        }
        
        return $resource;
    }
    
    /**
     * Get the resource templates list in array value or file path
     *
     * @return array | path | boolean
     */
    public function getTemplateList() {
        $result = $this->rest()->resource()->getResourceTemplates();
        if ( $result === false ) {
            Utils::CLIerror( "Failed to retrieve list of resource templates" );
            return false;
        }
        
        if ( $this->return === "file" ) {
            return substr($result,1,-1);
        } else {
            return Utils::outputToArray( $result );
        }
    }
    
    /**
     * 
     * Recursively remove all the deprecated resource templates
     *
     * @param array $oldTemplate
     * @param array $newTemplate
     */
    protected function recursiveTemplateDelete ( $oldTemplate, $newTemplate ) {
        $result = array();
        foreach ( $oldTemplate as $key => $value ) {
            $search = false;
            foreach ( $newTemplate as $newKey => $newValue ) {
                if ( $value['name'] == $newValue['name'] ) {
                    $search = true;
                    break;
                }
            }
            
            if ( $search === false ) {
                $result[$key] = $this->rest()->resource()->deleteResourceTemplate ( $value['id'] );
            } else {
                if ( array_key_exists( 'children', $value ) ) {
                    if ( ! array_key_exists( 'children', $newValue ) ) {
                        $newValue['children'] = array();
                    }
                    $result[$key] = $this->recursiveTemplateDelete ( $value['children'], $newValue['children'] );
                }
            }
        }
        
        return $result;
    }
    
    /**
     * 
     * Recursively check and create resource template children, comparing origin template with the new one
     *
     * @param string $templateId
     * @param string $parentId
     * @param array $oldTemplate
     * @param array $newTemplate
     * @return array|boolean
     */
    protected function recursiveTemplateUpgrade ( $templateId, $parentId, $oldTemplate, $newTemplate ) {
        $result = array();
        // Loop new template and compare to original one
        foreach ( $newTemplate as $key => $value ) {
            // Check if the child exists
            $search = false;
            foreach ( $oldTemplate as $oldId => $oldTemplateChild ) {
                if ( $value['name'] == $oldTemplateChild['name'] ) {
                    $search = true;
                    break;
                }
            }
            
            // Create if missing, upgrade if exists
            if ( $search === false ) {
                Utils::CLIdebug( "Creating resource template '{$value['name']}' ");
                // Tricking the array
                $create = array($value);
                $result[$key] = $this->createTemplateChildren( $templateId, $parentId, $create );
            } else {
                // Assign tags
                if ( ! empty ( $value['tags'] ) ) {
                    foreach ( $value['tags'] as $tag ) {
                        $result[$key]['tags'][$tag['name']] = $this->addTag( $tag['name'], $oldTemplate[$oldId]['id'] );
                    }
                }
                Utils::CLIdebug("Create children for resource template '{$value['name']}'");
                if ( !empty($value['children']) ) {
                    $result[$key]['children'] = $this->recursiveTemplateUpgrade( 
                            $templateId, 
                            $oldTemplate[$oldId]['id'], 
                            $oldTemplate[$oldId]['children'], 
                            $value['children'] 
                    );
                }
            }
        }
        
        return $result;
    }
    
    /**
     * Upgrade an existing resourceTemplate
     *
     * @param string $template
     * @return boolean|array
     */
    public function upgradeTemplate ( $template ) {
        $returnType = $this->return;
        $this->setReturn('json');
        $file_name = null;
        $return = array();
        
        // Check if arg is file path or array obj
        if ( ! is_array( $template ) && ! is_object( $template )) {
            if ( file_exists( $template ) === false ) {
                Utils::CLIerror( "Json file does not exist" );
                Utils::CLIdebug( __CLASS__ . '::' . __FUNCTION__.": Json file '{$template}' does not exist" );
                return false;
            }
            if (Utils::validateJSON( file_get_contents( $template ) ) === false) {
                Utils::CLIerror( "Json file has wrong format" );
                Utils::CLIdebug( __CLASS__ . '::' . __FUNCTION__.": Json file '{$template}' has wrong format" );
                return false;
            }
            $file_name = $template;
            unset( $template );
            $template = json_decode( file_get_contents( $file_name ), true );
        }
        
        // Retrieve list of resource templates
        $templateList = $this->getTemplateList();
        $templateId = null;
        foreach ( $templateList as $single ) {
            if ( $single['name'] == $template['name'] ) {
                $templateId = $single['id'];
                break;
            }
        }
        
        if ( empty ( $templateId ) ) {
            Utils::CLIdebug("Template does not exist. Creating it.");
            $result['create'] = $this->createTemplate ( $template );
        } else {
            $originTemplate = $this->getTemplate( $templateId );
            if ( $originTemplate === false ) {
                // Error message handled in getTemplate
                return false;
            }
            $result['upgrade'] = $this->recursiveTemplateUpgrade ( 
                    $originTemplate['resourceTemplate']['id'], 
                    $originTemplate['id'], 
                    $originTemplate['children'], 
                    $template['children']
            );
            
            // Clean unused templates
            $result['delete'] = $this->recursiveTemplateDelete( $originTemplate['children'], $template['children'] );
        }
        
        $this->setReturn($returnType);
        return $result;
    }
}