<?php
/****************************************************************
 * IBM Confidential
 *
 * SFA100-Collaboration Source Materials
 *
 * (C) Copyright IBM Corp. 2014
 *
 * The source code for this program is not published or otherwise
 * divested of its trade secrets, irrespective of what has been
 * deposited with the U.S. Copyright Office
 *
 ***************************************************************/
require_once ('include/EntryPoint.php');
require_once ('include/Layers/Application.php');
require_once ('include/Layers/ApplicationProcess.php');
require_once ('include/Layers/Component.php');
require_once ('include/Layers/Process.php');
require_once ('include/Layers/Resource.php');

/**
 * Backup uDeploy generating webUI-like export files (json), using udclient's APIs
 *
 * @class BackupUCD
 * @author Marco Bassi marcobas@ie.ibm.com
 * @author Kostadin Ivanov kostadii@ie.ibm.com
 */
class BackupUCD extends EntryPoint {
    protected $resource = null;

    public function __construct($alias = null) {
        parent::__construct($alias);
        $this->setReturn('json');
        $this->resource = new Resource();
    }

    /**
     * Export all Applications.
     * Will get a complete list of available Application on specified in $configs UCD Server.
     * Will export each UCD Application to its dedicated backup folder.
     *
     * @param boolean $useName
     *            [OPTIONAL] If set to TRUE - Application Name will be used
     *            for each Application export.
     *            If set to FALSE - Application ID will be used.
     *            DEFAULT value: TRUE
     *            
     * @param boolean $$enviroment
     *            [OPTIONAL] If set to TRUE - the application will be exported with Environments included
     *            - really slow
     *            DEFAULT value: FALSE
     *            
     * @return boolean NULL array in case of an ERROR
     *         in retrieving Applications to backup from Server.
     *         NULL if there was NOT Available Applications to backup.
     *         Array in the form:
     *         array[name][path] = 'path'
     *         array[name[blueprints] = 'path'
     */
    public function applications($useName = true, $enviroment = false)
    {
        $return = array(); // return value
        $applicationAPIs = new Application($this->alias);
        $resourceAPIs = new Resource($this->alias);
        
        $applicationAPIs->setReturn('array');
        // getList will return an array of the applications in UCD
        $apps = $applicationAPIs->getList();
        if ($apps === false) {
            Utils::CLIwarning('There was a problem obtaining Applications to be backed up.');
            return false;
        }
        if (count($apps) == 0) {
            Utils::CLIwarning('There is NO Applications to be backed up.');
            return null;
        }
        // Loop for all the Applications
        foreach ($apps as $app) {
            Utils::CLIinfo('Exporting Application "' . $app['name'] . '":');
            // Set dedicated Application backup sub-directory
            $applicationAPIs->setOutput($this->output . '/' . $app['name']);
            $applicationAPIs->setReturn('file');
            // Get function from Application layer will return a full object(env, components, processes...)
            Utils::CLIinfo("This can take a while.....", false, 3);
            $application_path = $applicationAPIs->get($app['id'], true);
            
            // Removing envs
            if ($enviroment === false) {
                $application_path = $applicationAPIs->cleanEnvironment($application_path);
                if (empty($application_path)) {
                    Utils::CLIerror("Error removing environments from application {$app['name']}.");
                    Utils::CLIwarning('Application: "' . $app['name'] . '" backup FAILED to complete');
                    $return[$app['name']] = false;
                    continue;
                }
            }
            
            // Success - Rename the file
            if ($useName) {
                $application_path = Utils::renameJson($application_path, $app['name']); // Error message handled by function
            }
            $return[$app['name']]['path'] = $application_path;
            
            // Backup application blueprints
            $resourceAPIs->setOutput("{$applicationAPIs->output}/blueprints");
            $resourceAPIs->setReturn("file");
            $bluePrint_json = $resourceAPIs->getBlueprintList($app['name'], true);
            // Validate result
            if ($bluePrint_json !== false) {
                Utils::CLIinfo('Blue print list for "' . $app['name'] . '" saved into:', false, 3);
                Utils::CLIdest($bluePrint_json, false, 3);
                $return[$app['name']]['blueprints'] = $bluePrint_json;
            } else {
                Utils::CLIwarning('Blue print list for "' . $app['name'] . '" backup - FAILED');
                $return[$app['name']]['blueprints'] = false;
            }
            
            Utils::CLIinfo('Application: "' . $app['name'] . '" backup completed - SUCCESS');
        }
        return $return;
    }

    /**
     * Run a complete backup of uDeploy
     */
    public function backupAll($environments = false) {
        $processAPIs = new Process($this->alias);
        $resourceAPIs = new Resource($this->alias);
        // Link objects to parent server
        
        $backUp_watch = time();
        StopWatch::getInstance()->start($backUp_watch);
        
        $return = array();
        // Prompt message for Backup Process
        Utils::CLIinfo('### START of UrbanCode BACKUP process ###');
        Utils::CLIstart('BACKUP process: ' . date("d-m-Y h:i:s"));
        Utils::CLIinfo('Server to backup:', false, 4);
        // create a api to save weburl and output based on config file
        Utils::CLIdest($this->weburl, false, 6);
        Utils::CLIinfo('Backup folder:', false, 4);
        Utils::CLIdest($this->output, false, 6);
        
        /*
         * Backup the backups - root directory. 
         * This is getting cha  nged during each of below steps
         */
        $backupsRootDir = $this->output;
        
        /* ### Step 1: Backup ComponentTemplates ### */
        $this->setOutput($this->output . "/componentTemplates");
        $backupComponentTemplates_watch = time();
        StopWatch::getInstance()->start($backupComponentTemplates_watch);
        $return['componentTemplates'] = $this->componentTemplates();
        StopWatch::getInstance()->stop($backupComponentTemplates_watch);
        Utils::CLIcompleted('Backup Component Templates: ' . StopWatch::getInstance()->get($backupComponentTemplates_watch));
        $this->setOutput($backupsRootDir);
        
        /* ### Step 2: Backup ALL Components ### */
        $this->setOutput($this->output . "/components");
        $backupComponents_watch = time();
        StopWatch::getInstance()->start($backupComponents_watch);
        $return['components'] = $this->components(null, true);
        StopWatch::getInstance()->stop($backupComponents_watch);
        Utils::CLIcompleted('Backup Components: ' . StopWatch::getInstance()->get($backupComponents_watch));
        $this->setOutput($backupsRootDir);
        
        /* ### Step 3: Backup Resource Templates ### */
        // Set Resource Templates backup sub-folder
        $this->setOutput($this->output . "/resourceTemplates");
        $backupResourceTemplates_watch = time();
        StopWatch::getInstance()->start($backupResourceTemplates_watch);
        $return['resourceTemplates'] = $this->resourceTemplates();
        StopWatch::getInstance()->stop($backupResourceTemplates_watch);
        Utils::CLIcompleted('Backup Resource Templates: ' . StopWatch::getInstance()->get($backupResourceTemplates_watch));
        // Re-Set output to the Backup root directory for the next step
        $this->setOutput($backupsRootDir);
        
        /* ### Step 4: Backup ALL Applications ### */
        $this->setOutput($this->output . "/applications");
        $backupApplications_watch = time();
        StopWatch::getInstance()->start($backupApplications_watch);
        $return['applications'] = $this->applications(true, $environments);
        StopWatch::getInstance()->stop($backupApplications_watch);
        Utils::CLIcompleted('Backup Applications: ' . StopWatch::getInstance()->get($backupApplications_watch));
        $this->setOutput($backupsRootDir);
        
        /* ### Step 5: Backup ALL Post Processing Scripts ### */
        $processAPIs->setOutput($this->output . "/settings");
        $processAPIs->setReturn('file');
        $backupPostProcessingScripts_watch = time();
        StopWatch::getInstance()->start($backupPostProcessingScripts_watch);
        $return['postProcessingScripts'] = $processAPIs->getPostProcessingScripts();
        StopWatch::getInstance()->stop($backupPostProcessingScripts_watch);
        Utils::CLIcompleted('Backup Post Processing Scripts: ' . StopWatch::getInstance()->get($backupPostProcessingScripts_watch));
        $processAPIs->setReturn('json');
        $processAPIs->setOutput($backupsRootDir);
        
        /* ### Step 6: Backup ALL Tags ### */
        $resourceAPIs->setOutput($this->output . "/tags");
        $resourceAPIs->setReturn('file');
        $backupTags_watch = time();
        StopWatch::getInstance()->start($backupTags_watch);
        $return['tags']['component'] = $resourceAPIs->getTag('component');
        $return['tags']['resource'] = $resourceAPIs->getTag('resource');
        StopWatch::getInstance()->stop($backupTags_watch);
        Utils::CLIcompleted('Backup Tags: ' . StopWatch::getInstance()->get($backupTags_watch));
        $resourceAPIs->setReturn('json');
        $resourceAPIs->setOutput($backupsRootDir);
        
        /* ### Step 7: Backup ALL Generic Processes ### */
        $this->setOutput($this->output . "/processes");
        $backupGenericProcesses_watch = time();
        StopWatch::getInstance()->start($backupGenericProcesses_watch);
        $return['processes'] = $this->processes();
        StopWatch::getInstance()->stop($backupGenericProcesses_watch);
        Utils::CLIcompleted('Backup Generic Processes: ' . StopWatch::getInstance()->get($backupGenericProcesses_watch));
        $this->setReturn('json');
        $this->setOutput($backupsRootDir);
        StopWatch::getInstance()->stop($backUp_watch);
        Utils::CLIcompleted('BACKUP process: ' . StopWatch::getInstance()->get($backUp_watch));
        Utils::CLIinfo('### END of UrbanCode BACKUP process ###');
        
        return $return;
    }

    /**
     * @param string $componentID            
     * @param string    $componentName  optional
     *            Export a component
     */
    public function component($componentID, $componentName = null) {
        $componentAPIs = new Component($this->alias);
        $componentAPIs->setReturn('file');
        $componentAPIs->setOutput($this->output);
        $output = $componentAPIs->get($componentID, false);
        
        if (! empty($componentName)) {
            # mv from ID.json to file_name.json
            $new_file_name = $this->output . '/' . $componentName . '.json';
            rename($output, $new_file_name);
        } else {
            $new_file_name = $output;
        }
        
        if ($output === false) {
            Utils::CLIwarning('There was a problem with obtaining a Component to backup.');
            return false;
        }
        Utils::CLIdest($new_file_name, false, 6);
        return $output;
    }

    /**
     * Export all components or components in a specified Application.
     *
     * If $application is NOT provided, ALL components will be backed up.
     *
     * If $application is SPECIFIED, ONLY components which belong to this Application, will be backed up.
     *
     * @param string    $application    [OPTIONAL] String holding the Application Name or ID,
     *            for the Application which components are to be backed up.
     *            DEFAULT value: NULL
     * @param bool      $useName        [OPTIONAL] If is set to true,
     *            will use component name instead of ID to export Component.
     *            DEFAULT value: FALSE
     *            
     */
    public function components($application = null, $useName = false) {
        $componentAPIs = new Component($this->alias);
        // connect the Api to the right server
        $componentAPIs->setReturn('array');
        $components = $componentAPIs->getList($application);
        
        if (empty($components)) {
            Utils::CLIwarning('There was a problem with obtaining of Components to be backed up.');
            return false;
        }
        
        if (count($components) == 0) {
            Utils::CLIwarning('There is NO Components to be backed up.');
            return null;
        }
        
        $return = array();
        $componentOutput = $this->output;
        $tagOutput = $this->output . "/tags";
        foreach ($components as $comp) {
            Utils::CLIinfo("Exporting Component '{$comp['name']}':", true, 3);
            if ($useName) {
                $compKey = $comp['name'];
            } else {
                $compKey = $comp['id'];
            }
            $return[$comp['id']]['name'] = $comp['name'];
            $return[$comp['id']]['return'] = $this->component($comp['id'], $compKey);
            
            // Export tags
            $componentAPIs->setOutput($tagOutput);
            $componentAPIs->setReturn('file');
            $return[$comp['id']]['tags'] = $componentAPIs->getTags($comp['id'], $compKey);
            $componentAPIs->setOutput($componentOutput);
        }
        
        return $return;
    }

    /**
     * Store a ComponentTemplate as a JSON formatted string into a File.
     *
     * @param String|Array|Object $template_data JSON formatted String OR Array OR Object, 
     *            which holds the DATA of a ComponentTemplate to be exported.
     * @return boolean Boolean TRUE/FALSE, based on whether the provided ComponentTemplate,
     *         has been saved to file or not.
     */
    public function componentTemplate($template_data) {
        // Use $template_data as Object
        $template_obj = Utils::getAsObject($template_data);
        
        Utils::CLIinfo('Exporting Component-Template "' . $template_obj->name . '": ');
        
        Utils::CLIinfo('Save Component-Template "' . $template_obj->name . '" into:', false, 3);
        $templateFileName = str_replace(array(' ', '/'), array('_', '-'), $template_obj->name);
        Utils::CLIdest($this->output . '/' . $templateFileName . '.json', false, 3);
        
        // Set $template_obj to a JSON formatted string, before to save it to file
        $template_json = json_encode($template_obj);
        
        $return = Utils::writeToFile($this->output . '/' . $templateFileName . '.json', $template_json, false);
        
        return $return;
    }

    /**
     * Exports ALL available ComponentTemplates in UrbanCode as JSON formatted sctrings into a File.
     * In case if there is NO ComponentTemplates to be exported, this function will return an empty Array.
     *
     * @return array Array in the form:
     *         array(
     *         'componentTemplateID' => array(
     *         'name' => 'ComponentTemplate Name',
     *         'return' => 'result from function componentTemplate()'
     *         )
     *         )
     */
    public function componentTemplates() {
        $componentAPIs = new Component($this->alias);
        
        $componentAPIs->setReturn('array');
        # listTemplates will return array obj
        $templates = $componentAPIs->listTemplates();
        
        if ($templates === false) {
            Utils::CLIwarning('There was a problem with obtaining of ComponentTemplates to be backed up.');
            return false;
        }
        
        if (count($templates) == 0) {
            Utils::CLIwarning('There is NO ComponentTemplates to be backed up.');
            return null;
        }
        
        $return = array();
        foreach ($templates as $template) {
            // Get Details for this $template and export it
            $template_data = $componentAPIs->getTemplate($template['id'], $template['name']);
            if ($template_data !== false) {
                // Set return and Export ComponentTemplate
                $return[$template['id']]['name'] = $template['name'];
                $return[$template['id']]['return'] = $this->componentTemplate($template_data);
            } else {
                $return[$template['id']]['name'] = $template['name'];
                $return[$template['id']]['return'] = false;
            }
        }
        
        // Return
        return $return;
    }

    /**
     * Exports all generic processes
     * @return array|bool
     */
    public function processes()
    {
        $processAPIs = new Process($this->alias);
        Utils::CLIinfo("Exporting Processes");
        // Get all processes in the server
        $processAPIs->setReturn('array');
        $processes = $processAPIs->getList();
        // Export processes
        $processAPIs->setOutput($this->output);
        $processAPIs->setReturn('file');
        foreach ($processes as $process) {
            $file_path = $processAPIs->export($process['id']);
            if ($file_path !== false) {
                Utils::CLIout("  [{$process['name']}]");
                $file_path = Utils::renameJson($file_path, $process['name']);
                $result[$process['id']] = $file_path;
                Utils::CLIout("  $file_path");
            } else {
                Utils::CLIerror("{$process['name']}");
                $result[$process['id']] = false;
            }
        }
        return $result;
    }

    /**
     * Exports all resource Templates
     * @return array|bool
     */
    public function resourceTemplates() {
        $resourceAPIs = new Resource($this->alias);
        // connect the Api to the right server
        $resourceAPIs->setReturn('json');
        $resourceAPIs->setOutput($this->output);
        $templates = $resourceAPIs->getTemplateList();
        
        $resourceAPIs->setReturn('file');
        foreach ($templates as $key => $template) {
            $tmpFile = $resourceAPIs->getTemplate($template['id']);
            $fileName = str_replace(" ", "_", $template['name']);
            $destFile = Utils::renameJson($tmpFile, $fileName); // error message handled by function
            Utils::CLIinfo("{$template['name']}");
            Utils::CLIdest("{$destFile}", false);
            $result[$key]['id'] = $template['id'];
            $result[$key]['name'] = $template['name'];
            $result[$key]['file'] = $destFile;
        }
        return $result;
    }
}