<?php
/**
 * **************************************************************
 * IBM Confidential
 *
 * SFA - Collaboration Source Materials
 *
 * (C) Copyright IBM Corp. 2015
 *
 * The source code for this program is not published or otherwise
 * divested of its trade secrets, irrespective of what has been
 * deposited with the U.S. Copyright Office
 *
 * *************************************************************
 */
require_once ('include/EntryPoint.php');
require_once ('BackupUCD.php');
require_once ('RestoreUCD.php');

/**
 * Fork class is used to fork an existing UCD application.
 * The functions implemented in this class will generate a copy
 * of the existing application where all components and processes
 * are marked with a defined suffix
 *
 * @class Fork
 *
 * @author Marco Bassi              marcobas@ie.ibm.com
 */
class Fork extends EntryPoint {
    
    protected $suffix;
    
    public function __construct ($alias = null) {
        parent::__construct($alias);
        $this->suffix['new'] = '';
        $this->suffix['old'] = '';
    }
    
    /**
     * 
     * Set the suffix to be used during fork
     *
     * @param string $suffix
     */
    public function setNewSuffix ( $suffix ) {
        $this->suffix['new'] = $suffix;
    }
    
    /**
     *
     * Set the old suffix to be removed during fork
     *
     * @param string $suffix
     */
    public function setOldSuffix ( $suffix ) {
        $this->suffix['old'] = $suffix;
    }
    
    /**
     *
     * Get the suffix to be used during fork
     *
     * @return string
     */
    public function getSuffix () {
        return $this->suffix;
    }
    
    /**
     * 
     * Given the application name, fork it into the UCD server
     *
     * @param string $application
     */
    public function forkApplication ( $application ) {
         
        $appLayer = new Application();
        
        if(empty($application) || is_null($application)) {
            Utils::CLIerror("The application name is empty.");
            return false;
        }
        if(Utils::isUuid($application)) {
            Utils::CLIerror("Cannot fork using an application ID '{$application}'.");
            return false;
        }
        
        Utils::CLIinfo('Exporting the application with the forked name.');
        $appFile = $this->exportForkApplication($application);
        if($appFile === false) {
            Utils::CLIerror('The export of the forked application failed!');
            return false;
        }
        
        Utils::CLIinfo('Check existing applications, components and processes');
        $appArray = json_decode( file_get_contents( $appFile ), true );
        $checkResult = $this->checkExistingData( $appArray );
        if ( $checkResult === false ) {
            Utils::CLIerror("Some data in the forked application already exist. Please verify the parameters for Fork.");
            return false;
        } else {
            Utils::CLIout("  [OK]");
        }
        
         Utils::CLIinfo('Creating the forked application on the server.');
         $result = $appLayer->import($appFile, false);
         if($result === false) {
             Utils::CLIerror('The import of the forked application failed!');
             return false;
         } else {
             Utils::CLIinfo('The import of the forked application completed successfully.');
         }
        return true;
    }
    
    /**
     * 
     * Given a name as argument, return the name with the new suffix:
     * NAME => NAME_suffix
     * NAME_oldSuffix => NAME_suffix
     *
     * @param string $name
     * @return string
     */
    protected function getForkName ( $name ) {
        if ( ! empty ($this->suffix['old']) ) {
            // Get old suffix lenght and extract it from name
            $suffixLen = strlen( $this->suffix['old'] );
            $suffixLen = $suffixLen * (-1);
            $nameSuffix = substr( $name, $suffixLen );
            
            if ( $nameSuffix === false ||
                $nameSuffix != $this->suffix['old'] ) {
                return false;
            }
            
            // Remove old suffix
            $position = strpos( $name, $this->suffix['old'] );
            $name = substr( $name, 0, $position );
        }
        
        // Append new suffix
        $name = $name . $this->suffix['new'];
        return $name;
    }
    
    /**
     * 
     * Rename all the occurrences of application name, within the 
     * application array. Returns the edited array, false if name
     * is not found (invalid array)
     *
     * @param array $application
     * @return array|boolean
     */
    protected function renameApplication ( $application ) {
        Utils::CLIinfo ( "Renaming application" );
        if ( array_key_exists( 'name', $application ) ) {
            $name = $application['name'];
            $name = $this->getForkName( $name );
            if($name === false) {
                Utils::CLIerror("The application '{$application['name']}' cannot be renamed!");
                return false;
            }
            Utils::CLIout( "  {$application['name']} => {$name}" );
            $application['name'] = $name;
        } else {
            return false;
        }
        
        return $application;
    }
    
    /**
     *
     * Rename all the occurrences of component names, within the
     * application array. Returns the edited array.
     *
     * @param array $application
     * @return array
     */
    protected function renameComponents ( $application ) {
        Utils::CLIinfo("Renaming components");
        // Start renaming all components
        if (array_key_exists( 'components', $application )) {
            foreach ( $application['components'] as $index => $component ) {
                if (array_key_exists( 'name', $component )) {
                    $name = $component ['name'];
                    $name = $this->getForkName( $name );
                    if ($name !== false) {
                        $application['components'][$index]['name'] = $name;
                        Utils::CLIout ( "  {$component['name']} => {$name}" );
                    } else {
                        Utils::CLIwarning("The component '{$component['name']}' was not renamed!");
                    }
                }
                if (array_key_exists( 'componentTemplate', $component )) {
                    $name = $component ['componentTemplate'] ['name'];
                    $name = $this->getForkName( $name );
                    if ($name !== false) {
                        $application ['components'] [$index] ['componentTemplate'] ['name'] = $name;
                        Utils::CLIout( 
                                "    {$component['componentTemplate']['name']} => {$name}" );
                    } else {
                        Utils::CLIwarning( 
                                "The component template '{$component['name']}' was not renamed!" );
                    }
                }
            }
        }
        
        return $application;
    }
    
    /**
     * 
     * Recursively scan the array and search for index defined in indexList
     *
     * @param array $application
     * @param string|array $index
     * @return array
     */
    protected function recursiveRenameByIndex ( $array, $indexList ) {
        // Convert to array
        if ( ! is_array($indexList) ) {
            $indexList = array( $indexList );
        }
        
        // Scan the array
        foreach ( $array as $index => $value ) {
            if ( is_array ($value) ) {
                $result = $this->recursiveRenameByIndex ( $value, $indexList );
            } else {
                if ( in_array( $index, $indexList ) ) {
                    $result = $this->getForkName ( $value );
                    if($result === false) {
                        Utils::CLIwarning("{$value} was not renamed!");
                        $result = $value;
                    } else {
                        Utils::CLIout ( "  {$value} => {$result}" );
                    }
                } else {
                    $result = $value;
                }
            }
            
            $array[$index] = $result;
        }
        
        return $array;
    }
    
    /**
     *
     * Rename all the occurrences of process names, within the
     * application array. Returns the edited array.
     *
     * @param array $application
     * @return array
     */
    protected function renameProcess ( $application ) {
        Utils::CLIinfo( 'Renaming generic processes' );
        if (array_key_exists( 'components', $application )) {
            foreach ( $application ['components'] as $compNo => $component ) {
                if (array_key_exists( 'genericProcesses', $component )) {
                    foreach ( $component ['genericProcesses'] as $processNo => $genericProcess ) {
                        if (array_key_exists( 'name', $genericProcess )) {
                            $name = $this->getForkName( 
                                    $genericProcess ['name'] );
                            if ($name !== false) {
                                $application ['components'] [$compNo] ['genericProcesses'] [$processNo] ['name'] = $name;
                                Utils::CLIout( 
                                        "  {$genericProcess['name']} => {$name}" );
                            } else {
                                Utils::CLIwarning( 
                                        "The generic process '{$genericProcess['name']}' was not renamed!" );
                            }
                        }
                    }
                }
            }
        }
        
        unset( $processNo );
        unset( $genericProcess );
        if (array_key_exists( 'genericProcesses', $application )) {
            foreach ( $application ['genericProcesses'] as $processNo => $genericProcess ) {
                if (array_key_exists( 'name', $genericProcess )) {
                    $name = $this->getForkName( $genericProcess ['name'] );
                    if ($name !== false) {
                        $application ['genericProcesses'] [$processNo] ['name'] = $name;
                        tils::CLIout( 
                                "  {$genericProcess['name']} => {$name}" );
                    } else {
                        Utils::CLIwarning( 
                                "The generic process '{$genericProcess['name']}' was not renamed!" );
                    }
                }
            }
        }
        
        return $application;
    }
    
    /**
     * 
     * Given the application name, exports the forked version of the
     * application in a json file. Returns file path of the json file
     *
     * @param string $application
     * @return string
     */
    protected function exportForkApplication ( $application ) {
        // Get the application
        $application = $this->rest()->application()->getApplicationRest( $application );
        if($application === false) {
            Utils::CLIerror("The application doesn't exist");
            return false;
        }
        $application = Utils::outputToArray( $application );
        
        // Remove environments
        $application ['environments'] = array();
        
        // Rename application, component and process names
        $application = $this->renameApplication ( $application );
        if ( $application === false ) {
            return false;
        }
        $application = $this->renameComponents ( $application );
        $application = $this->renameProcess ( $application );
        
        // Rename component and process names in the application
        $indexList = array( 'componentName', 'processName', 'componentTemplateName' );
        Utils::CLIinfo("Renaming values for indexes ", false);
        foreach ( $indexList as $indexName ) {
            Utils::CLIout("'{$indexName}', ", false);
        }
        $application = $this->recursiveRenameByIndex( $application, $indexList );
        
        return Utils::createJsonFile( $application );
    }
    
    /**
     * 
     * Check if application, component and process names are unique
     *
     * @param array $application
     */
    protected function checkExistingData( $application ) {
        $existing['applications'] = array();
        $existing['components'] = array();
        $existing['processes'] = array();
        $existing['componentTemplates'] = array();
        $return = true;
        
        // List of applications
        $result = Utils::outputToArray( $this->udclient()->application()->getApplications() );
        foreach ( $result as $element ) {
            array_push( $existing['applications'], $element['name'] );
        }
        // List of components
        $result = Utils::outputToArray( $this->udclient()->component()->getComponents() );
        foreach ( $result as $element ) {
            array_push( $existing['components'], $element['name'] );
        }
        // List of processes
        $result = Utils::outputToArray( $this->rest()->process()->getProcesses() );
        foreach ( $result as $element ) {
            array_push( $existing['processes'], $element['name'] );
        }
        // List of componentTemplates
        $result = Utils::outputToArray( $this->rest()->component()->getComponentTemplates() );
        foreach ( $result as $element ) {
            array_push( $existing['componentTemplates'], $element['name'] );
        }
        
        // Check applications
        Utils::CLIout("\n  Checking Applications");
        if ( in_array ( $application['name'], $existing['applications'] ) ) {
            $return = false;
            Utils::CLIout("    Application '{$application['name']}' already exists");
        }
        
        // Check components and templates
        Utils::CLIout("\n  Checking Components");
        foreach ( $application['components'] as $component ) {
            if ( in_array ( $component['name'], $existing['components'] ) ) {
                $return = false;
                Utils::CLIout("    Component '{$component['name']}' already exists");
            }
            if (array_key_exists( 'componentTemplate', $component )) {
                $template = $component ['componentTemplate'];
                if ( in_array ( $template['name'], $existing['componentTemplates'] ) ) {
                    $return = false;
                    Utils::CLIout("    ComponentTemplate '{$template['name']}' already exists");
                }
            }
        }
        
        // Check processes
        Utils::CLIout("\n  Checking Processes");
        foreach ( $application['genericProcesses'] as $process ) {
            if ( in_array ( $process['name'], $existing['processes'] ) ) {
                $return = false;
                Utils::CLIout("    Process '{$process['name']}' already exists");
            }
        }
        foreach ( $application ['components'] as $compNo => $component ) {
            foreach ( $component ['genericProcesses'] as $processNo => $genericProcess ) {
                if ( in_array ( $genericProcess['name'], $existing['processes'] ) ) {
                    $return = false;
                    Utils::CLIout("    Process '{$genericProcess['name']}' already exists");
                }
            }
        }
        
        return $return;
    }
    
    /**
     * 
     * Retrieve list of blueprints and resource templates and do the fork
     *
     * @param string $oldApplication
     * @param string $newApplication
     * @return boolean|array
     */
    protected function forkResourceTemplate ( $oldApplication, $newApplication ) {
        // Get blueprint list
        Utils::CLIinfo("Retrieving blueprints and resource templates");
        $blueprints = $this->rest()->resource()->getBlueprintsInApplication ( $oldApplication );
        $blueprints = Utils::outputToArray( $blueprints );
        if ( empty ($blueprints) ) {
            Utils::CLIwarning("No blueprints in application '{$oldApplication}'. ");
            return null;
        }
        // Get resource template for each blueprint
        $resources = array();
        foreach ( $blueprints as $bp ) {
            $blueprint = $this->rest()->resource()->getBlueprint ( $bp['id'] );
            if ( $blueprint === false ) {
                Utils::CLIerror("Failed to retrieve blueprint '{$bp['name']}' ({$bp['id']})");
                continue;
            }
            $blueprint = Utils::outputToArray( $blueprint );
            if ( array_key_exists( 'resource', $blueprint ) && 
                 array_key_exists( 'id', $blueprint['resource']) ) {
                if ( ! array_key_exists( $blueprint['resource']['id'], $resources ) ) {
                    $resources[ $blueprint['resource']['id'] ] = $blueprint['resource']['name'];
                }
            }
        }
        if ( empty ($resources) ) {
            Utils::CLIwarning("No resource templates to fork.");
            return null;
        }
        // Export resource templates
        Utils::CLIinfo("Forking resource templates");
        $bkp = new BackupUCD();
        $templatesDir = getcwd() . '/tmp';
        $bkp->setOutput ( $templatesDir, true );
        $result = $bkp->resourceTemplates();
        if ( $result === false ) {
            Utils::CLIerror("Failed to retrieve resource templates.");
            return false;
        }
        
        // Fork the resource templates
        $templatesFiles = scandir( $templatesDir );
        
        foreach ( $templatesFiles as $file ) {
            $filePath = $templatesDir . '/' . $file;
            if ( $file == '.' ||
                 $file == '..' || 
                 is_dir( $filePath) ) {
                continue;
            }
            $fileName = basename( $file, '.json' );
            
            if ( ! in_array( $fileName, $resources )) {
                unlink ( $filePath );
            } else {
                $resourceFile = file_get_contents ( $filePath );
                $resourceFile = json_decode ( $resourceFile, true );
                $resourceFile = $this->recursiveRenameByIndex( $resourceFile, 'name' );
                $result = Utils::createJsonFile ( $resourceFile, $filePath, true );
                if ( $result === false ) {
                    Utils::CLIerror ("Failed to write resource template file '{$filePath}'.");
                    return false;
                }
            }
        }
        
        // Fork the resource template resources
        $resourcesDir = $templatesDir . '/resources';
        if ( is_dir ( $resourcesDir ) ) {
            $resourceFiles = scandir( $resourcesDir );
            
            foreach ( $resourceFiles as $file ) {
                $filePath = $resourcesDir . '/' . $file;
                if ( $file == '.' ||
                     $file == '..' ||
                     is_dir( $filePath) ) {
                    continue;
                }
                
                $resourceFile = file_get_contents ( $filePath );
                $resourceFile = json_decode ( $resourceFile, true );
                $resourceFile = $this->recursiveRenameByIndex( $resourceFile, 'name' );
                $result = Utils::createJsonFile ( $resourceFile, $filePath, true );
                if ( $result === false ) {
                    Utils::CLIerror ("Failed to write resource file '{$filePath}'.");
                    return false;
                }
            }
        }
        
        // Recreate the resource templates
        $return['resources'] = false;
        $return['blueprints'] = array();
        
        $restore = new RestoreUCD();
        $result = $restore->resourceTemplates( $templatesDir );
        if ( $result === false ) {
            Utils::CLIerror ("Failed to create resource templates from directory '{$templatesDir}'.");
            return false;
        } else {
            $return['resources'] = true;
        }
        
        // Get all resource templates
        $resourceTemplates = $this->rest()->resource()->getResourceTemplates();
        if ( $resourceTemplates === false ){
            Utils::CLIerror("Failed to retrieve list of resource templates.");
            $return['blueprints'] = false;
            return $return;
        }
        $resourceTemplates = Utils::outputToArray( $resourceTemplates );
        
        // Get application
        $newApp = $this->udclient()->application()->getApplication( $newApplication );
        if ( $newApp === false ) {
            Utils::CLIerror("Failed to retrieve application '{$newApplication}'");
            $return['blueprints'] = false;
            return $return;
        }
        $newApp = Utils::outputToArray( $newApp );
        
        // Recreate the blueprints
        foreach ( $blueprints as $bp ) {
            Utils::CLIinfo("Creating blueprint '{$bp['name']}' ");
            $blueprint = $this->rest()->resource()->getBlueprint ( $bp['id'] );
            if ( $blueprint === false ) {
                Utils::CLIerror("Failed to retrieve blueprint '{$bp['name']}' from application '{$oldApplication}'.");
                $return['blueprints'][$bp['name']] = false;
                continue;
            }
            $blueprint = Utils::outputToArray( $blueprint );
            $blueprintResourceName = $this->getForkName( $blueprint['resource']['name'] );
            $blueprintResourceId = null;
            
            foreach ( $resourceTemplates as $template ) {
                if ( $template['name'] == $blueprintResourceName ) {
                    $blueprintResourceId = $template['id'];
                    break;
                }
            }
            
            if ( empty ($blueprintResourceId) ) {
                Utils::CLIerror("Cannot find resource template '{$blueprintResourceName}' for blueprint '{$bp['name']}'.");
                $return['blueprints'][$bp['name']] = false;
                continue;
            }
            
            $result = $restore->restoreBlueprint( $newApp['id'], $bp['name'], $blueprintResourceId, $bp['description'] );
            if ( $result === false ) {
                Utils::CLIerror("Failed to create blueprint '{$bp['name']}'");
                $return['blueprints'][$bp['name']] = false;
            } else {
                $return['blueprints'][$bp['name']] = true;
            }
        }
        
        return $return;
    }
    
}