<?php
/****************************************************************
 * IBM Confidential
 *
 * SFA100-Collaboration Source Materials
 *
 * (C) Copyright IBM Corp. 2015
 *
 * The source code for this program is not published or otherwise
 * divested of its trade secrets, irrespective of what has been
 * deposited with the U.S. Copyright Office
 *
 * **************************************************************/

require_once ('include/EntryPoint.php');

/**
 * Provides tools for creating and managing Processes requests
 *
 * @class Process
 * @author Felice Geracitano
 */
class Process extends EntryPoint {
    
    public function __construct($alias = null){
        parent::__construct($alias);
        $this->setReturn('json');
    }

    /**
     * Get the process info in an array or in a file path
     * 
     * @param string $processId
     * @return array|path|boolean
     */
    public function get($processId) {
        // Save user preference
        $return_type = $this->return;
        $return_value = null;
        if (! Utils::isUuid( $processId )) {
            Utils::CLIerror( "Parameter is not a process Id" );
            Utils::CLIdebug( __CLASS__ . '::' . __FUNCTION__.": parameter '{$processId}' is not a process Id" );
            return false;
        }
        // Failure check allowed only if return is json
        $this->setReturn( 'json' );
        $return_value = $this->rest()->process()->getProcessRest( $processId );
        if (empty( $return_value )) {
            Utils::CLIerror( "Process does not exist" );
            Utils::CLIdebug( __CLASS__ . '::' . __FUNCTION__.": process with id '{$processId}' does not exist" );
            return false;
        }
        $return_value = Utils::outputToArray( $return_value );
        // Return the right type
        if ($return_type === 'file'){
            $destination_path = $this->output."/Process_".$processId."_".microtime(true).".json";
            return (Utils::createJsonFile( $return_value, $destination_path ));
        }
        return $return_value;
    }
    
    /**
     * Get the post processesing script in an array or in a file path
     *
     * @return array|path|boolean
     */
    public function getPostProcessingScripts() {
        $return_value = $this->rest()->process()->getPostProcessingScripts();
        if (empty( $return_value )) {
            Utils::CLIerror( "Post Processing Scripts do not exist" );
            Utils::CLIdebug( __CLASS__ . '::' . __FUNCTION__.": Post Processing Scripts do not exist" );
            return false;
        }
        // Return the right type
        if ($this->return == 'file'){
            $destination_path = $this->output."/ProcessingScripts_".microtime(true).".json";
            //path from api has quotes
            $file_path = trim( $return_value, "'" );
            rename($file_path, $destination_path );
            return ($destination_path);
        }
        return Utils::outputToArray( $return_value );
    }
    
    /**
     * Create a Process from scratch or import a Process from a file path
     * 
     * @param array|json_file_path $process
     * @return boolean
     */
    public function create($process) {
        $file_name = null;
        // Check if arg is file path or array obj
        if (is_array( $process ) || is_object( $process )) {
            $file_name = Utils::createJsonFile( $process );
        } else {
            if (file_exists($process) === false) {
                Utils::CLIerror( "Json file does not exist" );
                Utils::CLIdebug( __CLASS__ . '::' . __FUNCTION__.": Json file '{$process}' does not exist" );
                return false;
            }
            if (Utils::validateJSON( file_get_contents( $process ) ) === false) {
                Utils::CLIerror( "Json file has wrong format" );
                Utils::CLIdebug( __CLASS__ . '::' . __FUNCTION__.": Json file '{$process}' has wrong format" );
                return false;
            }
            $file_name = $process;
        }
        if ($this->rest()->process()->importProcess( $file_name, false ) === false) {
            Utils::CLIerror( "Process already exist" );
            return false;
        }
        is_string( $process ) === false ? unlink( $file_name ) : null;
        return true;
    }
    
    
    /**
     * Create a PostProcessingScripts from scratch or import a Process from a file path
     *
     * @param array|json_file_path $process
     * @return boolean
     */
    public function createPostProcessingScript($script) {
        $file_name = null;
        // Check if arg is file path or array obj
        if (is_array( $script ) || is_object( $script )) {
            $file_name = Utils::createJsonFile( $script );
        } else {
            if (file_exists($script) === false) {
                Utils::CLIerror( "Json file does not exist" );
                Utils::CLIdebug( __CLASS__ . '::' . __FUNCTION__.": Json file '{$script}' does not exist" );
                return false;
            }
            if (Utils::validateJSON( file_get_contents( $script ) ) === false) {
                Utils::CLIerror( "Json file has wrong format" );
                Utils::CLIdebug( __CLASS__ . '::' . __FUNCTION__.": Json file '{$script}' has wrong format" );
                return false;
            }
            $file_name = $script;
        }
        if ($this->rest()->process()->createPostProcessingScript( $file_name ) === false) {
            Utils::CLIerror( "Process already exist" );
            return false;
        }
        is_string( $script ) === false ? unlink( $file_name ) : null;
        return true;
    }
    
    
    /**
     * Delete a process using processId
     *
     * @param string $processId
     * @return boolean
     */
    public function delete($processId) {
        if (! Utils::isUuid( $processId )) {
            Utils::CLIerror( "Parameter is not a process Id" );
            return false;
        }
        $response = $this->rest()->process()->deleteGenericProcess( $processId );
        if ( $response === false ) {
            Utils::CLIerror( "Process does not exist" );
            return false;
        }
        return true;
    }

    /**
     * Delete a Post Processesing Script using processId
     *
     * @param string $processId
     * @return boolean
     */
    public function deletePostProcessingScript($ppsId)
    {
        if (! Utils::isUuid($ppsId)) {
            Utils::CLIerror("Parameter is not a process Id");
            return false;
        }
        $result = $this->rest()->process()->deletePostProcessingScript($ppsId);
        if ($result === false) {
            Utils::CLIerror("Process does not exist");
            return false;
        }
        return true;
    }
    
    /**
     * Export process
     *
     * @param string $processId
     * @return array path boolean
     */
    public function export($processId) {
        // Save user preference
        if (empty( $processId )) {
            Utils::CLIerror( "Missign  process Id." );
            return false;
        }
        $return = $this->rest()->process()->exportProcess( $processId );
        if (empty( $processId )) {
            Utils::CLIerror( "Failed to retrieve process." );
            Utils::CLIdebug( __CLASS__ . '::' . __FUNCTION__ .": Failed to retrieve process: $processId" );
            return false;
        }
        // Return the right type
        if ($this->return == 'file') {
            $destination_path = $this->output."/Process_".$processId."_".microtime(true).".json";
            //path from api has quotes
            $file_path = trim( $return, "'" );
            rename($file_path, $destination_path );
            return ($destination_path);
        }
        return Utils::outputToArray( $return );
    }
    
    
    /**
     * Upgrade an existing Process
     *
     * @param string $json_file_path
     * @return boolean
     */
    public function upgrade($json_file_path) {
        if (Utils::validateJSON( file_get_contents( $json_file_path ) ) === false) {
            Utils::CLIerror( "Json file has wrong format" );
            Utils::CLIdebug( __CLASS__ . '::' . __FUNCTION__ .": Json file has wrong format" );
            return false;
        }
        if ($this->rest()->process()->importProcess( $json_file_path, true ) === false) {
            Utils::CLIerror( "Failure during upgrade of process" );
            Utils::CLIdebug( __CLASS__ . '::' . __FUNCTION__ .": Failure during upgrade of process described in {$json_file_path}" );
            return false;
        }
        return true;
    }
    
    /**
     * Get the Processes list in array value or file path
     *
     * @return array | path | boolean
     */
    public function getList() {
        $return = $this->rest()->process()->getProcesses();
        if ($return === false){
            Utils::CLIerror( "Failed to retrieve processes list" );
            return false;
        }
        if ($this->return == 'file') {
            $file_path = trim( $return, "'" );
            if (Utils::validateJSON( file_get_contents( $file_path ) ) === true){
                return $file_path; 
            }
        } else if (!empty( $return )) {
            return Utils::outputToArray( $return );
        } else {
            return null;
        }
        return false;
    }
}