<?php
/****************************************************************
 * IBM Confidential
*
* SFA100-Collaboration Source Materials
*
* (C) Copyright IBM Corp. 2014
*
* The source code for this program is not published or otherwise
* divested of its trade secrets, irrespective of what has been
* deposited with the U.S. Copyright Office
*
***************************************************************/

require_once('include/Extensions/RestoreUCD.php');
require_once('include/Extensions/BackupUCD.php');
require_once('include/Layers/Component.php');


/**
 * Manages snapshot promotion between uDeploy servers
 *
 * @class   Snapshot
 * @author  Marco Bassi         marcobas@ie.ibm.com
*/
class Snapshot extends EntryPoint {
    protected  $component; 
    public function __construct($alias = null){
        parent::__construct($alias);
        $this->setReturn('json');
        $this->component = new Component();
    }

    
    /**
     *
     * Check snapshot config before proceeding with the creation.
     * Returns the config in PHP format if successful, false if an error occurs.
     *
     * @param array|string $config
     * @return array|boolean
     */
    protected function checkSnapshotConfig ( $config ) {
        if ( !is_array($config) && file_exists($config) ) {
            // Argument is a file
            $ext = pathinfo($config, PATHINFO_EXTENSION);
            if ($ext == 'php') {
                include ($config);
            } else if ($ext == 'json') {
                $config = json_decode(file_get_contents($config),true);
            } else {
                Utils::CLIerror("Invalid file. Please pass a php or json file.");
                return false;
            }
        } else if ( !is_array($config )) {
            Utils::CLIerror("Invalid argument. Please use a php or json file or a php array.");
            return false;
        }
        
        // Check required values for snapshot creation
        $missingValue = false;
        if ( ! array_key_exists( 'application', $config ) ) {
            Utils::CLIerror("Missing value for 'application'.");
            $missingValue = true;
        }
        if ( ! array_key_exists( 'snapshot', $config ) ) {
            Utils::CLIerror("Missing value(s) for 'snapshot'.");
            $missingValue = true;
        } else {
            if ( ! array_key_exists( 'name', $config['snapshot'] ) ) {
                Utils::CLIerror("Missing value for 'name' in snapshot.");
                $missingValue = true;
            }
        }
        if ( ! array_key_exists( 'components', $config ) ) {
            Utils::CLIerror("Missing value(s) for 'components'.");
            $missingValue = true;
        }
        
        // Something is missing
        if ( $missingValue ) {
            Utils::CLIinfo ("Required format for configuration file (user defined values are in uppercase): ");
            if ($ext == 'json') {
                Utils::CLIout('{');
                Utils::CLIout('  "application": "APPLICATION NAME",');
                Utils::CLIout('  "snapshot": {');
                Utils::CLIout('    "name": "SNAPSHOT NAME",');
                Utils::CLIout('    "description": "SNAPSHOT DESCRIPTION"');
                Utils::CLIout('  }');
                Utils::CLIout('  "components": {');
                Utils::CLIout('    "COMPONENT NAME": "COMPONENT VERSION", ... ');
                Utils::CLIout('  }');
                Utils::CLIout('}');
            } else {
                Utils::CLIout('$config["application" = "APPLICATION NAME";');
                Utils::CLIout('$config["snapshot"] = array(');
                Utils::CLIout('    "name" => "SNAPSHOT NAME",');
                Utils::CLIout('    "description" => "SNAPSHOT DESCRIPTION"');
                Utils::CLIout(');');
                Utils::CLIout('$config["components" = array(');
                Utils::CLIout('    "COMPONENT NAME" => "COMPONENT VERSION", ... ');
                Utils::CLIout(')');
            }
            
            return false;
        }
        
        // All good, return the config
        return $config;
    }
    
    /**
     * Create a Snapshot based on an array object or json file path
     */
    public function create($snapshot) {
        $file_path = null;
        // Check if $snapshot arg is file path or array obj
        if (is_array( $snapshot ) || is_object( $snapshot )) {
            $file_path = Utils::createJsonFile( $snapshot );
        } else {
            if (file_exists($snapshot) === false) {
                Utils::CLIerror( "Json file does not exist" );
                Utils::CLIdebug( __CLASS__ . '::' . __FUNCTION__.": Json file '{$snapshot}' does not exist" );
                return false;
            }
            if (Utils::validateJSON( file_get_contents( $snapshot ) ) === false) {
                Utils::CLIerror( "Json file has wrong format" );
                Utils::CLIdebug( __CLASS__ . '::' . __FUNCTION__.": Json file '{$snapshot}' has wrong format" );
                return false;
            }
            $file_path = $snapshot;
        }
        $result = Utils::outputToArray( $this->udclient()->snapshot()->createSnapshot( $file_path ) );
        if ($result) {
            is_string( $snapshot ) === false ? unlink( $file_path ) : null;
            return true;
        } else {
            Utils::CLIerror("Failed to create snapshot " );
            return false;
        }
        return false;
    }
    
    /**
     *
     * Creates component versions and add them into snapshot. 
     * Accepts an array or file for configuration 
     * 
     * Config (php or json version) should look like
     * 
     * $config['application'] = "Application Name"
     * $config['snapshot'] = array(
     *                          "name" => "Snapshot name",
     *                          "description" => "Snapshot description",
     *                      )
     * $config['components'] = array(
     *                          "Component Name" => "Component Version",
     *                          ...
     *                      )
     *
     * @param array|string $config
     */
    public function createVersionsAndSnapshot($config, $force = true) {
        $componentAPIs = new Component;
        
        // Check the snapshot configuration
        $config = $this->checkSnapshotConfig( $config );
        if ( $config === false ) {
            return false;
        }
        
        // Retrieve component list for application
        $appDestComponents = $this->udclient()->component()->getComponentsInApplication($config['application']);
        $appDestComponentList = array();
        if ($appDestComponents) {
            $appDestComponents = Utils::outputToArray($appDestComponents);
            foreach ($appDestComponents as $comp) {
                array_push($appDestComponentList, $comp['name']);
            }
        } else {
            Utils::CLIerror("Error retrieving components for application '{$config['application']}' on destination server.");
            return false;
        }
        
        // Check if all components in snapshot are available in application
        $missingComp = false;
        foreach ($config['components'] as $name => $versions) {
            if (! in_array($name, $appDestComponentList)) {
                Utils::CLIerror("Component '{$name}' is not available in application '{$config['application']}'");
                $missingComp = true;
            }
        }
        if ($missingComp) {
            Utils::CLIout("\n");
            Utils::CLIerror("Cannot create Snapshot due to missing components.");
            return false;
        }
        
        // Snapshot creation
        $createSnapshot = false;
        $fatal_error = false;
        
        $snapshot['name'] = $config['snapshot']['name'];
        $snapshot['application'] = $config['application'];
        $snapshot['description'] = $config['snapshot']['description'];
        $snapshot['versions'] = array();
        
        // Check if snapshot already exits
        $snapshotExists = false;
        $output = $this->rest()->snapshot()->getSnapshotsInApplication($snapshot['application']);
        $checkSnapshots = Utils::outputToArray($output);
        
        foreach ($checkSnapshots as $checkSnapshot) {
            if ($checkSnapshot['name'] == $snapshot['name']) {
                $snapshotExists = true;
                break;
            }
        }
        if ($snapshotExists) {
            Utils::CLIerror("Requested snapshot '{$snapshot['name']}' is already available in '{$config['application']}' application.");
            return true;
        }
        
        // Create component version for every component in config
        Utils::CLIinfo("Creating component versions \n");
        
        // Login component object to the current server
        $componentAPIs->setup($this->alias);
        foreach ($config['components'] as $name => $versions) {
            if (!is_array($versions)) {
                $versions = array($versions);
            }
            foreach ($versions as $version) {
                // Check if the component version already exists
                if ($componentAPIs->versionExists($name,$version)) {
                    Utils::CLIout("  [SKIPPING] {$name} ({$version})");
                    array_push($snapshot['versions'], array( $name => $version ));
                    continue;
                }
                // Version is missing: create it
                $result['components'][$name] = $componentAPIs->createComponentVersion($name, $version);
                if ($result['components'][$name] !== false) {
                    Utils::CLIout("  [SUCCESS] {$name} ({$version})");
                    array_push($snapshot['versions'], array( $name => $version ));
                    $createSnapshot = true;
                    
                    // Upload artifacts, if any
                    if (array_key_exists('artifacts', $config)) {
                        $artifacts = $config['artifacts'];
                        if (array_key_exists($name, $artifacts)) {
                            // Check if more than one version
                            if (array_key_exists($version, $artifacts[$name])) {
                                $artifact = $artifacts[$name][$version];
                            } else {
                                $artifact = $artifacts[$name];
                            }
                            
                            if (array_key_exists('base', $artifact)) {
                                // Check artifacts base directory
                                $base = $artifact['base'];
                                if (!is_dir($base)) {
                                    Utils::CLIerror("Artifacts base directory '{$base}' is missing or not accessible");
                                    return false;
                                }
                                
                                // Check artifacts to include
                                $include = null;
                                $includeArray = array();
                                if (array_key_exists('include', $artifact)) {
                                    $include = $artifact['include'];
                                    $includeError = false;
                                    if (! is_array($include)) {
                                        array_push($includeArray, $include);
                                    } else {
                                        $includeArray = $include;
                                    }
                                    foreach ($includeArray as $includeFile) {
                                        $includeFile = str_replace('*','',$includeFile);
                                        $includeFile = str_replace('\/','/',$includeFile);
                                        $path = "{$base}/{$includeFile}";
                                        if (! file_exists($path)) {
                                            Utils::CLIerror("Artifacts file/dir '{$includeFile}' is missing or not accessible in base directory '{$base}'");
                                            $includeError = true;
                                        }
                                    }
                                    if ($includeError) {
                                        return false;
                                    }
                                }
                                
                                // Artifacts to exclude
                                $exclude = null;
                                if (array_key_exists('exclude', $artifact)) {
                                    $exclude = $artifact['exclude'];
                                }
                                // Upload Artifacts
                                $result['artifacts'][$name][$version] = $this->udclient()->component()->addVersionFiles($name, $version, $base, $include, $exclude);
                                // If an error occurs during file upload, delete component version and exit
                                if ($result['artifacts'][$name][$version] === false){
                                    Utils::CLIerror("Error during the artifacts upload, removing version: $version, from component $name.");
                                    // createComponentVersion save version into $result['components'][$name]
                                    $componentAPIs->deleteVersion( $result['components'][$name]['id']);
                                    $fatal_error = true;
                                    continue 2;
                                }
                                
                            } else {
                                $this->log->debug("Trying to upload artifacts for component '{$name}', version '{$version}', but no base path has been set. Skipping.");
                            }
                        }
                    } // End of artifacts upload
                } else {
                    // Error during component version creation
                    Utils::CLIerror("  [FAILURE] {$name} ({$version})");
                    // Failed to create the version of the component for some reason
                    return false;
                }
            }
        }
        
        // check for fatal error
        if ($fatal_error) {
            Utils::CLIout("\n\n  [SKIPPED] Snapshot not created.");
            return false;
        }
        
        // At least one new component version has been created. Create a new snapshot
        if ($createSnapshot || $force) {
            $file = Utils::createJsonFile($snapshot);
            $result['snapshot'] = Utils::outputToArray($this->udclient()->snapshot()->createSnapshot($file));
            $this->log->log(print_r($result['snapshot'],true));
            if ($result['snapshot']) {
                Utils::CLIout("\n\n  [SUCCESS] Snapshot created ({$snapshot['name']})");
                Utils::CLIout("\n\n  [LINK] {$this->weburl}/#snapshot/{$result['snapshot']['id']}");
            } else {
                Utils::CLIout("\n\n  [FAILURE] Snapshot was not created ({$snapshot['name']})");
                $result['snapshot'] = false;
            }
            unlink($file);
        } else {
            Utils::CLIout("\n\n  [SKIPPED] Snapshot not created. All component versions are included in a previous snapshot.");
        }
        
        return $result;
    }

    /**
     * Export Snapshot
     *
     * @param snapshot ID
     * @param Application ID|Name
     * @param snapshot Name (optional)
     *            
     * @return string
     */
    public function get($snapshotId, $application, $snapshotName = null)
    {
        if (empty($snapshotId) || empty($application)) {
            Utils::CLIerror( "Invalid Paramtets." );
            return false;
        }
        $result = $this->rest()->snapshot()->exportSnapshot($snapshotId, $application, $snapshotName);
        
        if ($result === false) {
            Utils::CLIerror("Cannot download snapshot zip file from origin server.");
            return false;
        }
        if ($this->return === 'file'){
            // delete quotes and return path
            return substr($result,1,-1);
        }
        return Utils::outputToArray( $result );
    }

    /**
     * Return a list of snapshot based on application id or name
     *
     * @param string $json_file_path
     * @return boolean | array
     */
    public function getList($sapplication) {
        $this->setReturn('array');
        $return = $this->rest()->snapshot()->getSnapshotsInApplication($sapplication);
        if ($return === false){
            Utils::CLIerror( "Failed to retrieve snapshots list" );
            Utils::CLIdebug( __CLASS__ . '::' . __FUNCTION__ .": Failed to retrieve snapshots list for application '{$$sapplication}'" );
            return false;
        }
        if (empty($return)){
            return null;
        }
        return Utils::outputToArray($return);
    }
    
    /**
     * Return all the snapshots with matchs with name in param
     *
     * @param snapshots array
     * @param snapshot name
     *
     * @return array
     */
    private function matchSnapshot($snapsArray, $snapshotName) {
        $snapshotReturn = array ();
        foreach ( $snapsArray as $entry ) {
    
            if ($entry ['name'] === $snapshotName) {
                $snapshotReturn [] = $entry;
                continue;
            }
            $status = substr( $entry ['name'], - 6 );
            $nameNoStatus = substr( $entry ['name'], 0, strlen($entry ['name']) - 7 );
            if (($status == "FAILED" || $status == "PASSED") &&
            ($nameNoStatus == $snapshotName)) {
                $snapshotReturn [] = $entry;
            }
        }
        return $snapshotReturn;
    }
    
    /**
     * Print out snapshots available
     */
    private function printSnapshots($snapsArray) {
        foreach ( $snapsArray as $entry ) {
            Utils::CLIout($entry['name']);
        }
        Utils::CLIout('');
    }
    
    /**
     * Set a snapshot PASSED or FAILED
     *
     * @param snapshot name
     * @param snapshot status [PASSED | FAILED] case insensitive
     * 
     * @return boolean
     */
    public function setSnapshotStatus($snapshotName, $status) {
        $error = false;
        // Check status argument
        if (empty($status)) {
            Utils::CLIerror( "Missing snapshot status." );
            $errors = true;
        }
        
        $status = strtoupper( $status );
        if ($status != "PASSED" && $status != "FAILED") {
            Utils::CLIerror( "Passed Status: '$status' is not supported" );
            $errors = true;
        }
        
        if ($errors === true){
            Utils::CLIinfo( "Possible options:" );
            Utils::CLIout( "  PASSED" );
            Utils::CLIout( "  FAILED".PHP_EOL );
            return false;
        }
        
        Utils::CLIinfo( "Getting snapshot info.." );
        $ucc = new EntryPoint();
        $this->setReturn( 'json' );
        $snapJson = $ucc->rest()->snapshot()->getSnapshotsInApplication( $this->application );
        if ($snapJson == false) {
            Utils::CLIerror( 
                    __CLASS__ . '::' . __FUNCTION__ .
                             ":: Failed to get info about snapshots from application: {$this->application}" );
            return false;
        }
        // Output to Array for easy editing
        $snapsArray = Utils::outputToArray( $snapJson );
        // Looking for conflicts snapshot
        $snapsArray = $this->matchSnapshot( $snapsArray, $snapshotName );
        if (empty($snapsArray)) {
            Utils::CLIinfo( 
                    __CLASS__ . '::' . __FUNCTION__ .
                             ":: No snapshot named: $snapshotName from application: {$this->application}" );
            return false;
        }
        if (count( $snapsArray ) > 1) {
            Utils::CLIinfo( 
                    __CLASS__ . '::' . __FUNCTION__ .
                             ":: Conflict detected for '{$snapshotName}', please use a more specific name. Snapshot collected: " );
            $this->printSnapshots($snapsArray);
            return false;
        }
        $snapArray = $snapsArray[0];
        // Try to override the old status
        $statusOld = substr( $snapArray ['name'], - 6 );
        if ($statusOld === "PASSED" || $statusOld === "FAILED") {
            $snapArray ['name'] = str_replace( $statusOld, $status, 
                    $snapArray ['name'] );
        } else {
            $snapArray ['name'] = "{$snapArray ['name']}_{$status}";
        }
        $snapArray ['existingId'] = $snapArray ['id'];
        $nameResult = $snapArray ['name'];
        $snapJson = json_encode( $snapArray );
        $response = $ucc->rest()->snapshot()->setSnapshotName( $snapJson );
        if ($response == false) {
            Utils::CLIerror( 
                    __CLASS__ . '::' . __FUNCTION__ .
                             ":: Failed to edit the snapshot: $snapshotName, in : {$snapshotName}_{$status}" );
            return false;
        }
        $response = $ucc->rest()->snapshot()->addSnapshotStatus( $snapArray ['id'], $status );
        if (! is_array( $response )) {
            Utils::CLIerror( 
                    __CLASS__ . '::' . __FUNCTION__ .
                             ":: Failed to add the snapshot status: $status to $nameResult" );
            return false;
        }
        
        Utils::CLIresult( 
                "Success. Snapshot: '$snapshotName' now marked : '$nameResult' and in status: '$status'" );
        return true;
    }
    
}