<?php
/****************************************************************
 * IBM Confidential
 *
 * SFA100-Collaboration Source Materials
 *
 * (C) Copyright IBM Corp. 2014
 *
 * The source code for this program is not published or otherwise
 * divested of its trade secrets, irrespective of what has been
 * deposited with the U.S. Copyright Office
 *
 ***************************************************************/

/**
 * @class   Utils
 * @author  Marco Bassi         marcobas@ie.ibm.com
 * 
 * This class is providing some utils for editing and managing the output in EntryPoint 
 */
define("ERROR",'error');
define("INFO",'info');
define("WARNING",'warning');
define("DEST",'dest');
define("RESULT",'result');
define("API",'api');
define("LOG",'log');
define("DEBUG",'debug');
define("START",'start');
define("COMPLETED",'completed');



require_once('Logger.php');

class Utils{
    public static $silent;
    public static $log;
    /*
     * Executes a EntryPoint command
     */
    public static function ucdExec($command){
        exec( $command, $output, $return);
        outputToJson( $output );
    }
    
    /**
     * Convert output from exec into JSON compatible var
     */
    public static function outputToJson($output){
        if (!is_array($output)){
            return false;
        }
        $json = '';
        foreach($output as $key=>$value){
            $pos = strpos($value,'<textarea>');
            if ($pos === 0) {
                str_replace('<textarea>','',$value);
            }
            $pos = strpos($value,'}</textarea>');
            if ($pos === 0) {
                str_replace('}</textarea>','}',$value);
            }
            $json .= trim($value);
        }
        
        return $json;
    }
    
    /**
     * Convert output from exec into array
     */
    public static function outputToArray($output){
        $json = self::outputToJson($output);
        
        // $output holds correct: JSON formatted string, returned from UCD Server
        if ( self::validateJSON($json) ) {
            $array = json_decode($json, true);
            
        // UCD Server returned some 'error' message, i.e. Request was not processes successfully
        } else {
            $array = array('error' => $json);
        }
        
        return $array;
    }
    
    /**
     * Export JAVA_HOME path, required by udclient
     */
    public static function setJavaHome($javaHome){
        // If javaHome is empty, get System default
        if ( empty( $javaHome ) ) {
            $command = 'ls ${JAVA_HOME}';
            exec($command, $output, $result);
            
            if ( $result !== 0 ){
                return false;
            } else {
                $javaHome = $output[0];
            }
        }
        
        // Check for bin files
        if ( is_file ( "{$javaHome}/bin/java" ) ) {
            return $javaHome;
        } else {
            return false;
        }
    }
    
    /**
     * @param bool      $bool
     * If set to true, output is suppressed. 
     */
    public static function setSilent($bool){
        if (is_bool($bool)){
            self::$silent = $bool;
            return true;
        }
        return false;
    }
    
    /**
     * Prints messages in the console with a particular format.
     * Will also implement $message indentation, when parameters: 
     * $indent_sign and $indent_count are provided.
     * 
     * @param string    $message        Message to be prompted
     * @param bool      $newline        If set to true, the message will be prompted in a new line
     * @param string    $type           Add messages such as [ERROR] or [LOG] before the message
     * @param number $indent_count      A counter to be used for indentation of a message.
     *                                  If specified, the defined message will be indented by the
     *                                  specified number of EMPTY SPACES, to the right.
     * 
     * @return boolean
     */
    public static function CLIout($message,$newline=true,$type=null,$indent_count=0){
        // Set logger
        if (empty(self::$log)) {
            self::$log = new Logger();
            if (file_exists('config/logger.config.php')) {
                include('config/logger.config.php');
                self::$log->setLogFile($config['log']);
            }
        }
        
        // Print message in the CLI
        if (self::$silent != true) {
            // Implement indentation
            if ($indent_count > 0) {
                $message = str_pad($message, (strlen($message) + $indent_count), ' ', STR_PAD_LEFT);
            }
            
            // Set message type
            if (!empty($type)){
                $message = '[' . strtoupper($type) . '] ' . $message;
            }
            
            // Set new line
            if ($newline){
                $message = "\n" . $message;
            }
            
            // Print message
            echo $message;
        }
        // Redirect to error log if silent/error
        else if (self::$silent == true || $type = ERROR){
            self::$log->log($message,$type);
        }
        
        // Redirect stdout to output log
        self::$log->stdoutLog($message);
        
        return true;
    }
    
    /**
     * Print an error message to the console.
     * When $errorType is provided (example: 'system'), the output will look like:
     * [ERROR][SYSTEM] message.
     * 
     * @param string $message
     * @param string $newline
     * @param number $indent_count
     * @param string $errorType String used to add Error Type to the Error message.
     *                                  
     * @return boolean
     */
    
    public static function CLIbold($message,$newline=true,$indent_count=0){
        $message = "\033[1m".$message."\033[0m";
        return self::CLIout($message,$newline, null ,$indent_count);
    }
    
    public static function CLIerror($message,$newline=true,$indent_count=0,$errorType=null){
        $error = 'error';
        if (!empty($errorType)){
            $error =  $error . '][' . strtoupper($errorType);
        }
        $message .= "\n";
        return self::CLIout($message,$newline,$error,$indent_count);
    }
    
    public static function CLIinfo($message,$newline=true,$indent_count=0){
        $message .= "\n";
        return self::CLIout($message,$newline, INFO ,$indent_count);
    }
    
    public static function CLIdest($message,$newline=true,$indent_count=0){
        $message .= "\n";
        return self::CLIout($message,$newline, DEST ,$indent_count);
    }
    
    public static function CLIresult($message,$newline=true,$indent_count=0){
        $message .= "\n";
        return self::CLIout($message,$newline, RESULT ,$indent_count);
    }
    
    public static function CLIwarning($message,$newline=true,$indent_count=0){
        $message .= "\n";
        return self::CLIout($message,$newline, WARNING ,$indent_count);
    }
    
    public static function CLIlog($message,$newline=true,$indent_count=0){
        $message .= "\n";
        return self::CLIout($message,$newline, LOG ,$indent_count);
    }
    
    public static function CLIapi($message,$newline=true,$indent_count=0){
        $message .= "\n";
        return self::CLIout($message,$newline, API ,$indent_count);
    }
    
    public static function CLIdebug($message,$newline=true,$indent_count=0){
        $message .= "\n";
        if (empty(self::$log)) {
            self::$log = new Logger();
            if (file_exists('config/logger.config.php')) {
                include('config/logger.config.php');
                self::$log->setLogFile($config['log']);
            }
        }
        self::$log->log($message,DEBUG);
        if (self::$log->isDebug()) {
            return self::CLIout($message,$newline, DEBUG ,$indent_count);
        }
    }
    public static function CLIstart($message,$newline=true,$indent_count=0){
        $message .= "\n";
        return self::CLIout($message,$newline, START ,$indent_count);
    }
    public static function CLIcompleted($message,$newline=true,$indent_count=0){
        $message .= "\n";
        return self::CLIout($message,$newline, COMPLETED ,$indent_count);
    }
    
    /**
     *
     * Check that array passed as input cotains a false value
     *
     * @param array $result
     * @return boolean
     */
    public function resultHasFailed($result = array())
    {
        if ($result === false) {
            return true;
        }
    
        $errors = false;
        if (is_array($result)) {
            foreach ($result as $key => $value) {
                $tmp = self::resultHasFailed($value);
                $errors = ($errors || $tmp);
            }
        }
        return $errors;
    }
    
    /**
     * 
     * Check if a content is a valid json file
     *
     * @param string $content
     * @return boolean
     */
    public static function validateJSON($content) {
        json_decode($content);
        return (json_last_error() == JSON_ERROR_NONE);
    }
    
    /**
     * Giving a directory as argument, recursively scans it and verify that all *.json files are valid 
     *
     * @param string $dir
     */
    public static function validateDirectoryJSON($dir) {
        $is_valid = true;
        
        // Check directory
        if (!is_dir($dir)) {
            self::CLIerror("{$dir} is not a valid directory");
            $is_valid = false;
        }
        
        // Check Files within specified directory
        if ($is_valid) {
            $files = scandir($dir);
            foreach ($files as $file) {
                if ( $file === '.' || $file === '..' ) {
                    continue;
                }
                // Call this function recursively when there is a sub-dir in the inspected dir
                if ( is_dir("{$dir}/{$file}") ){
                    $is_valid = self::validateDirectoryJSON("{$dir}/{$file}");
                    
                // Validate file
                } else if (is_file("{$dir}/{$file}")) {
                    // Skip if not a JSON files
                    $ext = explode(".", $file) ;
                    $n = count($ext)-1;
                    $ext = $ext[$n];
                    if (strtolower($ext) != 'json') {
                        continue;
                    }
                    // Validate JSON file
                    $file_content = file_get_contents("{$dir}/{$file}");
                    $is_valid = self::validateJSON($file_content);
                    if ( !$is_valid ) {
                        self::CLIwarning("File {$dir}/{$file} didn't pass JSON validation process.");
                    }
                }
            }
        }
        
        // Return
        return $is_valid;
    }
    
    /**
     * Write a string to a file.
     * This function is used to store a UCD Application, Component or a Process as a JSON formatted string into a file.
     *
     * On completion, this function will print a message for SUCCESS / FAIL for file creation,
     * and will return booolean TRUE / FALSE correspondingly.
     *
     * Please note, if $config['silent'] flag is set to true, in the ucd.config.php,
     * no messages will be printed on screen, instead these will be Logged to a file.
     *
     * @param string $filename          FULL Path + filename for the file to be created.
     * @param string $file_content      JSON formatted strinf of the UCD APP, Component / Process to be stored
     * @param string $print_filename    A Boolean flag, used to specify, whether to print specified $filename,
     *                                  in the Report messages from this function, or not.
     * @param boolean $toPrettyString   Boolean flag used to define, whether to save the passed $file_content
     *                                  (json string) on single line or as a formatted (multiple lines) string.
     *                                  NOTE: if passed $file_content string is larger than allowed length,
     *                                        currently this is set to 20 000 characters,
     *                                        the $file_content string will be FORMATTED regardless of
     *                                        whether $toPrettyString was set to true or not.
     *
     * @return boolean Boolean TRUE / FALSE, depending on whether the specified file was created or not.
     */
    public static function writeToFile ($filename, $file_content, $print_filename=true, $toPrettyString=false) {
        
        $file_written = true;
        $max_allowed_string_length = 20000;
        
        // Ensure $file_content is in a json format
        if (is_array($file_content) || is_object($file_content) ) {
            $file_content = json_encode($file_content);
        }
        
        /*
         * Passed $file_content string is larger than allowed and $toPrettyString is not set
         * -=> prompt user that String is longer than allowed and will be FORMATTED,
         *     before to be written to a file.
         * This is required in order to prevent file corruption by writing
         * very large (2mln + characters) string on 1 line.
         */
        if (!$toPrettyString && strlen($file_content) > $max_allowed_string_length) {
           self::CLIdebug('Length of passed JSON string is longer than allowed: '
                   . $max_allowed_string_length . ' characters', true, 3);
           self::CLIdebug('Passed JSON String will be FORMATTED to a pretty string', false, 6);
           self::$log->info("Length of passed JSON string is longer than allowed: {$max_allowed_string_length} characters");
           self::$log->info("Passed JSON String will be FORMATTED to a pretty string");
           $toPrettyString = true;
        }
        
        if ($toPrettyString) {
            $file_content = self::formatJsonString($file_content);
        }
        
        // Write to file
        $file_written = file_put_contents($filename, $file_content);
        
        // Print a message to console and return
        if ($file_written === false) {
            // set message
            $err_msg = str_replace(
                    '[filename_placeholder]', 
                    ($print_filename ? ': "' . $filename . '"' : ''), 
                    'An error occurred while writing to file[filename_placeholder].');
            // print message
            self::CLIerror($err_msg, false, 3);
            // return
            return false;
        } else {
            $return_msg = str_replace(
                    '[filename_placeholder]', 
                    ($print_filename ? ': "' . $filename . '"' : ''), 
                    'File[filename_placeholder] saved - SUCCESS.');
            self::CLIinfo($return_msg, false, 3);
            return true;
        }
    }
    
    /**
     * Used to retrieve configuration settings for implementing of User Impersonation.
     * Will return an Array, holding the settings for User Impersonation.
     * In case of a problem with the specified configuration, this function will return Boolean FALSE.
     * 
     * @return mixed: Array, holding the configuration for User Impersonation or
     *                Boolean FALSE in case of a problem with the Configuration file.
     */
    public static function getUserImpersonationConfig () {
        $ui_config = array();
        // Check if Impersonation config file is present
        if (is_file('config/ucd.impersonation.config.php')) {
            // Load impersonation config file
            include 'config/ucd.impersonation.config.php';

            // Validate configuration, before to read it
            if (
                isset($config['impersonation_config']) && 
                self::validateUserImpersonationConfig($config['impersonation_config'])
            ) {
                $ui_config = $config['impersonation_config'];
            } // else: validateUserImpersonationConfig() will prompt message(s) for error(s) in configuration
        } else {
            self::CLIerror('uDeploy - user impersonation config file: '
                    . '"config/ucd.impersonation.config.php" is missing.');
        }
    
        // Return
        return (count($ui_config) > 0) ? $ui_config : false;
    }
    
    /**
     * Validate the configuration for Component::Process::Step - User Impersonation.
     * Will check if specified config variables are correct and in the correct format and values.
     * 
     * @return boolean TRUE/FALSE when Configuration was OK or NOT OK, correspondingly.
     */
    protected function validateUserImpersonationConfig($impersonation_configuration) {
        $ui_config_valid = true;
        
        self::CLIinfo('Validate "User Impersonation" configuration:', true, 3);
        
        // 1. Validate if impersonation template is set up
        if (
                !isset($impersonation_configuration['template']) || 
                count($impersonation_configuration['template']) == 0
        ) {
            self::CLIerror('- please set up User Impersonation Configuration "template".', false, 11);
            $ui_config_valid = false;
        }
        
        // 2. Validate impersonation template variables
        if ($ui_config_valid) {
            // 2.1 Validate if useImpersonation is present
            if (!isset($impersonation_configuration['template']['useImpersonation'])) {
                self::CLIerror('- "useImpersonation" is REQUIRED.', false, 11);
                $ui_config_valid = false;
            }
            // 2.2 Validate if impersonationUsername is present and not empty
            if (
                !isset($impersonation_configuration['template']['impersonationUsername']) ||
                (
                    isset($impersonation_configuration['template']['impersonationUsername']) &&
                    strlen(trim($impersonation_configuration['template']['impersonationUsername'])) == 0
                )
            ) {
                self::CLIerror('- "impersonationUsername" is REQUIRED and CANNOT be EMPTY.', false, 11);
                $ui_config_valid = false;
            }
            // 2.3 Validate if impersonationGroup is present
            if ( !isset($impersonation_configuration['template']['impersonationGroup'])) {
                self::CLIwarning('- please set up: "impersonationGroup".', false, 9);
                $ui_config_valid = false;
            }
            // 2.4 Validate if impersonationPassword is present
            if ( !isset($impersonation_configuration['template']['impersonationPassword'])) {
                self::CLIwarning('- please set up: "impersonationPassword".', false, 9);
                $ui_config_valid = false;
            }
            // 2.5 Validate if impersonationUseSudo is present and not false
            if (
                !isset($impersonation_configuration['template']['impersonationUseSudo']) ||
                (
                    isset($impersonation_configuration['template']['impersonationUseSudo']) &&
                    !$impersonation_configuration['template']['impersonationUseSudo']
                )
            ) {
                self::CLIerror('- "impersonationUseSudo" is REQUIRED and MUST be set to boolean "true".', false, 11);
                $ui_config_valid = false;
            }
        }
        
        // 3. Validate mode (online/offline)
        if (
            !isset($impersonation_configuration['mode']) ||
            (
                isset($impersonation_configuration['mode']) &&
                !in_array($impersonation_configuration['mode'], array('online','offline'))
            )
        ) {
            self::CLIerror('- Impersonation "mode" is REQUIRED.', false, 11);
            self::CLIinfo('Possible values: (online/offline).', false, 14);
            $ui_config_valid = false;
        }
        
        // 4. Validate: overwrite (true/false)
        if (
            !isset($impersonation_configuration['overwrite']) ||
            (
                isset($impersonation_configuration['overwrite']) &&
                !is_bool($impersonation_configuration['overwrite'])
            )
        ) {
            self::CLIerror('- Impersonation "mode" is REQUIRED.', false, 11);
            self::CLIinfo('Possible values: (true/false).', false, 14);
            $ui_config_valid = false;
        }
        
        // Print additional info message in case of an error
        if (!$ui_config_valid) {
            self::CLIinfo('For more details in "User Impersonation" settings, '
                    . 'please check: "urbancode/README" file.', true, 3);
        } else {
            self::CLIinfo('- "User Impersonation" configuration: OK', false, 12);
        }
        
        /*
         * 5. Validate Variables Creation configurations 
         *    Use this validation Step ONLY when 'create_variables' is SET and is equal to TRUE,
         *    AND Impersonation validation ($ui_config_valid) is TRUE
         */
        if (
            $ui_config_valid === true &&
            isset($impersonation_configuration['create_variables']) && 
            $impersonation_configuration['create_variables'] === true
        ) {
            $ui_config_valid = self::validateCreateEnvironmentVariablesConfig($impersonation_configuration);
        }
        
        // Return
        return $ui_config_valid;
    }
    
    /**
     * Validate the configuration for Creation of Component-Environment variable(s).
     * Will check if specified config variables are correct and with the correct format and values.
     *
     * @return boolean TRUE/FALSE when Configuration was OK or NOT OK, correspondingly.
     */
    protected function validateCreateEnvironmentVariablesConfig($impersonation_configuration) {
        
        self::CLIinfo('Validate "Create Component-Environment Variables" configuration:', true, 3);
        
        $create_component_env_vars = true;
        // Validate variable(s) to create
        if (!isset($impersonation_configuration['variables_to_create'])
        ) {
            self::CLIerror('- "variables_to_create" is REQUIRED.', false, 11);
            $create_component_env_vars = false;
        
        // 'variables_to_create' has been set -=> validate if it was set in the correct way
        } else {
            /*
             * The 'variables_to_create' MUST be set as Array 
             * and should have atleast 1 Variable set up for creation
             */
            if (
                !is_array($impersonation_configuration['variables_to_create']) || 
                (
                        is_array($impersonation_configuration['variables_to_create']) && 
                        count($impersonation_configuration['variables_to_create']) == 0
                )
            ) { 
                self::CLIerror('- "variables_to_create" MUST be set as Array ' 
                        . 'and should have at least 1 Variable to be created.', false, 11);
                $create_component_env_vars = false;
            
            /*
             * The 'variables_to_create' is set as required 
             * -=> check its elements (config for each variable to create)
             */
            } else {
                foreach ($impersonation_configuration['variables_to_create'] as $key => $component_env_var_config) {
                    /*
                     * Component-Environment Variable configuration should be in the form:
                     * Array (
                     *      'name'          => 'var_name',
                     *      'label'         => 'var_label',
                     *      'pattern'       => '',
                     *      'type'          => 'TEXT',
                     *      'value'         => 'var_value',
                     *      'required'      => true,
                     *      'description'   => ''
                     * )
                     */
                    if (
                        !is_array($component_env_var_config) || 
                        (
                            is_array($component_env_var_config) && 
                            count($component_env_var_config) == 0
                        )
                    ) {
                        self::CLIerror('- Variables Configuration: ' 
                                . $key . ' => ' . var_export($component_env_var_config) 
                                . ' should be set as Array and cannot be empty.', false, 11);
                        $create_component_env_vars = false;
                    
                    // Validate $component_env_var_config as required
                    } else {
                        // Validate 'name'
                        if (
                            !isset($component_env_var_config['name']) || 
                            (
                                isset($component_env_var_config['name']) && 
                                trim($component_env_var_config['name']) == ''
                            )
                        ) {
                            self::CLIerror('- Config value for Variable "name" is REQUIRED.', false, 11);
                            $create_component_env_vars = false;
                        }
                        // Validate 'label'
                        if (
                            !isset($component_env_var_config['label']) ||
                            (
                                isset($component_env_var_config['label']) &&
                                trim($component_env_var_config['label']) == ''
                            )
                        ) {
                            self::CLIerror('- Config value for Variable "label" is REQUIRED.', false, 11);
                            $create_component_env_vars = false;
                        }
                        // Validate 'value'
                        if (
                            !isset($component_env_var_config['value']) ||
                            (
                                isset($component_env_var_config['value']) &&
                                trim($component_env_var_config['value']) == ''
                            )
                        ) {
                            self::CLIerror('- Config value for Variable "value" is REQUIRED.', false, 11);
                            $create_component_env_vars = false;
                        }
                        // Validate 'pattern' [OPTIONAL]
                        if (!isset($component_env_var_config['pattern'])) {
                            self::CLIwarning('- Config value for "pattern" is not provided.', false, 9);
                            self::CLIinfo('Please Note: "pattern" will be defaulted to EMPTY_STRING.', false, 14);
                        }
                        // Validate 'type' [OPTIONAL]
                        if (
                            !isset($component_env_var_config['type']) ||
                            (
                                isset($component_env_var_config['type']) &&
                                trim($component_env_var_config['type']) == ''
                            )
                        ) {
                            self::CLIwarning('- Config value for "type" is not provided.', false, 9);
                            self::CLIinfo('Please Note: "type" will be defaulted to "TEXT".', false, 14);
                        }
                        // Validate 'required' [OPTIONAL]
                        if (
                            !isset($component_env_var_config['required']) ||
                            (
                                isset($component_env_var_config['required']) &&
                                !is_bool($component_env_var_config['required'])
                            )
                        ) {
                            self::CLIwarning('- Config value for "required" is not provided.', false, 9);
                            self::CLIinfo('Please Note: "required" will be defaulted to "false".', false, 14);
                        }
                        // Validate 'description' [OPTIONAL]
                        if (
                            !isset($component_env_var_config['description']) ||
                            (
                                isset($component_env_var_config['description']) &&
                                trim($component_env_var_config['description']) == ''
                            )
                        ) {
                            self::CLIwarning('- Config value for "description" is not provided.', false, 9);
                            self::CLIinfo('Please Note: "description" will be defaulted to EMPTY_STRING.', false, 14);
                        }
                    }
                }
            }
        }
        
        // Print additional info message in case of an error
        if (!$create_component_env_vars) {
            self::CLIinfo('For more details about ' 
                    . '"SET UP - Importing / Creation of Environment Variables to a Component" settings, '
                    . 'please check: "urbancode/README" file.', true, 3);
        } else {
            self::CLIinfo('- "Create Component-Environment Variables" configuration: OK', false, 12);
        }
        
        // Return
        return $create_component_env_vars;
    }
    
    /**
     * A HELPER Function used to transform a passed variable as Object, an instance of stdClass().
     *
     * @param mixed $variable_to_transform The variable to be transformed.
     *                                     This could be passed as:
     *                                         1. Object
     *                                         2. Array
     *                                         3. JSON formatted String
     * @return An Object instance of the passed Variable to transform or 
     *         NULL in case of a propblem.
     */
    public static function getAsObject ($variable_to_transform) {
        
        // variable is passed as array: json encode/decode it to convert it to object
        if (is_array($variable_to_transform)) {
            $var_object = json_decode(json_encode($variable_to_transform));
        // variable is passed as object: use it as it is
        } else if (is_object($variable_to_transform)) {
            $var_object = $variable_to_transform;
        // variable is passed as string: decode it from json to object
        } else {
            $var_object = json_decode($variable_to_transform);
        }
        
        // Return
        return $var_object;
    }

    /**
     * Function to rename a file, used in this scenario:
     * id.json -> name.json
     *
     * @param string path
     * @param new name
     * @return string path|bool
     */
    public function renameJson($full_path, $new_name)
    {
        if (empty($full_path) || empty($new_name)){
            self::CLIerror("Invalid parameters.");
            return false;
        }
        // Replace not allowed spaces
        $new_name = str_replace(' ', '_', $new_name);
        $path_parts = pathinfo($full_path);
        $new_name = $path_parts['dirname'] . '/' . $new_name . '.json';
        if (rename($full_path, $new_name) === false) {
            self::CLIwarning("Impossible to rename '$full_path' into '$new_name'");
            return false;
        }
        return $new_name;
    }

    /**
     * This function return an array with a list file
     * - if $ext is null|false, push all the files into return array
     * - if $ext has extension, push the file with that extension in return array
     *
     * @param   path
     * @param   $ext
     * @return  array
     */
    public static function listFiles($path, $ext = null)
    {
        // check if exist
        if (! is_dir($path)) {
            self::CLIerror("Ivalid path.");
            return null;
        }
        // get files from path
        $files = new FilesystemIterator($path, FilesystemIterator::SKIP_DOTS);
        $list = array();
        foreach ($files as $file) {
            if (empty($ext) || $file->getExtension() == $ext) {
                array_push($list, $file->getRealPath());
            }
        }
        return $list;
    }
    
    
    /**
     * Creates a json file from an array or an Object (stdClass()).
     * Retruns file path in case of success, false in case of error or if file exists and override is set to false.
     * 
     * @param array|object $itemToStore Array or Object, to be written to a FILE in a JSON formatted string
     * @param string       $fileToWrite [OPTIONAL] default: {cwd}/jsonfile_{YmdHis}.json
     * @param boolean      $override    [OPTIONAL] default: false
     * 
     * @return string|boolean String holding the path to the created JSON file OR
     *                        Boolean FALSE in case of an error
     * 
     */
    public static function createJsonFile($itemToStore, $fileToWrite = null, $override = false){
        if (!empty($fileToWrite) && is_file($fileToWrite)){
            if ($override){
                self::CLIinfo("Overriding existing file '{$fileToWrite}'.");
            } else {
                self::CLIerror("File '{$fileToWrite}' exists.");
                return false;
            }
        } else if (empty($fileToWrite)) {
            $cwd = getcwd();
            // create tmp if not exist
            if (!file_exists($cwd."/tmp")) {
                mkdir($cwd."/tmp", 0755, true);
            }
            $fileToWrite = $cwd . '/tmp/jsonfile_' . date('YmdHis') . '.json';
        }
        $content = json_encode($itemToStore);
        
        $file_written = true;
        $max_allowed_string_length = 20000;
        $toPrettyString = false;
        
        
        /*
         * Passed $file_content string is larger than allowed and $toPrettyString is not set
        * -=> prompt user that String is longer than allowed and will be FORMATTED,
        *     before to be written to a file.
        * This is required in order to prevent file corruption by writing
        * very large (2mln + characters) string on 1 line.
        */
        if (strlen($content) > $max_allowed_string_length) {
            self::CLIdebug('Length of passed JSON string is longer than allowed: '
                    . $max_allowed_string_length . ' characters', true, 3);
            self::CLIdebug('Passed JSON String will be FORMATTED to a pretty string', false, 6);
            $toPrettyString = true;
        }
        
        if ($toPrettyString) {
            $content = self::formatJsonString($content);
        }
        
        $result = file_put_contents($fileToWrite,$content);
        if ($result !== false) {
            return $fileToWrite;
        } else {
            // @TODO: LOG ERROR MESSAGE
            self::CLIerror('Could NOT write content: ');
            self::CLIout($content, false, 'content');
            self::CLIinfo('to file: ');
            self::CLIdest($fileToWrite);
            return false;
        }
    }
    
    /**
     * Format a JSON formatted string into a human friendly layout,
     * i.e. in a multiple lines with indented elements based on their level.
     * 
     * @param  string $json A JSON formatted string to be formatted 
     *                      in a human friendly (readable) way.
     *                      
     * @return string "Pretty formatted" JSON string or an Error Message,
     *                in case when the passed string is not a valid JSON string.
     */
    public static function formatJsonString($json) {
    
        $formattedJson  = '';
        $jsonLength     = strlen($json);
        $currentChar    = '';
        $prevChar       = '';
        $nextChar       = '';
        $indentCount    = 0;
        $indentString   = '  ';
        $inQuotes       = false;
        
        self::CLIdebug('Start of formatting of a JSON string to pretty string', true, 3);
        self::CLIdebug('Length of passed JSON string is: ' . $jsonLength . ' characters', false, 6);
        self::$log->info('Start of formatting of a JSON string to pretty string');
        self::$log->info('Length of passed JSON string is: ' . $jsonLength . ' characters');
        
        // Make sure passed $json string is valid JSON string
        if ( !self::validateJSON($json) ) {
            $err_msg = 'Passed JSON string is NOT a VALID JSON string.';
            self::CLIerror($err_msg, true, 5);
            return $err_msg;
        }
        
        // Go through provided json string
        for ($i=0; $i < $jsonLength; $i++) {
            // Get previous character
            if ($i > 0) {
                $prevChar = $json[$i-1];
            } else {
                $prevChar = '';
            }
            // Get current character
            $currentChar = $json[$i];
            // Get next character
            if ($i < ($jsonLength - 1)) {
                $nextChar = $json[$i+1];
            } else {
                $nextChar = '';
            }
            
            // Process current character
            switch ($currentChar) {
                // Handle start of an Object character
                case '{':
                    if (!$inQuotes) {
                        // Handle string in the form: '{}', i.e. empty object
                        if ($nextChar == '}') {
                            $formattedJson .= $currentChar;
                        } else {
                            $indentCount++;
                            $formattedJson .= $currentChar . PHP_EOL . str_repeat($indentString, $indentCount);
                        }
                    } else {
                        $formattedJson .= $currentChar;
                    }
                    break;
                // Handle end of an Object character
                case '}':
                    if (!$inQuotes) {
                        // Handle string in the form: '{}', i.e. empty object
                        if ($prevChar == '{') {
                            $formattedJson .= $currentChar;
                        } else {
                            $indentCount--;
                            $formattedJson .= PHP_EOL . str_repeat($indentString, $indentCount) . $currentChar;
                        }
                    } else {
                        $formattedJson .= $currentChar;
                    }
                    break;
                // Handle start of an Array character
                case '[':
                    if (!$inQuotes) {
                        // Handle string in the form: '[]', i.e. empty array
                        if ($nextChar == ']') {
                            $formattedJson .= $currentChar;
                        } else {
                            $indentCount++;
                            $formattedJson .= $currentChar . PHP_EOL . str_repeat($indentString, $indentCount);
                        }
                    } else {
                        $formattedJson .= $currentChar;
                    }
                    break;
                // Hanlde end of an Array character
                case ']':
                    if (!$inQuotes) {
                        // Handle string in the form: '[]'
                        if ($prevChar == '[') {
                            $formattedJson .= $currentChar;
                        } else {
                            $indentCount--;
                            $formattedJson .= PHP_EOL . str_repeat($indentString, $indentCount) . $currentChar;
                        }
                    } else {
                        $formattedJson .= $currentChar;
                    }
                    break;
                // Handle next element character. Example: ("element_name" : "elementr_value",)
                case ',':
                    if (!$inQuotes) {
                        $formattedJson .= $currentChar . PHP_EOL . str_repeat($indentString, $indentCount);
                    } else {
                        $formattedJson .= $currentChar;
                    }
                    break;
                // Handle start of DOUBLE QUOTES enclosed string
                case '"':
                    if ($prevChar == '\\') {
                        $formattedJson .= $currentChar;
                    } else {
                        $inQuotes = !$inQuotes;
                        $formattedJson .= $currentChar;
                    }
                    break;
                // Any other character will be just added as it is
                default:
                    $formattedJson .= $currentChar;
                    break;
            }
        }// end of formatting $json string
        
        if (self::validateJSON($formattedJson)) {
            self::CLIdebug('JSON string formatted - SUCCESS', false, 6);
        } else {
            self::CLIwarning('JSON string formatted - WITH ERRORS', false, 6);
        }
        
        // Return
        return $formattedJson;
    }

    /**
     * Returns true if $str is Uuid
     *
     * @param string $str            
     * @return boolean
     */
    public static function isUuid($str) {
        if (strlen( $str ) === 36 && $str [8] == '-' && $str [13] == '-' &&
                 $str [18] == '-' && $str [23] == '-') {
            return true;
        } else {
            return false;
        }
    }
}