<?php
/****************************************************************
 * IBM Confidential
*
* SFA100-Collaboration Source Materials
*
* (C) Copyright IBM Corp. 2014
*
* The source code for this program is not published or otherwise
* divested of its trade secrets, irrespective of what has been
* deposited with the U.S. Copyright Office
*
***************************************************************/

require_once('RestoreUCD.php');
require_once('Utils.php');
require_once('BackupUCD.php');

/**
 * Manages snapshot promotion between uDeploy servers
 *
 * @class   Snapshot
 * @author  Marco Bassi         marcobas@ie.ibm.com
*/
class Snapshot extends RestoreUCD {
    
    public function __construct(){
        parent::__construct();
        $this->setReturn('json');
    }
    
    /**
     *
     * Promotes a snapshot from a uDeploy server (origin) to another uDeploy server (destination).
     * Compares component versions in the snapshot between the two uDeploy servers and updates the destination server with the missing version.
     * Creates a copy of the snapshot into the destination server.
     *
     * @param string $snapshot      Snapshot name
     * @param array $config         array(
     *                                  'origin_weburl'         => 'weburl for origin uDeploy server',
     *                                  'origin_username'       => 'username for origin',
     *                                  'origin_password'       => 'password for origin',
     *                                  'destination_weburl'    => 'weburl for destination uDeploy server',
     *                                  'destination_username'  => 'username for destination',
     *                                  'destination_password'  => 'password for destination',
     *                                  'application'           => 'application name', // optional, will use default if missing
     *                                  'insecure'              => 'use insecure connection', // optional, will use default if missing
     *                                  'origin_certificate'    => 'if insecure set to false, certificate to use for origin server',
     *                                  'destination_certificate'    => 'if insecure set to false, certificate to use for destination server',
     *                              )
     * @param array $artifacts      $artifacts['component_name']['base'] = 'path of directory/file to upload to component version'
     *                              $artifacts['component_name']['include'] => array (
     *                                                                              'file to include',
     *                                                                              ...
     *                                                                         )
     *                               $artifacts['component_name']['exclude'] => array (
     *                                                                              'file to exclude',
     *                                                                              ...
     *                                                                              )
     * @return array|boolean
     */
    public function promoteSnapshot($snapshot, $config = null, $artifacts = null, $force = null) {
        if (empty($config)) {
            $config = 'config/servers.config.php';
            if (file_exists($config)) {
                include $config;
            } else {
                Utils::CLIerror("No server configuration set.");
            }
        }
        if (empty($artifacts)) {
            $artifacts = 'config/artifacts.config.php';
            if (file_exists($artifacts)) {
                include $artifacts;
            }
        }
        
        if (!self::validateServerConfig($config)) {
            Utils::CLIerror("Configuration error. Please check the messages above and provide a valid configuration.");
            return false;
        }
        
        $result = array();
        $snapshotFailures = 0;
        
        // Setup connection to first server (origin)
        if (!$this->setupServer('origin',$config)) {
            Utils::CLIerror("Cannot connect to origin server.");
            return false;
        }
        
        // Retrieve snapshot from origin uDeploy server
        Utils::CLIinfo("Retrieving requested snapshot '{$snapshot}' from '{$config['application']}' application on origin server");
        $output = $this->getSnapshotsInApplication($config['application']);
        $appSnapshots = Utils::outputToArray($output);
        $snapshotId = null;
        foreach ($appSnapshots as $appSnapshot) {
            if ($appSnapshot['name'] == $snapshot) {
                $snapshotId = $appSnapshot['id'];
                break;
            }
        }
        if (empty($snapshotId)) {
            Utils::CLIerror("Requested snapshot '{$snapshot}' in '{$config['application']}' application NOT FOUND on origin server.");
            return false;
        }
        
        // Get component and component version list, as defined in the snapshot
        $output = $this->getComponentVersionsInSnapshot($snapshotId);
        $originComponents = Utils::outputToArray($output);
        
        
        // Verify that snapshot doesn't already exist on destination server
        Utils::CLIinfo("Verifying that snapshot '{$snapshot}' doesn't exists on destination server");
        
        if (!$this->setupServer('destination',$config)) {
            Utils::CLIerror("Cannot connect to destination server.");
            return false;
        }
        
        $output = $this->getSnapshotsInApplication($config['application']);
        $checkSnapshots = Utils::outputToArray($output);
        $checkSnapshotId = null;
        foreach ($checkSnapshots as $checkSnapshot) {
            if ($checkSnapshot['name'] == $snapshot) {
                $checkSnapshotId = $checkSnapshot['id'];
                break;
            }
        }
        if (!empty($checkSnapshotId)) {
            Utils::CLIerror("Requested snapshot '{$snapshot}' in '{$config['application']}' application ALREADY EXISTS on destination server.");
            return false;
        }
        
        // Compare components between origin and destination, verify if every component exists in destination. Create if missing
        Utils::CLIinfo("Verifying existing components in destination server");
        $appDestComponents = $this->getComponentsInApplication($config['application']);
        if ($appDestComponents) {
            $appDestComponents = Utils::outputToArray($appDestComponents);
        } else {
            Utils::CLIerror("Error retrieving components for application '{$config['application']}' on destination server.");
            return false;
        }
        
        // All components in destination server
        $allDestComponents = $this->getComponents();
        if ($allDestComponents) {
            $allDestComponents = Utils::outputToArray($allDestComponents);
        } else {
            Utils::CLIerror("Error retrieving all components from destination server.");
            return false;
        }
        
        $toCreateComponents = array();
        $toAddComponents = array();
        // Loop components required for snapshot
        foreach ($originComponents as $component) {
            $componentExists = false;
            $componentIsAdded = false;
            // Check if current component is available in destination server
            foreach ($appDestComponents as $appDestComponent) {
                if ($component['name'] == $appDestComponent['name']) {
                    $componentExists = true;
                    // Check if current component is added to the target application
                    foreach($allDestComponents as $destComponent) {
                        if ($component['name'] == $destComponent['name']) {
                            $componentIsAdded = true;
                            break;
                        }
                    }
                    break;
                }
            }
            
            if (!$componentExists) {
                // Component doesn't exist: need to create it
                $toCreateComponents[$component['id']] = $component['name'];
            }
            if (!$componentIsAdded) {
                // Component is not added to target applicaiton: need to add it
                $toAddComponents[$component['id']] = $component['name'];
            } 
        } // END Loop components
        
        
        if (!empty($toCreateComponents)) {
            Utils::CLIwarning("The following components are missing and need to be created:");
            foreach ($toCreateComponents as $id => $name){
                Utils::CLIout("  $name");
            }
            
            return false;
        } else {
            Utils::CLIout("  All components are available on destination server");
        }
        
        if (!empty($toAddComponents)) {
            Utils::CLIwarning("The following components are missing from '{$config['application']}' application:");
            foreach ($toAddComponents as $id => $name){
                Utils::CLIout("  $name");
            }
            return false;
        } else {
            Utils::CLIout("  All components are associated to '{$config['application']}' application");
        }
        Utils::CLIout("");
        
        // Export artifacts
        Utils::CLIinfo("Exporting snapshot artifacts");
        if (!$this->setupServer('origin',$config)) {
            Utils::CLIerror("Cannot connect to origin server.");
            return false;
        }
        $this->setReturn('file');
        
        $backupOutput = $this->output;
        $cwd = getcwd();
        $this->setOutput("{$cwd}/tmp");
        
        $result['snapshotExport'] = $this->exportSnapshot($snapshotId, $this->application, $snapshot);
        if ($result['snapshotExport'] === false) {
            Utils::CLIerror("Cannot download snapshot zip file from origin server.");
            return false;
        }
        $snapshotZip = "{$this->output}/{$this->application}.{$snapshot}.zip";
        $this->log->info("Snapshot zip: {$snapshotZip}");
        
        $snapshotDir = "{$this->output}/" . time();
        Utils::CLIinfo("Extracting snapshot artifacts to {$snapshotDir}");
        $unzip = "unzip -q -o {$snapshotZip} -d {$snapshotDir}";
        exec($unzip, $output, $return);
        
        $this->log->info(print_r($output,true));
        
        if ($return != 0) {
            Utils::CLIerror("Cannot unzip snapshot artifacts.");
            return false;
        }
        
        $snapshotSubDirs = scandir($snapshotDir);
        
        $this->setOutput($backupOutput);
        
        $this->setReturn('json');
        
        // Preparing new Snapshot data
        Utils::CLIinfo("Preparing snapshot for destination server");
        if (!$this->setupServer('destination',$config)) {
            Utils::CLIerror("Cannot connect to destination server.");
            return false;
        }
        
        $newSnapshot['snapshot']['name'] = $snapshot;
        $newSnapshot['application'] = $config['application'];
        $newSnapshot['snapshot']['description'] = "{$config['application']} snapshot for build '{$snapshot}'";
        $newSnapshot['components'] = array();
        
        Utils::CLIout("  Name: {$newSnapshot['snapshot']['name']}");
        Utils::CLIout("  Application: {$newSnapshot['application']}");
        Utils::CLIout("  Description: {$newSnapshot['snapshot']['description']}");
        $this->log->info("Components: ");
        
        // Compare component versions set in snapshot with destination's component versions. Create if missing
        foreach ($originComponents as $component) {
            $name = $component['name'];
            foreach ($component['desiredVersions'] as $ver) {
                $version = $ver['name'];
                $newSnapshot['components'][$name] = $version;
                $this->log->log("  {$name} ({$version})");
                
                // Verify if any artifact and add it
                foreach ($snapshotSubDirs as $dir) {
                    if (strpos($dir,"{$version}-{$ver['id']}") !== false) {
                        $base =  "{$snapshotDir}/{$dir}";
                        $newSnapshot['artifacts'][$name] = array('base' => $base);
                        $this->log->log("    Artifacts: {$base}");
                        break;
                    }
                }
            }
        }
        Utils::CLIout("\n");
        $result['versionsAndSnapshot'] = self::createVersionsAndSnapshot($newSnapshot, $force);
        
        // Restore server info
        $this->logout();
        return $result;
    }
    
    /**
     * 
     * Validate server configuration used by promoteSnapshot() 
     *
     * @param array $config
     * @return boolean
     */
    public function validateServerConfig($config) {
        if (empty($config)) {
            Utils::CLIerror("Configuration is empty. Please provide a valid configuration");
            return false;
        }
        
        $return = true;
        
        if (empty($config['origin_weburl'])) {
            Utils::CLIerror("No weburl set for origin server");
            $return = false;
        }
        if (empty($config['origin_username'])) {
            // If empty, at login time user will be prompted for providing a username
            Utils::CLIwarning("No username for destination server");
        }
        if (empty($config['origin_password']) && empty($config['origin_authtoken'])) {
            // If empty, at login time user will be prompted for providing a password
            Utils::CLIwarning("No password or authtoken set for destination server");
        }
        if (empty($config['destination_weburl'])) {
            Utils::CLIerror("No weburl set for destination server");
            $return = false;
        }
        if (empty($config['destination_username'])) {
            // If empty, at login time user will be prompted for providing a username
            Utils::CLIwarning("No username for destination server");
        }
        if (empty($config['destination_password']) && empty($config['destination_authtoken'])) {
            // If empty, at login time user will be prompted for providing a password
            Utils::CLIwarning("No password or authtoken set for destination server");
        }
        if (empty($config['application'])) {
            Utils::CLIerror("No application set for snapshot promotion");
            $return = false;
        }
        if ($config['insecure'] === false) {
            if (empty($config['origin_certificate'])) {
                Utils::CLIerror("Secure connection selected but certificate for origin server is missing");
                $return = false;
            }
            if (empty($config['destination_certificate'])) {
                Utils::CLIerror("Secure connection selected but certificate for destination server is missing");
                $return = false;
            }
        }
        return $return;
    }
    
    /**
     *
     * Creates component versions and add them into snapshot. 
     * Accepts an array or file for configuration 
     * 
     * Config (php or json version) should look like
     * 
     * $config['application'] = "Application Name"
     * $config['snapshot'] = array(
     *                          "name" => "Snapshot name",
     *                          "description" => "Snapshot description",
     *                      )
     * $config['components'] = array(
     *                          "Component Name" => "Component Version",
     *                          ...
     *                      )
     *
     * @param array|string $config
     */
    public function createVersionsAndSnapshot($config, $force = true) {
        if (!is_array($config) && file_exists($config)) {
            // Argument is a file
            $ext = pathinfo($config, PATHINFO_EXTENSION);
            if ($ext == 'php') {
                include ($config);
            } else if ($ext == 'json') {
                $config = json_decode(file_get_contents($config),true);
            } else {
                Utils::CLIerror("Invalid file. Please pass a php or json file.");
                return false;
            }
        } else if (!is_array($config)) {
            Utils::CLIerror("Invalid argument. Please use a php or json file or a php array.");
            return false;
        }
        
        // Retrieve component list for application
        $appDestComponents = $this->getComponentsInApplication($config['application']);
        $appDestComponentList = array();
        if ($appDestComponents) {
            $appDestComponents = Utils::outputToArray($appDestComponents);
            foreach ($appDestComponents as $comp) {
                array_push($appDestComponentList, $comp['name']);
            }
        } else {
            Utils::CLIerror("Error retrieving components for application '{$config['application']}' on destination server.");
            return false;
        }
        
        // Check if all components in snapshot are available in application
        $missingComp = false;
        foreach ($config['components'] as $name => $versions) {
            if (! in_array($name, $appDestComponentList)) {
                Utils::CLIerror("Component '{$name}' is not available in application '{$config['application']}'");
                $missingComp = true;
            }
        }
        if ($missingComp) {
            Utils::CLIout("\n");
            Utils::CLIerror("Cannot create Snapshot due to missing components.");
            return false;
        }
        
        // Snapshot creation
        $createSnapshot = false;
        
        $snapshot['name'] = $config['snapshot']['name'];
        $snapshot['application'] = $config['application'];
        $snapshot['description'] = $config['snapshot']['description'];
        $snapshot['versions'] = array();
        
        // Create component version for every component in config
        Utils::CLIinfo("Creating component versions \n");
        foreach ($config['components'] as $name => $versions) {
            if (!is_array($versions)) {
                $versions = array($versions);
            }
            foreach ($versions as $version) {
                // Check if the component version already exists
                if (self::componentVersionExists($name,$version)) {
                    Utils::CLIout("  [SKIPPING] {$name} ({$version})");
                    array_push($snapshot['versions'], array( $name => $version ));
                    continue;
                }
                // Version is missing: create it
                $result['components'][$name] = $this->createVersion($name,$version);
                if ($result['components'][$name] !== false) {
                    Utils::CLIout("  [SUCCESS] {$name} ({$version})");
                    array_push($snapshot['versions'], array( $name => $version ));
                    $createSnapshot = true;
                    
                    // Upload artifacts, if any
                    if (array_key_exists('artifacts', $config)) {
                        $artifacts = $config['artifacts'];
                        if (array_key_exists($name, $artifacts)) {
                            if (array_key_exists('base', $artifacts[$name])) {
                                // Check artifacts base directory
                                $base = $artifacts[$name]['base'];
                                if (!is_dir($base)) {
                                    Utils::CLIerror("Artifacts base directory '{$base}' is missing or not accessible");
                                    return false;
                                }
                                
                                // Check artifacts to include
                                $include = null;
                                $includeArray = array();
                                if (array_key_exists('include', $artifacts[$name])) {
                                    $include = $artifacts[$name]['include'];
                                    $includeError = false;
                                    if (! is_array($include)) {
                                        array_push($include, $includeArray);
                                    } else {
                                        $includeArray = $include;
                                    }
                                    foreach ($includeArray as $includeFile) {
                                        $path = "{$base}/{$includeFile}";
                                        if (! file_exists($path)) {
                                            Utils::CLIerror("Artifacts file/dir '{$includeFile}' is missing or not accessible in base directory '{$base}'");
                                            $includeError = true;
                                        }
                                    }
                                    if ($includeError) {
                                        return false;
                                    }
                                }
                                
                                // Artifacts to exclude
                                $exclude = null;
                                if (array_key_exists('exclude', $artifacts[$name])) {
                                    $exclude = $artifacts[$name]['exclude'];
                                }
                                
                                $result['artifacts'][$name] = $this->addVersionFiles($name, $version, $base, $include, $exclude);
                            } else {
                                $this->log->debug("Trying to upload artifacts for component '{$name}', version '{$version}', but no base path has been set. Skipping.");
                            }
                        }
                    } // End of artifacts upload
                } else {
                    // Error during component version creation
                    Utils::CLIerror("  [FAILURE] {$name} ({$version})");
                }
            }
        }
        
        $result['snapshot'] = true;
        // At least one new component version has been created. Create a new snapshot
        if ($createSnapshot || $force) {
            $file = Utils::createJsonFile($snapshot);
            $result['snapshot'] = Utils::outputToArray($this->createSnapshot($file));
            $this->log->log(print_r($result['snapshot'],true));
            if ($result['snapshot']) {
                Utils::CLIout("\n\n  [SUCCESS] Snapshot created ({$snapshot['name']})");
                Utils::CLIout("\n\n  [LINK] {$this->weburl}/#snapshot/{$result['snapshot']['id']}");
            } else {
                Utils::CLIout("\n\n  [FAILURE] Snapshot was not created ({$snapshot['name']})");
                $result['snapshot'] = false;
            }
            unlink($file);
        } else {
            Utils::CLIout("\n\n  [SKIPPED] Snapshot not created. All component versions are included in a previous snapshot.");
        }
        
        return $result;
    }
    
    /**
     * 
     * Returns true if a component version exists, false if missing
     *
     * @param string $component
     * @param string $version
     */
    public function componentVersionExists($component, $version) {
        
        $versions = $this->getComponentVersions($component);
        $versions = Utils::outputToArray($versions);
        foreach ($versions as $ver) {
            if ($ver['name'] == $version) {
                return true;
            }
        }
        
        
        return false;
    }
    
    /**
     * 
     * DEPRECATED
     *
     */
    private function createAddMissingComponents($toCreateComponents, $toAddComponents, $snapshot, $config){
        $result = array();
        // If any component is missing, create it
        if (!empty($toCreateComponents)) {
            Utils::CLIinfo("Some components in snapshot '{$snapshot}' are missing in destination server. Downloading from origin and creating in destination server.");
            // Disconnect from second server (destination) and connect to first server (origin)
            if (!$this->setupServer('origin',$config)) {
                Utils::CLIerror("Cannot connect to origin server.");
                return false;
            }
        
            // Export missing components into temp files
            Utils::CLIinfo("Retrieving missing components from origin server.");
            $this->setReturn('file');
            $tempComponents = array();
            foreach ($toCreateComponents as $componentId => $componentName) {
                $tempComponentFile = $this->getComponentRest($componentId);
                if ($tempComponentFile) {
                    $tempComponentFile = str_replace(' -o ','',$tempComponentFile);
                    array_push($tempComponents,$tempComponentFile);
                    Utils::CLIdest("{$componentName} => {$tempComponentFile}", false);
                } else {
                    Utils::CLIwarning("Cannot download component '{$componentName}' ({$componentId}) from origin server.");
                }
            }
        
            // Reset output to json
            $this->setReturn('json');
        
            // Disconnect from first server (origin) and connect to second server (destination)
            if (!$this->setupServer('destination',$config)) {
                Utils::CLIerror("Cannot connect to destination server.");
                return false;
            }
        
            foreach ($tempComponents as $component) {
                Utils::CLIinfo("Restoring component from temp file '{$component}'");
                $output = self::restoreComponent($component);
                $result['restore']['component'][$output['component']['name']] = $output;
                if ($output) {
                    // Add component to application
                    $exec = $this->addComponentToApplication($output['component']['id'],$config['application']);
                    $result['restore']['add'][$output['component']['name']] = $exec;
                    if (!$exec) {
                        Utils::CLIwarning("Error when adding component {$output['component']['name']} to application '{$config['application']}'");
                    }
                } else {
                    Utils::CLIwarning("Cannot restore component from file '{$component}'");
                }
            }
        }
        
        foreach ($toAddComponents as $componentId => $componentName) {
            Utils::CLIinfo("Add component {$componentName} to application {$config['application']}");
            $exec = $this->addComponentToApplication($componentId,$config['application']);
            $result['restore']['add'][$componentName] = $exec;
            if (!$exec) {
                Utils::CLIwarning("Error when adding component {$componentName} ({$componentId}) to application '{$config['application']}'");
            }
        }
        
        return $result;
    }
    
    /**
     *
     * Download configuration from one origin server and promote it (upgrade) up to the destination server
     * @param boolean $cleanEnvironments        If true, will remove the environments 
     *                                          imported with the application upgrade, 
     *                                          keeping the existing ones available before upgrade
     *
     */
    public function promoteConfig( $sourceDir = null, $cleanEnvironments = false ){
        if (empty($sourceDir)) {
            $config = 'config/servers.config.php';
            if (file_exists($config)) {
                include $config;
            } else {
                Utils::CLIerror("No server configuration set.");
                return false;
            }
            if (!self::validateServerConfig($config)) {
                Utils::CLIerror("Configuration error. Please check the messages above and provide a valid configuration.");
                return false;
            }
            Utils::CLIinfo("Promoting from server {$config['origin_weburl']}");
        }
        else {
            Utils::CLIinfo("Promoting from directory {$sourceDir}");
        }
        
        $backupsRootDir = $this->output;
        $result = array();
        
        // Set backup/restore objects
        $backup = new BackupUCD();
        $restore = new RestoreUCD();
        
        if (empty($sourceDir)) {
            // Setup connection to first server (origin)
            if (!$backup->setupServer('origin',$config)) {
                Utils::CLIerror("Cannot connect to origin server.");
                return false;
            }
            
            // Get data from origin
            Utils::CLIinfo("Retrieving data from origin server");
            $sourceDir = getcwd() . '/tmp/' . date('Ymd-His');
            $backup->setOutput($sourceDir);
            
            $result['backup'] = $backup->backup();
            
            // Setup connection to second server (destination)
            if (!$restore->setupServer('destination',$config)) {
                Utils::CLIerror("Cannot connect to destination server.");
                return false;
            }
        }
        
        $restore->setOutput($sourceDir);
        $this->setOutput($sourceDir);
        
        // Get existing environments
        if ($cleanEnvironments) {
            Utils::CLIinfo("Retrieve existing environments");
            $this->setReturn('json');
            $existingEnvs = array();
            $apps = Utils::outputToArray($this->getApplications());
            foreach ($apps as $app) {
                $envs = Utils::outputToArray($this->getEnvironmentsInApplication($app['id']));
                foreach ($envs as $env) {
                    if (!in_array($env['id'], $existingEnvs)) {
                        array_push($existingEnvs, $env['id']);
                        Utils::CLIout("{$env['name']} ({$env['id']})");
                    }
                }
            }
        }
        
        // Upgrade generic processes
        Utils::CLIinfo("Upgrading processes");
        $processes = "{$this->output}/processes";
        $result['upgrade']['processes'] = $restore->restoreAllProcesses($processes, true);
        
        // Upgrade components
        Utils::CLIinfo("Upgrading components");
        $components = "{$this->output}/components";
        $result['upgrade']['components'] = $restore->importAllComponents($components, true);
        
        // Upgrade application
        Utils::CLIinfo("Upgrading applications");
        $applications = "{$this->output}/applications";
        $result['upgrade']['applications'] = $restore->upgradeAllApplications($applications);
        
        // Cleanup unwanted new environments
        if ($cleanEnvironments) {
            Utils::CLIinfo("Cleaning environments");
            $this->setReturn('json');
            $apps = Utils::outputToArray($this->getApplications());
            foreach ($apps as $app) {
                Utils::CLIout("  {$app['name']}");
                $envs = Utils::outputToArray($this->getEnvironmentsInApplication($app['id']));
                foreach ($envs as $env) {
                    if (!in_array($env['id'], $existingEnvs)) {
                        $result['remove_envs'][$env['id']] = $this->deleteEnvironment($env['id']);
                        Utils::CLIout("    [REMOVE] {$env['name']} ({$env['id']})");
                    }
                }
            }
        }
        
        Utils::CLIout("\n");
        
        // Restore backup dir
        $this->setOutput($backupsRootDir);
        
        return $result;
    }
}