<?php
/****************************************************************
 * IBM Confidential
 *
 * SFA100-Collaboration Source Materials
 *
 * (C) Copyright IBM Corp. 2014
 *
 * The source code for this program is not published or otherwise
 * divested of its trade secrets, irrespective of what has been
 * deposited with the U.S. Copyright Office
 *
 ***************************************************************/

require_once('Utils.php');
require_once('Logger.php');

/**
 * Implements udclient APIs in PHP
 * 
 * @class   UrbanCodeClient
 * @author  Marco Bassi         		marcobas@ie.ibm.com
 * @author  Kostadin Ivanov     		kostadii@ie.ibm.com
 * @author  Moises Dominguez Garcia     moisesdg@ie.ibm.com
 */
class UrbanCodeClient{
    protected $username;
    protected $password; // Deprecated
    protected $cookie_file; // Deprecated
    protected $authtoken;
    protected $b64_login;
    public $weburl;
    public $output;
    public $application;
    public $environment;
    public $udcpath;
    protected $java_home;
    protected $actions;
    protected $restActions;
    protected $args;
    protected $return;
    protected $destination;
    protected $connected;
    protected $insecure;
    protected $certificate;
    // User Impersonation related variables
    protected $apply_impersonation;
    protected $impersonation_config;
    
    // Logger
    protected $log;
    
    public function __construct(){
        $this->log = new Logger();
        $this->setup();
        $this->setReturn();
        $this->setActions();
        //$this->login();
    }
    
    /* Utils */
    
    /**
     * Get the weburl
     */
    
    public function getWeburl(){
        return $this->weburl;
    }
    
    /**
     * Setup the basic config values that will be used by this class
     */
    protected function setup(){
        include 'config/ucd.config.php';
        $this->username = $config['username'];
        $this->password = $config['password']; // Deprecated
        $this->cookie_file = $config['cookie_file']; // Deprecated
        $this->authtoken = $config['authtoken'];
        if (!empty($this->authtoken)) {
            $this->b64_login = base64_encode('PasswordIsAuthToken:{"token":"' . $this->authtoken . '"}');
        } else {
            $this->b64_login = null;
        }
        
        $this->weburl = $config['weburl'];
        $this->application = $config['application'];
        $this->environment = $config['environment'];
        $this->insecure = $config['insecure'];
        $this->certificate = $config['certificate'];
        if (empty($config['udcpath'])){
            $this->udcpath = '.';
        } else {
            $this->udcpath = $config['udcpath'];
        }
        // Chek if path for udcclient ends with '/'
        if ( substr($this->udcpath, -1) == '/' ){
            $this->udcpath = substr($this->udcpath, 0, -1);
        }
        // Check if it's a valid udcpath
        $checkUdclient = $this->checkUdclient();
        if ( !$checkUdclient ){
            Utils::CLIerror("Path for udclient is not valid. Missing zip or udclient files.");
            exit (1);
        }
        $this->java_home = $config['java_home'];
        $this->destination = '';
        $this->output = $config['output'];
        
        // Chek if path for folder ends with '/'
        if ( substr($this->output, -1) == '/' ){
            $this->output = substr($this->output, 0, -1);
        }
        
        // Default output to current working directory
        if ( trim( $this->output) == '' ) {
            Utils::CLIinfo('Default "output" to: ' . getcwd() . '/backups');
            if ( !$this->setOutput( getcwd() . '/backups' ) ) {
                Utils::CLIerror('There was a problem with setting up of default "output" directory.');
                Utils::CLIinfo('For more information please check "ucommand/README" file', false, 1);
                exit (1);
            }
        }
        
        // Set silent mode for outputs
        Utils::setSilent( $config['silent'] );
        // Set JAVA_HOME
        $result = Utils::setJavaHome( $this->java_home );
        if (!$result){
            Utils::CLIerror("Path for JAVA_HOME was not set correctly or you don't have permission on it.");
            exit (1);
        }
        // Set - Apply Impersonation flag
        $this->apply_impersonation = $config['apply_impersonation'];
        
        // Set Logger, if config file exists
        if (file_exists('config/logger.config.php')) {
            include('config/logger.config.php');
            $this->log->setLogFile($config['log']);
            $this->log->setDebug($config['debug']);
        }
    }
    
    public function setOutput($dir) {
        if (is_dir($dir)) {
            $this->output = $dir;
            return true;
        } else if (is_file($dir)) {
            return false;
        } else {
            if (mkdir($dir,0744,true)) {
                $this->output = $dir;
                return true;
            } else {
                return false;
            }
        }
    }
    
    /**
     * Check if the action passed as argument to exec is valid
     */
    public function isValidAction($action){
        $actions = $this->actions;
        return in_array($action, $actions);
    }
    
    public function isValidRestAction($action){
        $actions = $this->restActions;
        return in_array($action, $actions);
    }
    
    public function getActions(){
        return $this->actions;
    }
    
    public function getRestActions(){
        return $this->restActions;
    }
    
    /**
     * Populate the array of valid actions, which is used by isValidAction
     */
    protected function setActions(){
        $actionsFile = 'include/actions.php';
        if (is_file($actionsFile)) {
            include ($actionsFile);
            $this->actions = $actions;
            $this->restActions = $restActions;
            return true;
        } else {
            return false;
        }
    }
    
    protected function setUsername(){
        Utils::CLIout("Username: ");
        $username = readline( "" );
        $this->username = $username;
    }
    
    protected function setPassword(){
        Utils::CLIout("Password (WON'T BE HIDDEN): ");
        $password = readline( "" );
        $this->password = $password;
    }
    
    /**
     * @param string    $type
     * Set if the API are returning the json value or echoing the output 
     */
    public function setReturn($type = null){
        if (empty($type) || (strtolower($type) != 'json' && strtolower($type) != 'file' )){
            $this->return = 'json';
        } else {
            $this->return = $type;
        }
    }
    
    public function setArgs($arguments) {
        $this->args = $arguments;
    }
    
    /**
     * Set User Impersonation configuration, using: Utils::getUserImpersonationConfig(),
     * and return Boolean TRUE when configuration was obtained and valid, or
     * FALSE otherwise.
     * 
     * PLEASE NOTE: in case when this function is called more than once, 
     *              and the User Impersonation has been loaded with SUCCESS,
     *              THIS function will JUST return TRUE, otherwise it will return FALSE.
     * 
     * @return boolean
     */
    protected function setUpUserImpersonationConfig () {
        if ($this->impersonation_config === null) {
            $this->impersonation_config = Utils::getUserImpersonationConfig();
        }/* else:
          * $this->impersonation_config has already been loaded OR
          * there was a problem with the Configuration and
          * $this->impersonation_config was sert to FALSE,
          * so we don't call Utils::getUserImpersonationConfig(); MULTIPLE times.
          */
        
        // Return
        return ((is_null($this->impersonation_config) || is_bool($this->impersonation_config) ) ? false : true);
    }
    
    /**
     * @param string    $action
     * 
     * @return bool
     */
    public function exec($action){
        // Prepare the command to be executed
        if (!empty($this->authtoken) && ! empty($this->username)) {
            // Access with auth-token doesn't need an existing connection
            $command = "{$this->udcpath}/udclient -weburl {$this->weburl} -username {$this->username} -authtoken {$this->authtoken} {$action}";
        } else {
            // Check if a connection exists
            if ( !$this->connected ){
                $this->login();
            }
            $command = "{$this->udcpath}/udclient -weburl {$this->weburl} {$action}";
        }
        // Log details for the command to be executed
        $this->log->info(
            'Execute command: ' . PHP_EOL
                . '    Type  : weburl' . PHP_EOL
                . '    Path  : ' . $this->udcpath . '/udclient' . PHP_EOL
                . '    WebURL: ' . $this->weburl . PHP_EOL
                . '    Action: ' . $action
        );
        
        // Enable Debug
        if ($this->log->isDebug()){
            $debug = ' --verbose';
        } else {
            $debug = '';
        }
        
        // Process $command
        $processResult = self::processServerRequest($command.$debug);
        
        /*
         * Log success / fail to process request and return result
         * NOTE: error logs and messages to console are handled in: processServerRequest()
         */
        if ( $processResult === false ) {
            $this->log->error('Command processed with ERRORS');
        } else {
            $this->log->info('Command executed - SUCCESS');
        }
        
        // Return result
        return $processResult;
    }
    
    /**
     * 
     * Executes curl commands using REST APIs
     *
     * @param string $action
     */
    public function execRest($action){
            // Cookies
            if (!empty($this->cookie_file) && is_file($this->cookie_file) && !is_dir($this->cookie_file)) {
                $cookies = " -b '{$this->cookie}' ";
            } else {
                $cookies = '';
            }
            
            // Username:Password | Base_64 login
            if (!empty($this->b64_login)) {
                $login = "-H \"Authorization: Basic {$this->b64_login}\" ";
            } else {
                $login = "-u {$this->username}:{$this->password}";
            }
            
            // Secure login
            if ($this->insecure) {
                $ssl = "-k";
            } else {
                if (!empty($this->certificate) && file_exists($this->certificate)){
                    $ssl = "--cacert {$this->certificate}";
                } else {
                    Utils::CLIerror("Certificate is missing or cannot be read.");
                    return false;
                }
            }
            
            // Enable Debug
            if ($this->log->isDebug()){
                $debug = '--verbose';
            } else {
                $debug = '-s';
            }
            
            $command = "curl {$cookies} {$login} {$ssl} {$action} {$debug}";
            
            // Log details for the command to be executed
            $this->log->info(
                'Execute command: ' . PHP_EOL
                    . '    Type  : curl' . PHP_EOL
                    . '    SSL   : ' . $ssl . PHP_EOL
                    . '    Cookie: ' . $cookies . PHP_EOL
                    . '    Action: ' . $action
            );
            
            // Process $command
            $processResult = self::processServerRequest($command);
            
            /*
             * Log success / fail to process request
             * NOTE: error logs and messages to console are handled in: processServerRequest()
             */
            if ( $processResult === false ) {
                $this->log->error('Command processed with ERRORS');
            } else {
                $this->log->info('Command executed - SUCCESS');
            }
            
            // Return result
            return $processResult;
    }
    
    /**
     * Login into uDeploy using username/password passade as argument or provided by config file or from user cli input
     */
    public function login($username=null, $password=null){
        // Verify if a connection already exists
        if( $this->connectionCheck() ){
            if ( empty($this->username) || empty($this->password) ){
                // If there's a connection, but username or password is empty, logout and request login info again 
                $this->logout();
            } else {
                return true;
            }
        }
        // Max 3 attempts
        $counter = 3;
        do{
            $counter--;
            if ( empty($username) ){
                if (empty($this->username)){
                    $this->setUsername();
                    $this->setPassword();
                    $password = null;
                }
            } else {
                $this->username = $username;
            }
            if ( empty($password) ){
                if (empty($this->password)){
                    $this->setPassword();
                }
            } else {
                $this->password = $password;
            }
            $command = "{$this->udcpath}/udclient -username {$this->username} -password {$this->password} -weburl {$this->weburl} login";
            $return = exec($command);
            if ( empty($return) ){
                Utils::CLIresult("Login SUCCESFUL");
            } else {
                Utils::CLIresult("Login FAILED");
                if (!$counter) {
                    Utils::CLIerror("Maximum number of login attempts (3) reached. Exit.");
                    exit (6);
                }
                Utils::CLIout("Please, try again [{$counter}]");
                // Reset login info
                $this->password = null;
                $this->connected = false;
            }
        } while ( !empty($return) && $counter > 0 );
        
        $this->connected = true;
        return true;
    }
    
    /**
     * Logout from uDeploy
     */
    
    public function logout(){
        $command = "{$this->udcpath}/udclient -weburl {$this->weburl} logout";
        return exec($command);
    }
    
    /**
     * Check if a connection already exists
     */
    private function connectionCheck(){
        $command = "{$this->udcpath}/udclient -weburl {$this->weburl} login;";
        $return = exec($command);
        if ( empty($return) ){
            $this->connected = true;
            return true;
        } else {
            $this->connected = false;
            return false;
        }
    }
    
    /**
     * Verify that udclient files are in place.
     */
    protected function checkUdclient(){
        // Check if udclient files are valid
        $udclient_jar = $this->udcpath . '/udclient.jar';
        $udclient_jar = file_exists($udclient_jar);
        $udclient_exe = $this->udcpath . '/udclient';
        $udclient_exe = file_exists($udclient_exe);
        $udclient_cmd = $this->udcpath . '/udclient.cmd';
        $udclient_cmd = file_exists($udclient_cmd);
        
        // If any of the files is missing, check and extract zip
        if ( !$udclient_jar || !$udclient_exe || !$udclient_cmd ) {
            Utils::CLIdebug("Udclient files are missing: checking for zip");
            // Check if udclient zip exists and extract it
            $udclient_zip = $this->udcpath . '/udclient.zip';
            if ( !file_exists($udclient_zip) ) {
                Utils::CLIinfo("Udclient zip file {$udclient_zip} was not found. Downloading...");
                $download = $this->downloadUdclient();
                if ($download === false) {
                    Utils::CLIerror("Cannot download udclient.zip form UrbanCode Deploy server. Try to re-configure uCommand running ./setup");
                    return false;
                }
            }
            Utils::CLIdebug("Unzipping {$udclient_zip}");
            $command = "unzip -q -o {$udclient_zip} -d {$this->udcpath}";
            exec($command, $output, $result);
            $this->udcpath = "{$this->udcpath}/udclient";
            if ($result === 0) {
                // Check if extracted files are valid
                return $this->checkUdclient();
            } else {
                Utils::CLIerror("An error occured while extracting {$udclient_zip}");
                return false;
            }
        }
        return true;
    }
    
    public function downloadUdclient() {
        $action = "{$this->weburl}/tools/udclient.zip -o udclient.zip";
        return $this->execRest($action);
    }
    
/********************************************************************************************************************************
 * APIs - GET
 * http://pic.dhe.ibm.com/infocenter/ucdeploy/v6r0/index.jsp?topic=%2Fcom.ibm.udeploy.reference.doc%2Ftopics%2Fcli_commands.html
 *********************************************************************************************************************************/
    
    /**
     * @param string        $agent
     * @return string
     * 
     * Will return false if argument is incorrect
     */
    public function getAgent($agent = null) {
        if (empty($agent)){
            if (!empty($this->args[0])) {
                $agent = $this->args[0];
            } else {
                return false;
            }
        }
        if ($this->return == 'file') {
            $this->destination = " > {$this->output}/{$agent}.json";
        }
        $action = "getAgent -agent {$agent}{$this->destination}";
        return $this->exec($action);
    }
    
    /**
     * @param string        $pool
     * @return string
     * 
     * Will return false if argument is incorrect
     */
    public function getAgentPool($pool = null) {
        if (empty($pool)){
            if (!empty($this->args[0])) {
                $pool = $this->args[0];
            } else {
                return false;
            }
        }
        if ($this->return == 'file') {
            $this->destination = " > '{$this->output}/{$pool}.json'";
        }
        $action = "getAgentPool -pool {$pool}{$this->destination}";
        return $this->exec($action);
    }
    
    /**
     * @param bool        $active
     * @param bool        $inactive
     * @return string
     * 
     * Will return false if argument is incorrect
     */
    public function getAgentPools($active = null, $inactive = null) {
        if (empty($active) || !is_bool($active)){
            if (!empty($this->args[0])) {
                $active = $this->args[0];
            } else {
                $active = true;
            }
        }
        if (empty($inactive) || !is_bool($inactive)){
            if (!empty($this->args[1])) {
                $inactive = $this->args[1];
            } else {
                $inactive = true;
            }
        }
        $active = ' -active ' . $active;
        $inactive = ' -inactive ' . $inactive;
        
        if ($this->return == 'file') {
            $this->destination = " > '{$this->output}/agentPools.json'";
        }
        $action = "getAgentPools{$active}{$inactive}{$this->destination}";
        return $this->exec($action);
    }
    
    /**
     * @param string        $agent
     * @param string        $property
     * @return string
     * 
     * Will return false if argument is incorrect
     */
    public function getAgentProperty($agent = null, $property = null) {
        if (empty($agent)){
            if (!empty($this->args[0])) {
                $agent = $this->args[0];
            } else {
                return false;
            }
        }
        if (empty($property)){
            if (!empty($this->args[1])) {
                $agent = $this->args[1];
            } else {
                return false;
            }
        }
        if ($this->return == 'file') {
            $this->destination = " > '{$this->output}/{$agent}.{$property}.json'";
        }
        $action = "getAgentProperty -agent {$agent} -name {$property}{$this->destination}";
        return $this->exec($action);
    }
    
    /**
     * @param bool        $active
     * @param bool        $inactive
     * @return string
     * 
     * Will return false if argument is incorrect
     */
    public function getAgents($active = null, $inactive = null) {
        if (empty($active) || !is_bool($active)){
            if (!empty($this->args[0])) {
                $active = $this->args[0];
            } else {
                $active = true;
            }
        }
        if (empty($inactive) || !is_bool($inactive)){
            if (!empty($this->args[1])) {
                $inactive = $this->args[1];
            } else {
                $inactive = true;
            }
        }
        $active = ' -active ' . $active;
        $inactive = ' -inactive ' . $inactive;
        
        if ($this->return == 'file') {
            $this->destination = " > '{$this->output}/agents.json'";
        }
        $action = "getAgents{$active}{$inactive}{$this->destination}";
        return $this->exec($action);
    }
    
    /**
     * @param string    $application
     * @return string
     * 
     * Will return false if argument is incorrect
     */
    public function getApplication($application = null){
        if (empty($application)){
            if (!empty($this->args[0])) {
                $application = $this->args[0];
            } else if (!empty($this->application)){
                $application = $this->application;
            } else {
                return false;
            }
        }
        if ($this->return == 'file') {
            $this->destination = " > '{$this->output}/{$application}.json'";
        }
        $action = "getApplication -application {$application}{$this->destination}";
        return $this->exec($action);
    }
    
    public function exportSnapshot($snapshot = null, $application = null, $snapshotName = null) {
        if (empty($snapshot)){
            if (!empty($this->args[0])) {
                $snapshot = $this->args[0];
            } else {
                return false;
            }
        }
        if (empty($application)){
            if (!empty($this->args[1])) {
                $application = $this->args[1];
            } else if (!empty($this->application)){
                $application = $this->application;
            } else {
                return false;
            }
        }
        if (empty($snapshotName)){
            $snapshotName = $snapshot;
        }
        
        if ($this->return == 'file') {
            $this->destination = " > '{$this->output}/{$application}.{$snapshotName}.zip'";
        }
        
        $action = "{$this->weburl}/rest/deploy/application/{$application}/exportWithArtifacts?snapshotIds={$snapshot}{$this->destination}";
        return $this->execRest($action);
    }
    
    /**
     *
     * Export a generic process
     *
     * @param string $process
     * @return boolean|string
     */
    public function exportProcess($process = null) {
        if (empty($process)){
            if (!empty($this->args[0])) {
                $process = $this->args[0];
            } else {
                return false;
            }
        }
        
        if ($this->return == 'file') {
            $this->destination = " -o '{$this->output}/{$process}.json'";
        }
        
        $action = "{$this->weburl}/rest/process/{$process}/export {$this->destination}";
        return $this->execRest($action);
    }
    
    /**
     * 
     * Get tags for a specific type. if a name is set, get that particular tag
     *
     * @param string $type
     * @param string $name
     * @return boolean|string
     */
    public function getTag($type = null, $name = '*') {
        if (empty($type)){
            if (!empty($this->args[0])) {
                $type = $this->args[0];
            } else {
                return false;
            }
        }
        if ($this->return == 'file') {
            $this->destination = " > '{$this->output}/{$type}.tags.json'";
        }
        $action = "{$this->weburl}/rest/tag/type/{$type}/?name={$name} {$this->destination}";
        return $this->execRest($action);
    }
    
    public function getApplicationProcesses($application = null) {
        if (empty($application)){
            if (!empty($this->args[0])) {
                $application = $this->args[0];
            } else if (!empty($this->application)){
                $application = $this->application;
            } else {
                $this->log->error(__CLASS__ . '::' . __FUNCTION__ .': application: Name/ID is REQUIRED.');
                return false;
            }
        }
        
        if ($this->return == 'file') {
            $this->destination = " > '{$this->output}/{$application}.processes.json'";
        }
        
        $action = $this->weburl . '/rest/deploy/application/\'' . $application
                . '\'/processes/false' . $this->destination;
        return $this->exec($action);
    }
    
    /**
     * @param string        $application
     * @param string        $process
     * @return string
     * 
     * Will return false if argument is incorrect
     */
    public function getApplicationProcess($application = null, $process = null) {
        if (empty($application)){
            if (!empty($this->args[0])) {
                $application = $this->args[0];
            } else if (!empty($this->application)){
                $application = $this->application;
            } else {
                return false;
            }
        }
        if (empty($process)){
            if (!empty($this->args[1])) {
                $process = $this->args[1];
            } else {
                return false;
            }
        }
        if ($this->return == 'file') {
            $this->destination = " > '{$this->output}/{$process}.json'";
        }
        $action = "getApplicationProcess -application '{$application}' -applicationProcess '{$process}'{$this->destination}";
        return $this->exec($action);
    }
    
    public function getApplicationProcessRest($appProcessID = null, $appProcessName = null) {
        if (empty($appProcessID)){
            if (!empty($this->args[0])) {
                $appProcessID = $this->args[0];
            } else {
                $this->log->error(__CLASS__ . '::' . __FUNCTION__ .': Application Process ID is REQUIRED');
                return false;
            }
        }
        
        if (empty($appProcessName)){
            if (!empty($this->args[1])) {
                $appProcessName = $this->args[1];
            } else {
                // Set the ProcessID as Process name for the Process backup file
                $appProcessName = $appProcessID;
            }
        }
        
        if ($this->return == 'file') {
            $this->destination = " > '{$this->output}/{$appProcessName}.json'";
        }
        
        $action = $this->weburl . '/rest/deploy/applicationProcess/'
                . $appProcessID . '/-1' . $this->destination;
        return $this->execRest($action);
    }
    
    /**
     * @param string        $processRequest
     * @return string
     * 
     * Will return false if argument is incorrect
     */
    public function getApplicationProcessRequestStatus($processRequest = null) {
        if (empty($processRequest)){
            if (!empty($this->args[0])) {
                $processRequest = $this->args[0];
            } else {
                return false;
            }
        }
        if ($this->return == 'file') {
            $this->destination = " > '{$this->output}/{$processRequest}.json'";
        }
        $action = "getApplicationProcessRequestStatus -request {$processRequest}{$this->destination}";
        return $this->exec($action);
    }
    
    /**
     * @param string        $application
     * @return string
     * 
     * Will return false if argument is incorrect
     */
    public function getApplicationProperties($application = null) {
        if (empty($application)){
            if (!empty($this->args[0])) {
                $application = $this->args[0];
            } else if (!empty($this->application)){
                $application = $this->application;
            } else {
                return false;
            }
        }
        if ($this->return == 'file') {
            $this->destination = " > '{$this->output}/{$application}.properties.json'";
        }
        $action = "getApplicationProperties -application {$application}{$this->destination}";
        return $this->exec($action);
    }
    
    /**
     * @param string        $application
     * @param string        $property
     * @return string
     * 
     * Will return false if argument is incorrect
     */
    public function getApplicationProperty($application = null, $property = null) {
        if (empty($application)){
            if (!empty($this->args[0])) {
                $application = $this->args[0];
            } else if (!empty($this->application)){
                $application = $this->application;
            } else {
                return false;
            }
        }
        if (empty($property)){
            if (!empty($this->args[1])) {
                $property = $this->args[1];
            } else {
                return false;
            }
        }
        if ($this->return == 'file') {
            $this->destination = " > '{$this->output}/{$application}.{$property}.json'";
        }
        $action = "getApplicationProperty -application {$application} -name {$property}{$this->destination}";
        return $this->exec($action);
    }
    /**
     * Gel all application in uDeploy
     */
    public function getApplications(){
        if ($this->return == 'file') {
            $this->destination = " > {$this->output}/applications.json";
        }
        $action = "getApplications{$this->destination}";
        return $this->exec($action);
    }
    
    /**
     * @param string        $application
     * @return string
     * 
     * Will return false if argument is incorrect
     */
    public function getBlueprintsInApplication($application = null) {
        if (empty($application)){
            if (!empty($this->args[0])) {
                $application = $this->args[0];
            } else if (!empty($this->application)){
                $application = $this->application;
            } else {
                return false;
            }
        }
        if ($this->return == 'file') {
            $this->destination = " > '{$this->output}/{$application}.blueprints.json'";
        }
        $action = "getBlueprintsInApplication -application {$application}{$this->destination}";
        return $this->exec($action);
    }
    
    /**
     * @param string        $blueprint
     * @return string
     * 
     * Will return false if argument is incorrect
     */
    public function getBlueprintNodePropertiesTemplate($blueprint = null) {
        if (empty($blueprint)){
            if (!empty($this->args[0])) {
                $blueprint = $this->args[0];
            } else {
                return false;
            }
        }
        if (!empty($this->application)){
            $application = ' -application '.$this->application;
        } else {
            $application = '';
        }
        if ($this->return == 'file') {
            $this->destination = " > '{$this->output}/{$blueprint}.json'";
        }
        $action = "getBlueprintNodePropertiesTemplate -blueprint {$blueprint}{$application}{$this->destination}";
        return $this->exec($action);
    }
    
    /**
     * @param string    $component
     * @return string
     * 
     * Will return false if argument is incorrect
     */
    public function getComponent($component = null){
        if (empty($component)) {
            if (!empty($this->args[0])) {
                $component = $this->args[0];
            } else {
                return false;
            }
        }
        if ($this->return == 'file') {
            $this->destination = " > '{$this->output}/{$component}.json'";
        }
        $action = "getComponent -component '{$component}'{$this->destination}";
        return $this->exec($action);
    }
    
    /**
     * @param string        $component
     * @param string        $environment
     * @param string        $application
     * @return string
     * 
     * Will return false if argument is incorrect
     */
    public function getComponentEnvironmentProperties($component = null, $environment = null, $application = null){
        if (empty($component)){
            if (!empty($this->args[0])) {
                $component = $this->args[0];
            } else {
                return false;
            }
        }
        if (empty($environment)){
            if (!empty($this->args[1])) {
                $environment = $this->args[1];
            } else if (!empty($this->environment)){
                $environment = $this->environment;
            } else {
                return false;
            }
        }
        if (empty($application)){
            if (!empty($this->args[2])) {
                $application = $this->args[2];
            } else if (!empty($this->application)){
                $application = $this->application;
            } else {
                return false;
            }
        }
        if ($this->return == 'file') {
            $this->destination = " > '{$this->output}/{$environment}.{$component}.properties.json'";
        }
        $action = "getComponentEnvironmentProperties -component '{$component}' -environment {$environment} -application {$application}{$this->destination}";
        return $this->exec($action);
    }
    
    /**
     * @param string    $component
     * @param string    $process
     * @return string
     * 
     * Will return false if argument is incorrect
     */
    public function getComponentProcess($component = null, $process = null){
        if (empty($component)){
            if (!empty($this->args[0])) {
                $component = $this->args[0];
            } else {
                return false;
            }
        }
        if (empty($process)){
            if (!empty($this->args[1])) {
                $process = $this->args[1];
            } else {
                return false;
            }
        }
        if ($this->return == 'file') {
            $this->destination = " > '{$this->output}/{$component}.{$process}.json'";
        }
        $action = "getComponentProcess -component '{$component}' -componentProcess {$process}{$this->destination}";
        return $this->exec($action);
    }
    
    /**
     * @param string        $component
     * @return string
     * 
     * Will return false if argument is incorrect
     */
    public function getComponentProperties($component = null){
        if (empty($component)){
            if (!empty($this->args[0])) {
                $component = $this->args[0];
            } else {
                return false;
            }
        }
        if ($this->return == 'file') {
            $this->destination = " > '{$this->output}/{$environment}.{$component}.properties.json'";
        }
        $action = "getComponentProperties -component '{$component}'{$this->destination}";
        return $this->exec($action);
    }
    
    /**
     * @param string        $component
     * @param string        $property
     * @return string
     * 
     * Will return false if argument is incorrect
     */
    public function getComponentProperty($component = null, $property = null){
        if (empty($component)){
            if (!empty($this->args[0])) {
                $component = $this->args[0];
            } else {
                return false;
            }
        }
        if (empty($property)){
            if (!empty($this->args[1])) {
                $property = $this->args[1];
            } else {
                return false;
            }
        }
        if ($this->return == 'file') {
            $this->destination = " > '{$this->output}/{$component}.{$property}.json'";
        }
        $action = "getComponentProperty -component '{$component}' -name {$property}{$this->destination}";
        return $this->exec($action);
    }
    
    public function getComponents(){
        if ($this->return == 'file') {
            $this->destination = " > '{$this->output}/components.json'";
        }
        $action = "getComponents{$this->destination}";
        return $this->exec($action);
    }
    
    /**
     * @param string    $application
     * @return string
     * 
     * Will return false if argument is incorrect
     */
    public function getComponentsInApplication($application = null){
        if (empty($application)){
            if (!empty($this->args[0])) {
                $application = $this->args[0];
            } else if (!empty($this->application)){
                $application = $this->application;
            } else {
                return false;
            }
        }
        if ($this->return == 'file') {
            $this->destination = " > '{$this->output}/{$application}.components.json'";
        }
        $action = "getComponentsInApplication -application {$application}{$this->destination}";
        return $this->exec($action);
    }
    
    /**
     * Add a component to an application
     *
     * @param string        $component
     * @param string        $application
     * @return boolean|string
     */
    public function addComponentToApplication ($component = null, $application = null){
        if (empty($component)){
            if (!empty($this->args[0])) {
                $component = $this->args[0];
            } else {
                return false;
            }
        }
        if (empty($application)){
            if (!empty($this->args[1])) {
                $application = $this->args[1];
            } else if (!empty($this->application)){
                $application = $this->application;
            } else {
                return false;
            }
        }
        $action = "addComponentToApplication -component '{$component}' -application '{$application}'";
        return $this->exec($action);
    }
    
    /**
     * 
     * Associate a team to a component
     *
     * @param string $component
     * @param string $team
     * @return boolean|string
     */
    public function addComponentToTeam( $component = null, $team = null ) {
        if (empty($component)){
            if (!empty($this->args[0])) {
                $component = $this->args[0];
            } else {
                return false;
            }
        }
        if (empty($team)){
            if (!empty($this->args[1])) {
                $team = $this->args[1];
            } else {
                return false;
            }
        }
        $action = "addComponentToTeam -component '{$component}' -team '{$team}'";
        return $this->exec($action);
    }
    
    /**
     * 
     * Associate a team to a resource
     *
     * @param string $resource
     * @param string $team
     * @return boolean|string
     */
    public function addResourceToTeam( $resource = null, $team = null ) {
        if (empty($resource)){
            if (!empty($this->args[0])) {
                $resource = $this->args[0];
            } else {
                return false;
            }
        }
        if (empty($team)){
            if (!empty($this->args[1])) {
                $team = $this->args[1];
            } else {
                return false;
            }
        }
        $action = "addResourceToTeam -resource '{$resource}' -team '{$team}'";
        return $this->exec($action);
    }
    
    /**
     * 
     * Enter description here ...
     *
     * @param string $component
     * @param string $tag
     * @param string $color
     * @param string $description
     */
    public function addTagToComponent($component = null, $tag = null, $color = null, $description = null) {
        if (empty($component)){
            if (!empty($this->args[0])) {
                $component = $this->args[0];
            } else {
                return false;
            }
        }
        if (empty($tag)){
            if (!empty($this->args[1])) {
                $tag = $this->args[1];
            } else {
                return false;
            }
        }
        
        $optional = '';
        if (!empty($color)){
            $optional .= " -color {$color}";
        }
        if (!empty($description)) {
            $optional .= " -description {$description}";
        }
        
        $action = "addTagToComponent -component '{$component}' -tag '{$tag}'{$optional}";
        return $this->exec($action);
    }
    
    /**
     * 
     * Export an application to a file or returns a json
     *
     * @param string $application
     * @return boolean|string
     */
    public function getApplicationRest($application = null) {
        if (empty($application)){
            if (!empty($this->args[0])) {
                $application = $this->args[0];
            } else if (!empty($this->application)){
                $application = $this->application;
            } else {
                return false;
            }
        }
        if ($this->return == 'file') {
            $this->destination = " -o '{$this->output}/{$application}.json'";
        }
        $action = "{$this->weburl}/rest/deploy/application/{$application}/export {$this->destination}";
        return $this->execRest($action);
    }
    
    /**
     * 
     * Export a blueprint giving information about application, template and blueprint itself. 
     * More powerful tool compared to getblueprintsInApplication
     *
     * @param string $blueprintId
     * @param string $blueprintName
     * @return boolean|string
     */
    public function getBlueprint($blueprintId = null, $blueprintName = null) {
        if (empty($blueprintId)){
            if (!empty($this->args[0])) {
                $blueprintId = $this->args[0];
            } else {
                return false;
            }
        }
        if ($this->return == 'file') {
            if (empty($blueprintName)) {
                if (!empty($this->args[1])){
                    $blueprintName = $this->args[1];
                } else {
                    $blueprintName = $blueprintId;
                }
            }
            $this->destination =  "-o '{$this->output}/{$blueprintName}.json'";
        }
        $action = "{$this->weburl}/rest/deploy/blueprint/{$blueprintId} {$this->destination}";
        return $this->execRest($action);
    }
    
    public function getBlueprintResources($blueprintId = null, $blueprintName = null) {
        if (empty($blueprintId)){
            if (!empty($this->args[0])) {
                $blueprintId = $this->args[0];
            } else {
                return false;
            }
        }
        if ($this->return == 'file') {
            if (empty($blueprintName)) {
                if (!empty($this->args[1])){
                    $blueprintName = $this->args[1];
                } else {
                    $blueprintName = $blueprintId;
                }
            }
            $this->destination =  "-o '{$this->output}/{$blueprintName}.resources.json'";
        }
        $action = "{$this->weburl}/rest/deploy/blueprint/{$blueprintId}/resources {$this->destination}";
        
        return $this->execRest($action);
    }
    
    /**
     * 
     * Export a component to a file or returns json 
     *
     * @param string $componentID
     * @param string $componentName
     * @return boolean|string
     */
    public function getComponentRest($componentID = null, $componentName = null){
        if (empty($componentID)) {
            if (!empty($this->args[0])){
                $componentID = $this->args[0];
            } else {
                return false;
            }
        }
        if ($this->return == 'file') {
            if (empty($componentName)) {
                if (!empty($this->args[1])){
                    $componentName = $this->args[1];
                } else {
                    $componentName = $componentID;
                }
            }
            $this->destination = " -o '{$this->output}/{$componentName}.json'";
        }
        $action = "{$this->weburl}/rest/deploy/component/{$componentID}/export {$this->destination}";
        return $this->execRest($action);
    }
    
    /**
     * Set a uDeploy API request-command to obtain, ALL available Component Templates.
     * NOTE: The returned list with templates will ONLY hold a partial (general) Template data,
     *       for each ComponentTemplate.
     *
     * @return string - The uDeploy API request to be executed
     */
    public function getComponentTemplates() {
        if ($this->return == 'file') {
            $this->destination = " > '{$this->output}/componentTemplates.json'";
        }
        $action = $this->weburl . '/rest/deploy/componentTemplate' . $this->destination;
        return $this->execRest($action);
    }
    
    /**
     * Set a uDeploy API request to obtain FULL data for a specified ComponentTemplate.
     * Note: ComponentTemplate - Details can be obtaine only by using ComponentTemplate ID.
     *
     * @param integer $template_id      Integer, holding the ID of the ComponentTemplate to be retrieved.
     *                                  This is required to build the API request.
     * @param string $template_name     String, holding the name of the ComponentTemplate to obtain.
     *                                  This is required for when we are saving the Template into a file.
     *
     * @return boolean|string           The uDeploy API request to be executed or
     *                                  FALSE in case of a problem with passed arguments.
     */
    public function getComponentTemplateDetails($template_id=null, $template_name=null) {
        if (empty($template_id) || empty($template_name)) {
            if (!empty($this->args[0]) && !empty($this->args[1])) {
                $template_id    = $this->args[0];
                $template_name  = $this->args[1];
            } else {
                return false;
            }
        }
        if ($this->return == 'file') {
            $this->destination = " > '{$this->output}/{$template_name}.json'";
        }
        $action = $this->weburl . '/rest/deploy/componentTemplate/' . $template_id . $this->destination;
        return $this->execRest($action);
    }
    
    /**
     * @param string    $environment
     * @return string
     * 
     * Will return false if argument is incorrect
     */
    public function getComponentVersionPropDefs($component = null){
        if (empty($component)){
            if (!empty($this->args[0])) {
                $component = $this->args[0];
            } else {
                return false;
            }
        }
        if ($this->return == 'file') {
            $this->destination = " > '{$this->output}/{$component}.propDefs.json'";
        }
        $action = "getComponentVersionPropDefs -component '{$component}'{$this->destination}";
        return $this->execRest($action);
    }
    
    /**
     * 
     * Returns a paginated list of component versions
     *
     * @param string $component
     * @param number $rowsPerPage
     * @param number $pageNumber
     * @param string $orderField
     * @param string $sortType
     * @return boolean|string
     */
    public function getComponentVersions($component = null, $rowsPerPage = 10, $pageNumber = 1, $orderField = 'dateCreated', $sortType = 'desc'){
        if (empty($component)){
            if (!empty($this->args[0])) {
                $component = $this->args[0];
            } else {
                return false;
            }
        }
        if ($this->return == 'file') {
            $this->destination = " > '{$this->output}/{$component}.versions.{$pageNumber}.json'";
        }
        
        /* DEPRECATED ***
        $action = "'{$this->weburl}/rest/deploy/component/{$component}/versions/paginated/false?"
                . "rowsPerPage={$rowsPerPage}"
                . "&pageNumber={$pageNumber}"
                . "&orderField={$orderField}"
                . "&sortType={$sortType}' "
                . "{$this->destination}";
        ***/
        
        $action = "'{$this->weburl}/rest/deploy/component/{$component}/versions/paginated'";
                
        return $this->execRest($action);
    }
    
    /**
     * 
     * Return componentVersions in a snapshot
     *
     * @param string $snapshotId
     * @return boolean|string
     */
    public function getComponentVersionsInSnapshot($snapshotId = null) {
        if (empty($snapshotId)){
            if (!empty($this->args[0])) {
                $snapshotId = $this->args[0];
            } else {
                return false;
            }
        }
        if ($this->return == 'file') {
            $this->destination = "> '{$this->output}/{$snapshotId}.componentVersions.json'";
        }
        $action = $this->weburl . "/rest/deploy/snapshot/{$snapshotId}/versions {$this->destination}";
        return $this->execRest($action);
    }
    
    /**
     * 
     * get environment resources
     *
     * @param string $environment               Environment ID
     * @return boolean|string
     */
    public function getEnvironmentResources($environment){
        if ($this->return == 'file') {
            $this->destination = "> '{$this->output}/postProcessScripts.json'";
        }
        $action = $this->weburl . "/rest/deploy/environment/{$environment}/resources {$this->destination}";
        return $this->execRest($action);
    }
    
    /**
     * 
     * Get all port precessing scripts
     *
     */
    public function getPostProcessingScripts(){
        if ($this->return == 'file') {
            $this->destination = "> '{$this->output}/postProcessScripts.json'";
        }
        $action = $this->weburl . "/rest/script/postprocessing {$this->destination}";
        return $this->execRest($action);
    }
    
    /**
     * 
     * Returns list of all generic processes
     *
     * @return Ambigous <boolean, unknown>
     */
    public function getProcesses(){
        if ($this->return == 'file') {
            $this->destination = "> '{$this->output}/processes.json'";
        }
        $action = $this->weburl . "/rest/process? {$this->destination}";
        return $this->execRest($action);
    }
    
    public function getResourceRole( $name = null ) {
        if (empty($name)){
            if (!empty($this->args[0])) {
                $name = $this->args[0];
            } else {
                $name = '*';
            }
        }
        if ($this->return == 'file') {
            if ( $name == '*' ) {
                $nameForFile = 'resourceRoles';
            } else {
                $nameForFile = "{$name}.resourceRole";
            }
            $this->destination = "> '{$this->output}/{$nameForFile}.json'";
        }
        
        $action = $this->weburl . "/rest/resource/resourceRole/componentRoles/?name={$name} {$this->destination}";
        return $this->execRest($action);
    }
    
    /**
     * 
     * Retireve list of resource templates for a uDeploy server
     *
     */
    public function getResourceTemplates() {
        if ($this->return == 'file') {
            $this->destination = "> '{$this->output}/resourceTemplates.json'";
        }
        $action = $this->weburl . "/rest/resource/resourceTemplate/ {$this->destination}";
        return $this->execRest($action);
    }
    
    /**
     * 
     * Export a template resource
     *
     * @param string $templateId
     * @param string $templateName
     * @return boolean|string
     */
    public function getResourceTemplate($templateId = null, $templateName = null) {
        if (empty($templateId)){
            if (!empty($this->args[0])) {
                $templateId = $this->args[0];
            } else {
                return false;
            }
        }
        if (empty($templateName)){
            if (!empty($this->args[1])) {
                $templateName = $this->args[1];
            } else {
                $templateName = $templateId;
            }
        }
        if ($this->return == 'file') {
            $this->destination = "> '{$this->output}/{$templateName}.json'";
        }
        $action = $this->weburl . "/rest/resource/resourceTemplate/{$templateId}/resources {$this->destination}";
        return $this->execRest($action);
    }
    
    public function getResourceTemplateResources($resourceId = null, $resourceName = null) {
        if (empty($resourceId)){
            if (!empty($this->args[0])) {
                $resourceId = $this->args[0];
            } else {
                return false;
            }
        }
        if (empty($resourceName)){
            if (!empty($this->args[1])) {
                $resourceName = $this->args[1];
            } else {
                $resourceName = $resourceId;
            }
        }
        if ($this->return == 'file') {
            $this->destination = " > '{$this->output}/{$resourceName}.json'";
        }
        $action = "'{$this->weburl}/rest/resource/resource/{$resourceId}/resources?rowsPerPage=99999&pageNumber=1&orderField=name&sortType=asc'{$this->destination}";
        return $this->execRest($action);
    }
    
    /**
     * @param string    $environment
     * @return string
     * 
     * Will return false if argument is incorrect
     */
    public function getEnvironment($environment = null, $application = null){
        if (empty($environment)){
            if (!empty($this->args[0])) {
                $environment = $this->args[0];
            } else if (!empty($this->environment)){
                $environment = $this->environment;
            } else {
                return false;
            }
        }
        if (empty($application)){
            if (!empty($this->args[1])) {
                $application = " -application '{$this->args[1]}'";
            } else if (!empty($this->application)){
                $application = " -application '{$this->application}'";
            }
        }
        
        if ($this->return == 'file') {
            $this->destination = " > '{$this->output}/{$environment}.json'";
        }
        $action = "getEnvironment -environment '{$environment}'{$application}{$this->destination}";
        return $this->exec($action);
    }
    
    /**
     * @param string        $environment
     * @param string        $application
     * @return string
     * 
     * Will return false if argument is incorrect
     */
    public function getEnvironmentBaseResources($environment = null, $application = null){
        if (empty($environment)){
            if (!empty($this->args[0])) {
                $environment = $this->args[0];
            } else if (!empty($this->environment)){
                $environment = $this->environment;
            } else {
                return false;
            }
        }
        if (empty($application)){
            if (!empty($this->args[1])) {
                $application = $this->args[1];
            } else if (!empty($this->application)){
                $application = $this->application;
            }
        }
        if (!empty($application)){
            $application = ' -application ' . $application;
        }
        if ($this->return == 'file') {
            $this->destination = " > '{$this->output}/{$environment}.baseResources.json'";
        }
        $action = "getEnvironmentBaseResources -environment {$environment}{$application}{$this->destination}";
        return $this->exec($action);
    }
    
    /**
     * @param string        $environment
     * @param string        $application
     * @return string
     * 
     * Will return false if argument is incorrect
     */
    public function getEnvironmentProperties($environment = null, $application = null){
        if (empty($environment)){
            if (!empty($this->args[0])) {
                $environment = $this->args[0];
            } else if (!empty($this->environment)){
                $environment = $this->environment;
            } else {
                return false;
            }
        }
        if (empty($application)){
            if (!empty($this->args[1])) {
                $application = $this->args[1];
            } else if (!empty($this->application)){
                $application = $this->application;
            } else {
                return false;
            }
        }
        if ($this->return == 'file') {
            $this->destination = " > '{$this->output}/{$environment}.properties.json'";
        }
        $action = "getEnvironmentProperties -environment {$environment} -application {$application}{$this->destination}";
        return $this->exec($action);
    }
    
    /**
     * @param string        $environment
     * @param string        $property
     * @param string        $application
     * @return string
     * 
     * Will return false if argument is incorrect
     */
    public function getEnvironmentProperty($environment = null, $property = null, $application = null){
        if (empty($environment)){
            if (!empty($this->args[0])) {
                $environment = $this->args[0];
            } else if (!empty($this->environment)){
                $environment = $this->environment;
            } else {
                return false;
            }
        }
        if (empty($property)){
            if (!empty($this->args[1])) {
                $property = $this->args[1];
            } else {
                return false;
            }
        }
        if (empty($application)){
            if (!empty($this->args[2])) {
                $application = $this->args[2];
            } else if (!empty($this->application)){
                $application = $this->application;
            } else {
                return false;
            }
        }
        if ($this->return == 'file') {
            $this->destination = " > '{$this->output}/{$environment}.{$property}.json'";
        }
        $action = "getEnvironmentProperty -name {$property} -environment {$environment} -application {$application}{$this->destination}";
        return $this->exec($action);
    }
    
    /**
     * @param string    $application
     * @return string
     * 
     * Will return false if argument is incorrect
     */
    public function getEnvironmentsInApplication($application = null){
        if (empty($application)){
            if (!empty($this->args[0])) {
                $application = $this->args[0];
            } else if (!empty($this->application)){
                $application = $this->application;
            } else {
                return false;
            }
        }
        if ($this->return == 'file') {
            $this->destination = " > '{$this->output}/{$application}.environments.json'";
        }
        $action = "getEnvironmentsInApplication -application {$application}{$this->destination}";
        return $this->exec($action);
    }
    
    /**
     * @param string    $resource
     * @return string
     * 
     * Will return false if argument is incorrect
     */
    public function getResource($resource = null){
        if (empty($resource)){
            if (!empty($this->args[0])) {
                $resource = $this->args[0];
            } else {
                return false;
            }
        }
        if ($this->return == 'file') {
            $this->destination = " > '{$this->output}/{$resource}.json'";
        }
        $action = "getResource -resource {$resource}{$this->destination}";
        return $this->exec($action);
    }
    
    /**
     * @param string    $resource
     * @return string
     * 
     * Will return false if argument is incorrect 
     */
    public function getResourceProperties($resource = null){
        if (empty($resource)){
            if (!empty($this->args[0])) {
                $resource = $this->args[0];
            } else {
                return false;
            }
        }
        if ($this->return == 'file') {
            $this->destination = " > '{$this->output}/{$resource}.properties.json'";
        }
        $action = "getResourceProperties -resource {$resource}{$this->destination}";
        return $this->exec($action);
    }
    
    /**
     * @param string    $resource
     * @param string    $property
     * @return string
     * 
     * Will return false if argument is incorrect
     */
    public function getResourceProperty($resource = null, $property = null){
        if (empty($resource)){
            if (!empty($this->args[0])) {
                $resource = $this->args[0];
            } else {
                return false;
            }
        }
        if (empty($property)){
            if (!empty($this->args[1])) {
                $property = $this->args[1];
            } else {
                return false;
            }
        }
        if ($this->return == 'file') {
            $this->destination = " > '{$this->output}/{$resource}.{$property}.json'";
        }
        $action = "getResourceProperty -resource {$resource} -name {$property}{$this->destination}";
        return $this->exec($action);
    }
    
    /**
     * @return string
     * 
     * Will return false if argument is incorrect
     */
    public function getResources(){
        if ($this->return == 'file') {
            $this->destination = " > '{$this->output}/resources.json'";
        }
        $action = "getResources{$this->destination}";
        return $this->exec($action);
    }
    
    /**
     * @param string    $resource
     * @param string    $property
     * @return string
     * 
     * Will return false if argument is incorrect
     */
    public function getResourcesWithComponentVersion($component = null, $version = null){
        if (empty($component)){
            if (!empty($this->args[0])) {
                $component = $this->args[0];
            } else {
                return false;
            }
        }
        if (empty($version)){
            if (!empty($this->args[1])) {
                $version = $this->args[1];
            } else {
                return false;
            }
        }
        if ($this->return == 'file') {
            $this->destination = " > '{$this->output}/{$component}.{$version}.resources.json'";
        }
        $action = "getResourcesWithComponentVersion -component '{$component}' -version '{$version}'{$this->destination}";
        return $this->exec($action);
    }
    
    /**
     * 
     * Returns all the snapshots for an application
     *
     * @param unknown $application
     * @return boolean|string
     */
    public function getSnapshotsInApplication($application = null) {
        if (empty($application)){
            if (!empty($this->args[0])) {
                $application = $this->args[0];
            } else if (!empty($this->application)) {
                $application = $this->application;
            } else {
                return false;
            }
        }
        if ($this->return == 'file') {
            $this->destination = "> '{$this->output}/{$application}.snapshots.json'";
        }
        $action = $this->weburl . "/rest/deploy/application/{$application}/snapshots/false?sortType=desc {$this->destination}";
        return $this->execRest($action);
    }
    
    /**
     * 
     * Get all the teams in uDeploy
     *
     * @return boolean|string
     */
    public function getTeams() {
        if ($this->return == 'file') {
            $this->destination = "> '{$this->output}/teams.json'";
        }
        $action = $this->weburl . "/security/team {$this->destination}";
        return $this->execRest($action);
    }
    
    /**
     * @param string    $status
     * @param string    $type       optional
     * @return string
     * 
     * Will return false if argument is incorrect
     */
    public function getStatus($status = null, $type = null){
        if (empty($status)){
            if (!empty($this->args[0])) {
                $status = $this->args[0];
            } else {
                return false;
            }
        }
        $validTypes = array('inventory','version','snapshot');
        if (empty($type)){
            if (!empty($this->args[1])) {
                $type = $this->args[1];
            }
        }
        if (in_array($type,$validTypes)){
            $type = ' -type ' . $type;
        } else {
            $type = '';
        }
        if ($this->return == 'file') {
            $this->destination = " > '{$this->output}/{$component}.{$status}.json'";
        }
        $action = "getStatus -status {$status}{$type}{$this->destination}";
        return $this->exec($action);
    }
    
    /**
     * @param string    $type
     * @return string
     * 
     * Will return false if argument is incorrect
     */
    public function getStatuses($type = null){
        $validTypes = array('inventory','version','snapshot');
        if (empty($type)){
            if (!empty($this->args[0])) {
                $type = $this->args[0];
            } else {
                return false;
            }
        }
        if (!in_array($type,$validTypes)){
            return false;
        }
        if ($this->return == 'file') {
            $this->destination = " > '{$this->output}/{$type}.statuses.json'";
        }
        $action = "getStatuses -type {$type}{$this->destination}";
        return $this->exec($action);
    }
    
    /**
     * @return string
     * 
     */
    public function getSystemConfiguration(){
        if ($this->return == 'file') {
            $this->destination = " > '{$this->output}/system.configuration.json'";
        }
        $action = "getsystemConfiguration{$this->destination}";
        return $action;
    }
    
    /**
     * @return string
     * 
     */
    public function getSystemProperties(){
        if ($this->return == 'file') {
            $this->destination = " > '{$this->output}/system.properties.json'";
        }
        $action = "getsystemProperties{$this->destination}";
        return $this->exec($action);
    }
    
    /**
     * @param string    $property
     * @return string
     * 
     * Will return false if argument is incorrect
     */
    public function getSystemProperty($property = null){
        if (empty($property)){
            if (!empty($this->args[0])) {
                $property = $this->args[0];
            } else {
                return false;
            }
        }
        if ($this->return == 'file') {
            $this->destination = " > '{$this->output}/system.{$property}.json'";
        }
        $action = "getSystemProperty -name {$property}{$this->destination}";
        return $this->exec($action);
    }
    
    /**
     * 
     * Get all tags in a component
     *
     * @param string $component
     * @param string $componentName
     * @return boolean
     */
    public function getTagsOnComponent($component = null, $componentName = null) {
        if (empty($component)){
            if (!empty($this->args[0])) {
                $component = $this->args[0];
            } else {
                return false;
            }
        }
        
        if (empty($componentName)) {
            $componentName = $component;
        }
        
        if ($this->return == 'file') {
            $this->destination = " > '{$this->output}/{$componentName}.tags.json'";
        }
        $action = "getTagsOnComponent -component '{$component}'{$this->destination}";
        return $this->exec($action);
    }
    
    /**
     * @param string    $user
     * @return string
     * 
     * Will return false if argument is incorrect
     */
    public function getUser($user = null){
        if (empty($user)){
            if (!empty($this->args[0])) {
                $user = $this->args[0];
            } else {
                return false;
            }
        }
        if ($this->return == 'file') {
            $this->destination = " > '{$this->output}/{$user}.json'";
        }
        $action = "getUser -user {$user}{$this->destination}";
        return $this->exec($action);
    }
    
    /**
     * @param string    $version
     * @param string    $link
     * @param string    $component      optional
     * @return string
     * 
     * Will return false if argument is incorrect
     */
    public function getVersionLink($version = null, $link = null, $component = null){
        if (empty($version)){
            if (!empty($this->args[0])) {
                $version = $this->args[0];
            } else {
                return false;
            }
        }
        if (empty($link)){
            if (!empty($this->args[1])) {
                $link = $this->args[1];
            } else {
                return false;
            }
        }
        if (empty($component)){
            if (!empty($this->args[2])) {
                $component = $this->args[2];
            }
        }
        if (!empty($component)){
            $component = " -component '{$component}'";
        } else {
            $component = '';
        }
        if ($this->return == 'file') {
            $this->destination = " > '{$this->output}/{$version}.{$link}.json'";
        }
        $action = "getVersionLink -version {$version} -linkName {$link}{$component}{$this->destination}";
        return $this->exec($action);
    }
    
    /**
     * @param string    $version
     * @param string    $component      optional
     * @return string
     * 
     * Will return false if argument is incorrect
     */
    public function getVersionLinks($version = null, $component = null){
        if (empty($version)){
            if (!empty($this->args[0])) {
                $version = $this->args[0];
            } else {
                return false;
            }
        }
        if (empty($component)){
            if (!empty($this->args[1])) {
                $component = $this->args[1];
            }
        }
        if (!empty($component)){
            $component = " -component '{$component}'";
        } else {
            $component = '';
        }
        if ($this->return == 'file') {
            $this->destination = " > '{$this->output}/{$version}.links.json'";
        }
        $action = "getVersionLinks -version {$version}{$component}{$this->destination}";
        return $this->exec($action);
    }
    
    /**
     * 
     * Run application process
     *
     * @param string $process   Path to template json file
     * @return boolean|string
     */
    public function requestApplicationProcess($process = null){
        if (empty($process)){
            if (!empty($this->args[0])) {
                $process = $this->args[0];
            } else {
                return false;
            }
        }
        if ($this->return == 'file') {
            $this->destination = " > '{$this->output}/{$version}.links.json'";
        }
        $action = "requestApplicationProcess '{$process}'{$this->destination}";
        return $this->exec($action);
    }
    
    /*
     * CREATE FUNCTIONS
     */
    
    /**
     * 
     * Add files to a component version
     *
     * @param string $component
     * @param string $version
     * @param string $base
     * @param array  $include optional
     * @param array  $exclude optional
     */
    public function addVersionFiles($component = null, $version = null, $base = null, $include = null, $exclude = null) {
        if (empty($component)){
            if (!empty($this->args[0])) {
                $component = $this->args[0];
            } else {
                return false;
            }
        }
        if (empty($version)){
            if (!empty($this->args[1])) {
                $version = $this->args[1];
            } else {
                return false;
            }
        }
        if (empty($base)){
            if (!empty($this->args[2])) {
                $base = $this->args[2];
            } else {
                return false;
            }
        }
        $includeFiles = "";
        if (!empty($include)){
            if (is_string($include)) {
                $includeFiles = " -include {$include}";
            }
            else if (is_array($include)) {
                foreach ($include as $file) {
                    $includeFiles .= " -include {$file}";
                }
            }
        }
        $excludeFiles = "";
        if (!empty($exclude)){
            if (is_string($exclude)) {
                $excludeFiles = " -exclude {$exclude}";
            }
            else if (is_array($exclude)) {
                foreach ($exclude as $file) {
                    $excludeFiles .= " -exclude {$file}";
                }
            }
        }
        $action = "addVersionFiles -component '{$component}' -version {$version} -base {$base}{$includeFiles}{$excludeFiles}";
        return $this->exec($action);
    }
    
    /**
     * 
     * Create a blueprint giving a valid json file
     *
     * @param string $blueprint
     * @return boolean|string
     */
    public function createBlueprint($blueprint = null) {
        if (empty($blueprint)){
            if (!empty($this->args[0])) {
                $blueprint = $this->args[0];
            } else {
                return false;
            }
        }
        
        $action = "-X PUT ";
        $action .= '-H "Content-Type: application/json" ';
        $action .= "{$this->weburl}/rest/deploy/blueprint -d @'{$blueprint}'";
        
        return $this->execRest($action);
    }
    
    /**
     * 
     * Enter description here ...
     *
     * @param string $resource  Resource filename
     * @return boolean|string
     */
    public function createBlueprintResource($resource = null) {
        if (empty($resource)){
            if (!empty($this->args[0])) {
                $resource = $this->args[0];
            } else {
                return false;
            }
        }
        
        $action = "-X PUT ";
        $action .= '-H "Content-Type: application/json" ';
        $action .= "{$this->weburl}/rest/resource/resource -d @'{$resource}'";
        
        return $this->execRest($action);
    }
    
    /**
     * Create an application from a json file
     *
     * @param string     $application
     */
    public function createApplication($application = null) {
        if (empty($application)){
            if (!empty($this->args[0])) {
                $application = $this->args[0];
            } else {
                return false;
            }
        }
        $action = "createApplication '{$application}'";
        return $this->exec($action);
    }
    
    public function createApplicationProcess($appProcessFile = null) {
        if (empty($appProcessFile)){
            if (!empty($this->args[0])) {
                $appProcessFile = $this->args[0];
            } else {
                $this->log->error(__CLASS__ . '::' . __FUNCTION__
                        . ': Application Process File is required.');
                return false;
            }
        }
        
        // @TODO: make sure that the AppProcess backup file, holds CORRECT Application ID
        
        $action = "createApplicationProcess '{$appProcessFile}'";
        return $this->exec($action);
    }
    
    /**
     * Create a component from a json file
     *
     * @param string     $component
     */
    public function createComponent($component = null) {
        if (empty($component)){
            if (!empty($this->args[0])) {
                $component = $this->args[0];
            } else {
                return false;
            }
        }
        $action = "createComponent '{$component}'";
        return $this->exec($action);
    }
    
    /**
     * Set up a uDeploy API Request for Creation of a componentTemplate,
     * based on a JSON formatted file.
     * 
     * @param string $componentTemplate Path to the JSON File, to be used for ComponentTemplate creation
     * 
     * @return boolean|string String, holding the uDeploy API action to be executed OR
     *                        Boolean FALSE, when $componentTemplate File is not provided
     */
    public function createComponentTemplate($componentTemplate = null) {
        if (empty($componentTemplate)){
            if (!empty($this->args[0])) {
                $componentTemplate = $this->args[0];
            } else {
                return false;
            }
        }
        
        $action = "-X PUT ";
        $action .= '-H "Content-Type: application/json" ';
        $action .= "{$this->weburl}/rest/deploy/componentTemplate -d @'{$componentTemplate}'";
        
        return $this->execRest($action);
    }
    
    /**
     *
     * Create a resource template
     *
     * @param string $resourceTemplate      Path to the json file
     *                       description: ""
     *                       name: "Web server stack"
     *                       teamMappings: []
     *
     * @return boolean|string
     */
    public function createResourceTemplate($resourceTemplate = null) {
        if (empty($resourceTemplate)){
            if (!empty($this->args[0])) {
                $resourceTemplate = $this->args[0];
            } else {
                return false;
            }
        }
    
        $action = "-X PUT ";
        $action .= '-H "Content-Type: application/json" ';
        $action .= "{$this->weburl}/rest/resource/resourceTemplate -d @'{$resourceTemplate}'";
    
        return $this->execRest($action);
    }
    
    /**
     * 
     * Create a new post processing script from a json file
     *
     * @param string $scriptFile
     * @return boolean|string
     */
    public function createPostProcessingScript($scriptFile = null){
        if (empty($scriptFile)){
            if (!empty($this->args[0])) {
                $scriptFile = $this->args[0];
            } else {
                return false;
            }
        }
        
        $action = "-X PUT ";
        $action .= '-H "Content-Type: application/json" ';
        $action .= "{$this->weburl}/rest/script/postprocessing -d @'{$scriptFile}'";
        
        return $this->execRest($action);
    }
    
    /**
     *
     * Create a resource template child
     *
     * @param string $resourceTemplate      Path to the json file
     *                       description: ""
     *                       dynamic: "false"
     *                       inheritTeam: "true"
     *                       name: "Apache"
     *                       parentId: "9d18d9e7-1861-474e-8b78-6414b2fe42ae"
     *                       teamMappings: []
     *
     * @return boolean|string
     */
    public function createResourceTemplateChild($resourceTemplateChild = null) {
        if (empty($resourceTemplateChild)){
            if (!empty($this->args[0])) {
                $resourceTemplateChild = $this->args[0];
            } else {
                return false;
            }
        }
    
        $action = "-X PUT ";
        $action .= '-H "Content-Type: application/json" ';
        $action .= "{$this->weburl}/rest/resource/resource -d @'{$resourceTemplateChild}'";
    
        return $this->execRest($action);
    }
    
    public function createResource($file = null) {
        if (empty($file)) {
            if (!empty($this->args[0])) {
                $file = $this->args[0];
            } else {
                return false;
            }
        }
        
        $action = "createResource '{$file}'";
        return $this->exec($action);
    }
    
    /**
     *
     * Creates a snapshot using a json file
     *
     * @param string $args
     * @return string
     */
    public function createSnapshot($file = null){
        if (empty($file)){
            if (!empty($this->args[0])) {
                $file = $this->args[0];
            } else {
                return false;
            }
        }
        
        $action = "createSnapshot '{$file}'";
        return $this->exec($action);
    }
    
    /**
     *
     * Creates a snapshot of an environment with name and description. Udclient call will return a json response.
     *
     * @param string $environment
     * @param string $name
     * @param string $description   optional
     * @param string $application   optional
     * @return string
     */
    public function createSnapshotOfEnvironment($environment = null, $name = null, $description = null, $application = null){
        if (empty($environment)) {
            if (!empty($this->args[0])) {
                $environment = $this->args[0];
            } else if (!empty($this->environment)) {
                $environment = $this->environment;
            } else {
                return false;
            }
        }
        if (empty($name)) {
            if (!empty($this->args[1])) {
                $name = $this->args[1];
            } else {
                return false;
            }
        }
        if (empty($description)) {
            if (!empty($this->args[2])) {
                $description = " -description {$this->args[2]}";
            }
        } else {
            $description = " -description {$description}";
        }
        if (empty($application)) {
            if (!empty($this->args[3])) {
                $application = " -application {$this->args[3]}";
            } else if (!empty($this->application)) {
                $application = " -application {$this->application}";
            }
        } else {
            $application = " -application {$application}";
        }
        
        $action = "createSnapshotOfEnvironment -environment {$environment} -name {$name}{$description}{$application}";
        return $this->exec($action);
    }
    
    /**
     * 
     * Create a new version for the component
     *
     * @param string    $component
     * @param string    $versionName
     * @param string    $description
     * @return boolean|string
     */
    public function createVersion($component = null, $versionName= null, $description = null) {
        if (empty($component)){
            if (!empty($this->args[0])) {
                $component = $this->args[0];
            } else {
                return false;
            }
        }
        if (empty($versionName)){
            if (!empty($this->args[1])) {
                $versionName = $this->args[1];
            } else {
                return false;
            }
        }
        if (!empty($description)){
            $description = " -description {$description}";
        } else {
            $description = '';
        }
        
        $action = "createVersion -component '{$component}' -name {$versionName}{$description}";
        return $this->exec($action);
    }
    
    /**
     * 
     * Creates a component process from a json file
     *
     * @param string     $component
     * @param string     $process
     * @return boolean|string
     */
    public function createComponentProcess($component = null, $process = null) {
        if (empty($component)){
            if (!empty($this->args[0])) {
                $component = $this->args[0];
            } else {
                return false;
            }
        }
        if (empty($process)){
            if (!empty($this->args[1])) {
                $process = $this->args[1];
            } else {
                return false;
            }
        }
        // Get process content and turn it into an array
        $process_content = json_decode(file_get_contents($process));
        $process_content->component = $component;
        $process_content = json_encode($process_content);
        
        if (Utils::writeToFile($process, $process_content)) {
            $action = "createComponentProcess '{$process}'";
            return $this->exec($action);
        } else {
            return false;
        }
    }
    
    /**
     * 
     * Enter description here ...
     *
     * @param string $environment
     * @param string $application
     * @param string $deleteResources
     * @param string $deleteCloud
     * @return boolean
     */
    public function deleteEnvironment( $environment = null, $application = null, $deleteResources = false, $deleteCloud = false) {
        if (empty($environment)){
            if (!empty($this->args[0])) {
                $environment = $this->args[0];
            } else {
                return false;
            }
        }
        if (empty($application)){
            if (!empty($this->args[1])) {
                $application = $this->args[1];
            } else if (!empty($this->application)) {
                $application = $this->application;
            }
        }
        if (empty($deleteResources)){
            if (!empty($this->args[2])) {
                $deleteResources = $this->args[2];
            }
        }
        if (empty($deleteCloud)){
            if (!empty($this->args[3])) {
                $deleteCloud = $this->args[3];
            }
        }
        
        $action = "deleteEnvironment -environment '{$environment}'"
                ." -application '{$application}'"
                ." -deleteAttachedResources {$deleteResources}"
                ." -deleteCloudInstances {$deleteCloud}";
        
        return $this->exec($action);
    }
    
    /**
     * Function, used to IMPORT configuration for a 
     * Component/ComponentTemplate::Process->Step User Impersonation.
     *
     * @param Mixed $element The UrbanCode component/componentTemplate to be updated.
     *              This could be passed as:
     *              1. Object
     *              2. Array
     *              3. JSON formatted String
     *
     * @return string A JSON formatted string, of the updated UrbanCode Component/ComponentTemplate
     */
    public function impersonateElementProcesses( $element ) {
        
        // Ensure the passed $element variable is used as an Object
        $ucd_element = Utils::getAsObject($element);
        
        // Set element type (hint: only a Component-Template has variable: path
        $element_type = (isset($element->path)) ? 'Component-Template' : 'Component';
        
        if (isset($ucd_element->processes) && count($ucd_element->processes) > 0) {
            Utils::CLIinfo('Impersonate Process(es) of ' . $element_type . ' - "' . $ucd_element->name . '":', true, 6);
            
            // Get number of Processes
            $element_processes = count($ucd_element->processes);
            
            // Loop through this Component::Processes
            foreach ($ucd_element->processes as $proc_key => $process) {
                Utils::CLIinfo('Process #' . ($proc_key +1) . ': "' . $process->name . '": ', false, 9);
                
                // $process->rootActivity->children holds all Component Process steps (THESE ARE TO BE IMPERSONATED)
                if (isset($process->rootActivity->children) && count($process->rootActivity->children) > 0) {
                    
                    $process_steps = count($process->rootActivity->children);
                    
                    // loop through Component::Process->steps
                    foreach ($process->rootActivity->children as $step_key => $proc_step) {
                        Utils::CLIinfo('Step #' . ($step_key + 1) . ': "'
                                . $proc_step->name . '" (' . $proc_step->type . ')', false, 12);
                        
                        // process ONLY Steps which can be impersonated, i.e. have field: useImpersonation
                        if (isset($proc_step->useImpersonation)) {
                            // Ensure that Impersonation configs is loaded
                            if ($this->setUpUserImpersonationConfig()) {
                                // Impersonate this STEP now (either for first time or overwrite existent)
                                if ($proc_step->useImpersonation === false ||
                                (
                                        $proc_step->useImpersonation === true &&
                                        $this->impersonation_config['overwrite'] === true
                                )
                                ) {
                                    $template_variables = count($this->impersonation_config['template']);
                                    $counter = 0;
                                    // Implement Impersonation
                                    foreach ($this->impersonation_config['template'] as $ui_variable => $ui_value) {
                                        Utils::CLIinfo("Set: {$ui_variable} = '{$ui_value}'"
                                        . (($counter == ($template_variables-1)) ? "\n" : ''), false, 17);
                                        $proc_step->{$ui_variable} = $ui_value;
                                        $counter++;
                                    }
                                    
                                // This STEP has already been Impersonated
                                } else {
                                    Utils::CLIinfo("This step has already been IMPERSONATED.\n", false, 17);
                                }
                            } else {
                                Utils::CLIwarning("Process Step Impersonation - COULD NOT be implemented.\n", false, 17);
                            }
                        } else {
                            Utils::CLIinfo("Process Step cannot be impersonated.\n", false, 17);
                        }
                    }// end of looping through Component::Process->steps
                } else {
                    Utils::CLIinfo('Process does not have Steps to be impersonated.' 
                            . (($proc_key == ($element_processes-1)) ? "\n" : ''), false, 17);
                }
            }//end of looping through Component::Processes
        }
        
        // Return updated Component as a JSON formatted string
        return json_encode($ucd_element);
    }
    
    /**
     * Create a STRING hoolding a POST curl request to be used for
     * IMPORTING or UPGRADE of a ComponentTemplate.
     * 
     * @param string $componentTemplate [REQUIRED] Path to a ComponentTemplate JSON file
     *                                  If not provided => $this->args[0] will be used.
     *                                  If BOTH are not set => function will return FALSE
     * @param boolean $upgrade          [REQUIRED] Boolean TRUE/FALSE, used to set whether
     *                                  to set an UPGRADE or IMPORT POST request correspondingly.
     *                                  If not provided => $this->args[1] will be used.
     *                                  If BOTH are not set => function will return false
     *                                  
     * @return boolean|string   String holding a POST request for ComponentTemplate Import/Upgrade to be executed
     *                          or Boolean FALSE in case of either of the required parameters is NOT provided.
     */
    public function importComponentTemplate ($componentTemplate = null, $upgrade = null) {
       
       // Validate arguments
       if (empty($componentTemplate)){
           if (!empty($this->args[0])) {
               $componentTemplate = $this->args[0];
           } else {
               $this->log->error(__CLASS__ . '::' . __FUNCTION__ . ':: 1st parameter is REQUIRED');
               return false;
           }
       }
       if (empty($upgrade) && !is_bool($upgrade)){
           if (is_bool($this->args[1])) {
               $upgrade = $this->args[1];
           } else {
               $this->log->error(__CLASS__ . '::' . __FUNCTION__
                       . ':: 2ND parameter is REQUIRED and MUST be a BOOLEAN: true or false');
               return false;
           }
       }
       
       // Prepare POST Request-action to be executed
       $action = "-X POST ";  
       $action .= $this->weburl . '/rest/deploy/componentTemplate/';
       // Import or Upgrade ComponentTemplate
       $action .= $upgrade ? 'upgrade?' : 'import?';
       // Use existent Process or Create it if it is missing
       $action .= 'processUpgradeType=USE_EXISTING_IF_EXISTS';
       // Set parameters required for the POSt request
       $action .= ' -H "Content-Type: multipart/form-data"';
       $action .= " -F file=@'" . $componentTemplate . "'";
       
       // Return
       return $this->execRest($action);
    }
    
    /**
     * Create a STRING hoolding a POST curl request to be used for
     * IMPORTING or UPGRADE of a Component.
     * 
     * This function will CHECK if the component to be Imported,
     * is based on a Component Template and 
     * WILL ENSURE the corresponding Template is PRESENT on the Server,
     * before to prepare the IMPORT Component request string. 
     *
     * @param string $componentFile [REQUIRED] Path to a Component JSON file
     *                              If not provided => $this->args[0] will be used.
     *                              If BOTH are not set => function will return FALSE
     * @param boolean $upgrade      [REQUIRED] Boolean TRUE/FALSE, used to set whether
     *                              to set an UPGRADE or IMPORT POST request correspondingly.
     *                              If not provided => $this->args[1] will be used.
     *                              If BOTH are not set => function will return false
     *
     * @return boolean|string   String holding a POST request for Component Import/Upgrade to be executed
     *                          or Boolean FALSE in case of either of the required parameters is NOT provided.
     */
    public function importComponent ($componentFile = null, $upgrade = null) {
       
       // Validate arguments
       if (empty($componentFile)){
           if (!empty($this->args[0])) {
               $componentFile = $this->args[0];
           } else {
               $this->log->error(__CLASS__ . '::' . __FUNCTION__ . ':: 1st parameter is REQUIRED');
               return false;
           }
       }
       if (empty($upgrade) && !is_bool($upgrade)){
           if (is_bool($this->args[1])) {
               $upgrade = $this->args[1];
           } else {
               $this->log->error(__CLASS__ . '::' . __FUNCTION__
                       . ':: 2ND parameter is REQUIRED and MUST be a BOOLEAN: true or false');
               return false;
           }
       }
       
       /*
        * Check if Component is based on a Template and 
        * make sure the Template is present on server before to import the component
        */
       // STEP 1: Get contents of the Component to be imported and use it as Object
       $component = file_get_contents($componentFile);
       if ( $component === false ) {
           $this->log->error(__CLASS__ . '::' . __FUNCTION__
                   . ": There is a problem with reading Component backup file: \n" . $componentFile
                   . " Component IMPORT CANNOT be COMPLETED");
           return false;
       }
       // Convert $component to an Object
       $component = json_decode($component);
       
       // STEP 2: Check if the Component is based on a Template
       if ( isset($component->componentTemplate) ) {
           Utils::CLIinfo('Component: "' . $component->name . '" is based on a Template: "'
                   . $component->componentTemplate->name . '"');
           Utils::CLIinfo('Check if ComponentTemplate exists on Server, before to continue with Component IMPORT', false);
           $templateFound = false;
           self::setReturn('json');
           /* 
            * Check if ComponentTemplate is present on This Server
            *  1. Get All ComponentTemplates on the Server
            *  2. Check if the Template is present
            * NOTE: we cannot use: getComponentTemplateDetails() because it requires: 
            *       Template Name and ID, the Component does not hold its Template Id,
            *       and we cannot rely on the ID, as it definitely will differ on different Servers.
            *       Or between the IDs in the backups files and Server IDs.
            */
           $serverCompTemplates = self::getComponentTemplates();
           $serverCompTemplates = Utils::outputToArray($serverCompTemplates);
           // 2.1 Try to find the ComponentTemplate
           if (count($serverCompTemplates) > 0) {
               foreach ($serverCompTemplates as $compTemplate) {
                   if ($compTemplate['name'] == $component->componentTemplate->name) {
                       $templateFound = true;
                       // ComponentTemplate found -=> exit for loop and continue with import
                       break;
                   }
               }
           }
            
           // 2.2 ComponentTemplate was NOT found on this Server
           if ( !$templateFound ) {
               $templateImported = false;
               // Prompt message and try to import it to this Server
               Utils::CLIinfo('Component Template was NOT found on this Server.');
               
               // Extract the PATH to backups root directory from the $componentFile
               $componentFilePathParts = explode('/', $componentFile);
               // Remove Component file and Components Sub-Directory  (last 2 elements in the ComponentFile path)
               $backupRootPathParts = array_splice($componentFilePathParts, 0, (count($componentFilePathParts) -2));
               // Set backups path
               $backupsRootDir = implode('/', $backupRootPathParts);
               // Set Path for the ComponentTemplate backup file
               $componentTemplateBackup = $backupsRootDir . '/componentTemplates/'
                       . $component->componentTemplate->name . '.json';
               
               if ( is_file($componentTemplateBackup) ) {
                   Utils::CLIinfo('Import Component Template: "' . $component->componentTemplate->name . '"'
                           .' from its backup File, before Component Import.');
                   /*
                    * Use: RestoreUCD->importAComponentTemplate()
                    * NOTE: PHP alows us to use a Class's children functions as its own function.
                    *       i.e. RestoreUCD->importAComponentTemplate() can be called as:
                    *            $this->importAComponentTemplate()
                    */
                   $templateImported = $this->importAComponentTemplate($componentTemplateBackup);
               } else {
                   Utils::CLIinfo('Import Component Template: "' . $component->componentTemplate->name . '"'
                           .' from its copy in the Component: "' . $component->name . '" backup File.');
                   
                   // Set a temporary Component Template from the component backup data and import it
                   $templateTmpFile = $this->output . '/componentTemplate'
                           . $component->componentTemplate->name . 'tmpfile.json';
                   if ( Utils::writeToFile($templateTmpFile, $component->componentTemplate, false)) {
                       // Import Component Template from temporary file
                       $templateImported = $this->importAComponentTemplate($templateTmpFile);
                       // remove temporary file
                       unlink($templateTmpFile);
                   } else {
                       $this->log->error(__CLASS__ . '::' . __FUNCTION__ . ': problem with writing to file: ' 
                               ."\n" . $templateTmpFile);
                   }
               }
               
               // Check if Template was Imported and continue or return false
               if ( $templateImported === false ) {
                   Utils::CLIerror('There was a problem with Importing of Component Template: "'
                           . $component->componentTemplate->name . '"');
                   Utils::CLIinfo('For more details. please check: "' . $this->log->getLogFile() . '"', false, 1);
                   // return
                   return false;
               } else {
                   Utils::CLIinfo('Component Template: "'
                           . $component->componentTemplate->name . '" - IMPORTED to Server.');
                   Utils::CLIinfo('OK to continue with Component Import.', false);
               }
           } else {
               Utils::CLIinfo('Component Template EXISTS on this Server.');
               Utils::CLIinfo('Continue with Component Import.', false);
           }
       } // else: Component is Generic, i.e. NOT BASED on a Template -=> OK to Import it
       
       // Prepare POST Request-action to be executed
       $action = "-X POST ";
       // NOTE the whole URL part MUST be in quotes. Otherwise it will brake on the & sign
       $action .= "'" . $this->weburl . '/rest/deploy/component/';
       // Import or Upgrade ComponentTemplate
       $action .= $upgrade ? 'upgrade?' : 'import?';
       // Use existent ComponentTemplate or Create it if it is Missing
       $action .= 'upgradeType=USE_EXISTING_IF_EXISTS';
       // Use existent Process or Create it if it is missing
       $action .= '&processUpgradeType=USE_EXISTING_IF_EXISTS\'';
       // Set parameters required for the POSt request
       $action .= ' -H "Content-Type: multipart/form-data"';
       $action .= " -F file=@'" . $componentFile . "'";
       
       // Return
       return $this->execRest($action);
    }
    
    /**
     * Function used to import Component-Environment variable(s) into a Component/ComponentTemplate.
     * The variables to be created are defined in: /ucommand/config/ucd.impersonation.config.php.
     * For more details, check: /ucommand/README file.
     * 
     * @param Mixed $element The UrbanCode component/componentTemplate to be updated.
     *              This could be passed as:
     *              1. Object
     *              2. Array
     *              3. JSON formatted String
     *              
     * @return string A JSON formatted string, of the updated UrbanCode Component/ComponentTemplate
     */
    public function importElementEnvironmentVariables( $element ) {
        
        // Ensure the passed $element variable is used as an Object
        $ucd_element = Utils::getAsObject($element);
        
        // Set element type (hint: only a Component-Template has variable: path
        $element_type = (isset($element->path)) ? 'Component-Template' : 'Component';
        
        // Add configured Component Environment variables to: $ucd_element->envPropDefs[]
        if (isset($ucd_element->envPropDefs)) {
            
            // Get the names of current Environment variables of this component/componentTemplate
            $element_env_variables = array();
            foreach($ucd_element->envPropDefs as $ce_variable) {
                $element_env_variables[] = $ce_variable->name;
            }
            
            Utils::CLIinfo('Import "Component-Environment" variable(s) to ' 
                    . $element_type .' - "' . $ucd_element->name . '":', false, 6);
            
            foreach ($this->impersonation_config['variables_to_create'] as $var_to_create) {
                // Add variable to component/componentTemplate. ONLY if it has not been added yet
                if (!in_array($var_to_create['name'], $element_env_variables)) {
                    Utils::CLIinfo('- add variable: "' . $var_to_create['name'] . '".', false, 9);
                    
                    // Set variable as object
                    $comp_env_var = new stdClass();
                    $comp_env_var->name         = $var_to_create['name'];
                    $comp_env_var->label        = $var_to_create['label'];
                    $comp_env_var->pattern      = (isset($var_to_create['pattern']) ? $var_to_create['pattern'] : '');
                    // default type to TEXT
                    $comp_env_var->type         = (
                            (isset($var_to_create['type']) && trim($var_to_create['type']) != '') ? 
                            $var_to_create['type'] : 'TEXT'
                    );
                    $comp_env_var->value        = $var_to_create['value'];
                    // default requitred to false
                    $comp_env_var->required     = (
                            (isset($var_to_create['required']) && is_bool($var_to_create['required'])) ? 
                            $var_to_create['required'] : false
                    );
                    // default description to EMPTY_STRING
                    $comp_env_var->description  = (isset($var_to_create['description']) ? $var_to_create['description'] : '');
                    
                    // add the variable to $ucd_element Environment Variables
                    $ucd_element->envPropDefs[] = $comp_env_var;
                    
                    // clear the $comp_env_var object
                    unset($comp_env_var);
                } else {
                    Utils::CLIinfo('- variable: "' . $var_to_create['name'] . '"' 
                            . ' already exists in this ' . $element_type 
                            . '\'s "Component-Environment" variables.', false, 9);
                }
            }
        }
        
        // Return updated Component as a JSON formatted string
        return json_encode($ucd_element);
    }
    
    /**
     *
     * Deletes a component using name or ID
     *
     * @param unknown $component
     */
    public function deleteComponent($component){
        $command = "-X DELETE ";
        $command .= "'{$this->weburl}/rest/deploy/component/{$component}'";
        
        return $this->execRest($command);
    }
    
    /**
     * Set a uDeploy DELETE action to be executed for 
     * Deletion of a Component Template using its ID
     *
     * @param integer $componentTemplateID
     *
     * @return string String holding the uDeploy action for the Deletion of a Component Template
     */
    public function deleteComponentTemplate($componentTemplateID){
        $command = "-X DELETE ";
        $command .= "{$this->weburl}/rest/deploy/componentTemplate/{$componentTemplateID} ";
        
        return $this->execRest($command);
    }
    
    /**
     * Set up a string holding the REST CURL request,
     * to be sent to UCD server in irder to Upgrade a UCD Application based on a JSON formatted backup file.
     * Will either upgrade, Application related: componentTemplate(s) and application-Process(es)
     * or use existent ones, based on the related parameters.
     * 
     * @param string  $application               Full Path to a JSON formatted Application backup file
     * @param boolean $upgradeComponentTemplate  Boolean flag, used to set, whether to UPGRADE or USE EXISTENT
     *                                           componentTemplate(s) for the Application being upgraded.
     *                                           DEFAULT value = TRUE (Upgrade componentTemplates)
     * @param boolean $upgradeProcess            Boolean flag, used to set, whether to UPGRADE or USE EXISTENT
     *                                           Application Process(es) for the Application being upgraded.
     *                                           DEFAULT value = TRUE (Upgrade Application Process(es))
     * @return boolean|string   
     */
    public function upgradeApplication($application = null, $upgradeComponentTemplate = true, $upgradeProcess = true) {
        if (empty($application)){
            if (!empty($this->args[0])) {
                $application = $this->args[0];
            } else {
                return false;
            }
        }
        $command = "-F file=@'{$application}' ";
        $command .= "'{$this->weburl}/rest/deploy/application/upgrade?upgradeType=UPGRADE_IF_EXISTS";
        if ($upgradeComponentTemplate) {
            $command .= "&compTempUpgradeType=UPGRADE_IF_EXISTS";
        } else {
            $command .= "&compTempUpgradeType=USE_EXISTING_IF_EXISTS";
        }
        if ($upgradeProcess) {
            $command .= "&processUpgradeType=UPGRADE_IF_EXISTS";
        } else {
            $command .= "&processUpgradeType=USE_EXISTING_IF_EXISTS";
        }
        $command .= "'";
        
        return $this->execRest($command);
    }
    
    /**
     * 
     * Upgrade a component using a json file. 
     * First argument is the path of the component file. uDeploy will then upgrade the component with the same name
     * If upgradeProcess is set to true (by default, TRUE) also component processes are upgraded 
     *
     * @param string $component
     * @param string $upgradeProcess 
     * @param string $upgradeProcess 
     */
    public function upgradeComponent($component = null, $upgradeTemplate = false, $upgradeProcess = true) {
        if (empty($component)){
            if (!empty($this->args[0])) {
                $component = $this->args[0];
            } else {
                return false;
            }
        }
        if (!empty($this->args[1])) {
            $upgradeTemplate = $this->args[1];
        }
        if (!empty($this->args[2])) {
            $upgradeProcess = $this->args[2];
        }
        
        $command = "-F file=@'{$component}' ";
        $command .= "'{$this->weburl}/rest/deploy/component/upgrade?";
        if ($upgradeTemplate) {
            $command .= "upgradeType=UPGRADE_IF_EXISTS";
        } else {
            $command .= "upgradeType=USE_EXISTING_IF_EXISTS";
        }
        if ($upgradeProcess) {
            $command .= "&processUpgradeType=UPGRADE_IF_EXISTS'";
        } else {
            $command .= "&processUpgradeType=USE_EXISTING_IF_EXISTS'";
        }
        
        return $this->execRest($command);
    }
    
    /**
     * Set up a uDeploy API Request for UPDATING of a specified ComponentTemplate
     * from a JSON formatted file.
     *  
     * @param string $componentTemplate Path to the JSON File, to be used for ComponentTemplate update
     * @param string $upgradeProcess    Boolean FLAG to set wheather to update ComponentTemplate Process(es) or Not
     *                                  DEFAULT value: TRUE
     * 
     * @return boolean|string   String, holding the uDeploy API UPDATE action to be executed OR
     *                          Boolean FALSE, when $componentTemplate File is not provided
     */
    public function upgradeComponentTemplate($componentTemplate = null, $upgradeProcess = true) {
        if (empty($componentTemplate)){
            if (!empty($this->args[0])) {
                $componentTemplate = $this->args[0];
            } else {
                return false;
            }
        }
        
        $command = "-F file=@'{$componentTemplate}' ";
        $command .= "{$this->weburl}/rest/deploy/componentTemplate/upgrade?";
        if ($upgradeProcess) {
            $command .= "&processUpgradeType=UPGRADE_IF_EXISTS ";
        } else {
            $command .= "&processUpgradeType=USE_EXISTING_IF_EXISTS";
        }
    
        return $this->execRest($command);
    }
    
    public function upgradeResource($resourceFile = null) {
        
        Utils::CLIinfo('upgradeResource - UNDER CONSTRUCTION');
        $this->log->info('upgradeResource - UNDER CONSTRUCTION');
        return null;
        
        // @TODO: complete functionality
//         if (empty($resourceFile)){
//             if (!empty($this->args[0])) {
//                 $resourceFile = $this->args[0];
//             } else {
//                 return false;
//             }
//         }
//         // below action is CREATING a resource, not updating it
//         $action = "{$this->weburl}/rest/resource/resource -X PUT --data @'{$resourceFile}'";
    
//         return $action;
    }
    
    /**
     * A helper function, which is used to support the $this->exec() and$this->execRest() functions.
     * Will process a specified command ($commandToExecute), 
     * which was generated for the specified: $requestedAction, using PHP: exec() command.
     * In case of an error in either: exec() command execution using or
     * if the Server returns an error message, this function will log corresponding message
     * and will return Boolean FALSE.
     * Otherwise it will return the result from the executed command
     * as defined in the: '$this->return' variable.
     * 
     * @param string $requestedAction   The name of the requested Action, for which
     *                                  corresponding $commandToExecute, has been generated
     * @param string $commandToExecute  The command to be executed
     * 
     * @return boolean|string|mixed Boolean FALSE - in case of an Error with execution
     *                              of the specified command, or when Server request returns with error message.
     *                              String holding the path to a file, when "$this->return" is set to: "file"
     *                              or the raw result from the executed command, otherwise.
     */
    private function processServerRequest($commandToExecute) {
        // Execute specified command
        exec($commandToExecute, $requestResult, $requestError);
        
        // Debug
        if ($this->log->isDebug()){
            $this->log->debug("Request exit code: {$requestError}");
            $this->log->debug("Request result: ");
            $this->log->debug(print_r($requestResult,true));
        }
        
        /*
         * Get destination and clean it up for next function call before to procede
         * NOTE: depending on ServerRequest destinations might begin with:
         *       ' > ', '> ', ' -o ' or '-o'
         *       We need to clean these in order to avoid any confusion in results
         */
        $fileDestination   = str_replace(
                array(' > ', '> ', ' -o ', '-o '),
                '',
                $this->destination);
        $this->destination = '';
        
        /*
         * Validate execution - result, log errors and return correspondingly.
         * NOTE: $command details have already been logged by the caller to this function
         */
        
        /* 1. Check for errors in the request execution,
         *    i.e. there was a problem with execution of the system command
         */
        if ( $requestError ){
            Utils::CLIerror(
                    'Command: "' . $commandToExecute . '" execution exited with error code: ' . $requestError,
                    true, 0, 'system');
            Utils::CLIinfo('For more details, please check: "' . $this->log->getLogFile() . '" file');
            
            /* 
             * Shell command errors are printed on the console
             * just log the error code
             */
            $this->log->error('System Error code: ' . $requestError . ' :: There was an ERROR in specified command string, '
                    .'or Request exited with a message "Broken Pipe".');
            return false;
        } // else request was processed OK -=> validate request result and return
        
        /* 2. Check for an error response from the Server
         *    command execution went OK -=> validate command result.
         *    Get $requestResult as a string using: Utils::outputToJson() and validate it.
         * NOTE: 1) in case when $command result was written to a file
         *          $output will be empty array which will be validated to TRUE
         *          by: Utils::validateJSON(Utils::outputToJson($output))
         *       2) $output might be in the form:
         *          - just a string with error message or
         *          - string in the form:
         *              '<textarea>{"status":"ok"...}</textarea>' - for Success
         *              '<textarea>{"status":"failed","error":"..."}</textarea>'
         */
        $resultToValidate = Utils::outputToJson($requestResult);
        // Server returned failure to process request
        if (
            !Utils::validateJSON($resultToValidate) &&
            strpos( $resultToValidate, '"status":"ok"' ) === false &&
            strpos( $resultToValidate, '"status": "ok"' ) === false &&
            strpos( $resultToValidate, 'Operation succeeded') === false
        ) {
            Utils::CLIerror('Command: "' . $commandToExecute . '" completed with an ERROR message.');
            Utils::CLIinfo('For more details, please check: "' . $this->log->getLogFile() . '" file');
            $this->log->warning('Command execution - COMPLETED with ERROR response:');
            $this->log->error('Server response :: ' . var_export($requestResult,true));
            return false;
            
        /*
         * Request was executed OK and Server returned success result 
         * -=> return as usual based on the $this->return type
         */
        } else {
            // $output has been set to be saved to a file
            if ($this->return == 'file'){
                // return $output - file destination
                return $fileDestination;
                
            // return $output as it is
            } else {
                return $requestResult;
            }
        } // END of validating result ($output) from $actionToExec
    }
    
    public function createTag($file = null) {
        if (empty($file)){
            if (!empty($this->args[0])) {
                $file = $this->args[0];
            } else {
                return false;
            }
        }
        
        if (!file_exists($file)) {
            return false;
        }
        
        if ($this->return == 'file') {
            $this->destination = " > '{$this->output}/tag_restore.json'";
        }
        
        $fileContent = json_decode(file_get_contents($file),true);
        $type = $fileContent['objectType'];
        
        $action = '-X PUT ';
        $action .= '-H "Content-Type: application/json" ';
        $action .= "{$this->weburl}/rest/tag/{$type} -d @'{$file}' {$this->destination}";
        
        return $this->execRest($action);
    }
    
    /**
     * 
     * Import a snapshot using a zip file
     *
     * @param string $application
     * @param string $file
     * @return boolean|string
     */
    public function importSnapshot($application = null, $file = null){
        if (empty($application)){
            if (!empty($this->args[0])) {
                $application = $this->args[0];
            } else {
                return false;
            }
        }
        if (empty($file)){
            if (!empty($this->args[1])) {
                $file = $this->args[1];
            } else {
                return false;
            }
        }
        
        $action = "-X POST ";
        $action .= "{$this->weburl}/rest/deploy/snapshot/importWithArtifacts/{$application}";
        $action .= ' -H "Content-Type: multipart/form-data"';
        $action .= " -F file=@'{$file}'";
         
        // Return
        return $this->execRest($action);
    }
    
    /**
     * 
     * Import new process from file. Upgrade an existing process if exists
     * 
     * @param string $file
     * @param boolean $upgrade
     * 
     * @return boolean|string
     *
     */
    public function importProcess($file = null, $upgrade = false) {
        if (empty($file)){
            if (!empty($this->args[0])) {
                $file = $this->args[0];
            } else {
                return false;
            }
        }
        
        if (is_bool($upgrade) && $upgrade) {
            $type = 'upgrade';
        } else {
            $type = 'import';
        }
        
        $action = "-X POST ";
        $action .= "{$this->weburl}/rest/process/{$type}";
        $action .= ' -H "Content-Type: multipart/form-data"';
        $action .= " -F file=@'{$file}'";
         
        // Return
        return $this->execRest($action);
    }
    
    /**
     *
     * Parse a environment properties file and it generates an array
     * compatible with the ucd import webservice
     *
     * @param string $file
     *
     * @return boolean|$arrayParsed
     *
     */
    public function parseEnvironmentPropertiesFile($file = null) {
    	$string = file_get_contents($file);
    	$jsonFile = json_decode($string,true);
    
    	$arrayOutput = '{';
    	for($i=0; $i<count($jsonFile); $i++) {
    		for($j=0; $j<count($jsonFile[$i]['components']); $j++) {
    
    			// I populate a row with this format
    			//"SCRIPTFOLDER/SCRIPTNAME":"VALUE"
    			$row =  ' "' .$jsonFile[$i]['components'][$j][name] . '/' .
    					$jsonFile[$i]['components'][$j][propValue][name] .'":"'.
    					$jsonFile[$i]['components'][$j][propValue][value] . '"';
    
    			$arrayOutput .=  $row;
    
    			if (!(($i ==count($jsonFile)-1 && $j == count($jsonFile[$i]['components'])-1))){
    				$arrayOutput = $arrayOutput .' ,';
    			}
    		}
    	}
    	$arrayOutput .= '}';
    	$arrayParsed = str_replace('\/', '/', $arrayOutput);
    
    	//The end of line should be removed because it might break the import format
    	$arrayParsed = preg_replace('~[\r\n]+~', '', $arrayParsed);
    
    	return $arrayParsed;
    
    }
    
    /**
     *
     * Import the environment properties of an application
     *
     * @param string $file
     * @param string $environment
     *
     * @return boolean|string
     *
     */
    public function importEnvironmentProperties($file = null, $environment = null) {
    	if (empty($file)){
    		if (!empty($this->args[0])) {
    			$file = $this->args[0];
    		} else {
    			return false;
    		}
    	}
    
    	if (empty($environment)){
    		if (!empty($this->args[1])) {
    			$environment = $this->args[1];
    		} else {
    			return false;
    		}
    	}
    
    	$action = "-X PUT ";
    	$action .= "{$this->weburl}/rest/deploy/environment/{$environment}/componentProperties";
    
    	$environmentData = $this->parseEnvironmentPropertiesFile($file);
    
    	$action .= " --data-binary '{$environmentData}' --compressed";
    
    	// Return
    	return $this->execRest($action);
    	return true;
    }
    
    /**
     *
     * Export the environment properties of an application
     *
     * @param string $application
     * @return boolean|string
     */
    public function exportEnvironmentProperties($environment = null) {
        if (empty($environment)){
            if (!empty($this->args[0])) {
                $environment = $this->args[0];
            } else {
                return false;
            }
        }
        $this->destination = " -o '{$this->output}/EnvironmentProperties_{$environment}.json'";
        
        $action = "{$this->weburl}/rest/deploy/environment/{$environment}/componentProperties {$this->destination}";
        return $this->execRest($action);
    }
    
    /**
     *
     * Logout from existing connection and setup a new connection to a new server
     *
     * @param string $server
     * @param boolean $logout
     * @return boolean
     */
    protected function setupServer($server, $config, $logout = true) {
        if (empty($server)) {
            Utils::CLIerror("No server set");
            return false;
        }
        if (empty($config)) {
            Utils::CLIerror("No configuration set");
            return false;
        }
    
    
        // Logout from an existing connection
        if ($logout) {
            if ($this->connected) {
                Utils::CLIout("  Disconnecting from server: {$this->weburl}");
                $this->logout();
            }
        }
    
        // Setup connection to first server (origin)
        $this->weburl = $config[$server.'_weburl'];
        $this->username = $config[$server.'_username'];
        $this->password = $config[$server.'_password'];
        $this->authtoken = $config[$server.'_authtoken'];
        Utils::CLIout("  Connecting to '{$server}' server: {$this->weburl}");
        if (!empty($this->authtoken)) {
            $this->b64_login = base64_encode('PasswordIsAuthToken:{"token":"' . $this->authtoken . '"}');
        } else {
            $this->b64_login = null;
            return $this->login();
        }
    
        return true;
    }
}