<?php
/****************************************************************
 * IBM Confidential
*
* SFA100-Collaboration Source Materials
*
* (C) Copyright IBM Corp. 2014
*
* The source code for this program is not published or otherwise
* divested of its trade secrets, irrespective of what has been
* deposited with the U.S. Copyright Office
*
***************************************************************/

require_once('UrbanCodeClient.php');
require_once('Utils.php');

// Define return values for process status
define("INITIALIZED",2);
define("EXECUTING",3);
define("PENDING",4);
define("CLOSED_CANCEL",5);
define("CLOSED_SUCCESS",0);
define("CLOSED_FAILURE",255);
define("AGENT_NOT_ONLINE",100);

/**
 * Provides tools for creating and managing application process requests
 *
 * @class   ApplicationProcess
 * @author  Marco Bassi         marcobas@ie.ibm.com
 */

class ApplicationProcess extends UrbanCodeClient{
    
    protected $id;
    protected $status;
    protected $result;
    protected $pool_time;
    protected $json;
    
    public function __construct(){
        parent::__construct();
        $this->setReturn('json');
        self::init();
    }
    
    protected function init(){
        $this->id = null;
        $this->status = null;
        $this->result = null;
        $this->pool_time = 60;
        $this->json = null;
    }
    
    public function get($key = null) {
        if (empty($key)) {
            $array = array(
                'id' => $this->id,
                'status' => $this->status,
                'result' => $this->result,
                'pool_time' => $this->pool_time,
                'json' => $this->json,
            );
            return $array;
        } else {
            if (isset($this->$key)) {
                return $this->$key;
            } else {
                return false;
            }
        }
    }
    
    public function request($json) {
        if (!is_file($json) || is_dir($json)) {
            Utils::CLIerror("File '{$json}' doesn't exist, is a directory or either you don't have access.");
            return false;
        } else if ( !Utils::validateJSON(file_get_contents($json)) ){
            Utils::CLIerror("File '{$json}' is not a valid JSON file.");
            return false;
        }
        $config = json_decode(file_get_contents($json), true);
        Utils::CLIout("Verifying that required agents for '{$config['environment']}' environment are online: \n");
        $agents = self::getEnvironmentAgentsStatus($config['environment']);
        $online = true;
        foreach ($agents as $agent => $status){
            $online = $online && $status;
            if ($status) {
                Utils::CLIout("[ONLINE] {$agent}");
            } else {
                Utils::CLIout("[OFFLINE] {$agent}");
            }
        }
        
        if (!$online) {
            Utils::CLIerror("Not all required agents are online\n");
            exit (AGENT_NOT_ONLINE);
        }
        
        $this->json = $json;
        $output = $this->requestApplicationProcess($json);
        if ($output === false) {
            Utils::CLIerror("There was an error while requesting application process from file '{$json}'.");
            return false;
        }
        
        $output = Utils::outputToArray($output);
        $this->id = $output['requestId'];
        
        return $this->id;
    }
    
    /**
     * 
     * Retrieve status of current application process request
     *
     * @param boolean $update
     * @return boolean|int
     */
    public function status($update = false) {
        if (empty($this->id)){
            Utils::CLIwarning("Currently there is no application process request available.");
            return false;
        }
        
        if ($update === true || empty($this->status)) {
            // Get current request status
            $output = $this->getApplicationProcessRequestStatus($this->id);
            if ($output === false) {
                Utils::CLIerror("There was an error while retrieving status for application process request (id: '{$this->id}').");
                $this->status = null;
                return false;
            }
            
            $output = Utils::outputToArray($output);
            switch ($output['status']) {
                case 'INITIALIZED':
                    $this->status = INITIALIZED;
                    $this->result = "INITIALIZED";
                    break;
                case 'EXECUTING':
                    $this->status = EXECUTING;
                    $this->result = "EXECUTING";
                    break;
                case 'PENDING':
                    $this->status = PENDING;
                    $this->result = "PENDING";
                    break;
                case 'CLOSED':
                    switch ($output['result']) {
                        case 'SUCCEEDED':
                            $this->status = CLOSED_SUCCESS;
                            $this->result = "CLOSED - SUCCESS";
                            break;
                        case 'FAULTED':
                            $this->status = CLOSED_FAILURE;
                            $this->result = "CLOSED - FAILURE";
                            break;
                        case 'CANCELLED':
                            $this->status = CLOSED_CANCEL;
                            $this->result = "CLOSED - CANCELLED";
                            break;
                        default:
                            Utils::CLIerror("Server returned an unexpected value for application process request result (id: '{$this->id}')");
                            Utils::CLIout("{$output['status']}",true,"status");
                            Utils::CLIout("{$output['result']}",true,"result");
                            $this->status = null;
                            $this->result = null;
                            return false;
                    }
                    break;
                default:
                    Utils::CLIerror("Server returned an unexpected value for application process request status (id: '{$this->id}')");
                    Utils::CLIout("{$output['status']}",true,"status");
                    Utils::CLIout("{$output['result']}",true,"result");
                    $this->status = null;
                    return false;
            }
        }
        
        return $this->status;
    }
    
    /**
     * 
     * Loop process request status until timeout time is reached or a COMPLETE value is returned.
     * If Timeout is not set, wait for completion.
     *
     * @param boolean|int $timeout
     * @return boolean|int
     */
    public function waitCompletion($timeout = true){
        $currentStatus = $this->status;
        do {
            $status = $this->status(true);
            // Status has changed: print the new status
            if ($status != $currentStatus) {
                $currentStatus = $status;
                Utils::CLIinfo($this->result);
            }
            if ($status == CLOSED_CANCEL || $status == CLOSED_FAILURE || $status == CLOSED_SUCCESS ) {
                return $status;
            }
            $timeout -= $this->pool_time;
            sleep($this->pool_time);
        } while ($timeout);
        
        return $status;
    }
    
    /**
     * 
     * Set current application process request ID
     *
     * @param string $id
     * @return boolean
     */
    public function setRequestId($id){
        if (!empty($id)) {
            $this->id = $id;
            return true;
        } else {
            return false;
        }
    }
    
    /**
     * 
     * Giving an environment, returns the list of it's agents and the related status
     *
     * @param string $environment
     */
    public function getEnvironmentAgentsStatus($environment){
        $envData = $this->getEnvironment($environment);
        $envData = Utils::outputToArray($envData);
        
        $resources = $this->getEnvironmentResources($envData['id']);
        $resources = Utils::outputToArray($resources);
        
        $agents_status = self::getAgentsInEnvironmentResources($resources);
        return $agents_status;
    }
    
    /**
     * 
     * Get agents for the environment resource and their status
     *
     * @param string $resources
     * @return array
     */
    protected function getAgentsInEnvironmentResources($resources) {
        $return = array();
        foreach ($resources as $resource) {
            if (array_key_exists("type",$resource) && $resource['type'] == "agent"){
                if ($resource['status'] == "ONLINE") {
                    $agent = array($resource['name'] => true);
                } else {
                    $agent = array($resource['name'] => false);
                }
                $return = array_merge($return, $agent);
            } 
            if (array_key_exists("children",$resource)) {
                $children_agents = self::getAgentsInEnvironmentResources($resource['children']);
                $return = array_merge($return, $children_agents);
            }
        }
        
        return $return;
    }
    
    /**
     * 
     * Returns true if the agent is online, false if offline
     *
     * @param string $agent
     * @return boolean
     */
    public function isAgentOnline($agent) {
        $result = $this->getAgent($agent);
        $result = Utils::outputToArray($result);
        
        if ($result['status'] == "ONLINE") {
            return true;
        } else {
            return false;
        }
    }
}