/*******************************************************************************
 * Licensed Materials - Property of IBM
 * (c) Copyright IBM Corporation 2009, 2012. All Rights Reserved.
 *  
 * U.S. Government Users Restricted Rights:
 * Use, duplication or disclosure restricted by GSA ADP Schedule
 * Contract with IBM Corp.
 ******************************************************************************/
package com.ibm.rqm.url.client;

import java.io.IOException;
import java.net.HttpURLConnection;
import java.net.MalformedURLException;
import java.net.URISyntaxException;
import java.net.URL;
import java.security.GeneralSecurityException;

import org.apache.commons.httpclient.Cookie;
import org.apache.commons.httpclient.Header;
import org.apache.commons.httpclient.HttpClient;
import org.apache.commons.httpclient.HttpException;
import org.apache.commons.httpclient.HttpMethodBase;
import org.apache.commons.httpclient.NameValuePair;
import org.apache.commons.httpclient.UsernamePasswordCredentials;
import org.apache.commons.httpclient.auth.AuthScope;
import org.apache.commons.httpclient.cookie.CookiePolicy;
import org.apache.commons.httpclient.methods.GetMethod;
import org.apache.commons.httpclient.methods.PostMethod;
import org.apache.commons.httpclient.params.HttpClientParams;
import org.apache.commons.httpclient.protocol.Protocol;

public class RQMUrlUtlyJRSHttpsClient extends HttpClient {

	private String protocol;
	private String host;
	private int port;
	private URL serverURL;
	private final static String JAZZ_LOGOUT_URL = "/service/com.ibm.team.repository.service.internal.ILogoutRestService";

	/**
	 * @param protocol
	 * @param host
	 * @param port
	 * @throws MalformedURLException
	 */
	public RQMUrlUtlyJRSHttpsClient(String protocol, String host, int port)
			throws MalformedURLException {
		super();
		this.protocol = protocol;
		this.host = host;
		this.port = port;

		// compute server url
		this.serverURL = new URL(this.protocol, this.host, this.port, "/");

		getParams().setParameter(HttpClientParams.SINGLE_COOKIE_HEADER, true);
		String cookiePolicy = System.getenv("RQM_CLIENT_COOKIE_POLICY"); //NON-NLS-1$
		if (cookiePolicy == null) 
			cookiePolicy = CookiePolicy.BROWSER_COMPATIBILITY;
		getParams().setCookiePolicy(cookiePolicy);
	}

	public void registerTrustingSSL() throws GeneralSecurityException {
		Protocol.registerProtocol("https", new Protocol("https",
				TrustingSSLProtocolSocketFactory.getInstance(), 443));
	}

	public void printCookies(String x) {
		System.out.println("RQMUrlUtility: -------------------PrintCookies "
				+ x);
		Cookie[] cookies = getState().getCookies();
		for (int i = 0; i < cookies.length; ++i) {
			System.out.println(cookies[i].getName() + " = "
					+ cookies[i].getValue() + " domain = "
					+ cookies[i].getDomain());
		}
	}

	public String getCookie(String cookieName) {
		Cookie[] cookies = getState().getCookies();
		for (int i = 0; i < cookies.length; ++i) {
			if (cookies[i].getName().equals(cookieName)) {
				return cookies[i].getValue();
			}
		}
		return null;
	}
	
	public int logout(String context)
	{
		HttpMethodBase authenticationMethod = new PostMethod(this.serverURL.toString() + context + JAZZ_LOGOUT_URL);
		authenticationMethod.setFollowRedirects(false);
		UsernamePasswordCredentials credentials = (UsernamePasswordCredentials)super.getState().getCredentials(new AuthScope(host,port));
		if(null != credentials
				&& ! (credentials.getUserName().isEmpty() && credentials.getPassword().isEmpty()) )
		{
			authenticationMethod.addRequestHeader("Authorization:", "Base " + credentials.getUserName() + ":" + credentials.getPassword());
		}
		String body = "";
		String status = "";
		int responseCode = 0;;
		try {
			responseCode = executeMethod(authenticationMethod);
			//Header[] authenticateHeader = authenticationMethod.getResponseHeaders();
			body = authenticationMethod.getResponseBodyAsString();
			status = authenticationMethod.getStatusText();
		} catch (Exception e) {
			System.err
			.println("RQMUrlUtility: Log out error (response code) - "
					+ responseCode + "\nStatus Code:" + status + "\nResponse Body: " + body);
		}
		
		return responseCode;
	}

	public int login(String user, String password, String context) throws URISyntaxException,
			HttpException, IOException {

		GetMethod get = new GetMethod(this.serverURL.toString()
				+ context + "/auth/authrequired");
		int responseCode = executeMethod(get);

		followRedirects(get, responseCode);

		GetMethod get2 = new GetMethod(this.serverURL.toString()
				+ context + "/authenticated/identity");
		responseCode = executeMethod(get2);
		followRedirects(get2, responseCode);

		HttpMethodBase authenticationMethod = null;
		Header authenticateHeader = get2.getResponseHeader("WWW-Authenticate"); //$NON-NLS-1$
		
		//Configure the HTTP client for basic authentication:
		if ((responseCode == HttpURLConnection.HTTP_UNAUTHORIZED) && (authenticateHeader != null) && (authenticateHeader.getValue().toLowerCase().indexOf("basic realm") == 0)) { //$NON-NLS-1$
				
			super.getState().setCredentials(new AuthScope(host, port), new UsernamePasswordCredentials(user, password));

			authenticationMethod = new GetMethod(this.serverURL.toString() + "/authenticated/identity");
			responseCode = super.executeMethod(get2); 	
		}
		
		//Configure the HTTP client for form authentication:
		else{
			
			authenticationMethod = new PostMethod(this.serverURL.toString()
					+ context + "/j_security_check");
	
			NameValuePair[] nvps = new NameValuePair[2];
			nvps[0] = new NameValuePair("j_username", user);
			nvps[1] = new NameValuePair("j_password", password);
			((PostMethod)(authenticationMethod)).addParameters(nvps);
			((PostMethod)(authenticationMethod)).addRequestHeader("Content-Type", "application/x-www-form-urlencoded; charset=utf-8"); //$NON-NLS-1$ //$NON-NLS-2$
	
			responseCode = executeMethod(authenticationMethod);
						
			//Manually redirect the request
	        String redirectLocation;
	        Header locationHeader = authenticationMethod.getResponseHeader("location");
	        if (locationHeader != null) {
	            redirectLocation = locationHeader.getValue();
	            authenticationMethod.releaseConnection();
	            authenticationMethod=new GetMethod(redirectLocation);
	            executeMethod(authenticationMethod);
	            }
	        
	        
			Header failure = authenticationMethod
					.getResponseHeader("X-com-ibm-team-repository-web-auth-msg");
			if (failure!= null && failure.getValue().indexOf("authfailed") >= 0) {
				responseCode = HttpURLConnection.HTTP_UNAUTHORIZED;
			}
		}
		
		// Only proceed if there were no problems posting the authentication.
		if ((responseCode != HttpURLConnection.HTTP_OK) && (responseCode != HttpURLConnection.HTTP_MOVED_TEMP)) {
			if (responseCode != HttpURLConnection.HTTP_UNAUTHORIZED) {
				String body = "";
				try {
					body = authenticationMethod.getResponseBodyAsString();
				} catch (Exception e) {

				}
				System.err
						.println("RQMUrlUtility: Login error (response code) - "
								+ responseCode + "\nResponse Body: " + body);
			}
		} else {
			followRedirects(authenticationMethod, responseCode);

			GetMethod get3 = new GetMethod(
					this.serverURL.toString()
							+ context + "/service/com.ibm.team.repository.service.internal.webuiInitializer.IWebUIInitializerRestService/initializationData");
			responseCode = executeMethod(get3);
			followRedirects(get3, responseCode);
		}

		return responseCode;
	}

	public void followRedirects(HttpMethodBase method, int responseCode)
			throws HttpException, IOException {
		Header location = method.getResponseHeader("Location");
		while (location != null
				&& responseCode == HttpURLConnection.HTTP_MOVED_TEMP) {
			GetMethod get3 = new GetMethod(location.getValue());
			responseCode = executeMethod(get3);
			if (responseCode != HttpURLConnection.HTTP_OK) {
				System.out
						.println("RQMUrlUtility: After login, attempt to access init data returned: "
								+ responseCode);
			}
			location = get3.getResponseHeader("Location");

		}
	}
}