/*******************************************************************************
 * (c) Copyright IBM Corporation 2009, 2010. All Rights Reserved.
 *******************************************************************************/
package com.ibm.rqm.url.client;

import java.io.IOException;
import java.net.Socket;
import java.security.KeyManagementException;
import java.security.NoSuchAlgorithmException;

import javax.net.ssl.SSLContext;
import javax.net.ssl.TrustManager;

/**
 * @since 2.0.1
 */
public class SSLContextUtil {
    
    /**
     * Encryption algorithm TLS then SSL - IBM JREs
     */
    public static final String SSL_TLS = "SSL_TLS"; //$NON-NLS-1$
    /**
     * Encryption algorithm TLS then SSL - Sun JREs
     */
    public static final String SSLV3 = "SSLv3"; //$NON-NLS-N$
    /**
     * Encryption algorithm TLS
     */
    public static final String TLS = "TLS"; //$NON-NLS-1$
    /**
     * Encryption algorithm SSL 
     */
    public static final String SSL = "SSL"; //$NON-NLS-1$
        
    
    /**
     * Creates an SSL context factory.
     * The returned SSLContext will be created so that it is compatible with the
     * current security environment.  If a FIPS environment is detected then a
     * FIPS 140-2 complaint context will be returned. 
     * 
     * @return a {@link SSLContext}
     * @since 2.0.1
     */
    public static SSLContext createSSLContext(TrustManager trustManager) {
        SSLContext context = createSSLContext(SSL_TLS, trustManager);
        
        if (context == null) {
            System.out.println("Unable to create SSL_TLS context, trying SSLv3");  //$NON-NLS-1$
            // When SSL_TLS doesn't work (e.g. under FIPS or Sun JRE), try SSLv3
            context = createSSLContext(SSLV3, trustManager);
        }
        
        if (context == null) {
            System.out.println("Unable to create SSLv3 context, trying TLS");  //$NON-NLS-1$
            // When SSLv3 doesn't work (e.g. under FIPS), try TLS
            context = createSSLContext(TLS, trustManager);
        }

        if (context == null) {
            System.out.println("Unable to create TLS context, trying SSL"); //$NON-NLS-1$
            // Fall back to just SSL when the above two are not available
            context = createSSLContext(SSL, trustManager);
        }

        if (context == null) {
            /* No encryption algorithm worked.  Give up.  This should never happen
             * in any of our supported configurations. */
            throw new RuntimeException("No acceptable encryption algorithm found"); //$NON-NLS-1$
        }

        return context;
    }
    
    // Returns null when the given algorithm fails
    private static SSLContext createSSLContext(String algorithm, TrustManager trustManager) {
        SSLContext context;
        try {
            context = SSLContext.getInstance(algorithm);
        } catch (NoSuchAlgorithmException e) {
            System.out.println(e);
            return null;
        }
        try {
            context.init(null, new TrustManager[] { trustManager }, null);
        } catch (KeyManagementException e) {
            System.out.println(e);
            return null;
        }

        /* Create a socket to ensure this algorithm is acceptable.  This will
         * correctly disallow certain configurations (such as SSL_TLS under FIPS) */
        try {
            Socket s = context.getSocketFactory().createSocket();
            s.close();
        } catch (IOException e) {
            System.out.println(e);
            return null;
        } catch (IllegalArgumentException e) {
            System.out.println(e);
            return null;
        }
        return context;
    }   
}