/*******************************************************************************
 * (c) Copyright IBM Corporation 2009, 2012. All Rights Reserved.
 *******************************************************************************/
package com.ibm.rqm.url;

import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.net.MalformedURLException;
import java.net.ProtocolException;
import java.net.URL;
import java.util.HashMap;

import com.ibm.rqm.url.client.RQMUrlDELETE;
import com.ibm.rqm.url.client.RQMUrlGET;
import com.ibm.rqm.url.client.RQMUrlPOST;
import com.ibm.rqm.url.client.RQMUrlPUT;
import com.ibm.rqm.url.client.RQMUrlUtlyJRSHttpsClient;
import com.ibm.rqm.url.client.RQMUrlUtlyLogin;

/**
 * 
 * The "RQMUrlUtility" allows user to upload/download/delete an Attachment or an
 * Artifact to/from the RQM Server. "RQMUrlUtility" is a sample, stand alone
 * Java command line application that users can use as the basis of their
 * development efforts to GET and PUT/POST records to an RQM server via the RQM
 * REST API. This utility allows user to upload/download/delete an Attachment or
 * an Artifact to/from the RQM Server and its goal is to provide the sample code
 * that grants the insight to a toolsmith.
 * 
 * The RQM Rest API allows a user to perform CRUD type operations via XML
 * sequences to the RQM server. This sample will provide a code that exposes
 * those CRUD operations for GET, PUT, POST and DELETE of RQM records via HTTPS.
 * The term CRUD stands for Create, Read, Update, Delete which corresponds to
 * the basic database operations.
 * 
 * Requirements 
 * ==================================== 
 * RQMUrlUtility requires a Java 6 (1.6) or greater.
 * 
 * Running RQM Copy Utility 
 * ==================================== 
 * java -jar RQMUrlUtility.jar [arguments]
 * 
 * Argument Reference 
 * ==================================== 
 * RQMUrlutility -command -user -password -filepath -url [-help]
 * 
 * Command Lines Arguments explained  
 * ====================================
 * 
 * -command    : HTTP CRUD method (GET/POST/PUT/DELETE) 
 * 
 * -user       : Login user name
 * 
 * -password   : Login Password 
 * 
 * -filepath   : Complete path of file to which data
 * will be written after GET operation, or complete path of a file whose
 * contents are to be written using HTTP PUT. In case of an attachment, it is
 * the path where an attachment will be downloaded or complete path of file that
 * is to be uploaded.
 *  
 * -url        : Https Url reference -help : Help
 * 
 * For uploading data to the server, user need to specify the request method
 * (PUT/POST in this case) corresponding to the -command argument tag and user
 * should be logged into that server with authorized user name(-user) and
 * password(-password).
 * <p>
 * If a user wants to upload an attachment, -command is POST and user has to
 * pass the complete path where the attachment is located as the -filepath
 * argument tag and the URL reference(-url) where it needs to be uploaded. For
 * uploading other Artifacts, -command is PUT and user has to specify the
 * complete path of the file(-filepath) containing the XML that needs to be
 * uploaced and the URL reference (-url) where it needs to be uploaded.
 * <p>
 * Similarly for downloading an artifact from RQM server, a user should give GET
 * request tagged by -command argument and user has to be logged into RQM server
 * with authorized user name and password. For downloading an
 * Attachment/Artifact, user has to specify the path where he wants the file to
 * be saved. If an attachment is being downloaded, attachement will be saved at
 * the path mentioned by user for -filepath tag. And if an artifact is being
 * downloaded, the artifact XML will be written and saved in the file mentioned
 * by user at command line.
 * <p>
 * For deleting an Artifact/Attachment, -command DELETE is given. User needs to
 * be logged in with authorized user name and password and needs to specify the
 * URL reference of the Artifact/Attachment that is to be deleted.
 * 
 * <p>
 * <b>Input:</b> HTTP Command, User name, Password, File path and URL.
 * 
 * java -jar RQMUrlUtility.jar -command PUT -user ADMIN -password ADMIN
 * -filepath C:\\RQMUrlUtility_PUT.txt -url
 * https://localhost:9443/qm/service /com.ibm.rqm.integration.service
 * .IIntegrationService/resources/Quality%20Manager/testsuite/tsuite_n1
 * 
 * java -jar RQMUrlUtility.jar -command GET -user ADMIN -password ADMIN
 * -filepath C:\\GET_0709_1.txt -url
 * https://localhost:9443/qm/service/com.ibm.rqm.integration .service
 * .IIntegrationService/resources/Quality%20Manager/testscript/urn
 * :com.ibm.rqm:testscript:2
 * 
 * java -jar RQMUrlUtility.jar -command DELETE -user ADMIN -password ADMIN -url
 * https://localhost:9443/qm/service/com.ibm.rqm.integration.service
 * .IIntegrationService/resources/Quality%20Manager/testscript/tscript1
 * 
 * java -jar RQMUrlUtility.jar -help
 */

public class UrlUtility {

	protected static RQMUrlUtlyJRSHttpsClient httpclient;
	static RQMUrlUtlyLogin RQMUrlUtlyLogin;

	/**
	 * RQMUrlUtlyCommandLine main method
	 */
	private static HashMap<String, String> methARGS = new HashMap<String, String>();
	private static String getArg(String name) {
		String val = methARGS.get(name); 
		if (val == null) {
			System.err.println("RQMUrlUtility: Missing command line option: "+ name);
			manual_Help();
		}
		return val;
	}
	public static void main(String[] args) {

		System.out
				.println("                           ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^");
		System.out
				.println("                                  <<<<<----- [ RQMUrlUtility ] ---->>>>>");
		System.out
				.println("                           ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^\n");

		/*
		 * Reading Command Line Arguments into HashMap "methARGS".
		 */

		if (args.length == 0) {
			manual_Help();
			return;
		}
		for (int i = 0; i < args.length; i = i + 2) {
			if (args.length > i + 1) {
				methARGS.put(args[i], args[i + 1]);
			}
			if (args[i].equals("-help")){
				manual_Help();
				return;
			}
		}
		
		String sCommand = getArg("-command");
		if (sCommand == null) { return; }

		String sPath = methARGS.get("-filepath");
		if (sCommand.equals("DELETE")) {
			if (sPath != null) {
				System.err.println("RQMUrlUtility: -filepath option not valid for DELETE command");
				manual_Help();
				return; 
			}
		} else {
			if (sPath == null) {
				System.err.println("RQMUrlUtility: Missing command line option: -filepath");
				manual_Help();
				return;
			}			
		}
		
		String sUser = getArg("-user");
		if (sUser == null) { return; }

		String sPassword = getArg("-password");
		if (sPassword == null) { return; }


		String sAttachURL = getArg("-url");
		if (sAttachURL == null) { return; }

		String sContext = methARGS.get("-context");
		if (sContext == null) {
			try {
				//Resolve the URL (https://<server>:<port>/<context root>/...):
				URL url = new URL(sAttachURL);

				//Resolve the URL path (/<context root>/...):
				String urlPath = url.getPath();

				//Resolve the URL context root (<context root>):
				sContext = urlPath.substring(1, urlPath.indexOf('/', 1));
			} catch (Exception e) {
				throw new IllegalArgumentException("RQMUrlUtility: URL not valid for parsing of context");
			}
		}

		sAttachURL = sAttachURL.replaceAll(" ", "+");
		sAttachURL = sAttachURL.replaceAll("%20", "+");

		URL url = null;
		try {
			url = new URL(sAttachURL);
		} catch (MalformedURLException e) {
			System.out
					.println("RQMUrlUtility: MalformedURLException found either due to illegal protocol or the string was not parsed : "
							+ e);
			e.printStackTrace();
			return;
		}

		String protocoL = url.getProtocol();
		String hosT = url.getHost();
		int porT = url.getPort();

		/*
		 * Extracting Project Area from URL reference
		 */

		String token = "/resources/";
		int startIndx = sAttachURL.indexOf(token);
		int tokenLen = token.length();
		token = sAttachURL.substring(startIndx + tokenLen);
		startIndx = token.indexOf("/");
		String sProjArea = token.substring(0, startIndx);

		/*
		 * Login to RQM server to ensure that there is a valid logged in session
		 * and session cookie
		 */

		RQMUrlUtlyLogin = new RQMUrlUtlyLogin(protocoL, hosT, porT, sProjArea);
		int ret = RQMUrlUtlyLogin.login(sUser, sPassword, sContext);
		if (ret != 0)
			return;

		try {
			
			httpclient = RQMUrlUtlyLogin.getHttpClient();
			/*
			 * If user gives PUT as request command, transfer the contents file
			 * specified to the URL specified as Command Line Argument.
			 */

			if (sCommand.equalsIgnoreCase("PUT")) {
				new RQMUrlPUT(sAttachURL, sPath, httpclient, RQMUrlUtlyLogin);
			}

			/*
			 * If user gives POST as request command, transfer the file
			 * specified to the URL specified as Command Line Argument.
			 */

			else if (sCommand.equalsIgnoreCase("POST")) {
				new RQMUrlPOST(sAttachURL, sPath, httpclient, RQMUrlUtlyLogin);
			}

			/*
			 * If user gives GET as request command, get the contents of the
			 * specified URL to the file specified as Command Line Argument.
			 */

			else if (sCommand.equalsIgnoreCase("GET")) {
				new RQMUrlGET(sAttachURL, sPath, httpclient, RQMUrlUtlyLogin);
			}

			/*
			 * If a user wants to DELETE an artifact, delete artifact's URL
			 * reference specified at Command Line.
			 */

			else if (sCommand.equalsIgnoreCase("DELETE")) {
				new RQMUrlDELETE(sAttachURL, RQMUrlUtlyLogin);
			}

			else
				System.out
						.println("RQMUrlUtility: **** Invalid HTTP Request ****");

		}

		catch (MalformedURLException e) {
			System.out
					.println("RQMUrlUtility: MalformedURLException found either due to illegal protocol or the string was not parsed : "
							+ e);
			e.printStackTrace();
		}

		catch (UnsupportedEncodingException e) {
			System.out
					.println("RQMUrlUtility: Got an UnsupportedEncodingException as Character Encoding is not supported. : "
							+ e);
			e.printStackTrace();
		}

		catch (ProtocolException e) {
			System.out
					.println("RQMUrlUtility: ProtocolException due to error in underlying protocol: "
							+ e);
			e.printStackTrace();
		}

		catch (IOException e) {
			System.out.println("RQMUrlUtility: Got an IOException: " + e);
			e.printStackTrace();
		}
		finally
		{
			RQMUrlUtlyLogin.logout(sContext);
		}
	}
	

	private static void manual_Help() {

		String MANUAL = "\"RQMUrlUtility\" is a sample, stand alone Java command line application that users can use as the basis of their "
				+ "development efforts to GET and PUT/POST records to an RQM server via the RQM REST API. This utility allows user to "
				+ "upload/download/delete an Attachment or an Artifact to/from the RQM Server and its goal is to provide the sample code that "
				+ "grants the insight to a toolsmith.\r\n\nThe RQM Rest API  allows a user to perform CRUD type operations via XML sequences "
				+ "to the RQM server.  This sample will provide a code that exposes those CRUD operations for GET, PUT, POST and DELETE of RQM "
				+ "records via HTTPS.  The term CRUD stands for Create, Read, Update, Delete which corresponds to the basic database "
				+ "operations.\r\n\n\nRequirements\n====================================\nRQMUrlUtility requires a Java 6 (1.6) or greater."
				+ "\r\n\n\nRunning RQM Copy Utility\n====================================\n"
				+ "java -jar RQMUrlUtility.jar [arguments]\r\n\n\n"
				+ "Argument Reference\n====================================\n"
				+ "RQMUrlutility -command -user -password -context -filepath -url [-help]\n"
				+ "\n\nCommand Lines Arguments explained -\n====================================\r\n" 
				+ "-command          : HTTP CRUD method (GET/POST/PUT/DELETE)\n"
				+ "-user             : Login user name\n"
				+ "-password         : Login Password\n"
				+ "-context          : context of server.  In 1.0 and 2.0 the context was \"jazz\", in 3.0 it is \"qm\".  If unspecified defaults to \"qm\".\n"
				+ "-filepath         : Complete path of file to which data will be written after GET operation, or complete path of a file whose contents are to "
				+ "be written using HTTP PUT.\n                    In case of an attachment, it is the path where an attachment will be "
				+ "downloaded or complete path of file that is to be uploaded.\n"
				+ "-url	          : Https Url reference\n"
				+ "-help             : Help\r\n\n"
				+ "*) For uploading data to the server, user need to specify the request method (PUT/POST in this case) corresponding to the "
				+ "-command argument tag and user should be logged into that server with authorized user name (-user) and password (-pass"
				+ "word).\r\n*) If a user wants to upload an attachment, -command is POST and user has to pass the complete path where the "
				+ "attachment is located as the -filepath argument tag and the URL reference (-url) where it needs to be uploaded.\r\n"
				+ "*) For uploading other Artifacts, -command is PUT and user has to specify the complete path of the file (-filepath) "
				+ "containing the XML that needs to be uploaded and the URL reference (-url) where it needs to be uploaded.\r\n*) Similarly "
				+ "for downloading an artifact from RQM server, a user should give GET request tagged by -command argument and user has to be "
				+ "logged into RQM server with authorized user name and password. For downloading an Attachment/Artifact, user has to specify "
				+ "the path where he wants the file to be saved. If an attachment is being downloaded, attachement will be saved at the path "
				+ "mentioned by user for -filepath tag.\n   And if an artifact is being downloaded, the artifact XML will be written and saved in "
				+ "the file mentioned by user at command line.\r\n*) For deleting an Artifact/Attachment, -command DELETE is given. User needs "
				+ "to be logged in with authorized user name and password and needs to specify the URL reference of the Artifact/Attachment "
				+ "that is to be deleted.\r\n\n";

		String sample_AGRS = "SAMPLE COMMAND LINE ARGUMENTS -\n====================================\n" +
		"Note: These examples use the default context root (e.g. /qm/) for Rational Quality Manager 3.0.1 or later.  For more information, see the RQM Reportable REST API (https://jazz.net/wiki/bin/view/Main/RqmApi#contextRoot)." + 
		"\n" +
		"Note: These examples may contain URL encoded project aliases in resource URL references (-url).  When invoking the RQM URL Utility from a batch file, enclose resource URL references (-url) in double quotes." + 
		"\n";

		String USAGE1 = "FOR UPLOADING AN ATTACHMENT/ATRIFACT -\nUsage: RQMUrlUtility -command <HTTP Method> -user <user name> "
				+ "-password <password> -context <context> -filepath <file path> -url <url> \n";

		String USAGE2 = "Example: RQMUrlUtility -command PUT/POST -user ADMIN -password ADMIN -filepath C:\\temp.txt "
				+ "-url https://localhost:9443/qm/service/com.ibm.rqm.integration.service.IIntegrationService/resources"
				+ "/Quality%20Manager/testscript/tscript1 \n";

		String USAGE3 = "FOR DOWNLOADING AN ARTIFACT/ATTACHMENT  -\nUsage: RQMUrlUtility -command <HTTP Method> -user <user name>"
				+ "-password <password> -context <context> -filepath <file path> -url <url> \n";

		String USAGE4 = "Artifact Example: RQMUrlUtility -command GET -user ADMIN -password ADMIN -context jazz -filepath C:\\temp.txt "
				+ "-url https://localhost:9443/qm/service/com.ibm.rqm.integration.service.IIntegrationService/resources"
				+ "/Quality%20Manager/testscript/tscript1 \n";

		String USAGE5 = "Attachment Example: RQMUrlUtility -command GET -user ADMIN -password ADMIN -context jazz -filepath C:\\ "
				+ "-url https://localhost:9443/qm/service/com.ibm.rqm.integration.service.IIntegrationService/resources"
				+ "/Quality%20Manager/attachment/attach1 \n";

		String USAGE6 = "FOR DELETING AN ARTIFACT/ATTACHMENT  -\nUsage: RQMUrlUtility -command <HTTP Method> -user <user name> "
				+ "-password <password> -context <context> -url <url> \n";

		String USAGE7 = "Example: RQMUrlUtility -command DELETE -user ADMIN -password ADMIN -context jazz "
				+ "-url https://localhost:9443/qm/service/com.ibm.rqm.integration.service.IIntegrationService/resources"
				+ "/Quality%20Manager/testscript/tscript1 \n";

		System.out.println(MANUAL);
		System.out.println(sample_AGRS);
		System.out.println(USAGE1 + USAGE2);
		System.out.println(USAGE3 + USAGE4 + USAGE5);
		System.out.println(USAGE6 + USAGE7);
	}
}
