/*
* Licensed Materials - Property of IBM Corp.
* IBM UrbanCode Build
* IBM UrbanCode Deploy
* IBM UrbanCode Release
* IBM AnthillPro
* (c) Copyright IBM Corporation 2002, 2017. All Rights Reserved.
*
* U.S. Government Users Restricted Rights - Use, duplication or disclosure restricted by
* GSA ADP Schedule Contract with IBM Corp.
*/

import java.util.regex.Pattern
import org.apache.commons.io.filefilter.WildcardFileFilter
import org.apache.commons.io.comparator.NameFileComparator

import com.urbancode.air.AirPluginTool

final AirPluginTool airPluginTool = new AirPluginTool(args[0], args[1])

final Properties props = airPluginTool.getStepProperties()
final def workDir = new File('.').canonicalFile

// Gather Properties
def AGENT_HOME = new File(System.getenv("AGENT_HOME").toString() + File.separator + "var" + File.separator + "temp")
def separator = '\n|[\n]|\r\n|[\r\n]'
def sqlPlusExecutable = props['sqlPlusExecutable']
println "Executable : " + sqlPlusExecutable
def user = props['user'].trim()
println "User : ${user.trim()?:'Oracle Wallet'}"
def password = props['password']
def connectionID = props['connectionID']
println "ConnectionID : " + connectionID
def unparsedSqlFiles = props['sqlFiles']
unparsedSqlFiles = unparsedSqlFiles.replaceAll(separator, ",")
def sqlFiles = unparsedSqlFiles.split(',')
def failOnORA = props['failOnORA']?.toBoolean()
println "SQL Files : "
sqlFiles.each {
    println "\t" + it
}

def existingScripts = new ArrayList<String>()
// Iterate through each given file and confirm existence
sqlFiles.each { sqlFileName ->
    sqlFileName = sqlFileName.trim()
    if (sqlFileName.contains("*") || sqlFileName.contains("?")) {
        def wildcardSqlFile = new File(sqlFileName)
        def wildcard = wildcardSqlFile.getName()
        FilenameFilter filter = new WildcardFileFilter(wildcard)
        def parentDir = wildcardSqlFile.getParentFile()

        if (parentDir == null) {
            parentDir = workDir
        }

        def filterFiles = parentDir.listFiles((FilenameFilter) filter)
        def matchingFiles = []
        if (filterFiles != null) {
            matchingFiles = Arrays.asList(filterFiles)
        }
        Comparator<File> nameComparator = NameFileComparator.NAME_COMPARATOR
        Collections.sort(matchingFiles, nameComparator)

        matchingFiles.each { matchingFile ->
            existingScripts.add(matchingFile.getCanonicalPath())
        }

    } else if (! (new File(sqlFileName)).exists()) {
        throw new RuntimeException("[Error] SQL File ${sqlFileName} does not exist!")
    } else {
        existingScripts.add(sqlFileName)
    }
}


// Create temporary files which include error-handling, since SQLPlus exits with code 0 by default
def editedFiles = []
existingScripts.each { sqlFile ->
    try {
        def line
        File sqlContents = new File(sqlFile)
        new File(sqlFile).withReader { line = it.readLine() }
        def temp = File.createTempFile('temp', '.txt', AGENT_HOME)
        temp.deleteOnExit()
        temp << sqlContents.text << System.getProperty("line.separator") << "SHOW ERRORS;"
        editedFiles.add(temp.absolutePath)
    }
    catch (e) {
        throw new RuntimeException("[Error] SQL File ${sqlFile} does not exist!")
    }
}

// Iterate through the additionalEnvVars property and confirm validity
def unparsedEnvVars = props['additionalEnvVars']
unparsedEnvVars = unparsedEnvVars.replace(separator, ",").split(',')
def envVars = unparsedEnvVars*.trim()
envVars.removeAll([""])
println "Environment Variables :"
for (var in envVars) {
    println "\t" + var
    def pair = var.split('=')
    if (pair.length > 2 || !pair[0].trim() || !var.contains('=')) {
        println "[Error] Invalid environment variable given."
        println "[Possible Solution] Environment variable pair does not contain an '=' chracter to separate the variable and value."
        println "[Possible Solution] Environment variable pair contains more than one '=' character."
        println "[Possible Solution] The environment variable is empty or null."
        println "[Possible Solution] Delimit each variable and value with an '=' and separate each pair with a new line or comma."
        System.exit(1)
    }
}
def oracleHome = props['oracleHome']
println "Oracle Home: ${oracleHome?:''}"

// Verify authorization credentials are specified
// Use oracle wallet if neither user and password are specified
if (!user && !password){
    println "[Ok] The User name and Password are empty. Using Oracle Wallet."
    // Fail if user/password, and oracleHome are not specified.
    if (!oracleHome){
        println "[Error] Authorization credentials must be specified."
        println "[Possible Solution] Specify both 'User name' and 'Password' for traditional login."
        println "[Possible Solution] Specify 'Oracle Home', and leave 'User name' and 'Password' blank, to use " +
                "Oracle Wallet authentication. Oracle Wallet must have been previously configured outside of IBM UrbanCode Deploy."
        System.exit(1)
    }
}
// If XOR User and Password, then crash
else if ((user && !password) || (!user && password)){
    println "[Error] Authorization credentials must be specified."
    println "[Possible Solution] Specify both 'User name' and 'Password' for traditional login."
    println "[Possible Solution] Specify 'Oracle Home', and leave 'User name' and 'Password' blank, to use " +
            "Oracle Wallet authentication. Oracle Wallet must have been previously configured outside of IBM UrbanCode Deploy."
    System.exit(1)
}

// Create and run the sqlplus command
int i = 0
editedFiles.each { existingFileName ->
    def cmdArgs = [sqlPlusExecutable, "${user}/${password}@${connectionID}"]
    println ""
    println "************************************************************************************************"
    println "[Action] Executing : " + cmdArgs.join(' ') + " @" + existingScripts[i]
    cmdArgs << '@'+existingFileName

    def proc
    if (oracleHome) {
        def env = []
        System.getenv().each {
            env << it.key + '=' + it.value
        }
        for (var in envVars) {
            env << var
        }
        env << "ORACLE_HOME=$oracleHome"
        proc = cmdArgs.execute(env, workDir)
    }
    else {
        proc = cmdArgs.execute()
    }
    proc.withWriter { writer ->
        writer.write("exit${System.getProperty("line.separator")}")
    }

    def out = new ByteArrayOutputStream()
    def err = new ByteArrayOutputStream()
    proc.waitForProcessOutput(out, err)
    println out
    println err
    proc.waitFor()
    if (proc.exitValue() != 0) {
        throw new RuntimeException("Process Failed!")
    }
    // Check for SQLPlus errors, since they will exit with exitValue 0
    def outString = out.toString()
    if (Pattern.compile("SP2-[\\d{4}]").matcher(outString).find()){
        println "[Error] SQLPlus error! See SQLPlus Error Documentation to see what action is required."
        System.exit(1)
    }
    if ((failOnORA) && (Pattern.compile("ORA-[\\d{5}]").matcher(outString).find())){
        println "[Error] SQLPlus Database Message! See SQLPlus Error Documentation to see what action is required."
        System.exit(1)
    }
    i++
}
System.exit(0)
