/*
* Licensed Materials - Property of IBM Corp.
* IBM UrbanCode Build
* IBM UrbanCode Deploy
* IBM UrbanCode Release
* IBM AnthillPro
* (c) Copyright IBM Corporation 2002, 2016. All Rights Reserved.
*
* U.S. Government Users Restricted Rights - Use, duplication or disclosure restricted by
* GSA ADP Schedule Contract with IBM Corp.
*/
import java.util.*
import java.io.*
import org.apache.commons.io.filefilter.WildcardFileFilter
import org.apache.commons.io.comparator.NameFileComparator

final def workDir = new File('.').canonicalFile
final def props = new Properties();
final def inputPropsFile = new File(args[0]);
try {
    inputPropsStream = new FileInputStream(inputPropsFile);
    props.load(inputPropsStream);
}
catch (IOException e) {
    throw new RuntimeException(e);
}

// Gather Properties
def sqlPlusExecutable = props['sqlPlusExecutable'];
println "Executable : " + sqlPlusExecutable;
def user = props['user'].trim();
println "User : ${user.trim()?:'Oracle Wallet'}";
def password = props['password'];
def connectionID = props['connectionID'];
println "ConnectionID : " + connectionID;
def unparsedSqlFiles = props['sqlFiles'];
unparsedSqlFiles = unparsedSqlFiles.replace('\n', ",");
def sqlFiles = unparsedSqlFiles.split(',');
println "SQL Files : ";
sqlFiles.each {
    println "\t" + it;
}

// Iterate through the additionalEnvVars property and confirm validity
def unparsedEnvVars = props['additionalEnvVars'];
unparsedEnvVars = unparsedEnvVars.replace('\n', ",").split(',')
def envVars = unparsedEnvVars*.trim()
envVars.removeAll([""])
println "Environment Variables :"
for (var in envVars) {
    println "\t" + var
    def pair = var.split('=')
    if (pair.length > 2 || !pair[0].trim() || !var.contains('=')) {
        println "[Error] Invalid environment variable given."
        println "[Possible Solution] Environment variable pair does not contain an '=' chracter to separate the variable and value."
        println "[Possible Solution] Environment variable pair contains more than one '=' character."
        println "[Possible Solution] The environment variable is empty or null."
        println "[Possible Solution] Delimit each variable and value with an '=' and separate each pair with a new line or comma."
        System.exit(1)
    }
}
def oracleHome = props['oracleHome']
println "Oracle Home: ${oracleHome?:''}"

// Verify authorization credentials are specified
// Use oracle wallet if neither user and password are specified
if (!user && !password){
    println "[Ok] The User name and Password are empty. Using Oracle Wallet."
    // Fail if user/password, and oracleHome are not specified.
    if (!oracleHome){
        println "[Error] Authorization credentials must be specified."
        println "[Possible Solution] Specify both 'User name' and 'Password' for traditional login."
        println "[Possible Solution] Specify 'Oracle Home', and leave 'User name' and 'Password' blank, to use " +
                    "Oracle Wallet authentication. Oracle Wallet must have been previously configured outside of IBM UrbanCode Deploy."
        System.exit(1)
    }
}
// If XOR User and Password, then crash
else if ((user && !password) || (!user && password)){
    println "[Error] Authorization credentials must be specified."
    println "[Possible Solution] Specify both 'User name' and 'Password' for traditional login."
    println "[Possible Solution] Specify 'Oracle Home', and leave 'User name' and 'Password' blank, to use " +
                "Oracle Wallet authentication. Oracle Wallet must have been previously configured outside of IBM UrbanCode Deploy."
    System.exit(1)
}
def existingScripts = new ArrayList<String>()

// Iterate through each given file and confirm existance
sqlFiles.each { sqlFileName ->

    if (sqlFileName.contains("*") || sqlFileName.contains("?")) {
        def wildcardSqlFile = new File(sqlFileName)
        def wildcard = wildcardSqlFile.getName()
        FilenameFilter filter = new WildcardFileFilter(wildcard)
        def parentDir = wildcardSqlFile.getParentFile()

        if (parentDir == null) {
            parentDir = workDir
        }

        def filterFiles = parentDir.listFiles((FilenameFilter) filter);
        def matchingFiles = [];
        if (filterFiles != null) {
            matchingFiles = Arrays.asList(filterFiles);
        }
        Comparator<File> nameComparator = NameFileComparator.NAME_COMPARATOR
        Collections.sort(matchingFiles, nameComparator)

        matchingFiles.each { matchingFile ->
            existingScripts.add(matchingFile.getCanonicalPath())
        }

    } else if (! (new File(sqlFileName)).exists()) {
        throw new RuntimeException("[Error] SQL File ${sqlFileName} does not exist!")
    } else {
        existingScripts.add(sqlFileName)
    }
}

// Create and run the sqlplus command
existingScripts.each { existingFileName ->
    def cmdArgs = [sqlPlusExecutable, "${user}/${password}@${connectionID}", '@'+existingFileName];
    println ""
    println "************************************************************************************************"
    println "[Action] Executing : " + cmdArgs.join(' ');

    def proc
    if (oracleHome) {
        def env = []
        System.getenv().each {
            env << it.key + '=' + it.value
        }
        for (var in envVars) {
            env << var
        }
        env << "ORACLE_HOME=$oracleHome"
        proc = cmdArgs.execute(env, workDir)
    }
    else {
        proc = cmdArgs.execute()
    }
    proc.withWriter { writer ->
       writer.write("exit\n");
    }
    
    proc.waitForProcessOutput(System.out, System.out);
    proc.waitFor();
    if (proc.exitValue() != 0) {
        throw new RuntimeException("Process Failed!");
    }
}
System.exit(0);

