/*
* Licensed Materials - Property of IBM Corp.
* IBM UrbanCode Build
* IBM UrbanCode Deploy
* IBM UrbanCode Release
* IBM AnthillPro
* (c) Copyright IBM Corporation 2002, 2013. All Rights Reserved.
*
* U.S. Government Users Restricted Rights - Use, duplication or disclosure restricted by
* GSA ADP Schedule Contract with IBM Corp.
*/
package com.urbancode.air.plugin.selenium
import com.urbancode.air.CommandHelper;

public class SeleniumHelper {

    final File resourceHome = new File(System.getenv()['PLUGIN_HOME'])
    final def workDir = new File('.').canonicalFile
    def ch = new CommandHelper(workDir);
    final def out = System.out;
    def seleniumJar;
    def webBrowser;
    def startingUrl;
    def testSuite;
    def testResults;
    def portNumber;
    def failurePercentThreshold = null;

    SeleniumHelper(def props) {
        seleniumJar = props['seleniumJar'];
        webBrowser = props['webBrowser'];
        startingUrl = props['startingUrl'];
        testSuite = props['testSuite'];
        testResults = props['testResults'];
        portNumber = props['portNumber'];
        if (props['failurePercentThreshold']) {
            failurePercentThreshold = Integer.valueOf(props['failurePercentThreshold']);
        }
        println failurePercentThreshold
        if (failurePercentThreshold) {
            if ((0 > failurePercentThreshold) || (failurePercentThreshold > 100)) {
                throw new IllegalStateException('failurePercentThreshold error: Failure Percent Threshold must be between 0 and 100')
            }
        }
        else {
            failurePercentThreshold = 100;
        }
        if (portNumber && !portNumber.isInteger()){
            throw new IllegalStateException('Port must be an integer');
        }
        ch.ignoreExitValue(true);
    }
    
    public def runTest() {
        try {
            def args = [];
            args = ['java', '-jar', seleniumJar, '-multiWindow', '-htmlSuite', webBrowser, startingUrl, testSuite, testResults];
            if (portNumber) {
                args << '-port'; 
                args << portNumber;
            }
            
            def initialSize = 4098;
            def errStream = new ByteArrayOutputStream(initialSize)
            def printStream = new PrintStream(errStream, true);

            def status = ch.runCommand("Running html test suite ${testSuite}", args) { proc ->
                proc.out.close();
                try {
                   proc.waitForProcessOutput(out, printStream);
                }
                finally {
                    out.flush();
                }
            }
			def errMessage = errStream
            
            if (status) {
                    if (errMessage =~ 'Tests failed, see result file for details'){
                         println("Selenium tests ran successfully\n")
                    }
                    else if (errMessage =~ 'SystemRoot apparently not set!') {
                        println ('Error may be caused by browser: ' + webBrowser + ' not being installed\n')
                        throw new Exception("Failed to run test server with exit code: ${status}")
                    }
                        else {
                        println ('Unknown Exception Occured')
                        throw new Exception("Failed to run test server with exit code: ${status}")
                    }
            }
            else {
                println("Selenium tests ran successfully")
            }
        }
        catch (Exception e) {
            println e
            System.exit(1)
        }
    }
    
    public def checkResults() {


        def outputFile = new File(testResults)
        def replaced = outputFile.text.replaceAll("&nbsp", " ")
        if (replaced.indexOf("&nbsp") > 0) {
            throw new Exception("not replaced!")
        }
        def status 
        def suiteName
        def failedTests = [] 
        def passedTests = []
        def slurper = new XmlSlurper()
        def htmlParser = slurper.parseText(replaced)

        htmlParser.'**'.findAll{ it.@class == 'title status_passed'}.each {
            passedTests.add(it)
        }

        htmlParser.'**'.findAll{ it.@class == 'title status_failed' || it.@class == 'title  status_failed' }.each {
            failedTests.add(it)
        }

        //this removes the overall status and gets just the case statuses
        if (failedTests.size() > 0) {
            status = 'Failed';
            suiteName = failedTests[0]
            failedTests.remove(0)
        }
        else {
            status = 'Success'
            suiteName = passedTests[0]
            passedTests.remove(0)
        }
        int successes = passedTests.size()
        int failures = failedTests.size()
        int failurePercent = failures / (failures + successes) * 100
        println "Done counting pass/fail ($successes passed, $failures failed) - about to generate test results XML"
        println ""
        println "****************************************************************************************************"
        println "****************************************************************************************************"
        println "****************************************************************************************************"
        println ""
        println ""
        println ""
        
        //print the results html here
        outputFile.eachLine { line ->
           println line
        }

        println ""
        println ""
        println ""
        println "****************************************************************************************************"
        println "****************************************************************************************************"
        println "****************************************************************************************************"
   
        if (failurePercent > failurePercentThreshold) {
           throw new Exception("Too many tests failed");
        }
    }
}
	
