/*
* Licensed Materials - Property of IBM Corp.
* IBM UrbanCode Build
* IBM UrbanCode Deploy
* IBM UrbanCode Release
* IBM AnthillPro
* (c) Copyright IBM Corporation 2002, 2013. All Rights Reserved.
*
* U.S. Government Users Restricted Rights - Use, duplication or disclosure restricted by
* GSA ADP Schedule Contract with IBM Corp.
*/

import com.urbancode.air.AirPluginTool
import com.urbancode.air.CommandHelper

final def workDir = new File('.').canonicalFile
final def out = System.out
final def env = System.getenv()

final def isWindows = (System.getProperty('os.name') =~ /(?i)windows/).find()
final def ahptool = isWindows ? 'ahptool.cmd' : 'ahptool'

final def DATE_READ_1 = new java.text.SimpleDateFormat("E MMM dd HH:mm:ss zzz yyyy") // java.util.Date.toString()
final def DATE_READ_2 = new java.text.SimpleDateFormat("yyyy-MM-dd HH:mm:ss z") // ISO-like format
final def SVN_DATE = new java.text.SimpleDateFormat("yyyy-MM-dd HH:mm:ss.S Z")
SVN_DATE.timeZone = TimeZone.getTimeZone("GMT")

//------------------------------------------------------------------------------
// GET ALL INPUT PARAMETERS 
//------------------------------------------------------------------------------
final def parseDate = { value ->
    if (!value) {
        return null
    }
    if (value.isLong()) {
        return new Date(value.toLong())
    }
    else {
        try {
            return DATE_READ_1.parse(value)
        }
        catch (java.text.ParseException e) {
            return DATE_READ_2.parse(value)
        }
    }
}


final def apTool = new AirPluginTool(this.args[0], this.args[1])
final def props = apTool.getStepProperties()

final def svnRoot      = props['repositoryUrl']
final def svnUsr       = props['username']
final def svnPwd       = props['password']
final def projectPath     = props['projectPath']
final def dir          = new File(workDir, props['dirOffset'] ?: '.').canonicalFile // get checkout directory
final def commandPath = props['commandPath'] ?: "svn"
final def trustCertificate     = props['trustCertificate']


final def revisionNum  = props['revisionNum']
final def revisionDate = parseDate(props['revisionDate'])

final boolean doExport = true
final boolean clean    = props['cleanWorkspace']?.toBoolean()

final def pb = new ProcessBuilder(['echo'] as String[]).directory(workDir)

//------------------------------------------------------------------------------
// PREPARE WORKING DIRECTORY
//------------------------------------------------------------------------------

  // clean && create the working directory
  if (clean && dir.isDirectory()) {
    new AntBuilder().delete(includeemptydirs:'true') {
        fileset(dir: workDir.path, includes:'**/*', defaultexcludes:'false')
    }
  }
  
  dir.mkdirs()
  
  if (!dir.exists()) {
    throw new Exception("Could not create working directory ${workDir}")
  }
  else if (dir.isFile()) {
    throw new Exception("Specified working directory is a file!! ${workDir}")
  }

//------------------------------------------------------------------------------
// PREPARE COMMAND LINE
//------------------------------------------------------------------------------
  
	def command = [commandPath, '--non-interactive', '--no-auth-cache']
	
    if (doExport) {
        command << 'export' << '--force' // need force because target directory does/may exist
    }
    else {
        command << 'checkout'
    }
	
    if (svnUsr) {
      command << '--username' << svnUsr

      if (svnPwd) {
          command << '--password' << svnPwd
      }
    }

    if (revisionNum) {
      command << '--revision' << revisionNum
    }
    else if (revisionDate) {
      command << '--revision' << ('{'+SVN_DATE.format(revisionDate)+'}')
    }

    if (trustCertificate=='true') {
      command << '--trust-server-cert'
    }

    command << (svnRoot+'/'+projectPath)

    command << dir.path // check out to workDir

    //------------------------------------------------------------------------------
    // EXECUTE
    //------------------------------------------------------------------------------

    new CommandHelper(workDir).runCommand('Subversion Export', command)
