/*
 * Licensed Materials - Property of IBM Corp.
 * IBM UrbanCode Deploy
 * (c) Copyright IBM Corporation 2016, 2017. All Rights Reserved.
 *
 * U.S. Government Users Restricted Rights - Use, duplication or disclosure restricted by
 * GSA ADP Schedule Contract with IBM Corp.
 */
import com.urbancode.air.AirPluginTool
import com.urbancode.air.plugin.wldeploy.ResourceHelper
import com.urbancode.air.plugin.wldeploy.WebLogicStatusHelper
import javax.management.ObjectName

AirPluginTool apTool = new AirPluginTool(args[0], args[1])
def props = apTool.getStepProperties(System.getenv("UCD_SECRET_VAR"))

def wlHelper = new WebLogicStatusHelper()

def user
def password
def configFile = props['configFile']
def keyFile = props ['keyFile']

if (configFile && keyFile) {
    user = wlHelper.getUser(configFile, keyFile)
    pass = wlHelper.getPass(configFile, keyFile)
}
else {
    user = props['user']
    password = props['password']
}

if (!user || !password) {
    throw new RuntimeException("Username/Password not specified.")
}

// create connection to weblogic jmx administration server
def url = props['url']
wlHelper.initConn(url, user, password)

// ucd setup
def ucdUser = apTool.getAuthTokenUsername()
def ucdPass = apTool.getAuthToken()
def ucdUri = new URI(System.getenv("AH_WEB_URL"))
def ucdHelper = new ResourceHelper(ucdUri, ucdUser, ucdPass)

def rootPath = props['resourcePath']

def domainFolderPath = rootPath + "/Domains"
def domainFolderDescription = "All existing domains managed by your WebLogic administration server at ${props['url']}."
def domainFolderResource = ucdHelper.getOrCreateSubResource(domainFolderPath, rootPath, "Domains", domainFolderDescription)

// create domains
ObjectName[] domains = wlHelper.getDomains()
def domainMap = [:]

for (def domain : domains) {
    def domainKeyProperties = domain.getKeyPropertyList()

    // eliminate duplicate entries with separate locations
    if (domainKeyProperties.containsKey("Location")) {
        continue
    }

    def domainName = domainKeyProperties.get("Name")
    def domainPath = domainFolderPath + "/${domainName}"
    def domainDescription = "WebLogic domain with name ${domainName}"
    def domainResource = ucdHelper.getOrCreateSubResource(domainPath, domainFolderPath, domainName, domainDescription)

    def domainProperties = new HashMap<String, String>()
    domainProperties.put("wldeploy.domain", domainName)
    domainMap.put(domainName, domainPath)
    ucdHelper.addRoleToResource(domainResource, "WebLogic-Domain", domainProperties)

    // create server folders for the domain resource
    def serverFolderPath = domainPath + "/Servers"
    def serverFolderDescription = "All existing servers in the ${domainName} domain."
    def serverFolder = ucdHelper.getOrCreateSubResource(serverFolderPath, domainPath, "Servers", serverFolderDescription)

    // create cluster folders for the domain resource
    def clusterFolderPath = domainPath + "/Clusters"
    def clusterFolderDescription = "All existing clusters in the ${domainName} domain."
    def clusterFolder = ucdHelper.getOrCreateSubResource(clusterFolderPath, domainPath, "Clusters", clusterFolderDescription)
}

// create servers
ObjectName[] servers = wlHelper.getServers()
for (def server : servers) {
    def domainName = server.getKeyProperty("Location")

    def serverFolderPath
    // only create server resources that belong to a domain, ignore irrelevant entries
    if (domainMap.containsKey(domainName)) {
        serverFolderPath = domainMap.get(domainName) + "/Servers"
    }
    else {
        continue
    }

    def serverName = server.getKeyProperty("Name")
    def serverPath = serverFolderPath + "/${serverName}"
    def serverDescription = "WebLogic server with name ${serverName}"
    def serverResource = ucdHelper.getOrCreateSubResource(serverPath, serverFolderPath, serverName, serverDescription)

    def serverProperties = new HashMap<String, String>()
    serverProperties.put("wldeploy.targets", serverName)
    ucdHelper.addRoleToResource(serverResource, "WebLogic-Server", serverProperties)
}

// create clusters
ObjectName[] clusters = wlHelper.getClusters()
for (def cluster : clusters) {
    def domainName = cluster.getKeyProperty("Location")

    def clusterFolderPath
    // only create cluster resources that belong to a domain, ignore irrelevant entries
    if (domainMap.containsKey(domainName)) {
        clusterFolderPath = domainMap.get(domainName) + "/Clusters"
    }
    else {
        continue
    }

    def clusterName = cluster.getKeyProperty("Name")
    def clusterPath = clusterFolderPath + "/${clusterName}"
    def clusterDescription = "WebLogic cluster with name ${clusterName}"
    def clusterResource = ucdHelper.getOrCreateSubResource(clusterPath, clusterFolderPath, clusterName, clusterDescription)

    def clusterProperties = new HashMap<String, String>()
    clusterProperties.put("wldeploy.targets", clusterName)
    ucdHelper.addRoleToResource(clusterResource, "WebLogic-Cluster", clusterProperties)
}

println("WebLogic Auto-Configuration has completed. Please close this window.")