package com.urbancode.air.plugin.wlst.helper

import com.urbancode.air.AirPluginTool
import com.urbancode.air.CommandHelper
import com.urbancode.air.FileSet

class WLSTCmdHelper {
    AirPluginTool apTool
    CommandHelper helper
    def props
    def isWindows
    def wlstFile
    def username
    def password

    WLSTCmdHelper(AirPluginTool apTool) {
        this.apTool = apTool
        helper = new CommandHelper()
        props = apTool.getStepProperties()
        isWindows = apTool.isWindows
        def wlstPath = props['wlstPath']
        def wlst = isWindows ? "wlst.cmd" : "wlst.sh"
        wlstFile = new File(wlstPath, wlst)
        username = props['username']
        password = props['password']
    }

    def runServerStateCommand(def state) {
        def url = props['serverUrl']
        def serverName = props['serverName']

        def cmdArgs = [wlstFile.absolutePath]
        def scriptFile = createTempFile("change_server_state")

        scriptFile.withWriter {out ->
            out << "connect('${username}','${password}'" + (url ? ",'${url}')\n" : ")\n")
            out << "${state}('${serverName}')"
        }

        cmdArgs << "${scriptFile.absolutePath}"

        println("Executing script file ${scriptFile.name}:")
        println(scriptFile.text)

        switch (state) {
            case 'shutdown':
                helper.runCommand("Shutting down server: ${serverName}", cmdArgs)
                break
            case 'suspend':
                helper.runCommand("Suspending server: ${serverName}", cmdArgs)
                break
            case 'resume':
                helper.runCommand("Resuming server: ${serverName}", cmdArgs)
                break
            default :
                println "Command not recognized."
                throw new Exception("${state} is not a known command.")
        }
    }

    def runStartServerCommand() {
        def domainName = props['domainName']
        def domainDir = props['domainDir']
        def serverName = props['serverName']
        def cmdArgs = [wlstFile.absolutePath]
        def scriptFile = createTempFile("start_server")

        scriptFile.withWriter {out ->
            out << "startNodeManager()\n"
            out << "nmConnect(username='${username}', password='${password}', domainName='${domainName}', domainDir='${domainDir}')\n"
            out << "nmStart('${serverName}')"
        }

        cmdArgs << "${scriptFile.absolutePath}"

        println("Executing script file ${scriptFile.name}:")
        println(scriptFile.text)

        helper.runCommand("Starting server: ${serverName}", cmdArgs)
    }

    def runApplicationCommand(def command) {
        final def workDir = new File('.').canonicalFile
        def appName = props['appName']
        def fileName = props['fileName']
        def url = props['url']
        def cmdArgs = [wlstFile.absolutePath]
        def scriptFile = createTempFile("app_command")

        def file = new File("${workDir.absolutePath}", "${fileName}")
        def cmdMessage
        def successMessage

        scriptFile.withWriter {out ->
            out << "connect('${username}','${password}'" + (url ? ",'${url}')\n" : ")\n")

            switch (command) {
                case 'deploy':
                    out << "deploy('${appName}', '${file.absolutePath}')"
                    cmdArgs << scriptFile.absolutePath
                    cmdMessage = "Deploying: ${fileName}"
                    successMessage = "Successfully deployed to application: ${appName}"
                    break
                case 'undeploy':
                    out << "undeploy('${appName}')"
                    cmdArgs << scriptFile.absolutePath
                    cmdMessage = "Undeploying: ${appName}"
                    successMessage = "Successfully undeployed application: ${appName}"
                    break
                case 'start':
                    out << "startApplication('${appName}')"
                    cmdArgs << scriptFile.absolutePath
                    cmdMessage = "Starting Application: ${appName}"
                    successMessage = "Successfully started application: ${appName}"
                    break
                case 'stop':
                    out << "stopApplication('${appName}')"
                    cmdArgs << scriptFile.absolutePath
                    cmdMessage = "Stopping Application: ${appName}"
                    successMessage = "Successfully stopped application: ${appName}"
                    break
                default :
                    println "Command not recognized."
                    throw new Exception("${command} is not a known command.", cmdArgs)
            }
        }

        println("Executing script file ${scriptFile.name}:")
        println(scriptFile.text)

        helper.runCommand(cmdMessage, cmdArgs)
        println(successMessage)
    }

    def runServerStatusCheck() {
        def serverName = props['serverName']
        def url = props['serverUrl']
        def cmdArgs = [wlstFile.absolutePath]
        def scriptFile = createTempFile("server_status_check")

        scriptFile.withWriter {out ->
            out << "connect('${username}', '${password}'" + (url ? ", '${url}')\n" : ")\n")
            out << "state('${serverName}')"
        }
        def inputClosure = {
            BufferedReader reader = new BufferedReader(new InputStreamReader(it.inputStream))
            def line
            while ((line = reader.readLine()) != null) {
                if (line.contains('RUNNING') | line.contains('ADMIN')) {
                    println line.toString()
                }
            }

            it.out.close() // close stdin
            def out = new PrintStream(System.out, true)
            try {
                it.waitForProcessOutput(out, out) // forward stdout and stderr to autoflushing output stream
            }
            finally {
                out.flush();
            }
        }

        cmdArgs << scriptFile.absolutePath

        println("Executing script file ${scriptFile.name}:")
        println(scriptFile.text)

        try {
            helper.runCommand("Checking server status of: ${serverName}", cmdArgs, inputClosure)
        }
        catch (Exception ex) {
            println "Server Name: ${serverName} is down"
            System.exit(1)
        }
    }

    def runExecuteScriptCommand() {
        final def workDir = new File('.').canonicalFile
        def scriptFiles = props['scriptFiles'].tokenize("\n") as String[]
        def fs = new FileSet(workDir)
        def cmdArgs = [wlstFile.absolutePath]

        for (file in scriptFiles) {
            fs.include(file)
        }
        def fileList = fs.files()

        println ("Executing ${fileList.size()} files")

        for (file in fileList) {
            cmdArgs << file.absolutePath
        }

        helper.runCommand("Executing scripts", cmdArgs)
    }

    def createTempFile(def filename) {
        def tempFile = File.createTempFile("${filename}", ".py")
        tempFile.deleteOnExit()
        return tempFile
    }
}