/*
* Licensed Materials - Property of IBM Corp.
* IBM UrbanCode Deploy
* (c) Copyright IBM Corporation 2017. All Rights Reserved.
*
* U.S. Government Users Restricted Rights - Use, duplication or disclosure restricted by
* GSA ADP Schedule Contract with IBM Corp.
*/

import com.urbancode.air.plugin.webspherebpm.WebSphereBPMRestAPIHelper
import com.urbancode.air.AirPluginTool
import com.urbancode.commons.util.IO
import com.urbancode.ud.client.ComponentClient
import com.urbancode.ud.client.VersionClient

import java.nio.charset.Charset
import java.util.ArrayList
import java.util.List
import javax.net.ssl.SSLHandshakeException

def apTool = new AirPluginTool(this.args[0], this.args[1])

def props = apTool.getStepProperties()

String serverUrl = props['serverUrl'].trim()
String procApp   = props['procApp'].trim()
String username  = props['username'].trim()
String password  = props['password']
String snapshotName = props['snapshotName'].trim()
boolean trustAllCerts  = Boolean.valueOf(props['trustAllCerts'])

String componentName = props['componentName']
boolean isUseVFS = Boolean.valueOf(props['isUseVFS'])
boolean preserveExecutePermissions = Boolean.valueOf(props['saveFileExecuteBits'])

Properties agentProps = new Properties()
agentProps.load(new FileInputStream(new File(System.getenv("AGENT_HOME"),
        "conf/agent/installed.properties")))
String charsetName = agentProps.getProperty("system.default.encoding")
Charset charset = null
if (charsetName != null) {
    charset = Charset.forName(charsetName)
}

String UDUsername = "PasswordIsAuthToken"
String UDPassword = String.format("{\"token\": \"%s\"}", System.getenv("AUTH_TOKEN"))

String webUrl = System.getenv("AH_WEB_URL")
URI url = new URI(webUrl)

VersionClient versionClient = new VersionClient(url, UDUsername, UDPassword)
ComponentClient componentClient = new ComponentClient(url, UDUsername, UDPassword)

WebSphereBPMRestAPIHelper helper = new WebSphereBPMRestAPIHelper(serverUrl, username, password, trustAllCerts)
try {
    // Gather complete Process Applications list and parse to the desired Process Application (json)
    def processAppJson = helper.getProcessApp(procApp)
    // Gather list of snapshots (aka versions)
    def installedSnapshots = processAppJson.installedSnapshots

    // End if no snapshots are found
    if (installedSnapshots.size()) {

        // Gather all known UCD component versions
        List<String> versions = componentClient.getComponentVersions(componentName, false)
        List<String> archivedVersions = componentClient.getArchivedComponentVersions(componentName)
        versions.addAll(archivedVersions)

        // Identify snapshots to import
        List<String> importVersions = new ArrayList<String>()
        // If snapshot was specified during manual import
        if (snapshotName) {
            installedSnapshots.each{ it ->
                if (it.name == snapshotName || it.acronym == snapshotName) {
                    importVersions.add(it.name)
                }
            }

            // Error if specified version is not found
            if (!importVersions.size()) {
                throw new Exception("[Error] Specified Snapshot name `${snapshotName}` could not be found!")
            }
        // Automated import or empty manual import
        } else {
            installedSnapshots.each{ it ->
                importVersions.add(it.name)
            }
        }
        importVersions.removeAll(versions)

        // End if no new snapshots are found
        if (importVersions.size()) {
            println "[Info] Found new `$procApp` Snapshot version(s): ${importVersions.toString()}"

            // Default includes, excludes, and extensions. Only one file to export from Websphere BPM
            String[] includes = ["**/*"] as String[]
            String[] excludes = [] as String[]
            String[] extensions = ["zip"] as String[] // Note: BPM doc says .twx file, but that's inside a .zip

            // Iterate through all versions, create new component versions, and download export.zip
            for (String importVersion : importVersions) {
                println ""
                println "[Action] Creating version: `${importVersion}`"

                // Create version and add files
                if (isUseVFS) {
                    String projectId  = processAppJson.ID
                    String snapshotId = helper.getSnapshotId(processAppJson, importVersion)
                    File tempDir = null
                    try {
                        tempDir = helper.downloadSnapshot(projectId, snapshotId)
                        String versionId = versionClient.createAndAddVersionFiles(componentName, importVersion,
                                "", tempDir, "", includes, excludes, preserveExecutePermissions, true, charset, extensions).toString()
                        apTool.setOutputProperty("VersionID", versionId)
                    } finally {
                        // Delete temporary download folder
                        try {
                            if (tempDir != null && tempDir.exists()) {
                                IO.delete(tempDir)
                            }
                        }
                        catch (IOException e) {
                            System.err.println(String.format("[Error] Unable to delete download directory", e.getMessage()))
                        }
                    }
                // Create version only
                } else {
                    String versionId = versionClient.createVersion(componentName, importVersion, " ").toString()
                    apTool.setOutputProperty("VersionID", versionId)
                    println "[Info] Not uploading artifact(s) to VFS."
                }

                // Set three output properties. Useful for the WebSphere BPM automation plugin
                try {
                    versionClient.setVersionProperty(importVersion, componentName, "procAppAcronym", processAppJson.shortName, false)
                    versionClient.setVersionProperty(importVersion, componentName, "snapshotAcronym", helper.getSnapshotAcronym(processAppJson, importVersion), false)
                    versionClient.setVersionProperty(importVersion, componentName, "trackAcronym", helper.getSnapshotBranchName(processAppJson, importVersion), false)
                } catch (IOException ex) {
                    println ""
                    println "[Warn] Unable to assign 'procAppAcronym', 'snapshotAcronym', and 'trackAcronym' component version properties. " +
                        "Create these three version properties manually in the $componentName or component template's configuration tab."
                }
            }

            println ""
            println "Import(s) completed successfully!"
        } else {
            println "[Info] No new Snapshots found."
        }
    } else {
        println "[Info] No Snapshots found for Process Application `${procApp}`"
    }
} catch (SSLHandshakeException ex) {
    throw new Exception("[Error] Certificate is invalid. Select " +
        "the `Trust All Certificates` checkbox for a temporary solution.", ex)
} finally {
    helper.closeClient()
    apTool.setOutputProperties()
}
