/*
* Licensed Materials - Property of IBM Corp.
* IBM UrbanCode Build
* IBM UrbanCode Deploy
* IBM UrbanCode Release
* IBM AnthillPro
* (c) Copyright IBM Corporation 2002, 2016. All Rights Reserved.
*
* U.S. Government Users Restricted Rights - Use, duplication or disclosure restricted by
* GSA ADP Schedule Contract with IBM Corp.
*/
package com.urbancode.air.plugin.websphereliberty;

import java.text.SimpleDateFormat

public abstract class AbstractWebSphereLibertyFileSystemHelper {
   def wlpHomeFile
   def wlpUserDirFile

   protected AbstractWebSphereLibertyFileSystemHelper(String wlpHome, String wlpUserDir=null) {
       wlpHomeFile = new File(wlpHome);
       if (!wlpHomeFile.isDirectory()) {
           System.out.println("WebSphere Liberty Home Directory does not exist or is not a file! " + wlpHome);
           System.exit(1);
       }

       if (wlpUserDir?.trim()) {
           wlpUserDirFile = new File(wlpUserDir?.trim());
       }
       else if (System.getenv("WLP_USER_DIR")) {
           wlpUserDirFile = new File(System.getenv("WLP_USER_DIR"));
       }
       else {
           wlpUserDirFile = new File(wlpHomeFile, "usr");
       }
       if (!wlpUserDirFile.isDirectory()) {
           System.out.println("WebSphere Liberty User Directory does not exist or is not a file! " + wlpUserDirFile.absolutePath);
           System.exit(1);
       }
   }

   protected File getSecurityUtilityExecutable(def isWindows) {
       def binDir = new File(wlpHomeFile, "bin");
       def secUtilExecutableName = "securityUtility" + (isWindows ? ".bat":"");
       def secUtilExecutable = new File(binDir, secUtilExecutableName);
       return secUtilExecutable
   }

   protected File getCollectiveExecutable(def isWindows) {
       def binDir = new File(wlpHomeFile, "bin");
       def collectiveExecutableName = "collective" + (isWindows ? ".bat":"");
       def collectiveExecutable = new File(binDir, collectiveExecutableName);
       return collectiveExecutable
   }

   protected File getServerExecutable(def isWindows) {
       def binDir = new File(wlpHomeFile, "bin");
       def serverExecutableName = "server" + (isWindows ? ".bat":"");
       def serverExecutable = new File(binDir, serverExecutableName);
       return serverExecutable
   }

   protected File getServerDirectory(String serverName) {
       def serversDir = new File(wlpUserDirFile, "servers");
       def serverDir = new File(serversDir, serverName);
       if (!serverDir.isDirectory()) {
           System.out.println("Expected Server Directory does not exist or is not a file! " + serverDir.absolutePath);
           System.exit(1);
       }
       return serverDir;
   }

   protected void runServerCommand(String serverName, def isWindows, String command, String message) {
       runServerCommand(serverName, isWindows, command, message, null);
   }

   protected void runServerCommand(String serverName, def isWindows, String command, String message, Closure clos) {
       def serverExecutable = getServerExecutable(isWindows);
       def cmdArgs = [serverExecutable.absolutePath, command, serverName];

       if (isWindows) {
           ch.addEnvironmentVariable("EXIT_ALL", "1");
       }

       ch.addEnvironmentVariable("WLP_USER_DIR", wlpUserDirFile.absolutePath);
       ch.runCommand(message, cmdArgs, clos);
   }

    protected void runCollectiveCommand(String serverName, def isWindows, String command, String options, String message) {
       runCollectiveCommand(serverName, isWindows, command, options, message, null);
   }

   protected void runCollectiveCommand(String serverName, def isWindows, String command, String options, String message, Closure clos) {
       def collectiveExecutable = getCollectiveExecutable(isWindows);
       def cmdArgs = [collectiveExecutable.absolutePath, command, serverName];
       // Can't tokenize on space because we need to handle options such as filenames that could contain spaces as part
       // of their value.  This assumes that all options for the collective command begin with "--".
       options = options.trim();  //Remove any trailing spaces
       options = options.replace(" --", ",--");
       cmdArgs.addAll(options.tokenize(","));

       if (isWindows) {
           ch.addEnvironmentVariable("EXIT_ALL", "1");
       }

       ch.addEnvironmentVariable("WLP_USER_DIR", wlpUserDirFile.absolutePath);
       ch.runCommand(message, cmdArgs, clos);
   }

   // Return String with current date and time
   protected String getDateTime() {
       def today = new Date()
       def formattedDateTime = new SimpleDateFormat("yyyy-MM-dd'T'HH-mm-ss.SSS").format(today)
   }

   // Use the securityUtility tool to encode/encrypt the specified string
   // May want to add support for specifying encoding_type so we can do AES encryption as well as default XOR encoding
   protected String encode(String value, def isWindows) {
       def securityUtilExecutable = getSecurityUtilityExecutable(isWindows);
       def cmdArgs = [securityUtilExecutable.absolutePath, "encode", value];

       if (isWindows) {
           ch.addEnvironmentVariable("EXIT_ALL", "1");
       }

       ByteArrayOutputStream os = new ByteArrayOutputStream();
       ByteArrayOutputStream es = new ByteArrayOutputStream();
       ch.runCommand(null, cmdArgs) { proc ->
           proc.waitForProcessOutput(os, es);
       }
       return new String(os.toByteArray());
   }
}
