package com.urbancode.air.plugin.websphereliberty;
import javax.management.remote.JMXConnector;
import javax.management.remote.JMXConnectorFactory;
import com.ibm.websphere.jmx.connector.rest.ConnectorSettings;
import javax.management.remote.JMXServiceURL;
import javax.management.ObjectName;
import javax.management.JMX;
import javax.management.MBeanServerConnection;

public class WebSphereLibertyJMXHelper {
    JMXConnector conn = null;
    JMXServiceURL url = null;
    Map<String, Object> env;
    MBeanServerConnection mbs = null;

    public WebSphereLibertyJMXHelper(def userName, 
                                     def userPassword,
                                     def server,
                                     def httpsPort,
                                     def trustStorePath,
                                     def trustStorePassword) 
    {
        File trustStoreFile = new File(trustStorePath);
        if (!trustStoreFile.isFile()) {
            System.out.println("Trust store file not found at path : " + trustStoreFile.absolutePath);
            Sysetm.exit(1);
        }


        try {
            httpsPort = Integer.valueOf(httpsPort);
        }
        catch (NumberFormatException e) {
            System.out.println("HTTPS Port must be an integer!");
        }
        System.setProperty("javax.net.ssl.trustStore", trustStoreFile.absolutePath);
        System.setProperty("javax.net.ssl.trustStorePassword", trustStorePassword);

        env = new HashMap<String, Object>();

        env.put("jmx.remote.protocol.provider.pkgs", "com.ibm.ws.jmx.connector.client");
        env.put("jmx.remote.protocol.provider.pkgs", "com.ibm.ws.jmx.connector.client");
        env.put(JMXConnector.CREDENTIALS,  [userName, userPassword] as String[]);
        env.put(ConnectorSettings.DISABLE_HOSTNAME_VERIFICATION, true);
        env.put(ConnectorSettings.READ_TIMEOUT, 2* 60 * 100);

        url = new JMXServiceURL("REST", server, httpsPort, "/IBMJMXConnectorREST");
    }

    public void createConn() {
        if (conn == null) {
           conn = JMXConnectorFactory.connect(url, env);
           mbs = conn.getMBeanServerConnection();
        }
    }

    public ObjectName createPluginConfigObjectName() {
        ObjectName fileMonitorObjName = new ObjectName(
            "WebSphere:name=com.ibm.ws.jmx.mbeans.generatePluginConfig");
        return fileMonitorObjName;
    }

    public ObjectName createFileMonitorObjectName() {
        ObjectName fileMonitorObjName = new ObjectName(
            "WebSphere:service=com.ibm.ws.kernel.filemonitor.FileNotificationMBean");
        return fileMonitorObjName;
    }

    public ObjectName createAppObjectName(String appName) {
        ObjectName myAppMBean = new ObjectName(
            "WebSphere:service=com.ibm.websphere.application.ApplicationMBean,name="+appName);
        return myAppMBean;
    }
   
    public boolean isMBeanRegistered(ObjectName mbean) {
        createConn();
        return mbs.isRegistered(mbean);
    }

    private isAppInState(ObjectName appMBean, String state) {
        createConn();
        boolean correctState = false;
        if (isMBeanRegistered(appMBean)) {
            String curState = (String) mbs.getAttribute(appMBean, "State");  
            if (curState == state) {
                correctState = true;
            }
            else {
                println ("Current state does not match expected!");
                println (curState + " : " + state);
            }
        }
        else {
                println ("Application not currently registered.");
        }
        return correctState;
    }

    public void waitForApplication(String appName, String state, Long waitTimeout) {
        createConn();
        ObjectName myAppMBean =createAppObjectName(appName); 
        def start = System.currentTimeMillis();
        def end = start + (waitTimeout * 1000)
        def started = false;
        while (!started && System.currentTimeMillis() < end) {
            if (isAppInState(myAppMBean, state)) {
                started=true;
            }
            else {
                println ("Sleeping for 3 seconds...");
                Thread.sleep(3*1000);
            }
        }
        
        if (started) {
            println "Application in the started state!";
        }
        else {
            println "Application never started!";
            System.exit(1);
        }
    }

    public void reloadServerXmlConfiguration() {
        ObjectName fileMonitorMBean = createFileMonitorObjectName();
        List createdFiles = new ArrayList<String>();
        List modifiedFiles = new ArrayList<String>();
        List deletedFiles = new ArrayList<String>();

        modifiedFiles.add("server.xml");

        createConn();
        System.out.println("Telling liberty to please reload configuration for server.xml...");
        println mbs.invoke(fileMonitorMBean, "notifyFileChanges", 
                     [createdFiles, modifiedFiles, deletedFiles] as Object[],
                     ["java.util.Collection", "java.util.Collection", "java.util.Collection"] as String[]);
    }

    public void generatePluginConfig(String serverName, String installRoot) {
        String method = "generatePluginConfig";
        String methodDefault = "generateDefaultPluginConfig";
        ObjectName pluginMBean = createPluginConfigObjectName();
        createConn();
        if (serverName && installRoot) {
            System.out.println("Generating plugin config for ${installRoot} and ${serverName}...");
            mbs.invoke(pluginMBean, method, [installRoot, serverName] as Object [],
                       [String.class.name, String.class.name] as String[]);
        }
        else if (!serverName && !installRoot) {
            System.out.println("Generating default plugin config...");
            mbs.invoke(pluginMBean, methodDefault, [] as Object [],
                       [] as String[]);
        }
        else {
            System.out.println("If Install Root or Server Name is configured, the other must be configured as well!");
            System.exit(1);
        }
    }

    private setAppState(String appName, String state, String method) {
        createConn();
        boolean failure = true;
        ObjectName appMBean = createAppObjectName(appName);
        if (!isMBeanRegistered(appMBean)) {
            System.out.println("No application MBean registered for ${appName}. Check that the application is installed in the server.");
        }
        else if (isAppInState(appMBean, state)) {
            System.out.println("Application ${appName} is already in state ${state}!");
            failure = false;
        }
        else {
            System.out.println("Moving application ${appName} to state ${state}...");
            mbs.invoke(appMBean, method, [] as Object[], [] as String[]);
            failure= false;
        }
        if (failure) {
            System.exit(1);
        }
    }
  
    public void startApp(String appName) {
        setAppState(appName, "STARTED", "start");
    }

    public void stopApp(String appName) {
        setAppState(appName, "STOPPED", "stop");
    }
}
