/*
* Licensed Materials - Property of IBM Corp.
* IBM UrbanCode Build
* IBM UrbanCode Deploy
* IBM UrbanCode Release
* IBM AnthillPro
* (c) Copyright IBM Corporation 2002, 2013. All Rights Reserved.
*
* U.S. Government Users Restricted Rights - Use, duplication or disclosure restricted by
* GSA ADP Schedule Contract with IBM Corp.
*/
package com.urbancode.air.plugin.websphereliberty;

import java.text.SimpleDateFormat

public abstract class AbstractWebSphereLibertyFileSystemHelper {
   def wlpHomeFile

   protected AbstractWebSphereLibertyFileSystemHelper(String wlpHome) {
       wlpHomeFile = new File(wlpHome);
       if (!wlpHomeFile.isDirectory()) {
           System.out.println("WebSphere Liberty Home Directory does not exist or is not a file! " + wlpHome);
           System.exit(1);
       }
   }

   protected File getSecurityUtilityExecutable(def isWindows) {
       def binDir = new File(wlpHomeFile, "bin");
       def secUtilExecutableName = "securityUtility" + (isWindows ? ".bat":"");
       def secUtilExecutable = new File(binDir, secUtilExecutableName);
       return secUtilExecutable
   }

   protected File getCollectiveExecutable(def isWindows) {
       def binDir = new File(wlpHomeFile, "bin");
       def collectiveExecutableName = "collective" + (isWindows ? ".bat":"");
       def collectiveExecutable = new File(binDir, collectiveExecutableName);
       return collectiveExecutable
   }

   protected File getServerExecutable(def isWindows) {
       def binDir = new File(wlpHomeFile, "bin");
       def serverExecutableName = "server" + (isWindows ? ".bat":"");
       def serverExecutable = new File(binDir, serverExecutableName);
       return serverExecutable
   }

   protected File getServerDirectory(String serverName) {
       def usrDir = new File(wlpHomeFile, "usr");
       def serversDir = new File(usrDir, "servers");
       def serverDir = new File(serversDir, serverName);
       if (!wlpHomeFile.isDirectory()) {
           System.out.println("Expected Server Directory does not exist or is not a file! " + serverDir.absolutePath);
           System.exit(1);
       }
       return serverDir;
   }

   protected void runServerCommand(String serverName, def isWindows, String command, String message) {
       runServerCommand(serverName, isWindows, command, message, null);
   }

   protected void runServerCommand(String serverName, def isWindows, String command, String message, Closure clos) {
       def serverExecutable = getServerExecutable(isWindows);
       def cmdArgs = [serverExecutable.absolutePath, command, serverName];

       if (isWindows) {
           ch.addEnvironmentVariable("EXIT_ALL", "1");
       }

       ch.runCommand(message, cmdArgs, clos);
   }

    protected void runCollectiveCommand(String serverName, def isWindows, String command, String options, String message) {
       runCollectiveCommand(serverName, isWindows, command, options, message, null);
   }

   protected void runCollectiveCommand(String serverName, def isWindows, String command, String options, String message, Closure clos) {
       def collectiveExecutable = getCollectiveExecutable(isWindows);
       def cmdArgs = [collectiveExecutable.absolutePath, command, serverName];
       cmdArgs.addAll(options.tokenize());

       if (isWindows) {
           ch.addEnvironmentVariable("EXIT_ALL", "1");
       }

       ch.runCommand(message, cmdArgs, clos);
   }

   // Return String with current date and time
   protected String getDateTime() {
       def today = new Date()
       def formattedDateTime = new SimpleDateFormat("yyyy-MM-dd'T'HH-mm-ss.SSS").format(today)
   }

   // Use the securityUtility tool to encode/encrypt the specified string
   // May want to add support for specifying encoding_type so we can do AES encryption as well as default XOR encoding
   protected String encode(String value, def isWindows) {
       def securityUtilExecutable = getSecurityUtilityExecutable(isWindows);
       def cmdArgs = [securityUtilExecutable.absolutePath, "encode", value];

       if (isWindows) {
           ch.addEnvironmentVariable("EXIT_ALL", "1");
       }

       ByteArrayOutputStream os = new ByteArrayOutputStream();
       ByteArrayOutputStream es = new ByteArrayOutputStream();
       ch.runCommand(null, cmdArgs) { proc ->
           proc.waitForProcessOutput(os, es);
       }
       return new String(os.toByteArray());
   }
}
