/*
* Licensed Materials - Property of IBM Corp.
* IBM UrbanCode Build
* IBM UrbanCode Deploy
* IBM UrbanCode Release
* IBM AnthillPro
* (c) Copyright IBM Corporation 2002, 2016. All Rights Reserved.
*
* U.S. Government Users Restricted Rights - Use, duplication or disclosure restricted by
* GSA ADP Schedule Contract with IBM Corp.
*/
package com.urbancode.air.plugin.websphereliberty;
import com.urbancode.air.CommandHelper;

public class WebSphereLibertyHelper extends AbstractWebSphereLibertyFileSystemHelper {
   def ch

   public WebSphereLibertyHelper(String wlpHome) {
       super(wlpHome);
       ch = new CommandHelper(new File(".").canonicalFile);
   }

   public void deployViaDropins(String serverName, String source) {
       def serverDir = getServerDirectory(serverName);
       def toDir = new File(serverDir, "dropins");
       def ant = new AntBuilder();
       ant.copy(verbose:"true", todir:toDir, overwrite:"true", file: source, failonerror:"true");
   }

   public void uninstallViaDropins(String serverName, String sourceName) {
       System.out.println("Deleting application ${sourceName}");

       def serverDir = getServerDirectory(serverName);
       def dropinsDir = new File(serverDir, "dropins");
       def ant = new AntBuilder();
       ant.delete(verbose:"true", failonerror:"true", includeemptydirs:'true') {
           fileset(dir:dropinsDir.absolutePath, defaultexcludes:'false', followsymlinks: 'true') {
               include(name:"${sourceName}");
           }
       }
   }

   public String createCollective(String serverName, String keystorePassword, String optionalArgs, def isWindows) {
       File configFile = new File(getServerDirectory(serverName), "collectiveCreate_" + getDateTime() + ".xml");

       def options = "${optionalArgs} --keystorePassword=${keystorePassword} --createConfigFile=" + configFile.absolutePath;
       runCollectiveCommand(serverName, isWindows, "create", options, "Creating collective controller configuration for " + serverName);

       return configFile.absolutePath;
   }

   public String joinCollective(String serverName, String controllerHostName, String controllerPort, String adminUser, String adminPassword, String keystorePassword, String optionalArgs, def isWindows) {
       File configFile = new File(getServerDirectory(serverName), "collectiveJoin_" + getDateTime() + ".xml");

       def options = "${optionalArgs} --host=${controllerHostName} --port=${controllerPort} --user=${adminUser} --password=${adminPassword} --keystorePassword=${keystorePassword} --autoAcceptCertificates --createConfigFile=" + configFile.absolutePath;
       runCollectiveCommand(serverName, isWindows, "join", options, "Adding ${serverName} to Collective");

       return configFile.absolutePath;
   }

   public void createServer(String serverName, def isWindows) {
       runServerCommand(serverName, isWindows, "create", "Creating new server named " + serverName);
   }

   public boolean startServer(String serverName, def isWindows) {
       boolean started = false
       runServerCommand(serverName, isWindows, "start", "Starting server named " + serverName) { Process proc ->
           proc.out.close()                           // close   the stdIn  of the process
           proc.consumeProcessErrorStream(System.out) // forward the stdErr of the process
           proc.in.withReader{
                String line = null
                // on Windows machines it run indefinitely when started, so kick out if server starts
                while ((line = it.readLine()) != null && !started)  {
                    println line
                    if (line.startsWith("Server ${serverName} started")) {
                        started = true
                    }
                }
           }
       }
       return started
   }

   public void stopServer(String serverName, def isWindows) {
       runServerCommand(serverName, isWindows, "stop", "Stopping server named " + serverName);
   }

   public void deleteServer(String serverName) {
       System.out.println("Deleting server ${serverName}");
       def serverDir = getServerDirectory(serverName);
       def ant = new AntBuilder();
       ant.delete(verbose:"true", failonerror:"true", includeemptydirs:'true') {
           fileset(dir:serverDir.parentFile.absolutePath, defaultexcludes:'false', followsymlinks: 'true') {
               include(name:"${serverName}/**/*");
               include(name:"${serverName}");
           }
       }
   }

   public void installServerArchive(String source, boolean license) {
       System.out.println("Unzipping server archive in : " + wlpHomeFile.absolutePath);
       String extension = ''
       int dotIndex = source.lastIndexOf('.')
       if (dotIndex >= 0) {
           extension = source.substring(dotIndex + 1)
       }
       if (extension.equals('jar')) {
           ArrayList args = ['java']
           args << '-jar'
           args << source
           if (license) {
               args << '--acceptLicense'
           }
           else {
               println ('[error]  Cannot extract jar file without accepting license.')
               println ('[possible solution]  Please update the step configuration by selecting "Accept License".')
               System.exit(1)
           }
           args << wlpHomeFile.absolutePath
           ch.runCommand('[action] Installing Liberty by extracting a Java archive file...', args)
       }
       else {
           def ant = new AntBuilder();
           ant.unzip(src:source, dest:wlpHomeFile.absolutePath);
       }
   }
}
