/**
* Licensed Materials - Property of IBM* and/or HCL**
* IBM UrbanCode Deploy
* (c) Copyright IBM Corporation 2002, 2017. All Rights Reserved.
* (c) Copyright HCL Technologies Ltd. 2018. All Rights Reserved.
*
* U.S. Government Users Restricted Rights - Use, duplication or disclosure restricted by
* GSA ADP Schedule Contract with IBM Corp.
*
* * Trademark of International Business Machines
* ** Trademark of HCL Technologies Limited
*/
package com.urbancode.air.plugin.webspheremq.helper;
import com.urbancode.air.AirPluginTool;
import com.urbancode.air.CommandHelper;

public class MQScriptHelper {

    public static MQScriptHelper newInstance(Properties props) {
        File scriptFile = new File(props['scriptFile']);
        File cmdDir = new File(props['commandDir']);
        String qMgrName = props['queueManagerName'].trim();
        String localQMgrName = props['localQueueManagerName'].trim();
        int waitTime = 0
        if (localQMgrName) {
            if (localQMgrName == qMgrName) {
                /*
                 * If the local and remote queue manager names are the same,
                 * don't use a local queue manager
                 */
                localQMgrName = null
            } else {
                try {
                    waitTime = Integer.valueOf(props['waitTime'].trim())
                } catch (NumberFormatException e) {
                    throw new IllegalArgumentException('Remote Queue Manager Wait Time must be an integer > 0.')
                }

                if (waitTime <= 0) {
                    throw new IllegalArgumentException('Remote Queue Manager Wait Time must be an integer > 0.')
                }
            }
        }
        boolean isBatch = Boolean.valueOf(props['isBatch']);
        boolean deleteOnExe = Boolean.valueOf(props['deleteOnExecute'])
        return new MQScriptHelper(scriptFile, cmdDir, qMgrName, localQMgrName, waitTime, isBatch, deleteOnExe);
    }

    def workDir = new File(".").canonicalFile
    final def cmdHelper = new CommandHelper(workDir)
    def out = System.out
    File scriptFile;
    File commandFile;
    boolean isBatch;
    boolean deleteOnExecute
    String queueManager;
    String localQueueManager
    int waitTime

    volatile StringBuilder curCommand = null;

    public MQScriptHelper(File scriptFile, File commandDir, String queueManager, String localQueueManager, int waitTime, boolean isBatch, boolean deleteOnExe) {
        this.isBatch = isBatch;
        this.scriptFile = scriptFile;
        if (!this.isBatch && scriptFile.exists()) {
            throw new IllegalStateException("Not in batch mode and script already exists at location : " + scriptFile.absolutePath);
        }
        this.deleteOnExecute = deleteOnExe;
        this.queueManager = queueManager;
        this.localQueueManager = localQueueManager
        this.waitTime = waitTime

        if (!commandDir.isDirectory()) {
            throw new IllegalArgumentException("Specified command directory is not a directory, does not exist, or is not readable : " + commandDir.absolutePath);
        }

        if (new File(commandDir, "runmqsc").isFile()) {
            commandFile = new File(commandDir, "runmqsc");
        }
        else if (new File(commandDir, "runmqsc.exe").isFile()) {
            commandFile = new File(commandDir, "runmqsc.exe");
        }
        else if (new File(commandDir, "runmqsc.sh").isFile()) {
            commandFile = new File(commandDir, "runmqsc.sh");
        }
        else {
            throw new IllegalArgumentException("Specified command directory does not seem to contain runmqsc executable : " + commandDir.absolutePath);
        }
    }


    private void checkNotStarted(String errMessage) {
        if (curCommand != null) {
            throw new IllegalStateException(errMessage);
        }
    }

    private void checkStarted(String errMessage) {
        if (curCommand == null) {
            throw new IllegalStateException(errMessage);
        }
    }

    synchronized public void startCommand(String commandStart) {
        checkNotStarted("Cannot start another command until the current is written");
        curCommand = new StringBuilder().append(commandStart.trim());
    }

    synchronized public void addSingletonElementToCommand(String element) {
        checkStarted("Cannot add element before starting command.");
        curCommand.append(" ").append(element.trim());
    }

    synchronized public void addKeyValueToCommand(String key, String value) {
        checkStarted("Cannot add key value before starting command.");
        value = escapeSingleQuotes(value);
        curCommand.append(" ").append(key.trim()).append("('").append(value?.trim()?value.trim():"").append("')");
    }

    synchronized public void addUnquotedKeyValueToCommand(String key, String value) {
        checkStarted("Cannot add key value before starting command.");
        curCommand.append(" ").append(key.trim()).append("(").append(value?.trim()?value.trim():"").append(")");
    }

    synchronized public void writeCommand() {
        checkStarted("Cannot write command before starting command.");
        curCommand.append("\n");
        scriptFile.append(curCommand.toString());
        curCommand = null;
    }

    synchronized public void execute() {
        if (!isBatch) {
            executeNow();
        }
    }

    synchronized public void executeNow() {
        executeNow(null);
    }

    synchronized public void executeNow(Closure clos) {
        try {
            checkNotStarted("Cannot execute until the current command is written");
            def cmdArgs = [commandFile.absolutePath]
            if (localQueueManager) {
                cmdArgs << '-w'
                cmdArgs << waitTime
                cmdArgs << '-m'
                cmdArgs << localQueueManager
            }
            cmdArgs << queueManager
            cmdHelper.runCommand("Executing final mqsc script " +scriptFile.absolutePath, cmdArgs) { proc ->
                proc.withWriter{it << scriptFile.text};
                if (clos == null) {
                    proc.waitForProcessOutput(out, out);
                }
                else {
                    clos(proc);
                }
            }

       }
       finally {
           try {
               if (deleteOnExecute) {
                   scriptFile.delete();
               }
           }
           catch (Throwable e) {
               System.out.println("Error deleting script file!");
           }
       }
    }

    private String escapeSingleQuotes(String input) {
        if (!input) {
            input = "";
        }
        String result = input;
        if (input && input.contains("\'")) {
            // Confirm even number of single quotes, simple check
            // Note: Will still continue if values == 'a'''
            if (((input.length() - input.replace("\'", "").length()) % 2) != 0) {
                throw new RuntimeException("[Error] Uneven number of single quotes for (${input}) value.");
            }

            result = input.replaceAll("\'\'", "'").replaceAll("\'", "''");

            if (input != result) {
                println "[Warning] Made dynamic updates to the value (${input}) to escape single quotes.";
            }
        }
        return result;
    }
}
