/*
* Licensed Materials - Property of IBM Corp.
* IBM UrbanCode Build
* IBM UrbanCode Deploy
* IBM UrbanCode Release
* IBM AnthillPro
* (c) Copyright IBM Corporation 2002, 2013. All Rights Reserved.
*
* U.S. Government Users Restricted Rights - Use, duplication or disclosure restricted by
* GSA ADP Schedule Contract with IBM Corp.
*/
import java.util.zip.*

import javax.xml.parsers.DocumentBuilderFactory
import javax.xml.transform.OutputKeys
import javax.xml.transform.TransformerFactory
import javax.xml.transform.dom.DOMSource
import javax.xml.transform.stream.StreamResult

import com.urbancode.air.AirPluginTool

final AirPluginTool airPluginTool = new AirPluginTool(this.args[0], this.args[1])
final Properties props = airPluginTool.getStepProperties()

def isEmpty(value) {
    return value == null || value.equals("")
}

def findFile(parent, fileToSearch) {
    File parentDir = new File(parent)
    File[] contents = parentDir.listFiles()
    def searchedFile = ""
    for (File file : contents) {
        if (file.isDirectory()) {
            if (file.getName().equals(fileToSearch)) {
                return file.getAbsolutePath()
            } else {
                searchedFile = findFile(file.getAbsolutePath(), fileToSearch)
                if (! isEmpty(searchedFile)) {
                    return searchedFile
                }
            }
        }
        else if (file.getName().equals(fileToSearch)) {
            return file.getAbsolutePath()
        }
    }
    return searchedFile
}

def extractWAR(war, workDirString) {
    def zis = new ZipInputStream(new FileInputStream(war))
    def warEntry = zis.getNextEntry()
    while (warEntry != null) {
        def fileName = warEntry.getName()
        def newFile = new File(workDirString + File.separator + fileName)
        if (warEntry.isDirectory()) {
            newFile.mkdirs()
        } else {
            new File(newFile.getParent()).mkdirs()

            def outstream = new FileOutputStream(newFile)
            def buf = new byte[1024]
            def len = 0;
            while ((len = zis.read(buf)) > 0) {
                outstream.write(buf, 0, len);
            }
            outstream.close()
        }
        zis.closeEntry()
        warEntry = zis.getNextEntry()
    }
    zis.close()
}

def getSubFoldersList(parent) {
    def subFolders = parent.list(new FilenameFilter() {
                @Override
                public boolean accept(File dir, String name) {
                    return new File(dir, name).isDirectory();
                }
            });
    return subFolders
}

def getSkinNames(war, workDir, templateDir) {
    def themesDirectory = findFile(workDir, "themes")
    def themesParent = new File(themesDirectory).getParent()
    File skins = new File(themesDirectory + File.separator + templateDir + File.separator + "skins")
    if (!skins.exists()) {
        skins = new File(themesParent + File.separator + "skins")
        if (!skins.exists()) {
            return []
        }
    }
    def skinFolders = getSubFoldersList(skins)
    return skinFolders
}

def findOID(name, tool, props, artifact) {
    def wpUser = props['wpUser']
    def wpPass = props['wpPass']

    File tempFile = new File("temp.py")
    tempFile.deleteOnExit();
    BufferedWriter temp = new BufferedWriter(new FileWriter(tempFile))
    temp.writeLine("Portal.login('" + wpUser + "', '" + wpPass + "')")
    temp.writeLine("try:")
    temp.writeLine("\tprint Look.find('" + artifact + "', 'uniquename', '" + name + "')")
    temp.writeLine("except:")
    temp.writeLine("\tprint 'Not Found'")
    temp.writeLine("Portal.logout()")
    temp.close()

    def commandArgs = []
    commandArgs << tool
    if (props["connectionPort"]) {
        commandArgs << "-port"
        commandArgs << props["connectionPort"]
    }
    commandArgs << "-username"
    commandArgs << wpUser
    commandArgs << "-password"
    commandArgs << wpPass
    commandArgs << "-lang"
    commandArgs << "jython"
    commandArgs << "-f"
    commandArgs << "temp.py"

    def isWindows = (System.getProperty('os.name') =~ /(?i)windows/).find()
    def procBuilder = new ProcessBuilder(commandArgs)
    if (isWindows) {
        def envMap = procBuilder.environment()
        envMap.put("PROFILE_CONFIG_ACTION", "true")
    }

    def statusProc = procBuilder.start()
    def reader = new BufferedReader(new InputStreamReader(statusProc.getInputStream()))
    def outputLines = []
    def line = reader.readLine()
    // Find the OID
    while (line != null) {
        outputLines << line
        line = reader.readLine()
    }

    // OID is the last line of the outputLines
    String result = outputLines.get(outputLines.size()-1)

    if (!(result.equals("Not Found"))) {
        /* No exception caught searching for the OID */
        return result
    } else {
        return '' /* An exception caught searching for the OID */
    }
}

def computeURL(props) {
    def host = props['vpHost']
    if (isEmpty(host)) {
        host = System.getenv("HOSTNAME");
    }

    def wpContextRoot = props['wpContextRoot']
    if (! isEmpty(wpContextRoot)) {
        if (!wpContextRoot.startsWith("/")) {
            wpContextRoot = "/" + wpContextRoot
        }
    } else {
        wpContextRoot = "/wps"
    }

    def urlString = "http://" + host + ":" + props['vpPort'] + wpContextRoot + "/config"

    def urlContext = props['vpContext']
    if (! isEmpty(urlContext)) {
        if (!urlContext.startsWith("/")) {
            urlContext = "/" + urlContext
        }
        urlString = urlString + urlContext
    }

    return urlString
}

////////////// MAIN ////////////////////
def xmlScript = '''
<request
    xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance"
    xsi:noNamespaceSchemaLocation="PortalConfig_7.0.0.xsd"
    type="update">

    <portal action="locate">
        <theme active="true">
        </theme>
    </portal>

</request>
'''

def skinXML = '''<skin active="true"/>'''

final def isWindows = (System.getProperty('os.name') =~ /(?i)windows/).find()
def wpscript = isWindows ? "wpscript.bat" : "wpscript.sh"
def xmlaccess = isWindows ? "xmlaccess.bat" : "xmlaccess.sh"


def componentName = new File(".").getCanonicalFile().name
def workDirString = new File(".").getCanonicalPath() + File.separator

def portalHome = props['portalHome']
if (!(portalHome.endsWith("\\") || portalHome.endsWith("/"))) {
    portalHome = portalHome + File.separator
}

def profHome = props['profHome']
if (!(profHome.endsWith("\\") || profHome.endsWith("/"))) {
    profHome = profHome + File.separator
}

def layoutRef = props['layoutRef']
if (!(layoutRef.endsWith("\\") || layoutRef.endsWith("/"))) {
    layoutRef = layoutRef + File.separator
}

def war = props['war']
if (isEmpty(war)) {
    war = componentName
}
if (!(war =~ /(?i)\.war$|\.zip$/)) {
    war = war + ".war"
}

def warfileName = war.substring(0, war.length() - 4)
def uniquename = props['uniquename']
if (isEmpty(uniquename)) {
    uniquename = 'com.ibm.urbancode.customTheme.' + warfileName
}

def resourceroot = props['resourceroot']
if (isEmpty(resourceroot)) {
    resourceroot = warfileName
}

def contextroot = props['contextroot']
if (isEmpty(contextroot)) {
    contextroot = "/" + warfileName
}
if (!contextroot.startsWith("/")) {
    contextroot = "/" + contextroot
}

def request = new XmlParser(false, false).parseText(xmlScript)
def portal = request.portal[0]
def theme = portal.children()[0]

def localeData = props['localedata']
if (!isEmpty(localeData)) {
    try {
        def pairs = localeData.split(",")
        for (pair in pairs) {
            def nameValue = pair.split("=")
            if (nameValue[0].equals(pair)) {
                def localeNode = theme.appendNode("localedata", [locale:"en"])
                localeNode.appendNode("title", nameValue[0].trim())
            } else {
                def localeNode = theme.appendNode("localedata", [locale:nameValue[0].trim()])
                localeNode.appendNode("title", nameValue[1].trim())
            }
        }
    }
    catch (Exception e) {
        throw new IllegalArgumentException("Options to the ConfigEngine command are not entered in the required format")
    }
} else {
    def localeNode = theme.appendNode("localedata", [locale:"en"])
    localeNode.appendNode("title", componentName)
}

theme.@"context-root" = contextroot
theme.@"resourceroot" = resourceroot
theme.@"uniquename" = uniquename
def theme_oid = findOID(uniquename, portalHome + File.separator + "bin" + File.separator + wpscript, props, 'theme')
if (!isEmpty(theme_oid)) {
    theme.@"action" = "update"
    theme.@"objectid" = theme_oid.toString()
} else {
    theme.@"action" = "create"
}

extractWAR(war, workDirString + warfileName)
def templateDir = props['templateDir']
if (isEmpty(templateDir)) {
    def themesDirectory = findFile(workDirString + warfileName, "themes")
    def themesParent = new File(themesDirectory).getParent()
    def themeSubFolders = getSubFoldersList(new File(themesParent + File.separator + "themes"))
    if (themeSubFolders == null || themeSubFolders.length < 1) {
        throw new Exception("There are no theme templates present under the '/themes' directory")
    }
    templateDir = themeSubFolders[0]
}

def skins = getSkinNames(war, workDirString + warfileName, templateDir)
for (skin in skins) {
    def skinuniquename = 'com.ibm.urbancode.customSkin.' + skin
    def skinOID = findOID(skinuniquename, portalHome + File.separator + "bin" + File.separator + wpscript, props, 'skin')
    if (isEmpty(skinOID)) {
        skinOID = skinuniquename+".objectID"
    }

    theme.appendNode("allowed-skin", [update:"set", skin:skinOID])

    def skinNode = new XmlParser(false, false).parseText(skinXML)
    def skinLocaleNode = skinNode.appendNode("localedata", [locale:"en"])
    skinLocaleNode.appendNode("title", componentName + " - " + skin)

    skinNode.@"uniquename" = skinuniquename
    skinNode.@"resourceroot" = resourceroot
    skinNode.@"context-root" = contextroot
    skinNode.@"action" = "update"
    skinNode.@"objectid" = skinOID

    skinNode.appendNode("parameter", [update:"set", type:"string", name:"com.ibm.portal.skintype"])
    skinNode.appendNode("parameter", [update:"set", type:"string", name:"com.ibm.portal.skin.template.ref", skinFolder:skin])
    skinNode.appendNode("parameter", [update:"set", type:"string", name:"com.ibm.portal.skin.template.file.name.html"])

    portal.children().add(0, skinNode)

}

def defaultSkin = props['defaultSkin']
if (isEmpty(defaultSkin)) {
    if (skins.size() > 0) {
        theme.@defaultskinref = 'com.ibm.urbancode.customSkin.' + skins[0] + ".objectID"
    }
} else {
  def defSkinOID = findOID(defaultSkin, portalHome + File.separator + "bin" + File.separator + wpscript, props, 'skin')
  if (!isEmpty(defSkinOID)) {
      theme.@defaultskinref = defSkinOID
  } else {
      theme.@defaultskinref = defaultSkin + ".objectID"
  }
}

theme.appendNode("parameter", [update:"set", type:"string", name:"com.ibm.portal.theme.dnd.sources"])
theme.appendNode("parameter", [update:"set", type:"string", name:"com.ibm.portal.theme.aggregationmodes"])
theme.appendNode("parameter", [update:"set", type:"string", name:"com.ibm.portal.layout.template.href"])
theme.appendNode("parameter", [update:"set", type:"string", name:"com.ibm.portal.theme.template.ref"])
theme.appendNode("parameter", [update:"set", type:"string", name:"resourceaggregation.profile"])

def inputScript = "xmlAccessScript.xml"
def xmlNodePrinter = new XmlNodePrinter(new PrintWriter(new FileWriter(inputScript)))
xmlNodePrinter.with {
    preserveWhitespace = true
}
xmlNodePrinter.print(request)

def cdata = ''
def dbf = DocumentBuilderFactory.newInstance();
dbf.setValidating(false);
def db = dbf.newDocumentBuilder();
def doc = db.parse(new FileInputStream(new File(inputScript)));

def skinElements = doc.getElementsByTagName("skin")
for (skinElement in skinElements) {
    def params = skinElement.getElementsByTagName("parameter")
    for (param in params) {
        def name = param.getAttribute("name")
        if (name.equalsIgnoreCase("com.ibm.portal.skintype")) {
            cdata = doc.createCDATASection("template")
        } else if (name.equalsIgnoreCase("com.ibm.portal.skin.template.file.name.html")) {
            cdata = doc.createCDATASection("skin.html")
        } else if (name.equalsIgnoreCase("com.ibm.portal.skin.template.ref")) {
            cdata = doc.createCDATASection("res:/" + warfileName + "/skins/" + param.getAttribute("skinFolder") + "/")
            param.removeAttribute("skinFolder")
        }
        param.appendChild(cdata)
    }
}

def themeElements = doc.getElementsByTagName("theme")
for (themeElement in themeElements) {
    def params = themeElement.getElementsByTagName("parameter")
    for (param in params) {
        def name = param.getAttribute("name")
        if (name.equalsIgnoreCase("com.ibm.portal.theme.dnd.sources")) {
            cdata = doc.createCDATASection("ibmDndColumn:com.ibm.pb.dnd.layout.LayoutColumnSource:vertical,ibmDndRow:com.ibm.pb.dnd.layout.LayoutRowSource:horizontal");
        } else if (name.equalsIgnoreCase("com.ibm.portal.theme.aggregationmodes")) {
            cdata = doc.createCDATASection("ssa");
        } else if (name.equalsIgnoreCase("com.ibm.portal.layout.template.href")) {
            cdata = doc.createCDATASection("war:" + warfileName + "/themes/" + templateDir + "/" + layoutRef)
        } else if (name.equalsIgnoreCase("com.ibm.portal.theme.template.ref")) {
            cdata = doc.createCDATASection("war:" + warfileName + "/themes/" + templateDir + "/")
        } else if (name.equalsIgnoreCase("resourceaggregation.profile")) {
            cdata = doc.createCDATASection("profiles/profile_deferred.json")
        }
        param.appendChild(cdata)
    }
}

def tf = TransformerFactory.newInstance().newTransformer();
tf.setOutputProperty(OutputKeys.INDENT, "yes");
Writer out = new PrintWriter(new FileWriter(inputScript))
tf.transform(new DOMSource(doc), new StreamResult(out));

def configPath = profHome + "PortalServer" + File.separator + "bin" + File.separator + xmlaccess
def deployWARCommandArgs = [configPath]
if (props['useSOAPProps'].equals("true")) {
    deployWARCommandArgs.add("-useEncryptedCredentials")
    deployWARCommandArgs.add(profHome + "properties" + File.separator + "soap.client.props")
} else {
    deployWARCommandArgs.add("-user")
    deployWARCommandArgs.add(props['wpUser'])
    deployWARCommandArgs.add("-password")
    deployWARCommandArgs.add(props['wpPass'])
}

def urlString = computeURL(props)

deployWARCommandArgs.add("-url")
deployWARCommandArgs.add(urlString)
deployWARCommandArgs.add("-in")
deployWARCommandArgs.add(workDirString + inputScript)
deployWARCommandArgs.add("-out")
deployWARCommandArgs.add(workDirString + (inputScript.substring(0, inputScript.length() - 4) + "_out.xml"))

println deployWARCommandArgs.join(' ');
def procBuilder = new ProcessBuilder(deployWARCommandArgs);

if (isWindows) {
    def envMap = procBuilder.environment();
    envMap.put("PROFILE_CONFIG_ACTION", "true");
}

def statusProc = procBuilder.start();
def outPrint = new PrintStream(System.out, true);
statusProc.waitForProcessOutput(outPrint, outPrint);
System.exit(statusProc.exitValue());
