/*
* Licensed Materials - Property of IBM Corp.
* IBM UrbanCode Deploy
* (c) Copyright IBM Corporation 2014. All Rights Reserved.
*
* U.S. Government Users Restricted Rights - Use, duplication or disclosure restricted by
* GSA ADP Schedule Contract with IBM Corp.
*/

package com.ibm.rational.air.plugin.worklight;

import com.urbancode.ud.client.UDRestClient;

import org.apache.http.HttpResponse;
import org.apache.http.client.ClientProtocolException;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpRequestBase;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.StringEntity;
import org.codehaus.jettison.json.JSONObject;

/**
* A utility class for helping to run the Worklight REST commands.
**/
public class WLRestClient extends UDRestClient {
    
    public WLRestClient(URI url, String clientUser, String clientPassword) {
        super (url, clientUser, clientPassword);
    }

    /**
    * Connects to the supplied Worklight Server and obtains the version.
    * Returns the Worklight Server version from the REST call.
    **/
    public String getServerVersion() {
        // When a user and password are provided, we attempt to authenticate to
        // the Worklight Console for Worklight 6.0 and 6.1 for the REST request.
        if(this.clientUser != null && this.clientPassword != null) {
            if(!authenticate()) {
                println "Failed to authenticate with the Worklight Server using REST.";
                return null;
            }
        }
        
        HttpGet getRequest = new HttpGet(this.url.toString() + "/console/api/ui/serverVersion");
        String result;
        HttpResponse response;
        try {
            response = this.invokeMethod(getRequest);
            JSONObject props = new JSONObject(getBody(response));
            result = props.getString("serverVersion");
        } catch (IOException e) {
            // Ignore when the system can be running on a WL 6.2 or newer server.
            // Possibly the server could be unreachable.
            println "Unable to connect to the server using REST to obtain the version " +
                "because of: " + e.getMessage();
        }
        return result;
    }
    
    /**
    * Authenticates with a secure Worklight console.
    **/
    private boolean authenticate() {
        // A GET request needs to be sent first to initiate the login request.
        HttpGet getRequest = new HttpGet(this.url.toString() + "/console");
        try {
            this.invokeMethod(getRequest);
        } catch (IOException e) {
            println "An error occurred while trying to initiate authentication " +
                "with the Worklight Server using REST: " + e.getMessage();
        }
        
        // Send a POST request for the authentication. Send the user credentials in
        // the body for security.
        HttpPost postRequest = new HttpPost(this.url.toString() + "/j_security_check");
        postRequest.setEntity(new StringEntity("j_username=" + this.clientUser + "&j_password=" + 
            this.clientPassword, ContentType.APPLICATION_FORM_URLENCODED));
        String result;
        HttpResponse response;
        try {
            response = this.invokeRedirectMethod(postRequest);
        } catch (IOException e) {
            // Ignore when the system can be running on a WL 6.2 or newer server.
            // Possibly the server could be unreachable.
            println "Unable to connect to the server using REST to authenticate " +
                "because of: " + e.getMessage();
        }
        // Make sure we got redirected to the logged in session.
        return (response?.getStatusLine()?.getStatusCode() == 302);
    }
    
    /**
    * Copied from UDRestClient, but updated to allow the redirect request
    * used by the Worklight Authentication.
    **/
    private HttpResponse invokeRedirectMethod(HttpRequestBase request)
    throws IOException, ClientProtocolException {
        HttpResponse response = client.execute(request);
        int status = response.getStatusLine().getStatusCode();
        // Allow the HTTP 302 redirect status.
        if (status > 299 && status != 302) {
            throw new IOException(String.format("%d %s\n%s",
                    response.getStatusLine().getStatusCode(),
                    response.getStatusLine().getReasonPhrase(),
                    getBody(response)));
        }
        return response;
    }
}