# IBM Worklight Server installation/uninstallation script for
# WebSphere Application Server.
#
# Licensed Materials - Property of IBM
# 5725-G92 (C) Copyright IBM Corp. 2011, 2013. All Rights Reserved.
# US Government Users Restricted Rights - Use, duplication or
# disclosure restricted by GSA ADP Schedule Contract with IBM Corp.
#

import fnmatch
import glob
import os
import sys

if len(sys.argv) > 0:
	# Note: On Unix, wsadmin does not accept empty arguments.
	# Arguments that start with a minus sign are dangerous as well.
	# Therefore all arguments are prefixed with '='.

	operation                                = sys.argv[0] [1:]

	appserver_selection                      = sys.argv[1] [1:]
	appserver_was_installdir                 = sys.argv[2] [1:]
	appserver_was_profile                    = sys.argv[3] [1:]
	appserver_was_profile_dir                = sys.argv[4] [1:]
	appserver_was_cell                       = sys.argv[5] [1:]
	appserver_was_node                       = sys.argv[6] [1:]
	appserver_was_scope                      = sys.argv[7] [1:]
	appserver_was_serverInstance             = sys.argv[8] [1:]
	appserver_was_nd_cluster                 = sys.argv[9] [1:]
	appserver_was_nd_node                    = sys.argv[10] [1:]
	appserver_was_nd_server                  = sys.argv[11] [1:]
	appserver_was_admin_name                 = sys.argv[12] [1:]
	appserver_was_admin_password             = sys.argv[13] [1:]

	worklight_database_selection             = sys.argv[14] [1:]
	worklight_database_derby_datadir         = sys.argv[15] [1:] # used only for derby
	worklight_database_db2_host              = sys.argv[16] [1:] # used only for db2
	worklight_database_db2_port              = sys.argv[17] [1:] # used only for db2
	worklight_database_name                  = sys.argv[18] [1:] # used only for derby, db2
	worklight_database_user_name             = sys.argv[19] [1:] # used only for db2, mysql, oracle
	worklight_database_user_password         = sys.argv[20] [1:] # used only for db2, mysql, oracle
	worklight_database_schema                = sys.argv[21] [1:] # used only for derby, db2
	worklight_database_jdbc_url              = sys.argv[22] [1:] # used only for mysql, oracle
	worklight_database_properties_py         = sys.argv[23] [1:]
	worklight_database_jndi_name             = sys.argv[24] [1:]

	worklightreports_database_selection      = sys.argv[25] [1:]
	worklightreports_database_derby_datadir  = sys.argv[26] [1:] # used only for derby
	worklightreports_database_db2_host       = sys.argv[27] [1:] # used only for db2
	worklightreports_database_db2_port       = sys.argv[28] [1:] # used only for db2
	worklightreports_database_name           = sys.argv[29] [1:] # used only for derby, db2
	worklightreports_database_user_name      = sys.argv[30] [1:] # used only for db2, mysql, oracle
	worklightreports_database_user_password  = sys.argv[31] [1:] # used only for db2, mysql, oracle
	worklightreports_database_schema         = sys.argv[32] [1:] # used only for derby, db2
	worklightreports_database_jdbc_url       = sys.argv[33] [1:] # used only for mysql, oracle
	worklightreports_database_properties_py  = sys.argv[34] [1:]
	worklightreports_database_jndi_name      = sys.argv[35] [1:]

	worklight_id                             = sys.argv[36] [1:]
	worklight_lib_basename                   = sys.argv[37] [1:]
	worklight_war                            = sys.argv[38] [1:]
	worklight_war_basename                   = sys.argv[39] [1:]
	worklight_war_context                    = sys.argv[40] [1:]
	worklight_war_appname                    = sys.argv[41] [1:]
	worklight_war_jndi_prop1_name            = sys.argv[42] [1:]
	worklight_war_jndi_prop1_value           = sys.argv[43] [1:]
	worklight_war_jndi_prop2_name            = sys.argv[44] [1:]
	worklight_war_jndi_prop2_value           = sys.argv[45] [1:]
	worklight_war_jndi_py                    = sys.argv[46] [1:]

	appserver_resourcesdir                   = sys.argv[47] [1:]

else:
	# Settings for Bruno's testing.

	operation = 'install'

	appserver_selection = 'was'
	appserver_was_installdir = '/n/java/webservers/was-8.0-express'
	appserver_was_profile = 'AppSrv01'
	appserver_was_profile_dir = '/n/java/webservers/was-8.0-express/profiles/AppSrv01'
	appserver_was_cell = 'bordeauxNode02Cell'
	appserver_was_node = 'bordeauxNode02'
	appserver_was_scope = 'server'
	appserver_was_serverInstance = 'server1'
	appserver_was_nd_cluster = ''
	appserver_was_nd_node = ''
	appserver_was_nd_server = ''
	appserver_was_admin_name = 'admin'
	appserver_was_admin_password = 'admin'

	worklight_database_selection = 'derby'
	worklight_database_derby_datadir = '/var/ibm/Worklight/derby'
	worklight_database_db2_host = ''
	worklight_database_db2_port = ''
	worklight_database_name = 'WRKLGHT'
	worklight_database_user_name = 'foo'
	worklight_database_user_password = ''
	worklight_database_schema = 'WORKLIGHT'
	worklight_database_jdbc_url = 'jdbc:derby:/var/ibm/Worklight/derby/WRKLGHT'
	worklight_database_properties_py = '/tmp/worklight-dbproperties.py'
	worklight_database_jndi_name = 'jdbc/WorklightDS'

	worklightreports_database_selection = 'derby'
	worklightreports_database_derby_datadir = '/var/ibm/Worklight/derby'
	worklightreports_database_db2_host = ''
	worklightreports_database_db2_port = ''
	worklightreports_database_name = 'WLREPORT'
	worklightreports_database_user_name = 'foo'
	worklightreports_database_user_password = ''
	worklightreports_database_schema = 'WORKLIGHT'
	worklightreports_database_jdbc_url = 'jdbc:derby:/var/ibm/Worklight/derby/WLREPORT'
	worklightreports_database_properties_py = '/tmp/worklightreports-dbproperties.py'
	worklightreports_database_jndi_name = 'jdbc/WorklightReportsDS'

	worklight_id = '7560103198'
	worklight_lib_basename = 'worklight-jee-library.jar'
	worklight_war = '/n/distrib/wl-mfee-5001-server/WorklightServer/worklight.war'
	worklight_war_basename = 'worklight.war'
	worklight_war_context = '/worklight'
	worklight_war_appname = 'IBM_Worklight_Console'
	worklight_war_jndi_prop1_name = 'publicWorkLightProtocol'
	worklight_war_jndi_prop1_value = 'http'
	worklight_war_jndi_prop2_name = 'publicWorkLightPort'
	worklight_war_jndi_prop2_value = '9080'
	worklight_war_jndi_py = '/tmp/worklight-jndi.py'

	appserver_resourcesdir = 'config/Worklight-5.0'

# Jython is Python 2.1 compatible, which did not have the boolean type.
True = 1
False = 0

# Determine whether the server (nodeName, serverName) is member of some cluster.
def isInCluster(nodeName, serverName):
	for clusterObject in AdminUtilities.convertToList(AdminConfig.list('ServerCluster')):
		# clusterObject is of type ServerCluster.
		for memberObject in AdminUtilities.convertToList(AdminConfig.showAttribute(clusterObject, 'members')):
			# memberObject is of type ClusterMember.
			if nodeName == AdminConfig.showAttribute(memberObject, 'nodeName'):
				if serverName == AdminConfig.showAttribute(memberObject, 'memberName'):
					return True
	return False

# Test whether WAS 8.0 or newer is used.
was8 = not glob.glob(appserver_was_installdir + os.sep + 'features' + os.sep + 'com.ibm.ws.osgi.applications_*') == []

# Scope dependent variables:
#
# appserver_was_scope == 'server':
#   means an unmanaged (isolated) server.
#   Extra variables: appserver_was_serverInstance.
#
# The other cases assume a cell and a deployment manager.
#
# appserver_was_scope == 'nd-cell':
#   means the entire cell.
#
# appserver_was_scope == 'nd-cluster':
#   means a cluster in the cell.
#   Extra variables: appserver_was_nd_cluster.
#
# appserver_was_scope == 'nd-node':
#   means a node in the cell.
#   Extra variables: appserver_was_nd_node.
#
# appserver_was_scope == 'nd-server':
#   means a server in a node in the cell.
#   Extra variables: appserver_was_nd_node, appserver_was_nd_server.


# Derived variables.

# affected_servers is the set of servers (tuples (node name, server name))
# which are affected by the operation.
# Note! A server name is not sufficient to identify a server:
# Different servers in different nodes can have the same name. But the pair
# (node name, server name) is sufficient.
affected_servers = []
if appserver_was_scope == 'server':
	affected_servers.append((appserver_was_node, appserver_was_serverInstance))
elif appserver_was_scope == 'nd-cell':
	cellObject = AdminConfig.getid('/Cell:' + appserver_was_cell + '/')
	if cellObject == '':
		raise 'Cell ' + appserver_was_cell + ' not found!'
	# cellObject is of type Cell.
	for nodeObject in AdminUtilities.convertToList(AdminConfig.list('Node', cellObject)):
		# nodeObject is of type Node.
		nodeName = AdminConfig.showAttribute(nodeObject, 'name')
		if nodeName != '':
			for serverObject in AdminUtilities.convertToList(AdminConfig.list('Server', nodeObject)):
				# serverObject is of type Server.
				# Include only those with serverType = 'APPLICATION_SERVER';
				# exclude those with serverType = 'DEPLOYMENT_MANAGER' or
				# = 'NODE_AGENT'.
				# Alternatively, we could also look at the components attribute
				# and exclude those which have a component of type CellManager
				# or a component of type NodeAgent.
				if AdminConfig.showAttribute(serverObject, 'serverType') == 'APPLICATION_SERVER':
					serverName = AdminConfig.showAttribute(serverObject, 'name')
					if serverName != '':
						affected_servers.append((nodeName, serverName))
elif appserver_was_scope == 'nd-cluster':
	clusterObject = AdminConfig.getid('/ServerCluster:' + appserver_was_nd_cluster + '/')
	if clusterObject == '':
		raise 'Cluster ' + appserver_was_nd_cluster + ' not found!'
	# clusterObject is of type ServerCluster.
	for memberObject in AdminUtilities.convertToList(AdminConfig.showAttribute(clusterObject, 'members')):
		# memberObject is of type ClusterMember.
		nodeName = AdminConfig.showAttribute(memberObject, 'nodeName')
		serverName = AdminConfig.showAttribute(memberObject, 'memberName')
		server = (nodeName, serverName)
		if not server in affected_servers:
			affected_servers.append(server)
elif appserver_was_scope == 'nd-node':
	nodeObject = AdminConfig.getid('/Node:' + appserver_was_nd_node + '/')
	if nodeObject == '':
		raise 'Node ' + appserver_was_nd_node + ' not found!'
	# nodeObject is of type Node.
	for serverObject in AdminUtilities.convertToList(AdminConfig.list('Server', nodeObject)):
		# serverObject is of type Server.
		# Include only those with serverType = 'APPLICATION_SERVER';
		# exclude those with serverType = 'DEPLOYMENT_MANAGER' or
		# = 'NODE_AGENT'.
		# Alternatively, we could also look at the components attribute
		# and exclude those which have a component of type CellManager
		# or a component of type NodeAgent.
		if AdminConfig.showAttribute(serverObject, 'serverType') == 'APPLICATION_SERVER':
			serverName = AdminConfig.showAttribute(serverObject, 'name')
			# Exclude servers that are part of a cluster, because WAS
			# does not allow to install an application on a server that is
			# part of a cluster without installing it on the entire cluster,
			# and the cluster can contain servers on different nodes.
			if serverName != '' and not isInCluster(appserver_was_nd_node, serverName):
				affected_servers.append((appserver_was_nd_node, serverName))
elif appserver_was_scope == 'nd-server':
	serverObject = AdminConfig.getid('/Node:' + appserver_was_nd_node + '/Server:' + appserver_was_nd_server + '/')
	if serverObject == '':
		raise 'Server ' + appserver_was_nd_server + ' not found in node ' + appserver_was_nd_node + '!'
	affected_servers.append((appserver_was_nd_node, appserver_was_nd_server))
print 'was-install.py determined set of affected servers:'
print affected_servers

# affected_nodes is the set of nodes (node names) which are affected by the
# operation.
affected_nodes = []
if appserver_was_scope == 'server':
	affected_nodes.append(appserver_was_node)
else:
	for server in affected_servers:
		(nodeName, serverName) = server
		if not nodeName in affected_nodes:
			affected_nodes.append(nodeName)
print 'was-install.py determined set of affected nodes:'
print affected_nodes

# scopeContainmentPath is a hierarchical search request that can be passed to AdminConfig.getid.
# scopeParam is a scope parameter for AdminTask.createJDBCProvider.
# scopeObject is an object that represents the entire scope (of type Server,
# Cell, ServerCluster, or Node).
if appserver_was_scope == 'server':
	scopeContainmentPath = '/Cell:' + appserver_was_cell + '/Node:' + appserver_was_node + '/Server:' + appserver_was_serverInstance + '/'
	scopeParam = 'Node=' + appserver_was_node + ',Server=' + appserver_was_serverInstance
elif appserver_was_scope == 'nd-cell':
	scopeContainmentPath = '/Cell:' + appserver_was_cell + '/'
	scopeParam = 'Cell=' + appserver_was_cell
elif appserver_was_scope == 'nd-cluster':
	scopeContainmentPath = '/Cell:' + appserver_was_cell + '/ServerCluster:' + appserver_was_nd_cluster + '/'
	scopeParam = 'Cluster=' + appserver_was_nd_cluster
elif appserver_was_scope == 'nd-node':
	scopeContainmentPath = '/Cell:' + appserver_was_cell + '/Node:' + appserver_was_nd_node + '/'
	scopeParam = 'Node=' + appserver_was_nd_node
elif appserver_was_scope == 'nd-server':
	scopeContainmentPath = '/Cell:' + appserver_was_cell + '/Node:' + appserver_was_nd_node + '/Server:' + appserver_was_nd_server + '/'
	scopeParam = 'Node=' + appserver_was_nd_node + ',Server=' + appserver_was_nd_server
scopeObject = AdminConfig.getid(scopeContainmentPath)

print 'was-install.py scope setting:'
print scopeContainmentPath
print scopeParam
print scopeObject
print AdminConfig.show(scopeObject)
print


# =============================================================================
# ================================ Library Code ===============================
# =============================================================================

# =========================== Authentication aliases ==========================

# -------- Get existing authentication alias --------

# Tests whether an authentication alias with the given name exists.
def authAliasExistsBAD(name):
	# Using AdminTask.getAuthDataEntry is bad because
	# - It leaves an exception in the log if the entry does not exist.
	# - In WAS 7.0, when invoked with argument ['-alias', name], the
	#   function may return the alias with name node+'/'+name.
	try:
		return AdminTask.getAuthDataEntry(['-alias', name]) != ''
	except:
		# Note: This leaves an exception
		#   FFDC Exception:com.ibm.websphere.management.cmdframework.CommandValidationException
		#   SECJ7732E: The specified auth data entry does not exist.
		# in the server's log, but it can be ignored.
		return False
def authAliasExists(name):
	for authEntry in AdminUtilities.convertToList(AdminConfig.list('JAASAuthData')):
		# authEntry is of type JAASAuthData.
		alias = AdminConfig.showAttribute(authEntry, 'alias')
		if alias != None and alias == name:
			return True
	return False

# Returns the AdminConfig object for an authentication alias with the given name.
# The result is of type JAASAuthData or empty.
def getAuthDataEntryObject(name):
	for authEntry in AdminUtilities.convertToList(AdminConfig.list('JAASAuthData')):
		# authEntry is of type JAASAuthData.
		alias = AdminConfig.showAttribute(authEntry, 'alias')
		if alias != None and alias == name:
			return authEntry
	return ''

# -------- Create an authentication alias --------

# Create the authentication alias (= credentials for a datasource).
# Documentation of the entirely bad AdminTask.*AuthDataEntry API:
# http://pic.dhe.ibm.com/infocenter/wasinfo/v7r0/index.jsp?topic=%2Fcom.ibm.websphere.base.doc%2Finfo%2Faes%2Fae%2Frxml_7securityconfig.html

# Creates an AdminConfig object for an authentication alias with the given name
# with the given user name, password, and description.
def createAuthDataEntryObjectBAD(name, user, password, description):
	# Using AdminTask.createAuthDataEntry is a bad idea, because
	# - In WAS 7.0, and
	# - in WAS 8.x when the checkbox "Prefix new alias names with the node
	#   name of the cell (for compatibility with earlier releases)" is enabled,
	#   i.e. property com.ibm.websphere.security.JAASAuthData.removeNodeNameGlobal
	#   has the value "false",
	# the alias name is prefixed with the node name of the cell implicitly.
	# This makes no sense, because the alias is attached to the cell. And
	# the behaviour depends on WAS version and user settings.
	AdminTask.createAuthDataEntry([
		'-alias', name,
		'-user', 'dummy',
		'-password', 'dummy'
	])
	# This statement depends on WAS version and user settings!
	name = appserver_was_node + '/' + name
	# Use AdminConfig.modify to set the password, because
	# AdminTask.createAuthDataEntry and AdminTask.modifyAuthDataEntry
	# don't support passwords that contain a '"' character.
	entryObject = getAuthDataEntryObject(name)
	if entryObject == '':
		raise "could not create authentication alias '" + name + "'"
	# entryObject is of type JAASAuthData.
	AdminConfig.modify(entryObject, [
		['userId', user],
		['password', password],
		['description', description]
	])
def createAuthDataEntryObject(name, user, password, description):
	# Use AdminConfig.
	securityObject = AdminConfig.getid('/Cell:' + appserver_was_cell + '/Security:/')
	if securityObject == '':
		raise "could not find Security object for cell '" + appserver_was_cell + "'"
	# securityObject is of type Security.
	AdminConfig.create('JAASAuthData', securityObject, [
		['alias', name],
		['userId', user],
		['password', password],
		['description', description]
	])

# Ensures an authentication alias with the given attributes exists.
def ensureAuthAlias(name, user, password, description):
	# If the authentication alias does not already exist, create it,
	# otherwise modify it.
	entryObject = getAuthDataEntryObject(name)
	if entryObject == '':
		createAuthDataEntryObject(name, user, password, description)
		entryObject = getAuthDataEntryObject(name)
		if entryObject == '':
			raise "could not create authentication alias '" + name + "'"
	else:
		AdminConfig.modify(entryObject, [
			['userId', user],
			['password', password],
			['description', description]
		])

# -------- Delete an authentication alias --------

# Documentation:
# http://pic.dhe.ibm.com/infocenter/wasinfo/v7r0/index.jsp?topic=%2Fcom.ibm.websphere.base.doc%2Finfo%2Faes%2Fae%2Frxml_7securityconfig.html

# Deletes the authentication alias with the given name.
def deleteAuthAliasBAD(name):
	# Using AdminTask.deleteAuthDataEntry is bad because
	# - It leaves an exception in the log if the entry does not exist.
	# - In WAS 7.0, when invoked with argument ['-alias', name], the
	#   function may delete the alias with name node+'/'+name.
	try:
		AdminTask.deleteAuthDataEntry(['-alias', name])
	except:
		# Note: This leaves an exception
		#   FFDC Exception:com.ibm.websphere.management.cmdframework.CommandValidationException
		#   SECJ7732E: The specified auth data entry does not exist.
		# in the server's log, but it can be ignored.
		pass
def deleteAuthAlias(name):
	# Use AdminConfig.
	entryObject = getAuthDataEntryObject(name)
	if entryObject != '':
		# entryObject is of type JAASAuthData.
		AdminConfig.remove(entryObject)

# =============================== JDBC Providers ==============================

# -------- Create a JDBC provider --------

# Create the JDBC provider.
# Documentation:
# http://pic.dhe.ibm.com/infocenter/wasinfo/v7r0/index.jsp?topic=%2Fcom.ibm.websphere.base.doc%2Finfo%2Faes%2Fae%2Frxml_atjdbcprovider.html
# http://pic.dhe.ibm.com/infocenter/wasinfo/v7r0/index.jsp?topic=%2Fcom.ibm.websphere.base.doc%2Finfo%2Faes%2Fae%2Frdat_minreq.html

# Converts a list of filenames to classpath syntax.
# This function deals with two problems:
# 1. conversion of the filenames to slashified syntax,
# 2. when a filename contains a space, this space should not be viewed as a separator
#    between classpath elements.
# Converts [a, b, c] to Tcl list syntax: '["a" "b" "c"]'.
def convertListToClasspathInTclSyntax(l):
	s = ''
	for x in l:
		s += '"' + x.replace(os.sep, '/') + '"' + ' '
	if s != '':
		s = s[0:len(s)-1]
	return '[' + s + ']'
# Converts [a, b, c] to semicolon syntax: 'a;b;c'.
def convertListToClasspathInSemicolonSyntax(l):
	s = ''
	for x in l:
		s += x.replace(os.sep, '/') + ';'
	if s != '':
		s = s[0:len(s)-1]
	return s

# Returns the set of files that matches a given pattern, in a given
# subdirectory of appserver_was_profile_dir, replacing
# appserver_was_profile_dir with ${USER_INSTALL_ROOT} in the result.
def glob_in_profile_dir(relativeDirName, pattern):
	l = []
	for fileName in os.listdir(appserver_was_profile_dir + '/' + relativeDirName):
		if fnmatch.fnmatch(fileName, pattern):
			l.append('${USER_INSTALL_ROOT}/' + relativeDirName + '/' + fileName)
	return l

# Ensures a given JDBC provider object exists.
def ensureJDBCProvider(jdbcProviderName, description, database_selection):
	implementationClassName = ''
	if database_selection == 'derby':
		databaseType = 'Derby'
		providerType = 'Derby JDBC Provider 40'
		implementationType = 'Connection pool data source'
		implementationClassName = 'org.apache.derby.jdbc.EmbeddedConnectionPoolDataSource40' # inferred by createJDBCProvider
		classpath = glob_in_profile_dir(appserver_resourcesdir + '/derby', '*.jar')
	elif database_selection == 'db2':
		databaseType = 'DB2'
		providerType = 'DB2 Universal JDBC Driver Provider'
		implementationType = 'Connection pool data source'
		implementationClassName = 'com.ibm.db2.jcc.DB2ConnectionPoolDataSource' # inferred by createJDBCProvider
		classpath = glob_in_profile_dir(appserver_resourcesdir + '/db2', '*.jar')
	elif database_selection == 'mysql':
		databaseType = 'User-defined'
		providerType = 'User-defined JDBC Provider'
		implementationType = 'User-defined'
		implementationClassName = 'com.mysql.jdbc.jdbc2.optional.MysqlConnectionPoolDataSource'
		classpath = glob_in_profile_dir(appserver_resourcesdir + '/mysql', '*.jar')
	elif database_selection == 'oracle':
		databaseType = 'Oracle'
		providerType = 'Oracle JDBC Driver'
		implementationType = 'Connection pool data source'
		implementationClassName = 'oracle.jdbc.pool.OracleConnectionPoolDataSource' # inferred by createJDBCProvider
		classpath = glob_in_profile_dir(appserver_resourcesdir + '/oracle', '*.jar')
	else:
		raise "unsupported database type"
	# If the JDBC provider does not already exist, create it,
	# otherwise modify it.
	jdbcProviderObject = AdminConfig.getid(scopeContainmentPath + 'JDBCProvider:' + jdbcProviderName + '/')
	if jdbcProviderObject != '' and AdminConfig.showAttribute(jdbcProviderObject, 'providerType') != providerType:
		# Since 'providerType' is a read-only attribute, there is no way to change it.
		# So remove the JDBC provider.
		# AdminTask.deleteJDBCProvider was only introduced in WAS 8.0.
		if was8:
			AdminTask.deleteJDBCProvider(jdbcProviderObject)
		else:
			AdminConfig.remove(jdbcProviderObject)
		jdbcProviderObject = ''
	if jdbcProviderObject == '':
		print 'was-install.py creating JDBC provider...'
		AdminTask.createJDBCProvider([
			'-scope', scopeParam,
			'-databaseType', databaseType,
			'-providerType', providerType,
			'-implementationType', implementationType,
			'-implementationClassName', implementationClassName,
			'-classpath', convertListToClasspathInTclSyntax(classpath),
			'-name', jdbcProviderName,
			'-description', description,
			'-isolated', 'true', # Avoid version conflicts with other JDBC providers
			'-nativePath', '' # Needed for -isolated true
		])
	else:
		# Note that the properties databaseType and implementationType are
		# not actually stored in the jdbcProviderObject.
		print 'was-install.py updating JDBC provider...'
		# When using AdminConfig.modify, we have to pass the classpath in
		# semicolon syntax. And, since this modify call augments (not replaces)
		# the classpath unless it is empty, we have to make two calls,
		# one to first clear the classpath, and another one to set it.
		AdminConfig.modify(
			jdbcProviderObject,
			[
				# ADMG0014E: Attribute providerType is a read-only attribute.
				#['providerType', providerType],
				['implementationClassName', implementationClassName],
				['classpath', ''],
				['name', jdbcProviderName],
				['description', description],
				['isolatedClassLoader', 'true'],
				['nativepath', '']
			])
		if len(classpath) > 0:
			AdminConfig.modify(
				jdbcProviderObject,
				[
					# ADMG0014E: Attribute providerType is a read-only attribute.
					#['providerType', providerType],
					['implementationClassName', implementationClassName],
					['classpath', convertListToClasspathInSemicolonSyntax(classpath)],
					['name', jdbcProviderName],
					['description', description],
					['isolatedClassLoader', 'true'],
					['nativepath', '']
				])
	jdbcProviderObject = AdminConfig.getid(scopeContainmentPath + 'JDBCProvider:' + jdbcProviderName + '/')
	print 'was-install.py JDBC provider done'
	print AdminConfig.show(jdbcProviderObject)
	print
	return jdbcProviderObject

# -------- Delete a JDBC provider --------

# Documentation:
# http://pic.dhe.ibm.com/infocenter/wasinfo/v8r0/index.jsp?topic=%2Fcom.ibm.websphere.base.doc%2Finfo%2Faes%2Fae%2Frxml_atjdbcprovider.html

# Delete a JDBC provider.
def deleteJDBCProvider(jdbcProviderName):
	print 'was-install.py deleting JDBC provider...'
	jdbcProviderObject = AdminConfig.getid(scopeContainmentPath + 'JDBCProvider:' + jdbcProviderName + '/')
	if jdbcProviderObject != '':
		# AdminTask.deleteJDBCProvider was only introduced in WAS 8.0.
		if was8:
			AdminTask.deleteJDBCProvider(jdbcProviderObject)
		else:
			AdminConfig.remove(jdbcProviderObject)
	print 'was-install.py deleting JDBC provider done'

# ============================= JDBC Data sources =============================

# -------- Create a JDBC datasource --------

# Create the datasources.
# Documentation:
# http://pic.dhe.ibm.com/infocenter/wasinfo/v7r0/index.jsp?topic=%2Fcom.ibm.websphere.base.doc%2Finfo%2Faes%2Fae%2Frxml_atjdbcprovider.html

# Shows the contents of a J2EEResourcePropertySet object.
def showJ2EEResourcePropertySet(obj):
	for prop in AdminUtilities.convertToList(AdminConfig.showAttribute(obj, 'resourceProperties')):
		print prop
		print AdminConfig.showAttribute(prop, 'name')
		print AdminConfig.showAttribute(prop, 'value')
		print

# Finds a property with the given name in a J2EEResourcePropertySet object.
def hasJ2EEResourcePropertySetProperty(obj, name):
	for prop in AdminUtilities.convertToList(AdminConfig.showAttribute(obj, 'resourceProperties')):
		if AdminConfig.showAttribute(prop, 'name') == name:
			return prop
	return None

# Ensures a J2EEResourcePropertySet object has a given property given by (name, tipe, value).
def setJ2EEResourcePropertySetProperty(obj, name, tipe, value):
	propertyObject = hasJ2EEResourcePropertySetProperty(obj, name)
	if propertyObject == None:
		AdminConfig.create('J2EEResourceProperty', obj,
			[
				['name', name],
				['type', tipe],
				['value', value]
			])
	else:
		AdminConfig.modify(propertyObject,
			[
				['type', tipe],
				['value', value]
			])

# Ensures that a resource object (e.g. a datasource object) has the given
# custom properties as J2EEResourcePropertySet.
# Documentation: http://pic.dhe.ibm.com/infocenter/wasinfo/v7r0/index.jsp?topic=%2Fcom.ibm.websphere.base.doc%2Finfo%2Faes%2Fae%2Ftxml_wascustom.html
def ensureJ2EEResourceProperties(resourceObject, customProperties):
	j2eePropertiesObject = AdminConfig.showAttribute(resourceObject, 'propertySet')
	if j2eePropertiesObject == None:
		j2eePropertiesObject = AdminConfig.create('J2EEResourcePropertySet', resourceObject, [])
	for customProperty in customProperties:
		setJ2EEResourcePropertySetProperty(j2eePropertiesObject,
			customProperty[0], customProperty[1], customProperty[2])
	#print AdminConfig.show(j2eePropertiesObject)
	#print
	#showJ2EEResourcePropertySet(j2eePropertiesObject)

# Constructs a declaration for a datasource property, in the format expected
# by the function ensureJ2EEResourceProperties.
def constructDataSourceCustomProperty(name, tipe, value):
	return [ name, tipe, value ]

# Ensures a given datasource object exists.
def ensureDatabase(datasourceName, description, database_selection, jndiName, database_name, database_jdbc_url, database_derby_datadir, database_db2_host, database_db2_port, database_user_name, database_user_password, database_schema, database_properties_py, authenticationAlias):
	if database_selection == 'derby':
		dataStoreHelperClassName = 'com.ibm.websphere.rsadapter.DerbyDataStoreHelper'
		properties = [
			['databaseName', 'java.lang.String', database_derby_datadir + '/' + database_name]
		]
		customProperties = [
			['user', 'java.lang.String', database_schema],
			['password', 'java.lang.String', '']
		]
	elif database_selection == 'db2':
		dataStoreHelperClassName = 'com.ibm.websphere.rsadapter.DB2UniversalDataStoreHelper'
		properties = [
			['driverType', 'java.lang.Integer', '4'],
			['serverName', 'java.lang.String', database_db2_host],
			['portNumber', 'java.lang.Integer', database_db2_port],
			['databaseName', 'java.lang.String', database_name]
		]
		customProperties = [
		]
		if database_schema != '':
			customProperties.append(['currentSchema', 'java.lang.String', database_schema])
	elif database_selection == 'mysql':
		dataStoreHelperClassName = 'com.ibm.websphere.rsadapter.GenericDataStoreHelper'
		properties = [
		]
		customProperties = [
			['URL', 'java.lang.String', database_jdbc_url],
			['user', 'java.lang.String', database_user_name],
			['password', 'java.lang.String', database_user_password],
			# Avoid runtime errors like
			# java.sql.SQLException: Can't call rollback when autocommit=true
			# java.sql.SQLException: Error open transaction is not closed
			# See <http://docs.oracle.com/cd/E19226-01/820-7695/gbhbr/index.html>.
			['relaxAutoCommit', 'java.lang.Boolean', 'true']
		]
	elif database_selection == 'oracle':
		dataStoreHelperClassName = 'com.ibm.websphere.rsadapter.Oracle11gDataStoreHelper'
		properties = [
			['URL', 'java.lang.String', database_jdbc_url]
		]
		customProperties = [
			['user', 'java.lang.String', database_user_name],
			['password', 'java.lang.String', database_user_password]
		]
	else:
		raise "unsupported database type"
	# Set the datasources to non-transactional.
	# - The advantage of transactional: Without "transactional", when you query
	#   the database while a transaction is in progress that modifies the database,
	#   the query retrieves the unmodified data in the database (since not committed).
	# - The advantage of using non-transactional is that the overhead for enlisting
	#   and delisting connections is avoided. But then you need to manage the
	#   transactions in the application code.
	# - For the Worklight datasources:
	#   The transactions of these datasources are managed by Worklight code;
	#   if the datasource attempts to do it as well, it leads to errors like
	#   J2CA0081E: Method cleanup failed while trying to execute method cleanup on ManagedConnection WSRdbManagedConnectionImpl@789caeb2 from resource WorklightDS/connectionManager-default. Caught exception: com.ibm.ws.rsadapter.exceptions.DataStoreAdapterException: DSRA0080E: An exception was received by the Data Store Adapter. See original exception message: Cannot call 'cleanup' on a ManagedConnection while it is still in a transaction..
	#   See http://www-01.ibm.com/support/docview.wss?uid=swg27038215; cf. also RTC 19344.
	# - For the Application Center datasource, same decision.
	customProperties.append(constructDataSourceCustomProperty('nonTransactionalDataSource', 'java.lang.Boolean', 'true'))
	if database_properties_py != '':
		# The file database_properties_py contains augmentation statements for customProperties.
		# It references the constructDataSourceCustomProperty function.
		execfile(database_properties_py)
	# If the data source does not already exist, create it,
	# otherwise modify it.
	datasourceObject = AdminConfig.getid(scopeContainmentPath + 'JDBCProvider:' + jdbcProviderName + '/DataSource:' + datasourceName + '/')
	if datasourceObject != '' and AdminConfig.showAttribute(datasourceObject, 'providerType') != AdminConfig.showAttribute(jdbcProviderObject, 'providerType'):
		# Since 'providerType' is a read-only attribute, there is no way to change it.
		# So remove the datasource.
		# AdminTask.deleteDatasource was only introduced in WAS 8.0.
		if was8:
			AdminTask.deleteDatasource(datasourceObject)
		else:
			AdminConfig.remove(datasourceObject)
		datasourceObject = ''
	if datasourceObject == '':
		print 'was-install.py creating datasource...'
		if len(properties) == 0:
			# -configureResourceProperties does not accept an empty argument:
			# "WASX8015E: Invalid option value for step configureResourceProperties"
			# -configureResourceProperties does not accept a '[]' argument:
			# java.lang.IllegalArgumentException
			# So, omit the option where there are no properties to configure.
			if authenticationAlias == '':
				AdminTask.createDatasource(
					jdbcProviderObject,
					[
						'-name', datasourceName,
						'-jndiName', jndiName,
						'-dataStoreHelperClassName', dataStoreHelperClassName,
						'-description', description
					]);
			else:
				AdminTask.createDatasource(
					jdbcProviderObject,
					[
						'-name', datasourceName,
						'-jndiName', jndiName,
						'-dataStoreHelperClassName', dataStoreHelperClassName,
						'-description', description,
						'-componentManagedAuthenticationAlias', authenticationAlias
					]);
		else:
			if authenticationAlias == '':
				AdminTask.createDatasource(
					jdbcProviderObject,
					[
						'-name', datasourceName,
						'-jndiName', jndiName,
						'-dataStoreHelperClassName', dataStoreHelperClassName,
						'-configureResourceProperties', properties,
						'-description', description
					]);
			else:
				AdminTask.createDatasource(
					jdbcProviderObject,
					[
						'-name', datasourceName,
						'-jndiName', jndiName,
						'-dataStoreHelperClassName', dataStoreHelperClassName,
						'-configureResourceProperties', properties,
						'-description', description,
						'-componentManagedAuthenticationAlias', authenticationAlias
					]);
	else:
		print 'was-install.py updating datasource...'
		AdminConfig.modify(
			datasourceObject,
			[
				['name', datasourceName],
				['jndiName', jndiName],
				['datasourceHelperClassname', dataStoreHelperClassName],
				['description', description],
				['provider', jdbcProviderObject]
				# ADMG0014E: Attribute providerType is a read-only attribute.
				#['providerType', AdminConfig.showAttribute(jdbcProviderObject, 'providerType')]
			]);
		if authenticationAlias != '':
			AdminConfig.modify(
				datasourceObject,
				[
					['authDataAlias', authenticationAlias]
				]);
		if len(properties) > 0:
			ensureJ2EEResourceProperties(datasourceObject, properties)
	print 'was-install.py modifying datasource properties...'
	datasourceObject = AdminConfig.getid(scopeContainmentPath + 'JDBCProvider:' + jdbcProviderName + '/DataSource:' + datasourceName + '/')
	ensureJ2EEResourceProperties(datasourceObject, customProperties)
	print 'was-install.py datasource done'
	print AdminConfig.show(datasourceObject)
	print
	#showJ2EEResourcePropertySet(AdminConfig.showAttribute(datasourceObject, 'propertySet'))

# -------- Delete a JDBC datasource --------

# Documentation:
# http://pic.dhe.ibm.com/infocenter/wasinfo/v8r0/index.jsp?topic=%2Fcom.ibm.websphere.base.doc%2Finfo%2Faes%2Fae%2Frxml_atjdbcprovider.html
# AdminTask.deleteDatasource was only introduced in WAS 8.0.

# Delete a datasource.
def deleteDatasource(jdbcProviderName, jdbcProviderObject, datasourceName):
	print 'was-install.py deleting datasource...'
	datasourceObject = AdminConfig.getid(scopeContainmentPath + 'JDBCProvider:' + jdbcProviderName + '/DataSource:' + datasourceName + '/')
	if datasourceObject != '':
		# AdminTask.deleteDatasource was only introduced in WAS 8.0.
		if was8:
			AdminTask.deleteDatasource(datasourceObject)
		else:
			AdminConfig.remove(datasourceObject)
	print 'was-install.py deleting datasource done'

# ============================== Shared Libraries =============================

# -------- Create a shared library --------

# Ensures a given shared library object exists.
# Documentation:
# http://pic.dhe.ibm.com/infocenter/wasinfo/v7r0/index.jsp?topic=%2Fcom.ibm.websphere.base.doc%2Finfo%2Faes%2Fae%2Ftxml_applibrary.html
def ensureSharedLibrary(sharedLibraryName, description, classpath):
	print 'was-install.py creating shared library...'
	sharedLibraryObject = AdminConfig.getid(scopeContainmentPath + 'Library:' + sharedLibraryName + '/')
	if sharedLibraryObject == '':
		AdminConfig.create('Library', scopeObject, [
			['name', sharedLibraryName],
			['description', description],
			['classPath', classpath],
			['isolatedClassLoader', 'true']
		])
	else:
		AdminConfig.modify(sharedLibraryObject, [
			['description', description],
			['classPath', classpath],
			['isolatedClassLoader', 'true']
		])
	print 'was-install.py creating shared library done'

# =========================== JVM system properties ===========================

# -------- Set a JVM system property --------

# Add system property.
# Documentation:
# http://pic.dhe.ibm.com/infocenter/wasinfo/v7r0/index.jsp?topic=%2Fcom.ibm.websphere.base.doc%2Finfo%2Faes%2Fae%2Frxml_atservermanagement.html

# Returns the Property object that encodes the JVM system property with the given name.
def getJVMSystemPropertyObject(name, nodeName, serverName):
	serverObject = AdminConfig.getid('/Cell:' + appserver_was_cell + '/Node:' + nodeName + '/Server:' + serverName + '/')
	# serverObject is of type Server.
	for jvmObject in AdminUtilities.convertToList(AdminConfig.list('JavaVirtualMachine', serverObject)):
		# jvmObject is of type JavaVirtualMachine.
		for propertyObject in AdminUtilities.convertToList(AdminConfig.list('Property', jvmObject)):
			# propertyObject is of type Property.
			if name == AdminConfig.showAttribute(propertyObject, 'name'):
				return propertyObject
	return ''

# Ensures there is a JVM system property with a given value.
# If the value is '', the JVM system property setting is removed.
# Documentation:
# http://pic.dhe.ibm.com/infocenter/wasinfo/v7r0/index.jsp?topic=%2Fcom.ibm.websphere.base.doc%2Finfo%2Faes%2Fae%2Frxml_atservermanagement.html
def ensureJVMSystemProperty(name, value):
	print 'was-install.py setting JVM system property ' + name + '...'
	for server in affected_servers:
		(nodeName, serverName) = server
		# In WAS 7, invoking AdminTask.setJVMSystemProperties always creates a
		# new Property object, even if there was already one with the same name.
		propertyObject = getJVMSystemPropertyObject(name, nodeName, serverName)
		if propertyObject == '':
			if value != '':
				AdminTask.setJVMSystemProperties([
					'-nodeName', nodeName,
					'-serverName', serverName,
					'-propertyName', name,
					'-propertyValue', value
				])
		else:
			if value != '':
				AdminConfig.modify(propertyObject, [['value', value]])
			else:
				AdminConfig.remove(propertyObject)
	print 'was-install.py setting JVM system property ' + name + ' done'

# -------- Delete a JVM system property --------

# Returns the Property object that encodes the JVM system property with the given name.
def getJVMSystemPropertyObject(name, nodeName, serverName):
	serverObject = AdminConfig.getid('/Cell:' + appserver_was_cell + '/Node:' + nodeName + '/Server:' + serverName + '/')
	for jvmObject in AdminUtilities.convertToList(AdminConfig.list('JavaVirtualMachine', serverObject)):
		for propertyObject in AdminUtilities.convertToList(AdminConfig.list('Property', jvmObject)):
			if name == AdminConfig.showAttribute(propertyObject, 'name'):
				return propertyObject
	return ''

# Removes a given JVM system property.
def deleteJVMSystemPropertyObject(name):
	for server in affected_servers:
		(nodeName, serverName) = server
		propertyObject = getJVMSystemPropertyObject(name, nodeName, serverName)
		if propertyObject != '':
			AdminConfig.remove(propertyObject)

# ========================== Web container properties =========================

# -------- Set a web container custom property --------

# Add web container custom property.
# Documentation:
# http://pic.dhe.ibm.com/infocenter/wasinfo/v7r0/index.jsp?topic=%2Fcom.ibm.websphere.base.doc%2Finfo%2Faes%2Fae%2Frweb_custom_props.html
def ensureWebContainerCustomProperty(name, value, description):
	print 'was-install.py setting web container custom property...'
	for server in affected_servers:
		(nodeName, serverName) = server
		serverObject = AdminConfig.getid('/Cell:' + appserver_was_cell + '/Node:' + nodeName + '/Server:' + serverName + '/')
		for webContainerObject in AdminUtilities.convertToList(AdminConfig.list('WebContainer', serverObject)):
			found = False
			for propertyObject in AdminUtilities.convertToList(AdminConfig.list('Property', webContainerObject)):
				if AdminConfig.showAttribute(propertyObject, 'name') == name:
					found = True
					AdminConfig.modify(propertyObject, [
						['value', value],
						['description', description]
					])
			if not found:
				AdminConfig.create('Property', webContainerObject, [
					['name', name],
					['value', value],
					['description', description]
				])
	print 'was-install.py setting web container custom property done'

# ========================== Installing applications ==========================

# -------- Install an application --------

# Install an application, given as a war file.
# Documentation:
# http://pic.dhe.ibm.com/infocenter/wasinfo/v7r0/index.jsp?topic=%2Fcom.ibm.websphere.base.doc%2Finfo%2Faes%2Fae%2Ftxml_callappinstall.html
# http://pic.dhe.ibm.com/infocenter/wasinfo/v7r0/index.jsp?topic=%2Fcom.ibm.websphere.base.doc%2Finfo%2Faes%2Fae%2Frxml_adminapp.html
# http://pic.dhe.ibm.com/infocenter/wasinfo/v7r0/index.jsp?topic=%2Fcom.ibm.websphere.base.doc%2Finfo%2Faes%2Fae%2Frxml_taskoptions.html
# otherOptions is a list of options (key-value pairs in a single list of even length).
def installApplication(war, war_basename, war_context, war_appname, otherOptions):
	print 'was-install.py installing ' + war_basename + '...'
	# Collect the options.
	options = []
	options.append('-cell'); options.append(appserver_was_cell)
	if appserver_was_scope == 'server':
		options.append('-node'); options.append(appserver_was_node)
		options.append('-server'); options.append(appserver_was_serverInstance)
		# Apparently redundant.
		#options.append('-target'); options.append('WebSphere:cell=' + appserver_was_cell + ',node=' + appserver_was_node + ',server=' + appserver_was_serverInstance)
	elif appserver_was_scope == 'nd-cell':
		# Here the '-target' option is necessary: without it, the application
		# gets installed in the deployment manager.
		# In the '-target' option, we cannot list servers that are members of some cluster:
		# It produces an error
		# "The target server ... specified for "+war_basename+"+WEB-INF/web.xml is a part of cluster ..., but cluster ... is not specified in the list of targets."
		# Therefore we have to specify all clusters and, separately, all servers
		# that are not members of any cluster.
		targets = []
		for clusterObject in AdminUtilities.convertToList(AdminConfig.list('ServerCluster')):
			# clusterObject is of type ServerCluster.
			name = AdminConfig.showAttribute(clusterObject, 'name')
			if name != '':
				targets.append('WebSphere:cell=' + appserver_was_cell + ',cluster=' + name)
		for server in affected_servers:
			(nodeName, serverName) = server
			if not isInCluster(nodeName, serverName):
				targets.append('WebSphere:cell=' + appserver_was_cell + ',node=' + nodeName + ',server=' + serverName)
		options.append('-target'); options.append('+'.join(targets))
	elif appserver_was_scope == 'nd-cluster':
		options.append('-cluster'); options.append(appserver_was_nd_cluster)
		# Apparently redundant.
		#options.append('-target'); options.append('WebSphere:cell=' + appserver_was_cell + ',cluster=' + appserver_was_nd_cluster)
	elif appserver_was_scope == 'nd-node':
		options.append('-node'); options.append(appserver_was_nd_node)
		# Here the '-target' option is necessary to avoid a "No valid target is specified in ObjectName" error.
		# And servers that are members of some cluster have been excluded from affected_servers, to avoid an error
		# "The target server ... specified for "+war_basename+"+WEB-INF/web.xml is a part of cluster ..., but cluster ... is not specified in the list of targets."
		targets = []
		for server in affected_servers:
			(nodeName, serverName) = server
			targets.append('WebSphere:cell=' + appserver_was_cell + ',node=' + nodeName + ',server=' + serverName)
		options.append('-target'); options.append('+'.join(targets))
	elif appserver_was_scope == 'nd-server':
		options.append('-node'); options.append(appserver_was_nd_node)
		options.append('-server'); options.append(appserver_was_nd_server)
		# Apparently redundant.
		#options.append('-target'); options.append('WebSphere:cell=' + appserver_was_cell + ',node=' + appserver_was_nd_node + ',server=' + appserver_was_nd_server)
	options.append('-contextroot'); options.append(war_context)
	options.append('-appname'); options.append(war_appname)
	# The options '-MapModulesToServers' [[war_basename, war_basename+',WEB-INF/web.xml', targets]]
	# are apparently redundant.
	for argument in otherOptions:
		options.append(argument)
	# Avoid error "ADMA0010E: Validation error in task Selecting virtual hosts for Web modules. A virtual host is not specified"
	options.append('-usedefaultbindings')
	# Perform the call.
	AdminApp.install(war, options)
	print 'was-install.py installing ' + war_basename + ' done'

# Constructs a declaration for a JNDI property.
# Documentation: http://pic.dhe.ibm.com/infocenter/wasinfo/v8r5/index.jsp?topic=%2Fcom.ibm.websphere.base.doc%2Fae%2Frxml_taskoptions.html
# section "MapEnvEntryForWebMod".
def constructJNDIEnvironmentEntry(war_basename, prop_name, prop_value):
	# If we don't put a wildcard in the first argument (the application name), we get an error
	# WASX7111E: Cannot find a match for supplied option: "..." for task "MapEnvEntryForWebMod"
	# Similarly for the type and the description: If the values that we pass don't match
	# the ones specified in the web.xml, we get the same error. So pass wildcards instead.
	# Since the description may contain newlines, use a more generic wildcard for it.
	return [ '.*', war_basename + ',WEB-INF/web.xml', prop_name, '.*', '(?s).*', prop_value ]

# Constructs a mapping for a JNDI datasource.
# Documentation: http://pic.dhe.ibm.com/infocenter/wasinfo/v8r5/index.jsp?topic=%2Fcom.ibm.websphere.base.doc%2Fae%2Frxml_taskoptions.html
# section "MapResRefToEJB".
# referenced_jndi_name is the name contained in a <res-ref-name> element in the
# war file's WEB-INF/web.xml.
# existing_jndi_name is the JNDI name of an existing datasource definition.
# If existing_jndi_name == referenced_jndi_name, this mapping is not needed,
# but it doesn't harm either.
def constructJNDIDatasourceMapping(war_basename, referenced_jndi_name, existing_jndi_name):
	# The first argument could also be given as 'Worklight' instead of a wildcard.
	return [ '.*', '', war_basename + ',WEB-INF/web.xml', referenced_jndi_name, 'javax.sql.DataSource', existing_jndi_name, '', '', '' ]

# Converts a list of lists of strings to Tcl syntax.
# Without it, the use of the -MapEnvEntryForWebMod option has the following
# problems:
# - Backslashes get doubled in the process of defining a JNDI environment entry:
#     abc\def    written as Python 'abc\\def'       becomes      abc\\def
# - Unicode character in the range U+0000..U+001F or U+007F..U+00FF get replaced
#   by an escape sequence:
#     U+0008     gets replaced by \b
#     U+0009     gets replaced by \t
#     U+000A     gets replaced by \n
#     U+000C     gets replaced by \f
#     U+000D     gets replaced by \r
#   and the others get replaced by hexadecimal escape sequences, e.g.
#     U+0001     gets replaced by \x01
#     U+007F     gets replaced by \x7F
#     U+00DF     gets replaced by \xDF
#     U+00FF     gets replaced by \xFF
# - If a string starts with a non-ASCII Unicode character that needs the \ u
#   syntax, an escaped copy of the string gets inserted at the beginning.
#   For example:
#     €def        becomes       \u20ACdef€def
#     日本語       becomes       \u65E5\u672C\u8A9E日本語
# - If a character right before a non-ASCII Unicode character is a space or a
#   comma, we see exceptions inside AdminApp.install.
#   Like this:
#     ' 日本語'    yields
#       java.lang.StringIndexOutOfBoundsException
#               at java.lang.String.substring(String.java:1148)
#               at com.ibm.ws.scripting.JythonUtilities.objectArrayToString(JythonUtilities.java:1122)
#               at com.ibm.ws.scripting.AdminAppClient.install(AdminAppClient.java:1396)
#   Or this:
#     'hello 日本語'     yields
#        java.lang.IllegalArgumentException
#                at com.ibm.ws.scripting.LanguageUtilities.throwIllegalArgException(LanguageUtilities.java:1174)
#                at com.ibm.ws.scripting.JythonUtilities.getString(JythonUtilities.java:705)
#                at com.ibm.ws.scripting.LanguageUtilities.getList(LanguageUtilities.java:499)
#                at com.ibm.ws.scripting.JythonUtilities.addOptionValue(JythonUtilities.java:894)
#                at com.ibm.ws.scripting.LanguageUtilities.optionsToHashtable(LanguageUtilities.java:294)
#                at com.ibm.ws.scripting.AdminAppClient.doInstall(AdminAppClient.java:2080)
#                at com.ibm.ws.scripting.AdminAppClient.install(AdminAppClient.java:1410)
#                at com.ibm.ws.scripting.AdminAppClient.install(AdminAppClient.java:1398)
# Problems not solved:
# - A string that contains both a single-quote and a double-quote is
#   unsupported.
#   This is because in the Tcl syntax, we can use either "..." or '...' to
#   protect the contents of a string. But in either syntax, backslash is not
#   special, and juxtaposition produces two separate list elements: 'abc'"def"
#   or "abc"'def' produces separate list elements abc and def.
# - Strings that contain invalid XML characters produce an exception when WAS
#   writes a "merged" web.xml to disk.
# Converts [[a, b, c], [d, e]] to Tcl list syntax: '[["a" "b" "c"] ["d" "e"]]'.
def convertListListToTclSyntax(ll):
	ss = ''
	for l in ll:
		s = ''
		for x in l:
			if x.find('"') >= 0:
				# Use single-quotes.
				s += '\'' + x + '\''
			else:
				# Use double-quotes.
				s += '"' + x + '"'
			s += ' '
		if s != '':
			s = s[0:len(s)-1]
		ss += '[' + s + ']' + ' '
	if ss != '':
		ss = ss[0:len(ss)-1]
	return '[' + ss + ']'

# -------- Set classloader mode --------

# Documentation:
# http://pic.dhe.ibm.com/infocenter/wasinfo/v7r0/index.jsp?topic=%2Fcom.ibm.websphere.base.doc%2Finfo%2Faes%2Fae%2Fcrun_classload.html

def ensureClassLoaderModeIsParentLast(appname):
	print 'was-install.py setting classloader mode for ' + appname + '...'
	# Verify that the server does not override the application-level classloader mode.
	# Keep this check consistent with the one done in the custom panels!
	for applicationServerObject in AdminUtilities.convertToList(AdminConfig.list('ApplicationServer', scopeObject)):
		# applicationServerObject is of type ApplicationServer.
		# For WAS ND, exclude the ApplicationServer that belongs to the deployment manager.
		serverObject = AdminConfig.showAttribute(applicationServerObject, 'server')
		# serverObject is of type Server.
		if AdminConfig.showAttribute(serverObject, 'serverType') == 'APPLICATION_SERVER':
			if AdminConfig.showAttribute(applicationServerObject, 'applicationClassLoaderPolicy') == 'SINGLE':
				if AdminConfig.showAttribute(applicationServerObject, 'applicationClassLoadingMode') == 'PARENT_FIRST':
					serverName = AdminConfig.showAttribute(serverObject, 'name')
					raise 'inappropriate server settings: server ' + serverName + ' has applicationClassLoaderPolicy = SINGLE and applicationClassLoadingMode = PARENT_FIRST'
	deploymentObject = AdminConfig.getid('/Deployment:'+appname+'/')
	if deploymentObject != '':
		# deploymentObject is not a list, because WAS guarantees that there is at most one application with a given name.
		# deploymentObject is of type Deployment.
		applicationDeploymentObject = AdminConfig.showAttribute(deploymentObject, 'deployedObject')
		if applicationDeploymentObject != '':
			# applicationDeploymentObject is of type ApplicationDeployment.
			# Set the classloader mode for the application to "Parent classloader last".
			classloaderObject = AdminConfig.showAttribute(applicationDeploymentObject, 'classloader')
			AdminConfig.modify(classloaderObject, [['mode', 'PARENT_LAST']])
			# Set the Class loader order of all modules to "Parent classloader last".
			for moduleObject in AdminUtilities.convertToList(AdminConfig.showAttribute(applicationDeploymentObject, 'modules')):
				# XXX Only if moduleObject.find('#WebModuleDeployment_') >= 0 ??
				AdminConfig.modify(moduleObject, [['classloaderMode', 'PARENT_LAST']])
	print 'was-install.py setting classloader mode for ' + appname + ' done'

# -------- Uninstall an application --------

# Uninstalls a web application.
# Documentation:
# http://pic.dhe.ibm.com/infocenter/wasinfo/v7r0/index.jsp?topic=%2Fcom.ibm.websphere.base.doc%2Finfo%2Faes%2Fae%2Ftxml_uninstall.html
def uninstallApplication(war_basename, war_appname):
	print 'was-install.py uninstalling ' + war_basename + '...'
	try:
		AdminApp.uninstall(war_appname)
	except:
		# Catch "com.ibm.ws.scripting.ScriptingException: WASX7280E: An application with name ... does not exist."
		pass
	print 'was-install.py uninstalling ' + war_basename + ' done'

# =========================== WAS ND synchorization ===========================

# Execute a "full synchronize".
# Documentation:
# http://www-01.ibm.com/support/docview.wss?uid=swg21108407
# http://www-01.ibm.com/support/docview.wss?uid=swg21233075
def fullSync():
	print 'was-install.py doing full synchronize...'
	# This is the old way (http://www.vm.ibm.com/events/2005-W25.PDF).
	#for node in affected_nodes:
	#	nodeSyncObject = AdminControl.completeObjectName('type=NodeSync,cell=' + appserver_was_cell + ',node=' + node + ',*')
	#	# For the node which contains just the deployment manager, nodeSyncObject is empty.
	#	if nodeSyncObject != '':
	#		AdminControl.invoke(nodeSyncObject, 'sync')
	# This is the way suggested by the WAS administrative console.
	for node in affected_nodes:
		configRepositoryObject = AdminControl.completeObjectName('type=ConfigRepository,name=repository,process=nodeagent,cell=' + appserver_was_cell + ',node=' + node + ',*')
		if configRepositoryObject != '':
			AdminControl.invoke(configRepositoryObject, 'refreshRepositoryEpoch')
	cellSyncObject = AdminControl.completeObjectName('type=CellSync,name=cellSync,process=dmgr,cell=' + appserver_was_cell + ',*')
	if cellSyncObject != '':
		for node in affected_nodes:
			AdminControl.invoke(cellSyncObject, 'syncNode', '[' + node + ']', '[java.lang.String]')
	print 'was-install.py full synchronize done'

# ======================= Starting/stopping applications ======================

# -------- Start an application --------

# Documentation:
# http://pic.dhe.ibm.com/infocenter/wasinfo/v7r0/index.jsp?topic=%2Fcom.ibm.websphere.base.doc%2Finfo%2Faes%2Fae%2Ftxml_startapplication.html
def startApplication(war_basename, war_appname):
	print 'was-install.py starting ' + war_basename + '...'
	# If appserver_was_scope is 'server' or 'nd-node' or 'nd-server',
	# it would be sufficient to make a single AdminControl.queryNames
	# call and iterate over the result. But this approach does not work
	# if appserver_was_scope is 'nd-cell' (the result contains also
	# the deployment manager process) or 'nd-cluster'.
	# Therefore loop over the predetermined set of servers.
	for server in affected_servers:
		(nodeName, serverName) = server
		appManager = AdminControl.queryNames('type=ApplicationManager,'
						     + 'cell=' + appserver_was_cell + ','
						     + 'node=' + nodeName + ','
						     + 'process=' + serverName + ','
						     + '*')
		# If the server is not running, appManager is empty.
		# Then the application will be started when the server starts.
		if appManager != '':
			AdminControl.invoke(appManager, 'startApplication', war_appname)
	print 'was-install.py starting ' + war_basename + ' done'

# -------- Stop an application --------

# Stops a web application.
# Documentation:
# http://pic.dhe.ibm.com/infocenter/wasinfo/v7r0/index.jsp?topic=%2Fcom.ibm.websphere.base.doc%2Finfo%2Faes%2Fae%2Ftxml_stopapplication.html
def stopApplication(war_basename, war_appname):
	print 'was-install.py stopping ' + war_basename + '...'
	for server in affected_servers:
		(nodeName, serverName) = server
		appManager = AdminControl.queryNames('type=ApplicationManager,'
						     + 'cell=' + appserver_was_cell + ','
						     + 'node=' + nodeName + ','
						     + 'process=' + serverName + ','
						     + '*')
		# If the server is not running, appManager is empty.
		if appManager != '':
			try:
				AdminControl.invoke(appManager, 'stopApplication', war_appname)
			except:
				# Catch "com.ibm.ws.exception.RuntimeWarning: Application ... not started"
				pass
	print 'was-install.py stopping ' + war_basename + ' done'


# =============================================================================
# =============================== Execution Code ==============================
# =============================================================================

# unique_suffix and space_suffix are suffixes used to distinguish applications
# from different Worklight project installations in the same cell. It is needed
# even when the corresponding .war files are deployed to disjoint set of servers
# (such as different clusters, or different nodes), because some entities have
# cell-wide scope (application names, authentication aliases, datasource JNDI
# names) and some entities are looked up by name, e.g. via AdminConfig.getid
# (JDBC providers, JDBC datasources, shared libraries).
if worklight_id != '':
	unique_suffix = '_' + worklight_id
	space_suffix = ' ' + worklight_id
else:
	unique_suffix = ''
	space_suffix = ''

# Add this suffix to all application names.
worklight_war_appname = worklight_war_appname + unique_suffix

# authenticationAliasForWorklight
# authenticationAliasForWorklightReports
# are the names of the authentication aliases (= credentials for a datasource).
if worklight_database_selection == 'derby' or worklight_database_selection == 'mysql' or worklight_database_selection == 'oracle':
	authenticationAliasForWorklight = ''
else:
	authenticationAliasForWorklight = 'Worklight' + worklight_database_selection.capitalize() + 'DatabaseCredentials' + unique_suffix
if worklightreports_database_selection == 'derby' or worklightreports_database_selection == 'mysql' or worklightreports_database_selection == 'oracle':
	authenticationAliasForWorklightReports = ''
else:
	authenticationAliasForWorklightReports = 'WorklightReports' + worklightreports_database_selection.capitalize() + 'DatabaseCredentials' + unique_suffix

# Add this suffix to all datasource JNDI names.
worklight_database_jndi_name = worklight_database_jndi_name + unique_suffix
worklightreports_database_jndi_name = worklightreports_database_jndi_name + unique_suffix


if operation == 'install':


	if appserver_was_scope[0:3] == 'nd-':
		# Execute a "full synchronize" of the resources to all nodes.
		fullSync()

	# ======================== Configuration for Worklight ========================

	print 'was-install.py creating authentication alias...'
	if authenticationAliasForWorklight != '':
		description = 'used by the IBM Worklight JDBC data source'
		ensureAuthAlias(authenticationAliasForWorklight, worklight_database_user_name, worklight_database_user_password, description)
	if authenticationAliasForWorklightReports != '':
		description = 'used by the IBM Worklight reports JDBC data source'
		ensureAuthAlias(authenticationAliasForWorklightReports, worklightreports_database_user_name, worklightreports_database_user_password, description)
	print 'was-install.py creating authentication alias done'
	print authenticationAliasForWorklight
	print authenticationAliasForWorklightReports
	print

	jdbcProviderName = 'Worklight JDBC Provider' + space_suffix
	description = 'JDBC provider for the IBM Worklight database'
	database_selection = worklight_database_selection
	jdbcProviderObject = ensureJDBCProvider(jdbcProviderName, description, database_selection)

	datasourceName = 'Worklight database' + space_suffix
	description = 'IBM Worklight database'
	database_selection = worklight_database_selection
	jndiName = worklight_database_jndi_name
	database_name = worklight_database_name
	database_jdbc_url = worklight_database_jdbc_url
	database_derby_datadir = worklight_database_derby_datadir
	database_db2_host = worklight_database_db2_host
	database_db2_port = worklight_database_db2_port
	database_user_name = worklight_database_user_name
	database_user_password = worklight_database_user_password
	database_schema = worklight_database_schema
	database_properties_py = worklight_database_properties_py
	ensureDatabase(datasourceName, description, database_selection, jndiName, database_name, database_jdbc_url, database_derby_datadir, database_db2_host, database_db2_port, database_user_name, database_user_password, database_schema, database_properties_py, authenticationAliasForWorklight)

	jdbcProviderName = 'Worklight reports JDBC Provider' + space_suffix
	description = 'JDBC provider for the IBM Worklight reports database'
	database_selection = worklightreports_database_selection
	jdbcProviderObject = ensureJDBCProvider(jdbcProviderName, description, database_selection)

	datasourceName = 'Worklight reports database' + space_suffix
	description = 'IBM Worklight reports database'
	database_selection = worklightreports_database_selection
	jndiName = worklightreports_database_jndi_name
	database_name = worklightreports_database_name
	database_jdbc_url = worklightreports_database_jdbc_url
	database_derby_datadir = worklightreports_database_derby_datadir
	database_db2_host = worklightreports_database_db2_host
	database_db2_port = worklightreports_database_db2_port
	database_user_name = worklightreports_database_user_name
	database_user_password = worklightreports_database_user_password
	database_schema = worklightreports_database_schema
	database_properties_py = worklightreports_database_properties_py
	ensureDatabase(datasourceName, description, database_selection, jndiName, database_name, database_jdbc_url, database_derby_datadir, database_db2_host, database_db2_port, database_user_name, database_user_password, database_schema, database_properties_py, authenticationAliasForWorklightReports)

	# Avoid using spaces in a shared library name, because it leads to problems
	# after a WAS "Update" application operation is performed (RTC 19834).
	sharedLibraryName = 'Worklight_Platform_Library' + unique_suffix
	description = 'IBM Worklight Server Platform Library'
	classpath = '${USER_INSTALL_ROOT}/' + appserver_resourcesdir + '/' + worklight_lib_basename
	ensureSharedLibrary(sharedLibraryName, description, classpath)

	# Needed as a workaround to defect 10294.
	# See http://www-01.ibm.com/support/docview.wss?uid=swg1PM50111
	ensureWebContainerCustomProperty('com.ibm.ws.webcontainer.invokeFlushAfterService', 'false', 'See http://www-01.ibm.com/support/docview.wss?uid=swg1PM50111')


	# Save the changes done so far.
	AdminConfig.save()


	# ================= Install Worklight console web application =================

	if appserver_was_scope[0:3] != 'nd-' and database_selection == 'db2' and 'IBM_Worklight_Console' in AdminUtilities.convertToList(AdminApp.list()):
		# Erase an application that was left over from an unsuccessful uninstallation
		# of Worklight Server 5.0.5 (defect 14739).
		stopApplication('worklight.war', 'IBM_Worklight_Console')
		uninstallApplication('worklight.war', 'IBM_Worklight_Console')

	environmentEntries = []
	# Later entries in environmentEntries override the earlier ones.
	# So put the default values first.
	environmentEntries.append(constructJNDIEnvironmentEntry(worklight_war_basename, worklight_war_jndi_prop1_name, worklight_war_jndi_prop1_value))
	environmentEntries.append(constructJNDIEnvironmentEntry(worklight_war_basename, worklight_war_jndi_prop2_name, worklight_war_jndi_prop2_value))
	# The file worklight_war_jndi_py contains augmentation statements for environmentEntries.
	# It references the constructJNDIEnvironmentEntry function.
	execfile(worklight_war_jndi_py)
	installApplication(
		worklight_war, worklight_war_basename, worklight_war_context,
		worklight_war_appname,
		[
			'-MapEnvEntryForWebMod', convertListListToTclSyntax(environmentEntries),
			'-MapResRefToEJB',
			[
				constructJNDIDatasourceMapping(worklight_war_basename, 'jdbc/WorklightDS', worklight_database_jndi_name),
				constructJNDIDatasourceMapping(worklight_war_basename, 'jdbc/WorklightReportsDS', worklightreports_database_jndi_name)
			],
			'-MapSharedLibForMod', [[worklight_war_appname, 'META-INF/application.xml', sharedLibraryName]]
		])

	# -------- Set classloader mode for Worklight console web application --------

	ensureClassLoaderModeIsParentLast(worklight_war_appname)


	# Save the changes done so far.
	AdminConfig.save()


	# ======================== Start the web applications ========================

	# Does not work in WAS ND:
	# It leads to "com.ibm.ws.exception.ConfigurationWarning: Application ... not installed"
	# error during startApplication. A fullSync() and a time.sleep(10) don't help.
	# TODO Need to restart the application servers? Or play with isAppReady and getDeployStatus?
	# http://pic.dhe.ibm.com/infocenter/wasinfo/v7r0/index.jsp?topic=%2Fcom.ibm.websphere.nd.doc%2Finfo%2Fae%2Fae%2Ftxml_callappinstall.html
	# Consider also
	# http://www-01.ibm.com/support/docview.wss?uid=swg21572889
	# http://www-01.ibm.com/support/docview.wss?uid=swg21161426
	if appserver_was_scope[0:3] != 'nd-':
		startApplication(worklight_war_basename, worklight_war_appname)


	# ==================================== Done ===================================


if operation == 'uninstall':


	# ======================== Stop the web applications ========================

	stopApplication(worklight_war_basename, worklight_war_appname)


	# ====================== Uninstall the web applications ======================

	uninstallApplication(worklight_war_basename, worklight_war_appname)


	# ========================== Delete the datasources ==========================

	jdbcProviderName = 'Worklight JDBC Provider' + space_suffix
	jdbcProviderObject = AdminConfig.getid(scopeContainmentPath + 'JDBCProvider:' + jdbcProviderName + '/')
	if jdbcProviderObject != '':
		datasourceName = 'Worklight database' + space_suffix
		deleteDatasource(jdbcProviderName, jdbcProviderObject, datasourceName)

	jdbcProviderName = 'Worklight reports JDBC Provider' + space_suffix
	jdbcProviderObject = AdminConfig.getid(scopeContainmentPath + 'JDBCProvider:' + jdbcProviderName + '/')
	if jdbcProviderObject != '':
		datasourceName = 'Worklight reports database' + space_suffix
		deleteDatasource(jdbcProviderName, jdbcProviderObject, datasourceName)


	# ========================= Delete the JDBC providers =========================

	jdbcProviderName = 'Worklight JDBC Provider' + space_suffix
	deleteJDBCProvider(jdbcProviderName)

	jdbcProviderName = 'Worklight reports JDBC Provider' + space_suffix
	deleteJDBCProvider(jdbcProviderName)


	# ======================= Delete authentication aliases =======================

	print 'was-install.py delete authentication alias...'
	if authenticationAliasForWorklight != '':
		deleteAuthAlias(authenticationAliasForWorklight)
	if authenticationAliasForWorklightReports != '':
		deleteAuthAlias(authenticationAliasForWorklightReports)
	print 'was-install.py delete authentication alias done'


	# ========================== Delete shared libraries ==========================

	sharedLibraryName = 'Worklight_Platform_Library' + unique_suffix
	print 'was-install.py delete shared library...'
	sharedLibraryObject = AdminConfig.getid(scopeContainmentPath + 'Library:' + sharedLibraryName + '/')
	if sharedLibraryObject != '':
		AdminConfig.remove(sharedLibraryObject)
	print 'was-install.py delete shared library done'


	# ==================================== Done ===================================


# Finally save all the changes.
AdminConfig.save()

if appserver_was_scope[0:3] == 'nd-':
	# Execute a "full synchronize" of the configuration to all nodes.
	fullSync()

