/**
* Licensed Materials - Property of IBM* and/or HCL**
* IBM UrbanCode Deploy
* (c) Copyright IBM Corporation 2011, 2017. All Rights Reserved.
* (c) Copyright HCL Technologies Ltd. 2018. All Rights Reserved.
*
* U.S. Government Users Restricted Rights - Use, duplication or disclosure restricted by
* GSA ADP Schedule Contract with IBM Corp.
*
* * Trademark of International Business Machines
* ** Trademark of HCL Technologies Limited
*/

import com.urbancode.air.AirPluginTool
import com.urbancode.air.CommandHelper
import com.urbancode.air.plugin.datastage.DataStageHelper;
import com.urbancode.air.plugin.datastage.TextBoxParser;

def apTool = new AirPluginTool(this.args[0], this.args[1])
def props = apTool.getStepProperties()

DataStageHelper helper = new DataStageHelper()
TextBoxParser tBox = new TextBoxParser()

// istool Export Properties
def istoolPath = props['istoolPath'].trim()
    // Authentication Parameters
def domain      = props['domain'].trim()
def authfile    = props['authfile'].trim()
def username    = props['username'].trim()
def password    = props['password']
    // Generic Parameters
def help        = Boolean.parseBoolean(props['help'])
def verbose     = Boolean.parseBoolean(props['verbose'])
def silent      = Boolean.parseBoolean(props['silent'])
def script      = props['script'].trim()
    // Common Parameters
def preview      = Boolean.parseBoolean(props['preview'])
def responseFile = props['responseFile'].trim()
def abortIfError = props['abortIfError']
    // Export Parameters
def updatearchive     = Boolean.parseBoolean(props['updatearchive'])
def server            = props['server'].trim()
def projects          = tBox.readFileorTextString(props['projects'].trim())
def includedependent  = Boolean.parseBoolean(props['includedependent'])
def includeexecutable = Boolean.parseBoolean(props['includeexecutable'])
def nodesign          = Boolean.parseBoolean(props['nodesign'])
if (nodesign && !includeexecutable) {
    println "[OK] Setting -includeexecutable flag as true because it must accompany -nodesign."
    includeexecutable = true
}
// To-do: '-base' parameter that is currently ignored

def workDir = new File(".")
def ch = new CommandHelper(workDir)
ch.ignoreExitValue = true

final def isWindows = System.getProperty('os.name').contains("Windows")

// Construct and find the location of the DataStage script
def istoolScript = isWindows ? "istool.bat" : "istool.sh"
def fullScriptPath = istoolPath + File.separator + istoolScript
File istoolFile = new File(fullScriptPath)
if (!istoolFile.isFile()) {
    throw new Exception ("ERROR ${fullScriptPath} does not exist!")
}

// If given, confirm authfile does exist
if (authfile) {
    File authfileFile = new File(authfile)
    if (!authfileFile.isFile()) {
        throw new FileNotFoundException ("ERROR ${authfile} does not exist!")
    }
}
else {
    if (!username && !password) {
        throw new NullPointerException ("ERROR Username and password are required fields without an authorization file.")
    }
}

// If given, confirm responseFile does exist
if (responseFile) {
    File responsefileFile = new File(responseFile)
    if (!responsefileFile.isFile()) {
        throw new FileNotFoundException ("ERROR ${responseFile} does not exist!")
    }
    if (preview) {
        throw new Exception ("ERROR Both a Response File and Preview properties may not be set. Please update one.")
    }
}

// If given, confirm script does exist
if (script) {
    File scriptFile = new File(script)
    if (!scriptFile.isFile()) {
        throw new FileNotFoundException ("ERROR ${script} does not exist!")
    }
}

// If project starts with a '/', remove it (prevents hidden archive file)
projects.each{ proj ->
    if (proj.startsWith("/")){
        proj = proj.replaceFirst("/", "")
    }
}

// Begin constructing the istool export script
def exitCodes = []
for (project in projects) {
    def args = []
    args << fullScriptPath
    args << "export"

    // Add domain
    if (domain) {
        args << "-domain"
        args << domain
    }
    // Add authorization
    if (authfile) {
        args << "-authfile"
        args << authfile
    }
    else {
        args << "-username"
        args << username
        args << "-password"
        args << password
    }

    // All Generic Parameters
    if (help) {
        args << "-help"
    }
    if (verbose) {
        args << "-verbose"
    }
    if (silent) {
        args << "-silent"
    }
    if (script) {
        args << "-script"
        args << script
    }

    // Create archive name based on the project name
    args << "-archive"
    File archive = File.createTempFile("archive", ".isx", workDir)
    args << archive.getCanonicalPath()
    apTool.setOutputProperty(project, archive.getCanonicalPath());

    // Optional: Add preview arg
    if (preview) {
        args << "-preview"
    }

    // Optional: Response File
    // Exclusive to Preview arg, checked above
    if (responseFile) {
        args << "-responseFile"
        args << responseFile
    }

    // Optional: Abort after X number of failures
    if (abortIfError > 0) {
        args << "-abortIfError"
        args << abortIfError
    }

    // Define the projec to import to...
    args << "-datastage"

    // True == only job executable are imported
    // Complete dsArg result: '-nodesign -includedependent -includeexecutable "server/project"'
    String dsArg = ""
    def dsList = []

    dsList << "'"
    if (nodesign) {
        dsArg  += "-nodesign "
        dsList << "-nodesign"
    }
    if (includedependent) {
        dsArg  += "-includedependent "
        dsList << "-includedependent"
    }
    if (includeexecutable) {
        dsArg  += "-includeexecutable "
        dsList << "-includeexecutable"
    }
    dsArg  += "\"${server}/${project}\""
    dsList << "\"${server}/${project}\""
    dsList << "'"

    // Create Windows specific command
    def argsWin = args + dsList

    // Create Unix specific command
    args << dsArg

    Map<String, Object> input = new HashMap<String, Object>()
    input.put("printCommand", args)

    // Windows requires all values (including single quotes) to be separate arguments.
    // This is different than what the documentation suggests.
    if (apTool.isWindows) {
    	input.put("command", argsWin)
    } else {
    	input.put("command", args)
    }

    // Run Command!
    println "Running istool export command..."
    def exitCode = ch.runCommand(input)
    exitCodes << exitCode
    if (exitCode == 0) {
        println "[OK] Successfully exported project: '${project}'."
    } else {
        println "[Error] Exit code of '${exitCode}' received when exporting project: '${project}'."
    }

    println "==========================================================================================="

}
apTool.storeOutputProperties();

// If any of the exit codes return a non-zero number, then some objects failed to import -> fail
for (codes in exitCodes) {
    if (codes != 0) {
        System.exit(1)
    }
}
