/**
* Licensed Materials - Property of IBM* and/or HCL**
* IBM UrbanCode Deploy
* (c) Copyright IBM Corporation 2011, 2017. All Rights Reserved.
* (c) Copyright HCL Technologies Ltd. 2018. All Rights Reserved.
*
* U.S. Government Users Restricted Rights - Use, duplication or disclosure restricted by
* GSA ADP Schedule Contract with IBM Corp.
*
* * Trademark of International Business Machines
* ** Trademark of HCL Technologies Limited
*/

import com.urbancode.air.AirPluginTool
import com.urbancode.air.CommandHelper
import com.urbancode.air.plugin.datastage.DataStageHelper;
import com.urbancode.air.plugin.datastage.TextBoxParser;

def apTool = new AirPluginTool(this.args[0], this.args[1])
def props = apTool.getStepProperties()

DataStageHelper helper = new DataStageHelper()
TextBoxParser tBox = new TextBoxParser()

// istool import Properties
def istoolPath = props['istoolPath'].trim()
    // Authentication Parameters
def domain      = props['domain'].trim()
def authfile    = props['authfile'].trim()
def username    = props['username'].trim()
def password    = props['password']
    // Generic Parameters
def help        = Boolean.parseBoolean(props['help'])
def verbose     = Boolean.parseBoolean(props['verbose'])
def silent      = Boolean.parseBoolean(props['silent'])
def script      = props['script'].trim()
    // Common Parameters
def prevOrRep    = props['prevOrRep']
def responseFile = props['responseFile'].trim()
def abortAfter   = props['abortAfter']
    // Import Parameters
def archives    = tBox.readFileorTextString(props['archives'].trim())
def server      = props['server'].trim()
def project     = props['project'].trim()
def nodesign    = Boolean.parseBoolean(props['nodesign'])

def workDir = new File(".")
def ch = new CommandHelper(workDir)
ch.ignoreExitValue = true

final def isWindows = System.getProperty('os.name').contains("Windows")

// Check that all files end with a DSX extension
helper.checkISXExtensions(archives)

// Construct and find the location of the DataStage script
def istoolScript = isWindows ? "istool.bat" : "istool.sh"
def fullScriptPath = istoolPath + File.separator + istoolScript
File istoolFile = new File(fullScriptPath)
if (!istoolFile.isFile()) {
    throw new Exception ("ERROR ${fullScriptPath} does not exist!")
}

// If given, confirm authfile does exist
if (authfile) {
    File authfileFile = new File(authfile)
    if (!authfileFile.isFile()) {
        throw new FileNotFoundException ("ERROR ${authfile} does not exist!")
    }
}
else {
    if (!username && !password) {
        throw new NullPointerException ("ERROR Username and password are required fields without an authorization file.")
    }
}

// If given, confirm responseFile does exist
if (responseFile) {
    File responsefileFile = new File(responseFile)
    if (!responsefileFile.isFile()) {
        throw new FileNotFoundException ("ERROR ${responseFile} does not exist!")
    }
    if (prevOrRep == "PREVIEW") {
        throw new Exception ("ERROR Both a Response File and Preview properties may not be set. Please update one.")
    }
}

// If given, confirm script does exist
if (script) {
    File scriptFile = new File(script)
    if (!scriptFile.isFile()) {
        throw new FileNotFoundException ("ERROR ${script} does not exist!")
    }
}

// Confirm all archives exist and end with .isx, else fail
helper.checkISXExtensions(archives)
def exitCodes = []
for (archive in archives) {
    def args = []
    args << fullScriptPath
    args << "import"

    // Add domain
    if (domain) {
        args << "-domain"
        args << domain
    }
    // Add authorization
    if (authfile) {
        args << "-authfile"
        args << authfile
    }
    else {
        args << "-username"
        args << username
        args << "-password"
        args << password
    }

    // All Generic Parameters
    if (help) {
        args << "-help"
    }
    if (verbose) {
        args << "-verbose"
    }
    if (silent) {
        args << "-silent"
    }
    if (script) {
        args << "-script"
        args << script
    }

    // Add archive
    args << "-archive"
    args << archive

    // Optional: Exclusive Preview or Replace args
    if (prevOrRep == "PREVIEW") {
        args << "-preview"
    }
    else if (prevOrRep == "REPLACE")  {
        args << "-replace"
    }

    // Optional: Response File
    // Exclusive to Preview arg, checked above
    if (responseFile) {
        args << "-responseFile"
        args << responseFile
    }

    // Optional: Abort after X number of failures
    if (abortAfter > 0) {
        args << "-abortAfter"
        args << abortAfter
    }

    // Define the projec to import to...
    args << "-datastage"

    String dsArg = ""
    def dsList = []
    dsList << "'"

    // True == only job executable are imported
    if (nodesign) {
        dsArg  += "-nodesign "
        dsList << "-nodesign"
    }
    dsArg  += "\"${server}/${project}\""
    dsList << "\"${server}/${project}\""
    dsList << "'"

    // Create Windows specific command
    def argsWin = args + dsList

    // Create Unix specific command
    args << dsArg

    Map<String, Object> input = new HashMap<String, Object>()
    input.put("printCommand", args)

    // Windows requires all values (including single quotes) to be separate arguments.
    // This is different than what the documentation suggests.
    if (apTool.isWindows) {
    	input.put("command", argsWin)
    } else {
    	input.put("command", args)
    }

    // Run Command!
    println "Running istool import command..."
    def exitCode = ch.runCommand(input)
    exitCodes << exitCode
    if (exitCode == 0) {
        println "[OK] Successfully imported archive: '${archive}'."
    } else {
        println "[Error] Exit code of '${exitCode}' received when importing archive: '${archive}'."
    }

    println "==========================================================================================="

}

// If any of the exit codes return a non-zero number, then some objects failed to import -> fail
for (codes in exitCodes) {
    if (codes != 0) {
        System.exit(1)
    }
}
