/**
* Licensed Materials - Property of IBM* and/or HCL**
* IBM UrbanCode Deploy
* (c) Copyright IBM Corporation 2011, 2017. All Rights Reserved.
* (c) Copyright HCL Technologies Ltd. 2018. All Rights Reserved.
*
* U.S. Government Users Restricted Rights - Use, duplication or disclosure restricted by
* GSA ADP Schedule Contract with IBM Corp.
*
* * Trademark of International Business Machines
* ** Trademark of HCL Technologies Limited
*/

import com.urbancode.air.AirPluginTool
import com.urbancode.air.CommandHelper
import com.urbancode.air.plugin.datastage.DataStageHelper;
import com.urbancode.air.plugin.datastage.TextBoxParser;

final LS = System.getProperty('line.separator')

def apTool = new AirPluginTool(this.args[0], this.args[1])

def props = apTool.getStepProperties()
DataStageHelper helper = new DataStageHelper()
TextBoxParser tBox = new TextBoxParser()

 // dscmdimport Properties
def authfile    = props['authfile']
def domain      = props['domain']
def dsScriptPath = props['dsScriptPath']
def hostname    = props['hostname']
def createLogfile = Boolean.parseBoolean(props['createLogfile'])
def username    = props['username']
def password    = props['password']
def project     = props['project']
def pathnames   = props['pathnames']
def nua         = Boolean.parseBoolean(props['nua'])
def noenv       = Boolean.parseBoolean(props['noenv'])
def verbose     = Boolean.parseBoolean(props['verbose'])

def workDir = new File(".")
def ch = new CommandHelper(workDir)

// Create a log file if that setting was enabled
def logfile
if (createLogfile) {
    logfile = new File('import-' + new Date().format("yyyyMMdd-HHmmss") + '.txt').canonicalFile
    apTool.setOutputProperty('logfile', logfile.name)

    // Fail if we can't create the file
    try {
        logfile.createNewFile()
    } catch (IOException e) {
        println "Cannot create output log file \"${logfile}\": " + e.getMessage()
        System.exit(1)
    }

    // Fail if the file isn't writable
    if (!logfile.canWrite()) {
        println "The specified output log file \"${logfile}\" is not writable."
        System.exit(1)
    }
}

final def isWindows = System.getProperty('os.name').contains("Windows")

// Construct and find the location of the DataStage script
def dsScript = isWindows ? "dscmdimport.exe" : "dscmdimport.sh"
def fullScriptPath = dsScriptPath + File.separator + dsScript
File scriptFile = new File(fullScriptPath)
if (!scriptFile.isFile()) {
    throw new Exception ("${fullScriptPath} does not exist!")
}

def args = [fullScriptPath]

if (authfile) {
    File authfileFile = new File(authfile)
    if (!authfileFile.isFile()) {
        throw new Exception ("${authfile} does not exist!")
    }
    args << "/AF=${authfile}"
}
if (domain) {
    args << "/D=${domain}"
}
if (hostname) {
    args << "/H=${hostname}"
}
if (username) {
    args << "/U=${username}"
}
if (password) {
    args << "/P=${password}"
}

// Used to determine manually whether to overwrite or ignore conflicting files. To-Do: Implement response capabilities
// Until functionality is added, always supress usage analysis
args << "/NUA"

if (noenv) {
    args << "/NOENV"
}
if (project) {
    args << "${project}"
}
if (pathnames) {
    def pathnamesArray = tBox.readFileorTextString(pathnames)
    helper.checkExtensions(pathnamesArray)
    for (path in pathnamesArray) {
        File pathDest = new File(path)
        if (!pathDest.isFile()) {
            throw new Exception ("${path} does not exist!")
        }
        args << path
    }
}
if (verbose) {
    args << "/V"
}

// RUN COMMAND
def ignoredFiles = ch.runCommand("Running dscmdimport command...", args, helper.getSystemOutput)
apTool.setOutputProperty("${project}.ignoredFiles", (String)ignoredFiles)

// Print system output
def output = helper.systemOutput
println "DataStage System output..."
println output

// Write output to log file if one was specified
if (logfile) {
    if (logfile.canWrite()) {
        println "Writing DataStage output to \"${logfile}\"."
        logfile.append("Import of project \"${project}\":" + LS)
        logfile.append(output)
        logfile.append(LS)
    } else {
        println "The specified output log file \"${logfile}\" is not writable."
        System.exit(1)
    }
}

//////// Parse output and return sensible output ////////

def exitCode
// Import failed before it began
if (output.contains("Failed to attach to the project")){
    println "Failed to reach DataStage Server with the given properties. Please confirm:"
    println "\t - Domain:Port \t - Username"
    println "\t - Password \t - Project"
    println "\t - Agent has admin privileges"
    println "\t - InfoSphere Server is up"
    exitCode = 1
}
// Failed during the .dsx import
else if (output.contains("dscmdimport failed")) {
    println "Import failed in the middle of the .dsx file import process."
    println "This is likely an error with the chosen DataStage .dsx file."
    exitCode = 1
}
// Import succeeded
else if (output.contains("DSImport ended at:")) {
    exitCode = 0
    if (output.contains("Warning:")){
        println "Import is successful with a warning!"
    }
    else {
        println "Import is successful!"
    }
}
// Catch all
else {
    println "Unknown dscmdimport failure. Please review the error file for a full report."
    exitCode = 1
}

apTool.setOutputProperties()
// Describe where additional error output can be found
if (exitCode) {
    print "Detailed error output created by the DataStage software can be found here: "
    if (isWindows) {
        println "'C:\\Users\\<USER_NAME>\\ds_logs\\*'"
    }
    else {
        println "'/home/<USER_NAME>/ds_logs/*'"
    }
    System.exit(exitCode)
}
