/**
* Licensed Materials - Property of IBM* and/or HCL**
* IBM UrbanCode Deploy
* (c) Copyright IBM Corporation 2011, 2017. All Rights Reserved.
* (c) Copyright HCL Technologies Ltd. 2018. All Rights Reserved.
*
* U.S. Government Users Restricted Rights - Use, duplication or disclosure restricted by
* GSA ADP Schedule Contract with IBM Corp.
*
* * Trademark of International Business Machines
* ** Trademark of HCL Technologies Limited
*/

import com.urbancode.air.AirPluginTool
import com.urbancode.air.CommandHelper
import com.urbancode.air.plugin.datastage.DataStageHelper;
import com.urbancode.air.plugin.datastage.TextBoxParser;

def apTool = new AirPluginTool(this.args[0], this.args[1])
def props = apTool.getStepProperties()

DataStageHelper helper = new DataStageHelper()
TextBoxParser tBox = new TextBoxParser()

// dscmdimportservice Properties
def dsScriptPath = props['dsScriptPath']
def ISHost      = props['ISHost']
def ISUser      = props['ISUser']
def ISPassword  = props['ISPassword']
def DSHost      = props['DSHost']
def DSProject   = props['DSProject']
def DSXFiles    = tBox.readFileorTextString(props['DSXFiles'])
def owChoice    = props['owChoice']
def StopOnError = Boolean.parseBoolean(props['StopOnError'])
def Verbose     = Boolean.parseBoolean(props['Verbose'])
def additionalArgs    = props['additionalArgs']
def ISAuthFile        = props['ISAuthFile']
def ISCredentialsFile = props['ISCredentialsFile']

def workDir = new File(".")
def ch = new CommandHelper(workDir)
ch.ignoreExitValue = true

final def isWindows = System.getProperty('os.name').contains("Windows")

// Check that all files end with a DSX extension
helper.checkDSXExtensions(DSXFiles)

// Construct and find the location of the DataStage script
def dsScript = isWindows ? "DSXImportService.bat" : "DSXImportService.sh"
def fullScriptPath = dsScriptPath + File.separator + dsScript
File scriptFile = new File(fullScriptPath)
if (!scriptFile.isFile()) {
    throw new Exception ("${fullScriptPath} does not exist!")
}
// If given, confirm ISAuthFile does exist
if (ISAuthFile) {
    File authfileFile = new File(ISAuthFile)
    if (!authfileFile.isFile()) {
        throw new FileNotFoundException ("${ISAuthFile} does not exist!")
    }
}
// If given, confirm ISCredentialsFile does exist
if (ISCredentialsFile) {
    File credentialsFile = new File(ISCredentialsFile)
    if (!credentialsFile.isFile()) {
        throw new FileNotFoundException ("${ISCredentialsFile} does not exist!")
    }
}

// Construct the DSXImportService command for each project and file pair
println "Project: ${DSProject}"
println "Files: ${DSXFiles}"
def ignoredFiles = [] // Tracks how many files during import were ignored
def failedDSXFiles = []  // Tracks which .dsx files were not found
for (DSXFile in DSXFiles) {
    File dsxFile = new File(DSXFile)
    int exitCode = -1
    println "[Action] Importing ${DSXFile}..."
    try {
        def args = [fullScriptPath]

        if (ISAuthFile) {
            args << "-ISAuthFile"
            args << ISAuthFile
        }
        if (ISCredentialsFile) {
            args << "-ISFile"
            args << ISCredentialsFile
        }
        if (ISHost) {
            args << "-ISHost"
            args << ISHost
        }
        if (ISUser) {
            args << "-ISUser"
            args << ISUser
        }
        if (ISPassword) {
            args << "-ISPassword"
            args << ISPassword
        }
        if (DSHost) {
            args << "-DSHost"
            args << DSHost
        }

        args << "-DSProject"
        args << DSProject

        if (!dsxFile.isFile()) {
            exitCode = 1
            throw new FileNotFoundException ("${DSXFile} does not exist! Ensure a path to a single file is specified.")
        }
        args << "-DSXFile"
        args << DSXFile

        if (owChoice == 'Overwrite') {
            args << "-Overwrite"
        }
        else if (owChoice == 'OverwriteReadOnly'){
            args << "-OverwriteReadOnly"
        }
        if (StopOnError) {
            args << "-StopOnError"
        }
        if (Verbose) {
            args << "-Verbose"
        }
        if (additionalArgs) {
            args << "${additionalArgs}"
        }

        // Run Command!
        exitCode = ch.runCommand("Running DSXImportService command...", args)

        println "[Success] ${DSXFile} imported successfully!"

    }
    catch (FileNotFoundException ex) {
        println "[Error] File does not exist! Skipping Project and File pair => Project:${DSProject}, File:${DSXFile}"
        failedDSXFiles << DSXFile
        if (StopOnError) {
            println "[Error] Stop on Error selected. Exiting because DSXFile was not found."
            System.exit(1)
        }
    }
    catch (Exception ex) {
        ex.printStackTrace()
        throw new Exception ("[Error] Failure during DSXImportService command construction or at runtime - ${ex.getMessage()}")
    }
    finally {
        // The exitCode equals the number of files not imported
        ignoredFiles << exitCode

        // Get only the file name, ignore extra path folders
        apTool.setOutputProperty("${DSProject}.${dsxFile.getName()}", (String)exitCode)
    }
    println "==========================================================================================="
}

apTool.setOutputProperties()

// Print out files that failed to import
if (failedDSXFiles.size() > 0){
    println "The following files failed to import: ${failedDSXFiles}"
}

// exitCode of -1 is a true failure
if (ignoredFiles.contains(-1)) {
    System.exit(1)
}
