/*
* Licensed Materials - Property of IBM Corp.
* IBM UrbanCode Build
* IBM UrbanCode Deploy
* IBM UrbanCode Release
* IBM AnthillPro
* (c) Copyright IBM Corporation 2002, 2016. All Rights Reserved.
*
* U.S. Government Users Restricted Rights - Use, duplication or disclosure restricted by
* GSA ADP Schedule Contract with IBM Corp.
*/
import java.io.File

import com.urbancode.air.XTrustProvider
import com.urbancode.air.plugin.docker.DockerUtils
import com.urbancode.air.AirPluginTool
import com.urbancode.air.CommandHelper
import com.urbancode.ud.client.ResourceClient
import com.urbancode.ud.client.ComponentClient
import com.urbancode.ud.client.ApplicationClient
import com.urbancode.ud.client.EnvironmentClient

XTrustProvider.install()
int MAX_RETRIES = 3
def COMPONENT_TAG='compose.service'

// Get step properties
def airTool = new AirPluginTool(this.args[0], this.args[1])
def props = airTool.getStepProperties()
String serverUrl                 = props['serverUrl']
String username                  = props['username']
String password                  = props['password']
String resourceId                = props['resource']
String applicationId             = props['application']
String componentProcessRequestId = props['componentProcessRequest']
String composeProjectName        = props['composeProjectName']
String composeOptions            = props['composeOptions']
String environmentId             = props['environment']
String propertyPrefix            = props['propertyPrefix']
String scriptPathsRaw            = props['scriptPaths']
String composeFilesRaw           = props['composeFiles']
String envPropValues             = props['envPropValues']

// Sanitize properties
def serverUri = DockerUtils.stringToUri(serverUrl)
if (serverUri.equals(new URI(''))) {
    System.exit(1)
}
if (!username) {
    username = airTool.getAuthTokenUsername()
}
if (!password) {
    password = airTool.getAuthToken()
}
def scriptPaths = DockerUtils.toTrimmedList(scriptPathsRaw, ',')
def composeFilePaths = DockerUtils.toTrimmedList(composeFilesRaw, '\n')
if (!composeFilePaths) {
    composeFilePaths << 'docker-compose.yml'
}

def versionOverridesFile = new File('ucd-version-overrides.yml')
if (versionOverridesFile.exists() && versionOverridesFile.text) {
    composeFilePaths << versionOverridesFile.path
}

def ch = new CommandHelper(new File('.'))

ComponentClient componentClient = new ComponentClient(serverUri, username, password)
ApplicationClient applicationClient = new ApplicationClient(serverUri, username, password)
EnvironmentClient environmentClient = new EnvironmentClient(serverUri, username, password)
ResourceClient resourceClient = new ResourceClient(serverUri,  username, password)

String deploymentRequestId = componentClient.getComponentProcessRequest(componentProcessRequestId)?.deploymentRequestId
def components = DockerUtils.parseTextAsJson(applicationClient.getApplicationComponents(applicationId))

def desiredVersionMap = [:]
def services = [:]
def componentMap = [:]

// Get services
components.each { component ->
    componentMap.put(component.name, component)
    def desiredInventoryEntry = DockerUtils.parseTextAsJson(environmentClient.getLatestEnvironmentInventoryByComponent(environmentId, applicationId, component.name.replace(' ', '%20')))
    if (desiredInventoryEntry) {
        desiredVersionMap.put(component.name, desiredInventoryEntry.version.name)
    }
    if (component.tags.find{ COMPONENT_TAG.equals(it.name) }) {
        def componentProps = componentClient.getComponentProperties(component.name)
        component.image = componentProps['docker.image.name']

        def version = desiredVersionMap.get(component.name) //Needs to be replaced with property for service name
        String imageTag = "${component.image}:${version}"
        services.put(component.name, ['image': imageTag])
    }
}

// compose up arguments
def composeUpArgs = ['docker-compose']
if (composeOptions) {
    composeUpArgs << composeOptions
}
if (composeProjectName) {
    composeUpArgs << '-p'
    composeUpArgs << composeProjectName
}
composeFilePaths.each { path ->
    composeUpArgs << '-f'
    composeUpArgs << path
}
composeUpArgs << 'up'
composeUpArgs << '-d'

// Environtment properties
def envVars = [:]
if(envPropValues) {
   if (propertyPrefix) {
        println "Looking for properties starting with ${propertyPrefix}"
    }
    //this is jeffs magic regex to split on ,'s preceded by even # of \ including 0
    envPropValues.split('(?<=(^|[^\\\\])(\\\\{2}){0,8}),').each { prop ->
        //split out the name
        def parts = prop.split('(?<=(^|[^\\\\])(\\\\{2}){0,8})=',2);
        def propName = parts[0];
        def propValue = parts.size() == 2 ? parts[1] : '';
        //replace \, with just , and then \\ with \
        propName = propName.replace('\\=', '=').replace('\\,', ',').replace('\\\\', '\\')
        propValue = propValue.replace('\\=', '=').replace('\\,', ',').replace('\\\\', '\\')

        if ((!propertyPrefix || propName.startsWith(propertyPrefix))) {
            envVars.put(propName, propValue)
        }
    }
}

// command helper arguments
def args = []
final def isWindows = System.getProperty('os.name').contains('Windows')
if (isWindows) {
    args << 'cmd'
    args << '/C'
}
else {
    args << '/bin/bash'
    args << '-c'
}

// Environment files
scriptPaths.each { filePath ->
    def envScript = new File(filePath)
    if (envScript.exists()) {
        envScript.setExecutable(true)
        args << ". ${filePath}\n"
    }
    else {
        throw new RuntimeException("Could not find env file at ${filePath}")
    }
}

// Run Compose
args << composeUpArgs.join(' ')
ch.runCommand("Executing Compose Up...", args)

//Update Actual (Resource) Inventory
def children = DockerUtils.parseTextAsJson(resourceClient.getResourceChildren(resourceId))
children.each { childResource ->
    def roles = DockerUtils.parseTextAsJson(resourceClient.getResourceRoles(childResource.id))
    roles.each { role ->
        //if child has role, update inventory
        def componentName = role.name
        def versions = DockerUtils.parseTextAsJson(componentClient.getComponentVersionsJsonArray(componentName, false))
        def versionId
        def resourceInventoryEntry = services.get(componentName)
        if (resourceInventoryEntry) {
            def versionName = desiredVersionMap.get(componentName)
            println "Updating inventory for ${componentName} - ${versionName}"
            versions.each {
                if (versionName.equals(it.name)) {
                    println "\t${it.name}"
                    versionId = it.id
                }
            }
            resourceClient.createResourceInventoryEntry(deploymentRequestId, childResource.id, componentMap.get(componentName).id, versionId, 'Active')
        }
    }
}
