/*
* Licensed Materials - Property of IBM Corp.
* IBM UrbanCode Build
* IBM UrbanCode Deploy
* IBM UrbanCode Release
* IBM AnthillPro
* (c) Copyright IBM Corporation 2002, 2016. All Rights Reserved.
*
* U.S. Government Users Restricted Rights - Use, duplication or disclosure restricted by
* GSA ADP Schedule Contract with IBM Corp.
*/
import org.codehaus.jettison.json.JSONArray
import org.codehaus.jettison.json.JSONObject
import org.yaml.snakeyaml.Yaml

import com.urbancode.air.AirPluginTool
import com.urbancode.air.XTrustProvider
import com.urbancode.air.plugin.docker.DockerUtils
import com.urbancode.ud.client.ApplicationClient
import com.urbancode.ud.client.ComponentClient
import com.urbancode.ud.client.EnvironmentClient
import com.urbancode.ud.client.ResourceClient
import com.urbancode.ud.client.VersionClient
import com.urbancode.ud.client.PropertyClient

XTrustProvider.install()
int MAX_RETRIES = 3
def COMPONENT_TAG='compose.service'

// Get step properties
def airTool = new AirPluginTool(this.args[0], this.args[1])
def props = airTool.getStepProperties()
String serverUrl                    = props['serverUrl']
String username                     = props['username']
String password                     = props['password']
String resourceId                   = props['resource']
String applicationId                = props['application']
String applicationProcessRequestId  = props['applicationProcessRequest']
String componentProcessRequestId    = props['componentProcessRequest']
String environmentId                = props['environment']
String composeComponentId           = props['component']
String composeFilesRaw              = props['composeFiles']
String sourceConfigType             = props['sourceConfigType']
String componentTemplate            = props['componentTemplate']

// Sanitize properties
def serverUri = DockerUtils.stringToUri(serverUrl)
if (serverUri.equals(new URI(''))) {
    System.exit(1)
}
if (!username) {
    username = airTool.getAuthTokenUsername()
}
if (!password) {
    password = airTool.getAuthToken()
}
def composeFilePaths = DockerUtils.toTrimmedList(composeFilesRaw, '\n')
if (!composeFilePaths) {
    composeFilePaths << 'docker-compose.yml'
}

def composeFiles = []
def envPropReferences = [] as Set
Yaml yaml = new Yaml()
ApplicationClient applicationClient = new ApplicationClient(serverUri, username, password)
ComponentClient componentClient = new ComponentClient(serverUri, username, password)
EnvironmentClient environmentClient = new EnvironmentClient(serverUri, username, password)
ResourceClient resourceClient = new ResourceClient(serverUri,  username, password)
VersionClient versionClient = new VersionClient(serverUri, username, password)
PropertyClient propertyClient = new PropertyClient(serverUri, username, password)

def applicationName = DockerUtils.parseTextAsJson(applicationClient.getApplication(applicationId)).name
String deploymentRequestId = componentClient.getComponentProcessRequest(componentProcessRequestId)?.deploymentRequestId

// Add all compose files and their env props
composeFilePaths.each { path ->
    def composeFile = new File(path)
    if (!composeFile.exists()) {
        throw new RuntimeException("Could not find compose file, ${path}")
    }
    composeFiles << composeFile
    envPropReferences.addAll(DockerUtils.getEnvReferences(composeFile.text))
}

// Environment property definitions
JSONArray propDefs = new JSONArray()
if (envPropReferences.size()) {
    println 'Creating environment property definitions for the following properties:'
    String propSheetDefPath = "components/${composeComponentId}/environmentPropSheetDef"
    envPropReferences.each { propReference ->
        println "\t${propReference}"
        JSONObject propDef = new JSONObject()
        propDef.put('name', propReference)
        propDef.put('label', propReference)
        propDef.put('type', 'TEXT')
        propDefs.put(propDef)
    }
    propertyClient.updatePropDefs(propSheetDefPath, propDefs, false)
}

// Get compose file version
def baseComposeFile = composeFiles[0]
println "Loading ${baseComposeFile}"
def composeYml = yaml.load(baseComposeFile.text)
def serviceList
def composeFileFormat
try {
    composeFileFormat = composeYml.version
    if ('2'.equals(composeFileFormat)) {
        println "Found compose file format version ${composeFileFormat}"
        serviceList = composeYml.services
    }
}
catch (MissingPropertyException e) {
    println 'Found compose file format version 1'
    serviceList = composeYml
}

// Build image specs, get versions
def manifest = [:]
def overrides = [:]
def userSelectedVersions = DockerUtils.parseTextAsJson(applicationClient.getApplicationProcessRequestVersions(applicationProcessRequestId))
userSelectedVersions.each{ version ->
    def versionProps = versionClient.getVersionProperties(version.name, version.component.name)
    if (versionProps.containsKey('dockerImageTag')) {
        manifest.put(version.component.name, [version.name]) //uDeployRestClient format
        Map <String, String> componentProps = componentClient.getComponentProperties(version.component.name)
        def imageName = componentProps.get('docker.image.name')
        def registry = componentProps.get('docker.registry.name')
        def serviceName = componentProps.get('compose.service')
        def tag = versionProps.get('dockerImageTag')

        if (imageName) {
            String imageSpec = DockerUtils.buildImageSpec(registry, imageName, tag)
            overrides.put(serviceName, ['image' : imageSpec]) //Docker compose format
        }
        else {
            println "Skipping override for ${version.component.name}"
        }
    }
}

// Create components
def existingResources = DockerUtils.parseTextAsJson(resourceClient.getResourceChildren(resourceId))
def applicationComponents = DockerUtils.parseTextAsJson(applicationClient.getApplicationComponents(applicationId))
serviceList.each{ service ->
    //create component for each service
    if (service.value.image) {
        String componentName = "${applicationName.replace(' ', '')}-${service.key}"

        def tagSpec = DockerUtils.parseImageSpec(service.value.image)

        def registry = tagSpec.registry
        def image = tagSpec.image
        def tag
        if (tagSpec.version) {
            tag = tagSpec.version
        }
        else {
            tag = 'latest'
        }

        def sourceProps = [:]
        if (registry) {
            sourceProps.put('DockerTagImport/dockerRegistryName', registry)
        }
        else {
            sourceProps.put('DockerTagImport/dockerRegistryName', '')
        }
        sourceProps.put('DockerTagImport/dockerImageName', image)

        def componentProps = new Properties()
        componentProps.setProperty('docker.image.name', image)
        if (registry) {
            componentProps.setProperty('docker.registry.name', registry)
        }
        else {
            componentProps.setProperty('docker.registry.name', '')
        }

        componentProps.setProperty('compose.service', service.key)

        String componentId = createComponent(componentClient, componentName, sourceProps, componentProps,
                                             sourceConfigType, componentTemplate).toString()

        //map components to this base resource
        //Currently assigns name componentName - if creating multiple instances of a container, should be unique

        if (!existingResources?.find{ componentName.equals(it.name) }) {
            println "Mapping compose service, ${componentName}"
            println(existingResources)
            resourceClient.createResource(componentName, null, null, resourceId, componentName)
        }

        if (applicationComponents && !applicationComponents.find{ resourceId.equals(it.id) }) {
            applicationClient.addComponentToApplication(applicationId, componentId)
        }

        componentClient.addTagToComponent(componentId, COMPONENT_TAG)

        def existingVersions = DockerUtils.parseTextAsJson(componentClient.getComponentVersionsJsonArray(componentName, false))


        def versionId

        existingVersions.each { def existingVersion ->
            if (tag.equals(existingVersion.name)) {
                versionId = existingVersion.id
            }
        }

        if (!versionId) {
            versionId = versionClient.createVersion(componentId, tag, "Compose service for image: ${image}")
        }

        versionClient.setVersionProperty(tag, componentId, 'dockerImageTag', tag, false)
        if (registry) {
            versionClient.setVersionProperty(tag, componentId, 'registry', registry, false)
        }
        else {
            versionClient.setVersionProperty(tag, componentId, 'registry', '', false)
        }

        if (!manifest.containsKey(componentName)){
            manifest.put(componentName, [tag])
            environmentClient.createDesiredInventoryEntry(deploymentRequestId, environmentId, componentId, versionId.toString(), 'Active')
        }
    }
    else {
        println "Skipping component creation for ${service.key} due to build specification"
    }
}

// Set invetory versions
if (userSelectedVersions) {
    println "Found the following user selected versions:\n${overrides}"
    def servicesYaml
    if ('1'.equals(composeFileFormat)) {
        servicesYaml = yaml.dump(overrides)
    }
    else {
        servicesYaml = yaml.dump(['version': '2', 'services': overrides])
    }

    def composeOverrideFile = new File('ucd-version-overrides.yml')
    if (composeOverrideFile.exists()) {
        composeOverrideFile.delete()
    }
    composeOverrideFile << servicesYaml
}
else {
    println 'Did not find any versions selected in UCD'
}

println ('Setting desired inventory to the following versions:')
overrides.each { println "\t[${it.key}:\t\t${it.value}]" }

/**
*  @param name The name of the componenet to create
*  @param sourceProps The list of source properties to set for the component
*  @param componentProps The list of properties to set for the componenet
*/
private def createComponent(def client, String name, def sourceProps, def componentProps, def configType, def template) {
    println "Creating component: ${name}"
    def result
    def image = sourceProps['DockerTagImport/dockerImageName']
    try {
        result = DockerUtils.parseTextAsJson(client.getComponent(name)).id
    }
    catch (IOException e){
        result = client.createComponent(
                name,
                "Compose service for image: ${image}",
                configType,
                'FULL',
                template,
                -1,
                true,
                false,
                sourceProps)
    }

    componentProps.each { componentProp ->
        client.setComponentProperty(result.toString(), componentProp.key, componentProp.value, false)
    }

    return result
}
